/******************************************************************************
 ******************************************************************************
 **
 ** COPYRIGHT (C) 1998 By Arthur Naseef
 **
 ** This file is covered under the GNU General Public License Version 2.  For
 **  more information, see the file COPYING.
 **
 **
 ** FILE: ovl_dir.c
 **
 ** DESCRIPTION: This file contains the source code for the directory
 **              operations of the overlay filesystem.
 **
 ** REVISION HISTORY:
 **
 ** DATE	AUTHOR		DESCRIPTION
 ** ==========	===============	==============================================
 ** 09/27/1997	art            	Added to source code control.
 ** 02/24/1998	ARTHUR N.	Added support for hiding magic directories and
 **				 added locking around call to lookup on the
 **				 storage filesystem's inode.
 ** 02/26/1998	ARTHUR N.	Added the blksize, blocks, version, and
 **				 nrpages to the inode information retained.
 ** 02/28/1998	ARTHUR N.	Fixed bug in read_into_dir() which would cause
 **				 directory entries to be missing if the readdir
 **				 operation of the real directory did not return
 **				 all entries in a single call.
 ** 03/01/1998	ARTHUR N.	Improved comments; added use of the term pseudo
 **				 filesystem.
 ** 03/01/1998	ARTHUR N.	Added support for the magic directory naming
 **				 option for the ovlfs (i.e. the pseudo fs).
 ** 03/01/1998	ARTHUR N.	Allocate GFP_KERNEL priority memory only.
 ** 03/03/1998	ARTHUR N.	Added support for the noxmnt option and fixed
 **				 crossing-mount points.
 ** 03/05/1998	ARTHUR N.	Added support for malloc-checking.
 ** 03/09/1998	ARTHUR N.	Added the copyright notice.
 ** 03/09/1998	ARTHUR N.	Update the access time of the directory after
 **				 the readdir() call completes.
 **
 ******************************************************************************
 ******************************************************************************
 **/

#ident "@(#) ovl_dir.c 1.10"

#ifdef MODVERSIONS
# include <linux/modversions.h>
# ifndef __GENKSYMS__
#  include "ovlfs.ver"
# endif
#endif

#include <linux/stddef.h>

#include <linux/fs.h>
#include <linux/malloc.h>
#include <linux/sched.h>
#include <linux/string.h>

#include "kernel_lists.h"
#include "ovl_fs.h"



/**
 ** STRUCTURE: dir_info_struct
 **
 ** NOTES:
 **	- Don't confuse this "temporary" structure with the ovlfs_dir_info_t
 **	  data structure.
 **/

struct dir_info_struct {
    void        *dirent;
    int         ent_num;
    filldir_t   callback;
    list_t      dlist;
} ;

typedef struct dir_info_struct  dir_info_t;

/**
 ** STATIC PROTOTYPES
 **/

static ovlfs_dir_info_t *ovlfs_ino_get_dirent(ovlfs_inode_t *, int);



/**
 ** FUNCTION: ovlfs_dir_readdir
 **
 **  PURPOSE: Read the overlay filesystem directory whose inode is given.
 **
 ** NOTES:
 **     - This is the second version of this method; the ovlfs storage is
 **       used here to determine the contents of the directory given.
 **
 **     - (CONSIDER) It might be a good idea to update the contents of the
 **       directory's storage since the actual base and/or storage filesystems
 **       may have changed.
 **
 **	- The entire contents of the directory are returned unless the filldir
 **	  callback returns a negative value.
 **/

static inline void  update_time (struct inode *inode)
{
    if ( ! IS_RDONLY(inode) )
    {
        inode->i_atime = ( inode->i_ctime = CURRENT_TIME );
        inode->i_dirt = 1;
    }
}

static int  ovlfs_dir_readdir (struct inode *inode, struct file *filp,
                               void *dirent, filldir_t filldir)
{
    int                 cur_ent_no;
    int                 error;
    ovlfs_inode_info_t  *i_info;
    ovlfs_inode_t       *ino;
    ovlfs_dir_info_t    *d_info;
    struct inode        *o_inode;
    _ulong              inum;


    if ( ( inode == (struct inode *) NULL ) || ( !S_ISDIR(inode->i_mode) ) ||
         ( ! is_ovlfs_inode(inode) ) )
    {
#if KDEBUG
        printk("OVLFS: ovlfs_dir_readdir called with invalid inode at 0x%x\n",
               (int) inode);
#endif

        return  -ENOTDIR;
    }

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_dir_readdir(inode %ld, f_pos %d)\n",
           inode->i_ino, (int) filp->f_pos);
#endif

    cur_ent_no = filp->f_pos;
    i_info = (ovlfs_inode_info_t *) inode->u.generic_ip;


        /* NOTE: All cases in the switch fall through (except default)! */

    switch ( cur_ent_no )
    {
        case 0:
            if ( filldir(dirent, ".", 1, cur_ent_no, inode->i_ino) < 0 )
            {
                update_time(inode);
                return  0;
            }
            cur_ent_no++;
            filp->f_pos++;

        case 1:
            if ( i_info->ovlfs_dir != INODE_NULL )
                inum = i_info->ovlfs_dir->i_ino;
            else
                inum = inode->i_ino;

            if ( filldir(dirent, "..", 2, cur_ent_no, inum) < 0 )
            {
                update_time(inode);
                return  0;
            }

            cur_ent_no++;
            filp->f_pos++;

        case 2:
#if USE_MAGIC
            if ( ( ovlfs_magic_opt_p(inode->i_sb, OVLFS_BASE_MAGIC) ) &&
                 ( ! ovlfs_magic_opt_p(inode->i_sb, OVLFS_HIDE_MAGIC) ) )
            {
                    /* Make sure there is a base inode for this inode */

                error = ovlfs_resolve_base_inode(inode, &o_inode);

                if ( error >= 0 )
                {
                    char    *name;
                    int     len;

                        /* Determine the directory's name from the ovlfs' */
                        /*  options, but use the defaults if not found.   */

                    name = ovlfs_sb_opt(inode->i_sb, Bmagic_name);
                    len = ovlfs_sb_opt(inode->i_sb, Bmagic_len);

                    if ( ( name == (char *) NULL ) || ( len <= 0 ) )
                    {
                        name = OVLFS_MAGICR_NAME;
                        len = OVLFS_MAGICR_NAME_LEN;
                    }

                    if ( filldir(dirent, name, len, cur_ent_no,
                                 o_inode->i_ino) < 0 )
                    {
                        update_time(inode);
                        return  0;
                    }

                    cur_ent_no++;
                    IPUT(o_inode);
                }
            }
            filp->f_pos++;
#endif


        case 3:
#if USE_MAGIC
            if ( ( ovlfs_magic_opt_p(inode->i_sb, OVLFS_OVL_MAGIC) ) &&
                 ( ! ovlfs_magic_opt_p(inode->i_sb, OVLFS_HIDE_MAGIC) ) )
            {
                    /* Make sure there is a storage inode for this inode */

                error = ovlfs_resolve_stg_inode(inode, &o_inode);

                if ( error >= 0 )
                {
                    char    *name;
                    int     len;

                        /* Determine the directory's name from the ovlfs' */
                        /*  options, but use the defaults if not found.   */

                    name = ovlfs_sb_opt(inode->i_sb, Smagic_name);
                    len = ovlfs_sb_opt(inode->i_sb, Smagic_len);

                    if ( ( name == (char *) NULL ) || ( len <= 0 ) )
                    {
                        name = OVLFS_MAGIC_NAME;
                        len = OVLFS_MAGIC_NAME_LEN;
                    }

                    if ( filldir(dirent, name, len, cur_ent_no,
                                 o_inode->i_ino) < 0 )
                    {
                        update_time(inode);
                        return  0;
                    }

                    cur_ent_no++;
                    IPUT(o_inode);
                }
            }
            filp->f_pos++;
#endif
    
        case 4:
                /***
                 *** Skip to file position 5 here whether or not magic is
                 ***  compiled in; this way, the default action always
                 ***  knows which file position is the first entry in its
                 ***  storage (that is 5).
                 ***/

            filp->f_pos = 5;
    
        default:
                /* Obtain the storage inode information for this directory */

            ino = ovlfs_get_ino(inode->i_sb, inode->i_ino);

            if ( ino == (ovlfs_inode_t *) NULL )
            {
#if KDEBUG
                printk("OVLFS: ovlfs_dir_readdir: unable to obtain directory's"
                       " storage inode\n");
#endif

                return  -ENOTDIR;
            }

                /* Now loop through any and all entries in this directory's */
                /*  stored list of entries.                                 */

            error = 0;

            while ( error == 0 )
            {
                    /* Get the nth entry (n >= 1) */

                d_info = ovlfs_ino_get_dirent(ino, filp->f_pos - 4);

                    /* Make sure the entry was obtained and that the */
                    /*  entry is not marked as being deleted.        */

                if ( d_info == (ovlfs_dir_info_t *) NULL )
                {
                        /* There must be no more entries. (not an error :)) */
                    error = 1;
                }
                else if ( d_info->flags & OVLFS_DIRENT_UNLINKED )
                        filp->f_pos++;
                else
                {
                    error = filldir(dirent, d_info->name, d_info->len,
                                    cur_ent_no, d_info->ino);

                    if ( error >= 0 )
                    {
                        error = 0;
                        cur_ent_no++;
                        filp->f_pos++;
                    }
                }
            }
            break;
    }

    update_time(inode);
    return  1;
}




/**
 ** FUNCTION: read_into_dir_cb
 **
 **  PURPOSE: Handle the callback from the readdir() call for the read_into_dir
 **           function.
 **
 ** NOTES:
 **     - This callback assumes that all entries in a directory reside on the
 **       same device as the directory itself.  This should be safe since one
 **       filesystem must reside on one device, and the filesystem's readdir()
 **       is the one passing the entries here.
 **/

struct read_into_cb_info_struct {
    struct super_block  *sb;
    struct super_block  *o_sb;
    struct inode        *o_dir;
    ovlfs_inode_t       *ino;
    kdev_t              dev;
    char                is_stg;
} ;

typedef struct read_into_cb_info_struct read_into_cb_info_t;

static int  read_into_dir_cb (void *d_ent, const char *name, int name_len,
                              off_t ent_no, ino_t inum)
{
    ovlfs_dir_info_t    *d_info;
    read_into_cb_info_t *cb_info;
    int                 ret_code;
    ovlfs_inode_t       *ino;
    _ulong              my_ino;


    cb_info  = (read_into_cb_info_t *) d_ent;


#if ! FASTER_AND_MORE_DANGEROUS
    if ( ( d_ent == NULL ) || ( cb_info->ino == NULL ) || ( name == NULL ) ||
         ( name_len <= 0 ) || ( cb_info->sb == NULL ) ||
         ( cb_info->sb->s_magic != OVLFS_SUPER_MAGIC ) )
    {
# if KDEBUG
        printk("OVLFS: read_into_dir_cb: invalid argument given\n");
# endif

        return  -ENOTDIR;
    }
#endif


        /* Check for "." and ".."; do not add either one */

    if ( ( name_len < 3 ) && ( name[0] == '.' ) )
        if ( ( name_len == 1 ) || ( name[1] == '.' ) )
            return  0;

    ret_code = 0;


        /***
         *** For each entry, update the information for the inode and add
         ***  it to the directory.  Note that a directory entry found here
         ***  may be marked as UNLINKED, but go ahead and update the info
         ***  in the same way as before.  In this way, it may be possible
         ***  to add the ability to restore unlinked entries easily.
         ***/

    d_info = ovlfs_ino_find_dirent(cb_info->ino, name, name_len);

    if ( ( d_info == (ovlfs_dir_info_t *) NULL ) || ( d_info->ino == 0 ) )
    {
            /* Did not find the directory entry, check to see if the inode */
            /*  given has pseudo fs storage information for it.            */

        if ( cb_info->is_stg )
            my_ino = ovlfs_lookup_ino(cb_info->sb, cb_info->dev, inum, 's');
        else
            my_ino = ovlfs_lookup_ino(cb_info->sb, cb_info->dev, inum, 'b');
    }
    else
        my_ino = d_info->ino;



        /***
         *** If the pseudo fs' inode number was not found, a new inode must
         ***  be created in the pseudo fs' storage; otherwise, obtain the
         ***  inode's information from storage and update its name and parent
         ***  directory.
         ***/

    if ( my_ino == 0 )
    {
        ino = ovlfs_add_inode(cb_info->sb, name, name_len);

        if ( ino == (ovlfs_inode_t *) NULL )
        {
#if KDEBUG
            printk("OVLFS: read_into_dir_cb: unable to add a new ovlfs "
                   "inode for a directory entry\n");
#endif

            ret_code = -ENOMEM;
        }
        else
            my_ino = ino->ino;
    }
    else
    {
            /* The inode was found; get its storage inode structure */

        ino = ovlfs_get_ino(cb_info->sb, my_ino);

        if ( ino == (ovlfs_inode_t *) NULL )
            ret_code = -ENOENT;
        else
        {
                /***
                 *** Make sure the name reference for the inode is set and
                 ***  set it if not.  This name is used when the inode must
                 ***  be created in the storage fs because it only exists
                 ***  in the base filesystem.
                 ***/

            if ( ( ino->name == (char *) NULL ) || ( ino->len <= 0 ) )
            {
                ino->name = (char *) MALLOC(name_len);

                if ( ino->name != (char *) NULL )
                {
                    ino->len = name_len;
                    memcpy(ino->name, name, name_len);
                }
#if KDEBUG
                else
                    printk("OVLFS: read_into_dir_cb: unable to allocate "
                           "%d bytes for the reference name\n", name_len);
#endif
            }
        }
    }


        /* If the storage information was obtained without error, copy all */
        /*  of the information from the real inode into the storage.       */

    if ( ( my_ino != 0 ) && ( ret_code >= 0 ) )
    {
        if ( (  ( cb_info->is_stg ) && ( ino->stg_dev == NODEV )  ) ||
             (  ( ! cb_info->is_stg ) && ( ino->base_dev == NODEV )  ) )
        {
            struct inode    *inode;
            kdev_t          dev;

                /* Set the parent directory of the new entry to the */
                /*  directory being read.                           */

            ino->parent_ino = cb_info->ino->ino;


                /* Default to the device number given by the callback */
                /*  information, and override with the device number  */
                /*  of the inode, if it is found.                     */

            dev = cb_info->dev;


                /* Lookup the inode from the pool */

            inode = __iget(cb_info->o_sb, inum, ovlfs_sb_xmnt(cb_info->sb));

            if ( inode != INODE_NULL )
            {
                dev = inode->i_dev;
                inum = inode->i_ino;


                    /* Only copy the attributes if they are not already */
                    /*  set or the "real" inode is from the storage fs. */

                if ( ( ino->mode == 0 ) || ( cb_info->is_stg ) )
                {
                    ino->mode    = inode->i_mode;
                    ino->size    = inode->i_size;
                    ino->uid     = inode->i_uid;
                    ino->gid     = inode->i_gid;
                    ino->atime   = inode->i_atime;
                    ino->mtime   = inode->i_mtime;
                    ino->ctime   = inode->i_ctime;
                    ino->nlink   = inode->i_nlink;
                    ino->blksize = inode->i_blksize;
                    ino->blocks  = inode->i_blocks;
                    ino->version = inode->i_version;
                    ino->nrpages = inode->i_nrpages;

                    if ( S_ISBLK(ino->mode) || S_ISCHR(ino->mode) )
                        ino->u.rdev = inode->i_rdev;
                    else
                        ino->u.generic = NULL;
                }

                IPUT(inode);
            }
#if KDEBUG
            else
                printk("OVLFS: read_into_dir_cb: lookup error %d\n       name "
                       "%.*s, len %d in directory dev %d, ino %ld = %d\n",
                       ret_code, name_len, name, name_len,
                       cb_info->o_dir->i_dev, cb_info->o_dir->i_ino, ret_code);
#endif


                /* Set the mapping for the storage to the real inode */

            if ( cb_info->is_stg )
                ret_code = ovlfs_inode_set_stg(cb_info->sb, ino, dev, inum);
            else
                ret_code = ovlfs_inode_set_base(cb_info->sb, ino, dev, inum);

            if ( ret_code == -EINVAL )
                ret_code = -ENOTDIR;
        }
    }

        /* Determine if this entry already exists in the pseudo fs' */
        /*  directory and add it if not.                            */

    if ( ( my_ino != 0 ) && ( ret_code >= 0 ) &&
         ( d_info == (ovlfs_dir_info_t *) NULL ) )
    {
            ret_code = ovlfs_ino_add_dirent(cb_info->ino, name, name_len,
                                            my_ino, OVLFS_DIRENT_NORMAL);
    }

    return  ret_code;
}



/**
 ** FUNCTION: read_into_dir
 **
 **  PURPOSE: Read the contents of the given "real" directory, o_dir, and add
 **          the contents to the given pseudo fs directory.
 **/

static int  read_into_dir (struct inode *dir, ovlfs_inode_t *ino,
                           struct inode *o_dir, char is_stg)
{
    struct file         file;		/* MUST be a stack variable */
    int                 ret;
    int                 last_pos;
    read_into_cb_info_t cb_info;	/* MUST be a stack variable */

#if ! FASTER_AND_MORE_DANGEROUS
    if ( ( dir == INODE_NULL ) || ( o_dir == INODE_NULL ) ||
         ( ino == (ovlfs_inode_t *) NULL ) )
    {
# if KDEBUG
        printk("OVLFS: read_into_dir: invalid argument\n");
# endif

        return  -ENOTDIR;
    }
#endif

        /* Make sure the real inode is a directory (that we can use) */

    if ( ( o_dir->i_op == NULL ) ||
         ( o_dir->i_op->default_file_ops == NULL ) ||
         ( o_dir->i_op->default_file_ops->readdir == NULL ) )
        return  -ENOTDIR;


        /* Setup a file to use to read the real directory */

    memset(&file, '\0', sizeof(struct file));

    if ( o_dir->i_op != NULL )
        file.f_op = o_dir->i_op->default_file_ops;

    file.f_count = 1;   /* Just in case anyone looks */
    file.f_op = o_dir->i_op->default_file_ops;

    ret = 0;
    last_pos       = -1;
    cb_info.ino    = ino;
    cb_info.sb     = dir->i_sb;
    cb_info.o_sb   = o_dir->i_sb;
    cb_info.o_dir  = o_dir;
    cb_info.dev    = o_dir->i_dev;
    cb_info.is_stg = is_stg;

        /* Keep calling the directory's readdir() function until     */
        /*  an error is encountered, or no more entries are returned */

    while ( ( ret >= 0 ) && ( file.f_pos > last_pos ) )
    {
        last_pos = file.f_pos;
        ret = file.f_op->readdir(o_dir, &file, &cb_info, read_into_dir_cb);
    }

    return  ret;
}



/**
 ** FUNCTION: ovlfs_read_directory
 **
 **  PURPOSE: Given a pseudo fs directory inode, determine the contents for the
 **           directory and fill in the directory's information.
 **
 ** NOTES:
 **	- The pseudo fs' storage must already contain the inode information
 **	  for the given directory.
 **
 **     - Unlike real physical directories, if the pseudo fs' directory does
 **       not exist in its storage, it must be created.
 **/

int ovlfs_read_directory (struct inode *dir)
{
    ovlfs_inode_t   *ino;
    struct inode    *o_inode;
    int             ret;

#if ! FASTER_AND_MORE_DANGEROUS
    if ( ( dir == INODE_NULL ) || ( dir->u.generic_ip == NULL ) ||
         ( ! is_ovlfs_inode(dir) ) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_read_directory: invalid inode given\n");
# endif

        return  -ENOTDIR;
    }
#endif

        /* Read the inode information from the pseudo fs' storage */

    ino = ovlfs_stg_read_ino(dir);

    if ( ino == (ovlfs_inode_t *) NULL )
    {
#if KDEBUG
        printk("OVLFS: ovlfs_read_directory: unable to obtain inode "
               "information from storage\n");
#endif

        ret = -ENOTDIR;
    }
    else
    {
            /***
             *** Read the entries to place in the directory's list of entries.
             ***  First look for the inode in the storage fs so that any
             ***  pseudo fs inodes created by read_into_dir() will take the
             ***  attributes of the storage fs' inode instead of the base fs'
             ***  inode.
             ***/


        ret = ovlfs_resolve_stg_inode(dir, &o_inode);

        if ( ret >= 0 )
        {
                /* Read the storage fs dir's contents into the pseudo fs */

            read_into_dir(dir, ino, o_inode, TRUE);
            IPUT(o_inode);
        }
        else if ( ret == -ENOENT )
            ret = 0;


        ret = ovlfs_resolve_base_inode(dir, &o_inode);

        if ( ret >= 0 )
        {
                /* Read the base fs dir's contents into the pseudo fs */

            read_into_dir(dir, ino, o_inode, FALSE);
            IPUT(o_inode);
        }
        else if ( ret == -ENOENT )
            ret = 0;
    }

        /* If successful, return the number of entries in the directory */

    if ( ( ret >= 0 ) && ( ino->dir_entries != (list_t) NULL ) )
        ret = klist_func(list_length)(ino->dir_entries);

    return  ret;
}



/**
 ** FUNCTION: ovlfs_ino_get_dirent
 **
 **  PURPOSE: Obtain the nth directory entry from the given pseudo fs
 **           directory.
 **
 ** NOTES:
 **     - A pointer to the actual entry in the list is returned; the caller
 **       must be careful NOT to free the returned value.
 **/

static ovlfs_dir_info_t *ovlfs_ino_get_dirent (ovlfs_inode_t *ino, int pos)
{
    ovlfs_dir_info_t    *result;

    if ( ( ino == (ovlfs_inode_t *) NULL ) ||
         ( ino->dir_entries == (list_t) NULL ) )
    {
#if KDEBUG
            /* Only print an error message if debugging is high enough or */
            /*  the inode is null since empty directories will get here.  */
# if KDEBUG < 2
        if ( ino == (ovlfs_inode_t *) NULL )
# endif

        printk("OVLFS: ovlfs_ino_get_dirent: invalid ovlfs inode given at "
               "0x%x\n", (int) ino);
#endif

        return  (ovlfs_dir_info_t *) NULL;
    }

    result = (ovlfs_dir_info_t *)
             klist_func(nth_element)(ino->dir_entries, pos);

    return  result;
}


struct file_operations  ovlfs_dir_fops = {
    (int (*)(struct inode *, struct file *, off_t, int)) NULL,
    (int (*)(struct inode *, struct file *, char *, int)) NULL,
    (int (*)(struct inode *, struct file *, const char *, int)) NULL,
    ovlfs_dir_readdir,
    (int (*)(struct inode *, struct file *, int, select_table *)) NULL,
    (int (*)(struct inode *, struct file *, unsigned int,
             unsigned long)) NULL,
    (int (*)(struct inode *, struct file *, struct vm_area_struct *)) NULL,
    (int (*)(struct inode *, struct file *)) NULL,
    (void (*)(struct inode *, struct file *)) NULL,
    (int (*)(struct inode *, struct file *)) NULL,
    (int (*)(struct inode *, struct file *, int)) NULL,
    (int (*)(kdev_t dev)) NULL,
    (int (*)(kdev_t dev)) NULL
} ;
