/*
 *  inode.c
 *
 *  Copyright (C) 1995 Martin von Lwis
 */

#include <stdio.h>
#include <fcntl.h>
#include "ntfs.h"
#include "config.h"

/* not used */
/* Index records and MFT records differ in fixup not only by the number
   of clusters to fix up, but also in the start of the fixup table */
int ntfs_fixup_record(char *block,int count)
{
	char *fixup=block+0x28;
	char *dest=block+NTFS_CLUSTERSIZE-2;
	for(;count;count--)
	{
		if(*(short*)dest!=*(short*)fixup)return 0;
		fixup+=2;
		*(short*)dest=*(short*)fixup;
		fixup+=2;
		dest+=NTFS_CLUSTERSIZE;
	}
	return 1;
}

/* mft records start with the string 'FILE'. Each cluster should end in the
   word at offset 0x2A, and needs to be replaced with the words following
   0x2A */
int ntfs_check_mft_record(char *record)
{
	short *fixup=(short*)(record+0x2A);
	char *dest=record+NTFS_CLUSTERSIZE-2;
	int i;
	if(!IS_MFT_RECORD(record))return 0;
	for(i=0;i<NTFS_MFT_CLUSTERS_PER_RECORD;i++)
	{
		if(*(short*)dest!=*fixup)return 0;
		*(short*)dest=fixup[i+1];
		dest+=NTFS_CLUSTERSIZE;
	}
	return 1;
}

/* find the location of an attribute in the inode. A name of NULL indicates
   unnamed attributes. Return pointer to attribute or NULL if not found */
char *ntfs_get_attr(char *ino,int attr,char *name)
{
	/* location of first attribute */
	char *it=ino+0x30;
	int type;
	int length;
	/* Only check for magic DWORD here, fixup should have happened before */
	if(!IS_MFT_RECORD(ino))return 0;
	do{
		type=*(int*)it;
		length=*(short*)(it+4);
		/* We found the attribute type. Is the name correct, too? */
		if(type==attr)
		{
			int namelen=*(it+9);
			char *name_it;
			/* match given name and attribute name if present,
				make sure attribute name is Unicode */
			for(name_it=it+*(int*)(it+10);namelen;
				name++,name_it+=2,namelen--)
				if(*name_it!=*name || name_it[1])break;
			if(!namelen)break;
		}
		it+=length;
	}while(type!=-1); /* attribute list end with type -1 */
	if(type==-1)return 0;
	return it;
}

/* Support is only for resident attribute list right now :-) */
char *ntfs_find_in_attr_list(char* alist,int attr,char *name,int segment)
{
	int length=*(short*)(alist+4);
	char *it=alist+0x18;
	while(it<(alist+length))
	{
		char *next=it+*(short*)(it+4);
		int namelen=*(it+0x6);
		if(*(int*)it==attr)
			if(!name)
				if(namelen==0)
					if(segment==0)
						return it;
					else
						segment--;
				else ;
			else{
				int i;
				/* compare both names if both are present */
				for(i=0;name[i] && i<namelen;i++)
					if(it[2*i+0x1A]!=name[i])
						break;
				if(i==namelen || name[i]==0)
					if(segment==0)
						return it;
					else
						segment--;
			}
		it=next;
	}
	return 0;
}


/* Answer true if a certain attribute is present in an inode */
int ntfs_has_attr(char *ino,int attr,char *name)
{
	char *alist;
	if(ntfs_get_attr(ino,attr,name))return 1;
	if(!(alist=ntfs_get_attr(ino,AT_ATTRIBUTE_LIST,NULL)))
		return 0;
	return ntfs_find_in_attr_list(alist,attr,name,0)!=0;
}

char *ntfs_get_ext_ino(void*vol,char*ino,int inum,
	int attr,char *name,int offset)
{
	char *alist;
	char *aitem,*lastitem;
	int section;
	char *result;
	int mftno;
	alist=ntfs_get_attr(ino,AT_ATTRIBUTE_LIST,NULL);
	/* should be only called if attribute list is present */
	if(!alist)
		return (char*)-1;
	section=0;
	aitem=ntfs_find_in_attr_list(alist,attr,name,section);
	/* no such attribute */
	if(!aitem)
		return (char*)-1;
	lastitem=aitem;
	do{
		if(*(int*)(aitem+0x8)*NTFS_CLUSTERSIZE>offset)
			break;
		section++;
		lastitem=aitem;
		aitem=ntfs_find_in_attr_list(alist,attr,name,section);
		if(!aitem)
			break;
	}while(1);
	mftno=*(int*)(lastitem+0x10);
	result=ntfs_allocate(NTFS_MFT_RECORDSIZE);
	if(ntfs_read_mft_record(vol,mftno,result)==-1)
		return (char*)-1;
	return result;
}

int ntfs_get_attr_size(void*vol,char*ino,int inum,int attr,char*name)
{
	char* data,*ext;
	int len;
	data=ntfs_get_attr(ino,attr,name);
	if(!data)
	{
		ext=ntfs_get_ext_ino(vol,ino,inum,attr,name,0);
		if(ext==(char*)-1 || ext==NULL)
			return -1;
		data=ntfs_get_attr(ext,attr,name);
	}else
		ext=0;
	/* fake size 0 for compressed files */
	if(COMPRESSED(data))
		return 0;
	len=DATASIZE(data);
	if(ext)
		ntfs_free(ext);
	return len;
}
	
int ntfs_attr_is_resident(void*vol,char*ino,int inum,int attr,char*name)
{
	char* data,*ext;
	int resident;
	data=ntfs_get_attr(ino,attr,name);
	if(!data)
	{
		ext=ntfs_get_ext_ino(vol,ino,inum,attr,name,0);
		if(ext==(char*)-1 || ext==NULL)
			return 0;
		data=ntfs_get_attr(ext,attr,name);
	}else
		ext=0;
	resident=RESIDENT(data);
	if(ext)
		ntfs_free(ext);
	return resident;
}
	

/* A run is coded as a type indicator, a length, and a cluster offset.
   The first run is relative to cluster 0, later runs are relative to
   the previous cluster.
   To save space, length and offset are integers of variable length.
   The low byte of the type indicates the width of the length :), the
   high byte the width of the offset. The offsets are interpreted signed,
   the length unsigned.
   length is an output parameter, data and cluster are inout parameters.
   As some type fields are probably reserved for sparse and compressed
   files, only those values known to be used are decoded */
int decompress_run(unsigned char **data,int *length,int *cluster,int compressed)
{
	unsigned char type=*(*data)++;
	int offset;
	switch(type & 0xF)
	{
		case 1: *length=*(*data)++;break;
		case 2: *length=*(short int*)*data;
				*data+=2;
				break;
		case 3: *length=(unsigned char)**data|
						((unsigned char)*(*data+1)<<8)+*(*data+2)<<16;
				*data+=3;
				break;
		default:
				ntfs_error("Can't decode run type field %x\n",type);
				return -1;
	}
	switch(type & 0xF0)
	{
		case 0x10: *cluster+=*((char*)*data)++;break;
		case 0x20: *cluster+=*(short int*)*data;
					*data+=2;
					break;
		case 0x30: 	offset=(*(int*)*data)&(0xffffff);
					if(offset>0x7fffff)offset-=0x1000000;
					*cluster+=offset;
					*data+=3;
					break;
		default:
				ntfs_error("Can't decode run type field %x\n",type);
				return -1;
	}
	if(compressed)
		*data+=2;
	return 0;
}

/* Reads l bytes of the attribute (attr,name) of ino starting at offset
   on vol into buf. Returns the number of bytes read. 0 is end of attribute,
   -1 is error */
int ntfs_read_attr(void *vol,char *ino,int inum,int attr,char *name,
	int offset,char *buf,int l,copyfunction pmemcpy)
{
	unsigned char *data;
	int datasize;
	int cluster=0;
	int length=0;
	int copied=0;
	int resident;
	int compressed;
	int namelen;
	int valueoffset;
	int startvcn,endvcn;
	if(inum!=-1)
	{
		/* The datasize is relative to the attribute start*/
		datasize=ntfs_get_attr_size(vol,ino,inum,attr,name);
		/* Nothing to do*/
		if(offset>=datasize)return 0;
		/* Make sure we don't read after end of attribute */
		if(offset+l>=datasize)l=datasize-offset;
	}
	/* -1 indicates that this is already the correct inode */
	if(inum!=-1 && ntfs_get_attr(ino,AT_ATTRIBUTE_LIST,NULL))
	{
		char *realino;
		int len,len1;
		realino=ntfs_get_ext_ino(vol,ino,inum,attr,name,offset);
		if((int)realino==-1)
			return -1;
		if(!realino)
			return 0;
		len=ntfs_read_attr(vol,realino,-1,attr,name,offset,buf,l,pmemcpy);
		if(len<=l){
			ntfs_free(realino);
			return len;
		}
		buf+=len;
		l-=len;
		len1=ntfs_read_attr(vol,ino,inum,attr,name,offset,buf,l,pmemcpy);
		ntfs_free(realino);
		if(len1==-1)
			return -1;
		return len+len1;
	}
	data=ntfs_get_attr(ino,attr,name);
	if(!data){
		ntfs_error("attribute %x not found\n",attr);
		/* Should we return -1 here ?*/
		return 0;
	}
	resident=RESIDENT(data);
	namelen=*(data+9);
	valueoffset=*(unsigned char*)(data+10);
	compressed=*(data+0xC);
	startvcn=*(int*)(data+0x10);
	endvcn=*(int*)(data+0x18);
	/* Most unnamed attributes have a valueoffset of zero, get it from 0x20*/
	if(!valueoffset)valueoffset=*(short*)(data+0x20);
	/* Now, advance to the data. Skip the name */
	data+=valueoffset+2*namelen;
	/* resident data are easy to access */
	if(resident)
	{
		pmemcpy(buf,data+offset,l);
		return l;
	}
	/* For non-resident data, walk through the runs 
	   and read from disk as necessary */
	if(offset<startvcn*NTFS_CLUSTERSIZE)
	{
		ntfs_error("Reading the wrong segment\n");
		return 0;
	}
	offset-=startvcn*NTFS_CLUSTERSIZE;
	if(offset+l>=(endvcn+1)*NTFS_CLUSTERSIZE)
		l=(endvcn+1)*NTFS_CLUSTERSIZE-offset;

	/* skip runs at the beginning that are before offset */
	do{
		offset-=length*NTFS_CLUSTERSIZE;
		if(decompress_run(&data,&length,&cluster,compressed)==-1)return -1;
	}while(offset>=length*NTFS_CLUSTERSIZE);

	while(l)
	{
		int s_cluster,chunk;
		
		/* Find the first cluster in the current run that needs to be read */
		s_cluster=cluster+offset/NTFS_CLUSTERSIZE;
		offset = offset % NTFS_CLUSTERSIZE;
		length -= s_cluster-cluster;

		chunk=min(length*NTFS_CLUSTERSIZE-offset,l);
		if(!ntfs_get_clusters(vol,s_cluster,offset,chunk,buf,pmemcpy))
		{
			ntfs_error("Read error\n");
			return copied;
		}
		buf+=chunk;
		l-=chunk;
		copied+=chunk;
		offset=0;
		/* proceed to next run, unless done */
		if(l)
			if(decompress_run(&data,&length,&cluster,compressed)==-1)return -1;
	}
	return copied;
}

int ntfs_vcn_to_lcn(char *ino,int vcn)
{
	char *data=ntfs_get_attr(ino,AT_DATA,NULL);
	int lcn,len;
	int valueoffset;
	/* It's hard to give an error code */
	if(!data)return -1;
	if(RESIDENT(data))return -1;
	if(DATASIZE(data)<vcn*NTFS_CLUSTERSIZE)return -1;
	if(COMPRESSED(data))return -1;

	/* name is NULL, so we can ignore namelen */
	valueoffset=*(int*)(data+10);
	if(!valueoffset)valueoffset=0x40;
	data+=valueoffset;

	for(lcn=len=0;vcn>=len;)
	{
		vcn-=len;
		/* only uncompressed data supported */
		decompress_run(&data,&len,&lcn,0);
	}

	return lcn+vcn;
}

/* copy len unicode characters from from to to :) */
void ntfs_uni2ascii(char *to,char *from,int len)
{
	int i;
	for(i=0;i<len;i++)
		to[i]=from[2*i];
	to[i]='\0';
}

/* Convert the NT UTC (based 1.1.1601, in hundred nanosecond units)
   into Unix UTC (based 1.1.1970, in seconds)
   Need to use floating point, because we cannot use libgcc for long
   long operations in the kernel */
time_t ntfs_ntutc2unixutc(long long ntutc)
{
	double sec=ntutc;
	sec=sec/10000000;
	return sec-((long long)369*365+89)*24*3600;
}

