#include <linux/ext2_fs.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>

/* (C) 1994 Scott Heavner (sdh@po.cwru.edu) */

/* find_ext2_fs.c -- greps a full disk devices for remnants of an ext2 partition.
 *                   Will be most useful to recover from a bad fdisk experience.
 *
 * Usage: find_ext2_fs --best-guess /dev/name
 *        find_ext2_fs /dev/name
 *        find_ext2_fs                           -- defaults to /dev/hda
 *
 * If it is called with 3 parameters as shown above (--best-guess can be anything),
 * it will only display one entry for the most likely filesystem cantidates.
 * Otherwise, it will print out any blocks which contain EXT2_MAGIC, which may or
 * may not be superblocks.  Also, there are many copies of the super block on 
 * each filesystem, they will all yield the same size, but will have different
 * starts and ends.  The first one should show the actual start and end of the
 * partition, the rest think that the partition begins just before this copy of
 * the superblock, so the start and end will be incorrect.  It may be necessary
 * to leave out the best guess option if nothing shows up on the first run.
 *
 * $Id: find_ext2_fs.c,v 2.5 2000/03/11 19:07:53 denis Exp $
 *
 * Changes:
 *
 * October 6, 1997 - modified to search 512 byte blocks instead of 1024.  If
 *   the filesystem didn't start on a 1024 byte interval, it would have been
 *   skipped.
 * - comment out EXT2_PRE_02B_MAGIC - must not be supported in kernels anymore
 * - compile with "gcc -O6 -s -o find_ext2_fs find_ext2_fs.c" for a 4k binary
 * - why is this thing so slow?
 *
 * October 7, 1997 - try to speed things up by reading in more blocks at a
 *   time
 * - NB=32 is still painfully slow, maybe we should revert back to old version
 *   that only read in 512 bytes at a time.  If NB=1, #ifdefs should revert
 *   back to old version.
 * - /usr/bin/time trials to search to ext2fs at sector 204624 of an ide disk:
 *        NB=     32             8              1
 *        real    1m17.916s      1m28.996s      1m40.012s   
 *        user    0m0.200s       0m0.530s       0m0.490s
 *        sys     0m57.270s      0m43.970s      0m49.060s
 * - Might be nice to write a new tool that starts at block 1 of a disk,
 *   figures out what's on it (msdos or linux partition), computes and displays
 *   its size/start/end, the jumps to the end and looks for the next bit of
 *   partition info.
 * - Fixed to report actual super block location and start and end of the
 *   partition (starting with block 0).
 * - Something is still wrong.  I need an lseek() to before I do anything.
 *   one of my drives seems to come up on byte 1 and maintains this 1-byte
 *   offset through the entire search process, so I never see the EXT_MAGIC
 *   in the right place.
 * - Be more cautious with the ranch wagers. Check that next copy of
 *   superblock matches current copy.
 */ 

/* The ext2 superblock begins 1024 bytes after the start of the ext2 file
 * system, we want to report the start of the filesystem, not the location
 * of the super block */
#define C -2UL

/* Speed enhancement - set the number of 512 byte blocks we read at a time
 * Feel free to twiddle this parameter, if you make it too big, performance
 * probably won't improve - it will also control how close to the end of
 * the partition you can get as the last NB sectors won't be checked, must be
 * at least 1 */
#define NB 32

#if NB < 1
#error Can you read?
#endif

/* Swiped next two routines out of no_fs.c for computing the size of
 * a given partition (or file I suppose) */
static int valid_offset (int fd, unsigned long offset)
{
  char ch;
  
  if (lseek (fd, offset, SEEK_SET) < 0)
    return 0;
  if (read (fd, &ch, 1) < 1)
    return 0;
  return 1;
}
/* Returns the number of blocks in a partition */
unsigned long NOFS_get_device_size(int fd, unsigned long blocksize)
{
  unsigned long high, low;

  /* Do binary search to find the size of the partition.  */
  low = 0;
  for (high = 1024; valid_offset (fd, high); high *= 2)
    low = high;
  while (low < high - 1)
    {
      const unsigned long mid = (low + high) / 2;
      
      if (valid_offset (fd, mid))
	low = mid;
      else
	high = mid;
    }
  valid_offset (fd, 0);
  return ((low + 1) / blocksize);
}

int main(int argc, char **argv)
{
  unsigned long i=0UL, lastblock, last_lastblock=0UL, blocksize, oldPos;
  unsigned long deviceSize, last_size=0UL, ranch_size=0UL, ranch_last=0UL;
  char *device_name="/dev/hdb", buffer[NB*512], b2[512];
  int fd, best_guess=0, this_is_it, j, noWay;
  struct ext2_super_block *sb, *sb2;

  /* To make things a little more readable */
#if NB<2
  sb = (void *) buffer;
#endif
  sb2 = (void *) b2;
  

  /* Parse command line arguments */
  if (argc==3) {
    best_guess = 1;
    device_name = argv[2];
  } else if (argc==2) {
    device_name = argv[1];
  } else if (argc>3) {
    printf("Usage: %s --best-only devicename\n",argv[0]);
    exit(-argc);
  }
  
  if ((fd=open(device_name,O_RDONLY))<0) {
    printf("Can't open device %s\n",device_name);
    exit(-1);
  }
  
  printf("\nIt would be wise to go grab a beverage, this will take a while . . .\n");

  printf("\nIt also wouldn't hurt to save the output next time:\n\t\"");
  for (j=0; j<argc; j++)
    printf("%s ",argv[j]);
  printf("2>&1 | tee DiskRemnants\"\n\n");

  printf("Checking device size . . .");
  deviceSize = NOFS_get_device_size(fd, 512UL);
  printf(" %ld sectors.\n",deviceSize);

  printf("Checking %s for ext2 filesystems %s\n",device_name,
	 (best_guess?"(Best guesses only)":"(Display even the faintest shreds)"));

  /* What the *uck is this?  I can't read one of my drives w/o seeking back to the start,
   * I'd be off by one character on all my read()s */
  lseek (fd, 0L, SEEK_SET);

  /* Make 1st block on disk #0 (where partition table lives), like fdisk would */  
  for(i=0UL;;) {

    /* Grab a block - dump it into memory defined as an ext2_super_block */
    if ( read(fd, buffer, NB*512) != NB*512 ) {
      if ( (i+NB) >= deviceSize) {
	printf("Reached end of device.\n");
	exit(0);
      }
      printf("Error reading device.  Last successful read on block %ld(including 0) of %ld.\n",i-1UL,deviceSize);
      exit(-2);
    }

#if NB>1
    /* Operate on all the blocks we've read into memory */
    for (j=0;j<NB;j++) {

      sb = (void *)(buffer+j*512);
#endif

      /* It probably isn't it */
      this_is_it = 0;
    
      /* Check if the ext2 magic number is in the right place */
      /* if ( (sb->s_magic== EXT2_SUPER_MAGIC)||(sb->s_magic == EXT2_PRE_02B_MAGIC) ) { */
      if (sb->s_magic== EXT2_SUPER_MAGIC) {

	/* For the moment, assume this really might be good */
	noWay = 0;

	/* When we get a match, figure out where this partiton would end */
	blocksize = (unsigned long)EXT2_MIN_BLOCK_SIZE << sb->s_log_block_size;
	/* END = START + (SIZE-1): i.e. 100 block fs starting at block 0 runs 0->99 */
	lastblock = (i+C)+(sb->s_blocks_count*(blocksize/512UL)-1UL);

	/* If the blocksize is bigger than the device, it can't be it */
	if ( (sb->s_blocks_count*blocksize) > (deviceSize*512UL) )
	  noWay = 1;

	if ( ((i+C) > last_lastblock) || (last_size != sb->s_blocks_count) ) {
	  if (!best_guess)
	    printf("THESE ENTRIES ARE PROBABLY ALL FOR THE SAME PARTITION:\n");
	  last_lastblock = lastblock;
	  last_size = sb->s_blocks_count;
	  if ( (blocksize >= EXT2_MIN_BLOCK_SIZE) && (blocksize <= EXT2_MAX_BLOCK_SIZE) && (!noWay) ) {
	    if ( !(ranch_size) || ((ranch_size==last_size)&&(ranch_last==last_lastblock)) ) {
#if 1
	      /* Check next copy of superblock */
	      /* Save original position - in case we tried to move past end of file */
	      oldPos = lseek(fd, 0, SEEK_CUR);
	      /* Jump ahead 8192 1024-byte blocks, adjust for our read ahead of j */
	      if (lseek (fd, (8192*2-(NB-j))*512, SEEK_CUR) >= 0) {

		if (read(fd, b2, 512)!=512)
		  noWay = 1;
		
		/* memcmp() fails everytime because the sb->s_state changes, check some of the easy ones */
		if (sb->s_magic != sb2->s_magic )
		  noWay = 1; 
		else
		  printf("Magic OK\n");
		if (sb->s_blocks_count != sb2->s_blocks_count )
		  noWay = 1;
		else
		  printf("Cnt OK\n");
		if (sb->s_log_block_size != sb2->s_log_block_size )
		  noWay = 1;
		else
		  printf("Log OK\n");
	      } else {
		printf("Seek failed\n");
	      }
	      
	      /* Restore saved position */
	      lseek (fd, oldPos, SEEK_SET);
#endif	      
	      if (!noWay) {
		if (!best_guess)
		  printf("**** I'D BET THE RANCH ON THIS NEXT ENTRY *************\n");
		this_is_it = 1;
		ranch_size = last_size;
		ranch_last = last_lastblock;
	      }
	    }
	  }
	}
	if ((!best_guess)||(this_is_it)) {
	  printf("   %c Found ext2_magic in sector %lu (512 byte sectors).\n",(noWay)?'-':'*',i);
	  printf("     This file system is %lu blocks long (each block is %lu bytes)\n",
		 sb->s_blocks_count,blocksize);
	  printf("     Filesystem runs %lu : %lu (512 byte sectors)\n",i+C,lastblock);
	}
      }
      i++; /* Needs to be inside j loop */
#if NB>1
    }
#endif
  }

  
  exit(0);
}





