/*   user2.cc : February 1, 1994    */

/* Copyright (C) 1994-1999  Sekhar Bala, Rama Bala, and
 *                          Alphax System, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <malloc.h>
#include <string.h>

#include "std.h"
#include "wildcard.h"
#include "parse.h"
#include "symtab.h"
#include "input.h"
#include "fnmatch.h"

#include "main.h"
#include "dump.h"
#include "ext2fs.h"
#include "diskio.h"


/*-------------------------------------------------------------------------*/

STATIC CHAR        *pblock       = NULL;
STATIC CHAR        *block        = NULL;
STATIC INT_2        entry_num    = 0;
STATIC ENTRY_TYP    entry        = { 0, };
STATIC INT_4        blocksize    = 0;

STATIC INODE_TYP   tinode        = {
   0,                                     // imode
   0,                                     // owner uid
   0,                                     // size
   851784735L,                            // access time
   851783644L,                            // creation time
   851783644L,                            // modification time
   0L,                                    // deletion time
   0,                                     // owner gid
   0,                                     // links count
   2,                                     // blocks count
   0,                                     // file flags
   0,                                     // reserved1
   {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,},      // 15 i_blocks
   0,                                     // file version
   0,                                     // file acl
   0,                                     // directory acl
   0,                                     // fragment address
   0,                                     // fragment number
   0,                                     // fragment size
   {0,0}                                  // reserved2
};

STATIC BOOLEAN dobckcpy( VOID );

/*-------------------------------------------------------------------------*/


STATIC VOID clrdir( VOID )
{

   if ( block != NULL )
      ::free( block );
   block     = NULL;
   entry_num = 0;
   memset( &entry, 0, sizeof(ENTRY_TYP) );
   return;

} /* clrdir */


STATIC BOOLEAN dodir( VOID )
{
 INT_4       value;
 INT_2       len;
 INT_2       i, j;
 ENTRY_TYP  *pentry;
 CHAR       *ptr;

   if ( P.get() )
      return( FALSE );

   if ( WC::streq(P.word, "BEGIN") ) {
      clrdir();
      block = (CHAR *) ::malloc( blocksize );
      if ( block == NULL ) {
         set_errhdl( ERR_SYS_NOMEM );
         return( FALSE );
      }
      memset( block, 0, blocksize );
      pblock = block;
      return( TRUE );
   }

   if ( WC::streq(P.word, "ENTRY") ) {
      if ( (pblock == NULL) || (block == NULL) ) {
         set_errhdl( ERR_SYS_NOINIT );
         return( FALSE );
      }
      if (
           (P.get())                                        ||
           ( (P.word[0] != '\'') && (P.word[0] != '\"') )   ||
           (P.delimited(FALSE))
         )
         return( FALSE );
      strcpy( entry.name, P.word );
      if ( !eval_expr(&value) )
         return( FALSE );
      entry.inode_num = value;
      len             = strlen(entry.name);
      entry.name_len  = len;
      len             = ( ((INT_U2) (len + 8 + 3)) & (~(INT_U2) 3) );
      if ( (len % 4) != 0 )
         len += (4 - (len % 4));
      entry.rec_len = len;
      memcpy( pblock, &entry, len );
      pblock = &pblock[len];
      memset( &entry, 0, sizeof(ENTRY_TYP) );
      entry_num++;
      return( TRUE );
   }
   
   if ( WC::streq(P.word, "END") ) {
      if ( (pblock == NULL) || (block == NULL) ) {
         set_errhdl( ERR_SYS_NOINIT );
         return( FALSE );
      }
      if ( !eval_expr(&value) )
         return( FALSE );
      write_LIBIO( block, blocksize, blocksize*value );
      clrdir();
      pblock = NULL;
      return( TRUE );
   }

   if ( WC::streq(P.word, "SHOW") ) {
      if ( (pblock == NULL) || (block == NULL) ) {
         set_errhdl( ERR_SYS_NOINIT );
         return( FALSE );
      }
      ptr    = block;
      for( i=0; (i < entry_num); i++ ) {
         pentry = (ENTRY_TYP *) ptr;
         printf( "%2d] %8lu - ", i, pentry->inode_num );
         for( j=0;  (j < pentry->name_len); j++ )
            printf( "%c", pentry->name[j] );
         printf( "\n" );
         ptr = &ptr[pentry->rec_len];
      }
      return( TRUE );
   }

   return( FALSE );

} /* dodir */


STATIC BOOLEAN dobad( VOID )
{
 INT_2        i;
 INT_4        idx;
 INT_4        value;
 INT_4        pinode_num;
 INODE_TYP   *pinode;

   if ( P.get() )
      return( FALSE );

   if ( WC::streq(P.word, "INODE") ) {
      // syntax is: # , $FILE or $DIR, size { b-#, b-#, ... }
      if ( !eval_expr(&value, ",") )
         return( TRUE );
      pinode_num = value;
      pinode     = &tinode;
      if ( !eval_expr(&value, ",") )
         return( TRUE );
      pinode->i_mode = (INT_U2) value;
      if ( !eval_expr(&value, "{") )
         return( TRUE );
      pinode->i_size   = (INT_U4) value;
      pinode->i_blocks = (value / 512 );
      if ( (value % 512) != 0 )
         pinode->i_blocks++;
      if ( P.word[0] != '{' )
         return( FALSE );
      for( i=0; (i < 15); i++ )
         pinode->i_block[i] = 0;
      i = 0;
      while ( P.word[0] != '}' ) {
         if ( !eval_expr(&value, "-,}") )
            return( TRUE );
         if ( i == 15 )
            return( FALSE );
         pinode->i_block[i++] = value;
         if ( P.word[0] == '-' ) {
            if ( !eval_expr(&value, ",}") )
               return( FALSE );
            i--;
            for( idx=pinode->i_block[i]; (idx <= value); idx++ ) {
               if ( i == 15 )
                  return( FALSE );
               pinode->i_block[i++] = idx;
            }
         }
      }
      for( i=0; (i < badinode_len); i++ ) {
         if ( pinode_num == badinode[i].inode_num )
            break;
      }
      if ( (i == badinode_len) && (badinode_len == MAX_BADINODES) ) {
         set_errhdl( ERR_SYS_NOMEM );
         return( TRUE );
      }
      memcpy( &badinode[i].inode, pinode, sizeof(INODE_TYP) );
      if ( i == badinode_len ) {
         badinode[i].inode_num = pinode_num;
         badinode_len++;
      }
      return( TRUE );
      // syntax is: # , $FILE or $DIR, size { b-#, b-#, ... }
   }

   return( TRUE );

} /* dobad */


GLOBAL BOOLEAN exec_fs( VOID )
{

   if ( !WC::streq(P.word, "FS") )
      return( FALSE );

   if ( sb == NULL ) {
      set_errhdl( ERR_SYS_NOINIT );
      return( TRUE );
   }
   blocksize = ( EXT2_BASE_BLOCK << sb->s_log_block_size );

   if ( P.get() )
      goto x_done;

   if ( WC::streq(P.word, "DIR") ) {
      if ( !dodir() )
         goto x_done;
      return( TRUE );
   }

   if ( WC::streq(P.word, "BCKCPY") ) {
      if ( !dobckcpy() )
         goto x_done;
      return( TRUE );
   }

   if ( WC::streq(P.word, "BAD") ) {
      if ( !dobad() )
         goto x_done;
      return( TRUE );
   }

x_done:

   set_errhdl( ERR_CMD_UNKNOWN );
   return( TRUE );

} /* exec_fs */


/*-------------------------------------------------------------------------*/

STATIC FILE   *fplst          = NULL;
STATIC FILE   *fpbad          = NULL;
STATIC INT_4   fcnt           = 0;
STATIC INT_4   dcnt           = 0;
STATIC CHAR    tfname[256]    = { 0, };
STATIC CHAR   *pbuf           = NULL;

/*-------------------------------------------------------------------------*/


STATIC CHAR *make_fname( CHAR *dest, CHAR *fname )
{

   strcpy( tfname, dest );
#if defined(GO32)
   if ( tfname[strlen(tfname)-1] != '\\' )
      strcat(tfname, "\\" );
#else
   if ( tfname[strlen(tfname)-1] != '/' )
      strcat(tfname, "/" );
#endif
   strcat( tfname, fname );
   return( tfname );

} /* make_fname */



STATIC CHAR *gen_fname( CHAR *dest, INT_2 cnt )
{
 CHAR    datfile[20];

   sprintf( datfile, "F%07d.DAT", cnt );
   return( make_fname(dest,datfile) );

} /* gen_fname */


STATIC INT_2 cpy(
  CHAR       *fname,
  INODE_TYP  *inode,
  BOOLEAN     udebug,
  BOOLEAN     uverbose )
{
 INT_2      l_ret       = 0;
 INT_4      total       = 0;
 INT_4      block;
 INT_4      block_len;
 INT_2      written;
 FILE      *ofp;

   ofp = fopen( fname, "wb" );
   if ( ofp == NULL ) {
      l_ret = set_errhdl(ERR_SYS_NOINIT);
      goto x_done;
   }

   l_ret = readblock_FS( inode, NULL, NULL );
   if ( l_ret != 0 )
      goto x_done;

   l_ret = readblock_FS( inode, &block, &block_len );
   while ( (block > 0) && (l_ret == 0) ) {

      if ( block_len > (INT_4) blocksize ) {
         set_errhdl( ERR_SYS_NOINIT );
         break;
      }

      l_ret = read_LIBIO( pbuf, blocksize, blocksize*block );
      if ( l_ret != 0 )
         break;

      if ( udebug ) {
         fprintf( ofp, "%ld %ld\n", block, block_len );
      } else {
         if ( uverbose )
            fprintf( ofp, "%ld %ld\n", block, block_len );
         written  = fwrite(pbuf, 1, (size_t) block_len, ofp);
         total   += written;
         if ( (INT_4) written != block_len ) {
             l_ret = set_errhdl(ERR_SYS_NOINIT);
             break;
         }
      }

      l_ret  = readblock_FS( inode, &block, &block_len );

   } /* endwhile */
   if ( l_ret == -1 )
      l_ret = 0;

x_done:

   printf( "(%8ld)", total );
   if ( fplst != NULL )
      fprintf( fplst, " %8ld", total );
   if ( ofp != NULL )
      fclose( ofp );
   return( l_ret );

} /* cpy */


STATIC BOOLEAN bckcpy(
   BOOLEAN   urecurse,
   BOOLEAN   udebug,
   BOOLEAN   uverbose, 
   CHAR     *pattern,
   CHAR     *dest      )
{
 DIR_TYP       *pdir       = NULL;
 INT_2          match      = FALSE;
// BOOLEAN        copied     = FALSE;
 DIR_TYP       *save_dir;
 INODE_TYP      save_inode;
 CHAR          *pfname;

   // make sure current direct is valid @@ & read
   if ( loaddir_FS() != 0 )
      return( FALSE );
   dcnt++;

   // for each dir entry
   for( pdir=dir; (pdir != NULL); pdir=pdir->next ) {

      // if subtree traversal required
      if ( urecurse ) {

         // make sure not to recurse into same dir @@
         // these conditions cannot be satisfied
         if ( (strlen(pdir->name) == 1) && (pdir->name[0] == '.') )
            continue;
         if ( (strlen(pdir->name) == 2) && (strcmp(pdir->name, "..") == 0) )
            continue;
         if (inode_no == pdir->inode_num)
            continue;
         
         if ( S_ISDIR(pdir->inode.i_mode) ) {
            save_dir = dir;
            memcpy( &save_inode, &inode, sizeof(INODE_TYP) );
            dir = NULL;
            memcpy( &inode, &pdir->inode, sizeof(INODE_TYP) );
            fprintf( fplst, "B %s\n", pdir->name );
            printf( "%24.24s -> %15.15s\n", pdir->name, "SUBDIR" );
            if ( !bckcpy(urecurse, udebug, uverbose, pattern, dest) ) {
               fprintf( fpbad, "D %s\n", pdir->name );
            }
            fprintf( fplst, "E\n" );
            dir = save_dir;
            memcpy( &inode, &save_inode, sizeof(INODE_TYP) );
            continue;
         }

      }

      // non-regular files are skipped
      if ( !S_ISREG(pdir->inode.i_mode) )
         continue;

      // match filename
      match = fnmatch( pattern, pdir->name );
      if ( match != FNM_MATCH )
         continue;

      pfname = gen_fname(dest,fcnt);
      fprintf( fplst, "F %-40.40s F%07ld.DAT", pdir->name, fcnt );
      printf( "%24.24s -> %15.15s ", pdir->name, pfname );
      if ( cpy(pfname, &pdir->inode, udebug, uverbose) != 0 ) {
         fprintf( fplst, " *" );
         fprintf( fpbad, "F %s\n", pdir->name );
      }
      fprintf( fplst, "\n" );
      printf( "\n" );
      fcnt++;

   } /* endfor */

   // done
   return( TRUE );

} /* bckcpy */


STATIC BOOLEAN dobckcpy( VOID )
{
 BOOLEAN    urecurse         = FALSE;
 BOOLEAN    udebug           = FALSE;
 BOOLEAN    uverbose         = FALSE;
 CHAR       cp_pattern[80];
 CHAR       cp_dest   [80];
 CHAR      *pfname;

   if ( !WC::streq(P.word, "BCKCPY") )
      return( FALSE );

   P.seperators( " " );
   if ( P.get((PARSE::_ASIS|PARSE::_STRIP)) )
      goto x_edone;

   while (TRUE) {
      if ( P.word[0] != '-' )
         break;
      if ( strchr(P.word, 'v') != 0 )
         uverbose = TRUE;
      if ( strchr(P.word, 'r') != 0 )
         urecurse = TRUE;
      if ( strchr(P.word, 'd') != 0 )
         udebug = TRUE;
      if ( P.get((PARSE::_ASIS|PARSE::_STRIP)) )
         goto x_edone;
   }

   strcpy( cp_pattern, P.word );

   cp_dest[0] = 0;
   if ( !P.get((PARSE::_ASIS|PARSE::_STRIP)) )
      strcpy( cp_dest, P.word );

   if ( pbuf == NULL ) {
      pbuf = (CHAR *) ::malloc( blocksize );
      if ( pbuf == NULL ) {
         set_errhdl( ERR_SYS_NOMEM );
         goto x_done;
      }
   }
   fcnt   = 1;
   dcnt   = 0;
   pfname = make_fname( cp_dest, "files.lst" );
   fplst  = fopen( pfname, "w" );
   if ( fplst == NULL ) {
      printf("Cannot open 'files.lst'!\n");
      set_errhdl(ERR_SYS_NOINIT);
      goto x_done;
   }
   pfname = make_fname( cp_dest, "files.bad" );
   fpbad = fopen( pfname, "w" );
   if ( fpbad == NULL ) {
      printf("Cannot open 'files.bad'!\n");
      set_errhdl(ERR_SYS_NOINIT);
      fclose( fplst );
      goto x_done;
   }
   if ( uverbose )
      printf( "Pattern='%s' Dest='%s'\n", cp_pattern, cp_dest );
   bckcpy( urecurse, udebug, uverbose, cp_pattern, cp_dest );
   fclose( fplst );
   fclose( fpbad );
   fplst = NULL;
   fpbad = NULL;
   printf( "%ld files, %ld directories\n", fcnt, dcnt );
   goto x_done;

x_edone:

   set_errhdl( ERR_CMD_UNKNOWN );

x_done:

   P.seperators( NULL );
   // never free pbuf
   return( TRUE );

} /* dobckcpy */


GLOBAL BOOLEAN exec_cp( VOID )
{
 BOOLEAN        udebug           = FALSE;
 BOOLEAN        uverbose         = FALSE;
 CHAR           cp_pattern[80];
 DIR_TYP       *pdir;
 INT_2          match;

   if ( !WC::streq(P.word, "CP") )
      return( FALSE );

   if ( blocksize == 0 ) {
      if ( sb == NULL ) {
         set_errhdl( ERR_SYS_NOINIT );
         return( TRUE );
      }
      blocksize = ( EXT2_BASE_BLOCK << sb->s_log_block_size );
   }

   P.seperators( " " );
   if ( P.get((PARSE::_ASIS|PARSE::_STRIP)) )
      goto x_edone;

   while (TRUE) {
      if ( P.word[0] != '-' )
         break;
      if ( strchr(P.word, 'd') != 0 )
         udebug = TRUE;
      if ( strchr(P.word, 'v') != 0 )
         uverbose = TRUE;
      if ( P.get((PARSE::_ASIS|PARSE::_STRIP)) )
         goto x_edone;
   }

   strcpy( cp_pattern, P.word );

   if ( pbuf == NULL ) {
      pbuf = (CHAR *) ::malloc( blocksize );
      if ( pbuf == NULL ) {
         set_errhdl( ERR_SYS_NOMEM );
         goto x_done;
      }
   }

   if ( loaddir_FS() != 0 )
      goto x_edone;

   fcnt = 0;
   for( pdir=dir; (pdir != NULL); pdir=pdir->next ) {

      // if not a regular file, skip
      if ( !S_ISREG(pdir->inode.i_mode) )
         continue;

      // test for pattern match
      match = fnmatch( cp_pattern, pdir->name );
      if ( match != FNM_MATCH )
         continue;

      printf( "-> %30.30s  ", pdir->name );
      if ( cpy(pdir->name, &pdir->inode, udebug, uverbose) != 0 )
         printf( " *" );
      printf( "\n" );

      fcnt++;

   } /* end for */

   goto x_done;

x_edone:

   set_errhdl( ERR_CMD_UNKNOWN );

x_done:

   P.seperators( NULL );
   // never free pbuf 
   return( TRUE );

} /* exec_cp */


/*-------------------------------------------------------------------------*/
// 
// cp  l-file  d-file
//   --> copies from single (nw) l-file (current dir) to d-file (full name)
/*-------------------------------------------------------------------------*/
// 
// 
// use option '-d' for debug mode
// use option '-v' for verbose mode
// 
// fs bckcpy  pattern  d-prefix
//   --> copies from multiple l-file (current dir) to d-file (sform)
// 
// fs bckcpy  -r  pattern  d-prefix 
//   --> copies from multiple l-file (current dir + lower) to d-file (sform)
// 
// sform & output is as follows:
// 
//    d-file -> "d:\dir\dir"     dir-name only and it better already exist
//    output is:                 files.lst and fxxxxxxx.dat
//    files.lst has follow record formats:
// 
//       x name  filename
// 
//       ; where x=(F,B,E) then
//       ; F=file          unix-fname      system-fname
//       ; B=begin-dir     unix-dname
//       ; E=end-dir
//
//    e.g.
// 
//       F afile1.cc       f0000001.dat
//       F afile2.cc       f0000002.dat
//         B dirname1
//           F afile3.cc   f0000xxx.dat
//           B subdir1
//             F afile4.cc f0000xxx.dat
//           E 
//           F afile4.cc   f0000xxx.dat
//         E
//       F afile3.cc       f0000xxx.dat
// 
/*-------------------------------------------------------------------------*/



