/*   input2.cc : May 5, 1994   */

/* Copyright (C) 1994-1999  Sekhar Bala,
 *                          Ramchander Balasubramanian, and
 *                          Alphax Systems, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "std.h"
#include "wildcard.h"
#include "parse.h"
#include "datetime.h"
#include "symtab.h"
#include "input.h"

/*------------------------------------------------------------------------*/

enum OPER_KEYS {
   OP_NONE   = 0,
   OP_LPAREN,
   OP_RPAREN,
   OP_MOD, 
   OP_TIMES,
   OP_DIVIDE,
   OP_OR,
   OP_AND,
   OP_XOR,
   OP_TILDE,
   OP_NOT,
   OP_SHIFTR,
   OP_SHIFTL,
   OP_PLUS,
   OP_MINUS,
   OP_EQUAL,
   OP_NOTEQUAL,
   OP_GTE,
   OP_LTE,
   OP_GT,
   OP_LT,
   OP_ANDAND,
   OP_OROR,
   OP_ARG    = 100
};

typedef struct {
   CHAR        ch;
   INT_2       key;
} OPER_TYP;

typedef struct {
   CHAR       *op;
   INT_2       key;
} COMPLEXOPER_TYP;

STATIC OPER_TYP           ops[] = {
   { '(',      OP_LPAREN      },
   { ')',      OP_RPAREN      },
   { '%',      OP_MOD         },
   { '*',      OP_TIMES       },
   { '/',      OP_DIVIDE      },
   { '|',      OP_OR          },
   { '&',      OP_AND         },
   { '^',      OP_XOR         },
   { '~',      OP_TILDE       },
   { '!',      OP_NOT         },
   { '+',      OP_PLUS        },
   { '-',      OP_MINUS       },
   { '>',      OP_GT          },
   { '<',      OP_LT          },
   { '=',      OP_EQUAL       },
   { 0,        OP_NONE        }
};

STATIC COMPLEXOPER_TYP    cops[] = {
   { "==",     OP_EQUAL       },
   { "=>",     OP_GTE         },
   { "=>",     OP_LTE         },
   { ">>",     OP_SHIFTR      },
   { "<<",     OP_SHIFTL      },
   { "!=",     OP_NOTEQUAL    },
   { ">=",     OP_GTE         },
   { "<=",     OP_LTE         },
   { "&&",     OP_ANDAND      },
   { "||",     OP_OROR        },
   { NULL,     0              },
};

#define MAX_STACK_LEN                     130

typedef struct {
   INT_4       value;
   INT_2       key;
} STACK_TYP;

STATIC INT_2               stack_len      = 0;
STATIC STACK_TYP           stack[MAX_STACK_LEN];

#define DT_DATE      0
#define DT_DATEY2    1
#define DT_DATEY4    2
#define DT_TIME      3
#define DT_DATETIME  4
#define SYS_ENV      5

typedef struct {
   INT_2       funct;
   CHAR       *kw;
} BUILTIN_TYP;

STATIC BUILTIN_TYP   builtin[] = {
   { DT_DATE,     "DT_DATE"      },
   { DT_DATEY2,   "DT_DATEY2"    },
   { DT_DATEY4,   "DT_DATEY4"    },
   { DT_TIME,     "DT_TIME"      },
   { DT_DATETIME, "DT_DATETIME"  },
   { SYS_ENV,     "SYS_ENV"      },
   { 0,           NULL           }
};

/*------------------------------------------------------------------------*/


GLOBAL BOOLEAN is_builtin( CHAR *kw )
{
 INT_2   i;
 CHAR   *pkw;

   if ( kw == NULL )
      return( FALSE );
   pkw = kw;
   if ( *pkw == '$' )
      pkw++;
   for( i=0; (TRUE); i++ ) {
      if ( builtin[i].kw == NULL )
         return( FALSE );
      if ( WC::streq(pkw, builtin[i].kw) )
         break;
   }
   return( TRUE );

} /* is_builtin */


GLOBAL BUILTIN_TYP  *get_builtin( CHAR *kw )
{
 INT_2  i;
 CHAR  *pkw;

   if ( kw == NULL )
      return( NULL );
   pkw = kw;
   if ( *pkw == '$' )
      pkw++;
   for( i=0; (TRUE); i++ ) {
      if ( builtin[i].kw == NULL )
         return( NULL );
      if ( WC::streq(pkw, builtin[i].kw) )
         break;
   }
   return( &builtin[i] );

} /* get_builtin */


/*+
 *      NAME : qpush
 *
 *  FUNCTION : push the next token on the term-stack (OP or ARG)
 *
 *     ENTRY : stack component contents (passed)
 *
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
GLOBAL BOOLEAN qpush( INT_4 value, INT_2 key )
{
 STACK_TYP   *pstack;

   if ( stack_len >= MAX_STACK_LEN ) {
      set_errhdl( ERR_EXP_QPUSH );
      return( FALSE );
   }
   pstack        = &stack[stack_len++];
   pstack->value = value;
   pstack->key   = key;
   return( TRUE );

} /* qpush */


/*+
 *      NAME : qpop
 *
 *  FUNCTION : pop the next token from the term-stack (OP or ARG)
 *
 *     ENTRY : stack component contents (returned)
 *
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
GLOBAL BOOLEAN qpop( INT_4 *value, INT_2 *key )
{
 STACK_TYP   *pstack;

   if ( stack_len == 0 ) {
      set_errhdl( ERR_EXP_QPOP );
      return( FALSE );
   }
   pstack = &stack[--stack_len];
   *value = pstack->value;
   *key   = pstack->key;
   return( TRUE );

} /*  qpop */


/*+
 *      NAME : eval
 *
 *  FUNCTION : eval all paired terms right to left, skipping
 *             an operator precendeces except for groupings "()"
 *
 *     ENTRY : lparen -> use "(" as the terminating argument
 *
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
STATIC BOOLEAN eval( BOOLEAN lparen=FALSE )
{
 INT_4    value;
 INT_2    key;
 INT_4    left;
 INT_4    right;
 INT_2    op;
 BOOLEAN  result;

   result = FALSE;
   while (TRUE) {

      // get right argument & test validity
      if ( !qpop(&value, &key) )
         break;
      if ( key != OP_ARG ) {
         set_errhdl( ERR_EXP_ARG1 );
         break;
      }
      right = value;

      // test for conclusion
      if ( (!lparen) && (stack_len == 0) ) {
         result = qpush( value, key );
         break;
      }

      // get operator
      if ( !qpop(&value, &key) )
         break;
      op = key;

      // test for conclusion
      if ( (lparen) && (op == OP_LPAREN) ) {
         result = qpush( right, OP_ARG );
         break;
      }

      // test for unary operator
      if ( (op == OP_TILDE) || (op == OP_NOT) ) {
         switch( op ) {
            case OP_TILDE:
               value = ~right;
               break;
            case OP_NOT:
               value = !right;
               break;
         }
         if ( !qpush(value, OP_ARG) )
            break;
         continue;
      }

      // get left argument & test validity
      if ( !qpop(&value, &key) )
         break;
      if ( key != OP_ARG ) {
         set_errhdl( ERR_EXP_ARG2 );
         break;
      }
      left = value;

      // evaluate
      switch( op ) {
         case OP_MOD:
            value = left % right;
            break;
         case OP_TIMES:
            value = left * right;
            break;
         case OP_DIVIDE:
            value = left / right;
            break;
         case OP_OR:
            value = left | right;
            break;
         case OP_AND:
            value = left & right;
            break;
         case OP_XOR:
            value = left ^ right;
            break;
         case OP_SHIFTR:
            value = left >> right;
            break;
         case OP_SHIFTL:
            value = left << right;
            break;
         case OP_PLUS:
            value = left + right;
            break;
         case OP_MINUS:
            value = left - right;
            break;
         case OP_EQUAL:
            value = (left == right);
            break;
         case OP_NOTEQUAL:
            value = (left != right);
            break;
         case OP_GTE:
            value = (left >= right);
            break;
         case OP_LTE:
            value = (left <= right);
            break;
         case OP_GT:
            value = (left > right);
            break;
         case OP_LT:
            value = (left < right);
            break;
         case OP_ANDAND:
            value = (left && right);
            break;
         case OP_OROR:
            value = (left || right);
            break;
         case OP_LPAREN:
         case OP_RPAREN:
         default:
            set_errhdl( ERR_EXP_ARG3 );
            return( result );
      }

      // push back to Q
      if ( !qpush(value, OP_ARG) )
         break;

   } /* end while */

   // done 
   return( result );

} /* eval_lparen */


/*+
 *      NAME : eval_expr
 *
 *  FUNCTION : parses and evaluates a numerical expression
 *
 *     ENTRY : uvalue <- resulting value
 *             useps  -> possible seperators terminating expression parse
 *
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
GLOBAL BOOLEAN eval_expr( INT_4 *uvalue, CHAR *useps )
{
 INT_2     i;
 INT_4     value;
 INT_2     key;
 CHAR     *ptr;

   // setup stack
   *uvalue   = 0;
   stack_len = 0;

   // loop thru the supplied expression
   while ( TRUE ) {

      // get next token, if none left, then done
      if ( (P.get()) || (P.word[0] == ';') || (P.word[0] == '#') )
         break;

      // test for expression seperator
      if ( (useps != NULL) && (strchr(useps, P.word[0]) != NULL) )
         break;

      // search for special simple ops
      for( i=0; (TRUE); i++ ) {
         if ( ops[i].ch == 0 )
            break;
         if ( P.word[0] == ops[i].ch )
            break;
      }

      // if special simple ops found
      if ( ops[i].ch != 0 ) {
         // we could have found a complex op
         ptr = P.cmd_ptr();
         value = 0;
         key   = ops[i].key;
         for( i=0; (TRUE); i++ ) {
            if ( cops[i].op == NULL )
               break;
            if (
                 (P.word[0] == cops[i].op[0]) &&
                 (*ptr      == cops[i].op[1])
               ) {
               P.get();
               key = cops[i].key;
            }
         }
         if ( key != OP_RPAREN ) {
            if ( !qpush(value, key) )
               return( FALSE );
            continue;
         }
         if ( !eval(TRUE) )
            return( FALSE );
         continue;
      }

      // test for delimited special op
      if ( P.word[0] == '\'' ) {
         if ( P.delimited(TRUE) )
            return( FALSE );
      }

      // otherwise must be arg
      key = OP_ARG;
      if ( !toINT(&value, P.word) )
         return( FALSE );
      if ( !qpush(value, key) )
         return( FALSE );

   } /* end while */

   // evaluate
   if ( !eval() )
      return( FALSE );

   // get the result
   if ( !qpop(uvalue, &key) )
      return( FALSE );

   // done ok
   return( TRUE );

} /* eval_expr */


/*+
 *      NAME : get_args
 *
 *  FUNCTION : parses argument list on a builtin function call
 *
 *     ENTRY : seps     -> list of seperators opening arg-list
 *
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
STATIC BOOLEAN get_args( CHAR *t_seps )
{
 BOOLEAN   noword;

   // setup
   noword = P.get();

   // test and skip leading seperator
   if ( t_seps != NULL ) {
      if ( noword ) {
         set_errhdl( ERR_SEM_BADARGS );
         return( FALSE );
      }
      if ( strchr(t_seps, P.word[0]) == NULL ) {
         set_errhdl( ERR_SEM_BADARGS );
         return( FALSE );
      }
      noword = P.get();
   }

   // done
   return( TRUE );

} /* get_args */


/*+
 *      NAME : eval_strexpr
 *
 *  FUNCTION : evaluate a string expression
 *
 *     ENTRY : uvalue      <- resulting value of expr
 *             uvalue_len  -> max byte len of resulting value
 *             useps       -> possible seperators terminating str-expr
 * 
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
GLOBAL BOOLEAN eval_strexpr( CHAR *uvalue, INT_2 uvalue_len, CHAR *useps )
{
 INT_2           i;
 BUILTIN_TYP    *pbuiltin;
 CHAR           *ptr;
 INT_2           uvalue_idx;
 INT_4           timevalue;

   *uvalue    = 0;
   uvalue_idx = 0;

   while (TRUE) {

      // get next token & test for termination
      if ( P.get() )
         break;
      if ( (P.word[0] == ';') || (P.word[0] == '#') )
         break;
      if ( (useps != NULL) && (strchr(useps, P.word[0]) != NULL) )
         break;

      // the default is to always string-concat
      if ( P.word[0] == '+' )
         continue;

      // test for delimited string
      if ( (P.word[0] == '\'') || (P.word[0] == '\"') ) {
         if ( P.delimited() )
            return( FALSE );
         for( i=0; (i < P.word_len); i++ ) {
            if ( uvalue_idx >= uvalue_len )
               goto x_overerr;
            uvalue[uvalue_idx++] = P.word[i];
         }
         continue;
      }

      // 
      // builtin functions
      //
      ptr      = smallbuf;
      pbuiltin = get_builtin(P.word);
      if ( pbuiltin != NULL ) {
         timevalue = (INT_4) time_sec();
         if ( !get_args( "(" ) )
            return( FALSE );
         switch( pbuiltin->funct ) {
            case DT_DATE:
            case DT_DATEY2:
               if ( P.word[0] != ')' ) {
                  if ( !toINT(&timevalue, P.word) )
                     return( FALSE );
                  if ( P.get() )
                     return( FALSE );
               }
               str_timdat( (INT_U4) timevalue, smallbuf, 2 );
               smallbuf[9] = 0;
               break;
            case DT_DATEY4:
               if ( P.word[0] != ')' ) {
                  if ( !toINT(&timevalue, P.word) )
                     return( FALSE );
                  if ( P.get() )
                     return( FALSE );
               }
               str_timdat( (INT_U4) timevalue, smallbuf, 4 );
               smallbuf[12] = 0;
               break;
            case DT_TIME:
               if ( P.word[0] != ')' ) {
                  if ( !toINT(&timevalue, P.word) )
                     return( FALSE );
                  if ( P.get() )
                     return( FALSE );
               }
               str_timdat( (INT_U4) timevalue, smallbuf, 2 );
               ptr = &smallbuf[10];
               break;
            case DT_DATETIME:
               if ( P.word[0] != ')' ) {
                  if ( !toINT(&timevalue, P.word) )
                     return( FALSE );
                  if ( P.get() )
                     return( FALSE );
               }
               str_timdat( (INT_U4) timevalue, smallbuf, 2 );
               break;
            case SYS_ENV:
               if ( (P.word[0] == '\'') || (P.word[0] == '\"') ) {
                  if ( P.delimited() )
                     return( FALSE );
                  ptr = getenv(P.word);
                  if ( P.get() )
                     return( FALSE );
               }
               break;
            default:
               goto x_badparse;
         }
         if ( P.word[0] != ')' )
            goto x_badparse;
      }

      while ( *ptr != 0 ) {
         if ( uvalue_idx >= uvalue_len )
            goto x_overerr;
         uvalue[uvalue_idx++] = *ptr;
         ptr++;
      }

   } /* end while */

   uvalue[uvalue_len-1] = 0;
   if ( uvalue_idx >= uvalue_len )
       goto x_overerr;
   uvalue[uvalue_idx] = 0;
   return( TRUE );

x_badparse:

   set_errhdl( ERR_SEM_BADARGS );
   return( FALSE );

x_overerr:

   set_errhdl( ERR_EXP_OVERFLOW );
   return( FALSE );

} /* eval_strexpr */


/*------------------------------------------------------------------------*/



