/*   input0.cc : May 5, 1994   */

/* Copyright (C) 1994-1999  Sekhar Bala,
 *                          Ramchander Balasubramanian, and
 *                          Alphax Systems, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <malloc.h>
#include <string.h>

#include "std.h"
#include "wildcard.h"
#include "symtab.h"
#include "parse.h"
#include "datetime.h"
#include "input.h"
#include "cgets.h"

/*-------------------------------------------------------------------------*/

#define MAX_TOKEN_SIZE     64

GLOBAL INPUT_TYP           input;

GLOBAL PARSE               P;
GLOBAL FILE               *outfp  = stdout;
GLOBAL BOOLEAN             udebug = FALSE;
GLOBAL BOOLEAN             uquote = FALSE;

GLOBAL CHAR                hugebuf  [MAX_HUGE_BUF];
GLOBAL CHAR                bigbuf   [MAX_LARGE_BUF];
GLOBAL CHAR                smallbuf [MAX_SMALL_BUF];

STATIC CHAR                tbuf[MAX_TOKEN_SIZE+1]    = { 0, };
STATIC INT_2               tbuf_len                  = 0;

EXTERN BOOLEAN    is_builtin( CHAR *kw );

/*-------------------------------------------------------------------------*/


/*+
 *      NAME : allocate
 *
 *  FUNCTION : allocate next input stack element
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
BOOLEAN INPUT_TYP::allocate( VOID )
{
 STREAM_TYP   *stream;
 INT_2         size;

   // allocate next cell
   size = sizeof(FILE *) * (input_len + 1);
   if ( input_len == 0 )
      input = (STREAM_TYP *(*)[]) ::malloc( size );
   else
      input = (STREAM_TYP *(*)[]) ::realloc( input, size );

   // test for error
   if ( input == NULL ) {
      set_errhdl( ERR_SYS_NOMEM );
      return( FALSE );
   }
   memset( &(*input)[input_len], 0, sizeof(FILE *) );

   // allocate cell
   stream = (STREAM_TYP *) ::malloc( sizeof(STREAM_TYP) );
   if ( stream == NULL ) {
      set_errhdl( ERR_SYS_NOMEM );
      return( FALSE );
   }
   memset( stream, 0, sizeof(STREAM_TYP) );

   // bump data structure
   (*input)[input_len] = stream;
   input_len++;

   // done
   return( TRUE );

} /* allocate */


/*+
 *      NAME : deallocate
 *
 *  FUNCTION : deallocates last input stack element
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
BOOLEAN INPUT_TYP::deallocate( VOID )
{
 INT_2     size;

   // test for validity
   if ( input_len <= 0 ) {
      set_errhdl( ERR_INP_UNDERFLOW );
      return( FALSE );
   }

   // deallocate cell
   if ( (*input)[input_len-1] != NULL )
      ::free( (*input)[input_len-1] );
   (*input)[input_len-1] = NULL;

   // allocate next cell
   size  = sizeof(FILE *) * (input_len - 1);
   if ( input_len == 1 ) {
      if ( input != NULL )
         ::free( input );
      input = NULL;
   } else {
      input = (STREAM_TYP *(*)[]) ::realloc( input, size );
      if ( input == NULL ) {
         set_errhdl( ERR_SYS_NOMEM );
         input_len--;
         return( TRUE );
      }
   }
   input_len--;

   // done
   return( TRUE );

} /* deallocate */


/*+
 *      NAME : stream_ptr
 *
 *  FUNCTION : returns ptr to last active stream
 *
 *     ENTRY : none
 *
 *   RETURNS : NULL on failure, STREAM_TYP ptr on success
 *
-*/
STREAM_TYP *INPUT_TYP::stream_ptr( BOOLEAN set_err )
{
 INT_2         idx;
 STREAM_TYP   *stream;

   idx = input_len -1;
   if ( (idx < 0) || (input == NULL) ) {
      if ( set_err )
         set_errhdl( ERR_INP_OVERFLOW );
      return( NULL );
   }
   stream = (*input)[idx];
   return( stream );

} /* stream_ptr */


/*+
 *      NAME : push
 *
 *  FUNCTION : push either function or keyboard device onto input stack
 *
 *     ENTRY : fname -> name of function or NULL for keyboard
 *
 *   RETURNS : TRUE on success; FALSE on error
 *
-*/
BOOLEAN INPUT_TYP::push( CHAR *fname, CHAR *fext )
{
 STREAM_TYP   *stream;
 CHAR          fnamebuf[MAX_PATHNAME+1];

   // allocate next cell
   if ( !allocate() )
      return( FALSE );

   // check for a new input stream
   if ( fname == NULL )
      return( TRUE );

   // open and setup file descriptors
   strncpy( fnamebuf, fname, sizeof(tbuf)-1 );
   stream     = (*input)[ (input_len-1) ];
   stream->fp = fopen( fnamebuf, "r" );
   if ( (stream->fp == NULL) && (fext != NULL) ) {
      strcat( fnamebuf, fext );
      stream->fp = fopen( fnamebuf, "r" );
   }
   if ( stream->fp == NULL ) {
      set_errhdl( ERR_INP_FNAME );
      return( FALSE );
   }

   // done
   return( TRUE );

} /* push */


/*+
 *      NAME : push
 *
 *  FUNCTION : push input buffer with special control code
 *             [ function, while, for stmts ]
 *
 *     ENTRY : buf      - datacode seperated by 0 and terminated with 0
 *             buf_len  - datacode len
 *             cond     - optional conditional code
 *             cond_len - optional conditional code len
 *             inc      - optional stmts to execution on loop increment
 *             inc_len  - optional inc stmts len
 *
 *   RETURNS : TRUE on success; FALSE on error
 *
-*/
BOOLEAN INPUT_TYP::push(
   CHAR     *buf,
   INT_2     buf_len,
   CHAR     *cond,
   INT_2     cond_len,
   CHAR     *inc,
   INT_2     inc_len   )
{
 STREAM_TYP   *stream;
 CHAR         *pbuf;
 CHAR         *pcond;
 CHAR         *pinc;

   // allocate buffers
   pbuf = (CHAR *) ::malloc( buf_len );
   if ( pbuf == NULL ) {
      set_errhdl( ERR_SYS_NOMEM );
      return( FALSE );
   }
   memset( pbuf, 0, buf_len );
   pcond = cond;
   if ( pcond != NULL ) {
      pcond = (CHAR *) ::malloc( cond_len+1 );
      if ( pcond == NULL ) {
         set_errhdl( ERR_SYS_NOMEM );
         return( FALSE );
      }
      memset( pcond, 0, cond_len+1 );
   }
   pinc = inc;
   if ( pinc != NULL ) {
      pinc = (CHAR *) ::malloc( inc_len );
      if ( pinc == NULL ) {
         set_errhdl( ERR_SYS_NOMEM );
         return( FALSE );
      }
      memset( pinc, 0, inc_len );
   }

   // allocate next cell
   if ( !allocate() ) {
      if ( pbuf != NULL )
         ::free( pbuf );
      if ( pcond != NULL )
         ::free( pcond );
      if ( pinc != NULL )
         ::free( pinc );
      return( FALSE );
   }

   // transfer values
   memcpy( pbuf,  buf,  buf_len  );
   memcpy( pcond, cond, cond_len );
   memcpy( pinc,  inc,  inc_len  );
   stream          = (*input)[ (input_len-1) ];
   stream->data    = pbuf;
   stream->cptr    = pbuf;
   stream->cond    = pcond;
   stream->inc     = pinc;
   if ( pinc != NULL )
      inc_len *= (-1);
   stream->inc_len = inc_len;
   if ( pcond != NULL )
      stream->cptr = &pbuf[buf_len-1];

   // done
   return( TRUE );

} /* push */


/*+
 *      NAME : pop
 *
 *  FUNCTION : removes input device from top of input-stack
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE on success; FALSE on error
 *
-*/
BOOLEAN INPUT_TYP::pop( VOID )
{
 STREAM_TYP   *stream;

   // setup index
   stream = stream_ptr();
   if ( stream == NULL )
      return( FALSE );

   // test for closure
   if ( stream->fp != NULL ) {
      fclose( stream->fp );
      stream->fp = NULL;
   } else if ( stream->data != NULL ) {
      ::free( stream->data );
      if ( stream->cond != NULL )
         ::free( stream->cond );
      if ( stream->inc != NULL )
         ::free( stream->inc );
      stream->data = NULL;
      stream->cptr = NULL;
      stream->cond = NULL;
      stream->inc  = NULL;
   }

   // release
   if ( !deallocate() )
       return( FALSE );

   // done
   return( TRUE );

} /* pop */


/*+
 *      NAME : is_file
 *
 *  FUNCTION : determines input redirection device type
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if a file is on the input stack, else FALSE
 *
-*/
BOOLEAN INPUT_TYP::is_file( VOID )
{
 STREAM_TYP  *stream;

   // setup index
   stream = stream_ptr(FALSE);
   if ( stream == NULL )
      return( FALSE );

   // return determination
   if ( stream->fp != NULL )
      return( TRUE );

   // done
   return( FALSE );

} /* is_file */


/*+
 *      NAME : is_funct
 *
 *  FUNCTION : determines input redirection device type
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if a function is on the input stack, else FALSE
 *
-*/
BOOLEAN INPUT_TYP::is_funct( VOID )
{
 STREAM_TYP    *stream;

   // setup index
   stream = stream_ptr(FALSE);
   if ( stream == NULL )
      return( FALSE );

   // return determination
   if ( stream->data != NULL )
      return( TRUE );

   // done
   return( FALSE );

} /* is_funct */


/*+
 *      NAME : is_keyboard
 *
 *  FUNCTION : determines input redirection device type
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if the keyboard is on the input stack, else FALSE
 *
-*/
BOOLEAN INPUT_TYP::is_keyboard( VOID )
{
 STREAM_TYP   *stream;

   // setup index
   stream = stream_ptr(FALSE);
   if ( stream == NULL )
      return( FALSE );

   // return determination
   if ( (stream->fp == NULL) && (stream->data == NULL) )
      return( TRUE );

   // done
   return( FALSE );

} /* is_keyboard */


/*+
 *      NAME : varsubst
 *
 *  FUNCTION : substitutes variables on input with appropriate values
 *
 *     ENTRY : ucmd -> cmd to interpret, eval -> flag for full evaluation
 *
 *   RETURNS : TRUE on success or FALSE on failure
 *
-*/
BOOLEAN INPUT_TYP::varsubst( CHAR *ucmd, BOOLEAN eval )
{
 // output string
 #define          obuf          bigbuf
 INT_2            oidx;
 // input string
 CHAR            *ibuf;
 INT_2            iidx;
 INT_2            ilen;
 INT_2            s_iidx;
 BOOLEAN          ibuf_free;
 CHAR            *ptr;
 INT_2            ptr_len;
 SYMBOL_TYP      *psym;
 // internal
 INT_2            i;
 INT_4            value;
 UCHAR            quotation;
 BOOLEAN          dquoted;
 BOOLEAN          squoted;
 BOOLEAN          skip;
 // bogus array reference parse using retrys
 BOOLEAN          retry;
 BOOLEAN          result;

   // no translations for special keywords FOR, WHILE
   skip = FALSE;
   ptr  = ucmd;
   while ( *ptr == ' ' )
      ptr++;
   if ( strncmpi(ptr, "FOR", 3) == 0 ) {
      if ( (ptr[3] == 0) || (strchr(";( #",ptr[3]) != NULL) )
         skip = TRUE;
   }
   if ( strncmpi(ptr, "WHILE", 5) == 0 ) {
      if ( (ptr[5] == 0) || (strchr(";( #",ptr[5]) != NULL) )
         skip = TRUE;
   }

   // make sure varsubst are necessary
   result    = FALSE;
   ibuf      = ucmd;
   ibuf_free = FALSE;
   if ( (skip) || (!eval) || (strchr(ibuf, '$') == NULL) ) {
      result = TRUE;
      ptr    = ucmd;
      goto x_done;
   }

x_retry:

   // init output string
   obuf[0] = 0;
   oidx    = 0;

   // init input string
   ilen    = strlen( ibuf );
   iidx    = -1;

   // setup for loop
   squoted  = FALSE;
   dquoted  = FALSE;
   skip     = FALSE;
   retry    = FALSE;

   // loop until found
   while (TRUE) {

      // transfer next character to output string
      if ( iidx != -1 )
         obuf[oidx++] = ibuf[iidx];
      iidx++;
      if ( (iidx >= ilen) || (ibuf[iidx] == 0) )
         break;

      // skip a single uninterpreted char
      if ( skip ) {
         skip = FALSE;
         continue;
      }

      // test for an uninterpreted character
      if ( ibuf[iidx] == '\\' ) {
         skip = TRUE;
         continue;
      }

      // test for quoted chars and string
      if ( ibuf[iidx] == '\"' ) {
         dquoted = !dquoted;
         continue;
      }
      if ( ibuf[iidx] == '\'' ) {
         squoted = !squoted;
         continue;
      }
      // if not '$' prefix, continue to next input char
      if ( (squoted) || (dquoted) || (ibuf[iidx] != '$') )
         continue;

      // 
      // otherwise '$' prefix found
      //
      s_iidx    = iidx;
      quotation = 0;

      // get the token to substitute
//    tbuf[0]  = 0;
      tbuf_len = 0;
      for( ; (iidx < ilen); iidx++ ) {
         if ( ibuf[iidx] == 0 )
            break;
         if ( strchr(P.seps, ibuf[iidx]) != NULL )
            break;
         if ( tbuf_len == MAX_TOKEN_SIZE ) {
            set_errhdl( ERR_INP_MAXTOKBUF );
            tbuf[tbuf_len-1] = 0;
            goto x_exit;
         }
         tbuf[tbuf_len++] = ibuf[iidx];
      }
      tbuf[tbuf_len] = 0;

      // default: let input goto output asis
      ptr     = tbuf;
      ptr_len = tbuf_len;

      // test for built-in reference
      if ( is_builtin(tbuf) ) {
         iidx = s_iidx;
         continue;
      }

      // determine the translation
      psym = symtab.find( &tbuf[1] );
      if ( psym == NULL ) {
         // tests special system functional variables
         if ( strcmpi(tbuf, "$NOW") == 0 ) {
            sprintf( tbuf, "0x%08lX", time_sec() );
            ptr     = tbuf;
            ptr_len = strlen( tbuf );
         }
      }
      if ( psym != NULL ) {
         switch( psym->type ) {
            case ST_TYPE_INT:
               sprintf( tbuf, "%d", (*(INT_2 *) psym->value) );
               ptr     = tbuf;
               ptr_len = strlen( ptr );
               break;
            case ST_TYPE_INT32:
               sprintf( tbuf, "%ld", (*(INT_4 *) psym->value) );
               ptr     = tbuf;
               ptr_len = strlen( ptr );
               break;
            case ST_TYPE_CHAR:
               quotation = '\"';
               ptr       = (CHAR *) psym->value;
               ptr_len   = psym->value_len;
               if ( (iidx == ilen) || (ibuf[iidx] == 0) )
                  break;
               // test for array reference
               i = iidx;
               while ( ibuf[i] == ' ' )
                  i++;
               if ( ibuf[i] != '[' )
                  break;
               // 
               // a array reference parsing problem:
               //   example:  $var1 [ $var2 ] 
               // so, wait for var2 translation first
               //
               i++;
               if ( !toINT(&value, &ibuf[i], "]") ) {
                  clr_errhdl();
                  quotation = 0;
                  retry     = TRUE;
                  ptr       = tbuf;
                  ptr_len   = tbuf_len;
                  break;
               }
               while ( ibuf[i] != ']' ) {
                  if ( (i >= ilen) || (ibuf[i] == 0) ) {
                     set_errhdl( ERR_INP_PARSE1 );
                     goto x_exit;
                  }
                  i++;
               }
               iidx      = i+1;
               quotation = '\'';
               ptr      += value;
               ptr_len   = 1;
               break;
            case ST_TYPE_FUNCT:
            default:
               // all others not interpreted
               break;
         }
      }

      // transfer to output string
      if ( (uquote) && (quotation != 0) )
         obuf[oidx++] = quotation;
      for( i=0; (i < ptr_len); i++ ) {
         if ( *ptr == 0 )
            break;
         obuf[oidx++] = *ptr++;
      }
      if ( (uquote) && (quotation != 0) )
         obuf[oidx++] = quotation;

   } /* end while */

   obuf[oidx] = 0;
   if ( ibuf_free ) {
      ibuf_free = FALSE;
      if ( ibuf != NULL )
         ::free( ibuf );
      ibuf = NULL;
   }
   if ( retry ) {
      ibuf      = NULL;
      ibuf_free = TRUE;
      if ( WC::strxfer(ibuf, obuf) != 0 )
         goto x_exit;
      goto x_retry;
   }
   ptr    = obuf;
   result = TRUE;

x_done:

   // setup parser
   if ( P.command(ptr) != 0 )
      return( FALSE );

x_exit:

   // free ibuf
   if ( ibuf_free ) {
      if ( ibuf != NULL )
         ::free( ibuf );
      ibuf = NULL;
   }

   // done
   return( result );

 #undef   obuf
} /* varsubst */


/*+
 *      NAME : setup
 *
 *  FUNCTION : decides source input direction, obtains input
 *             and evaluates interpretations
 *
 *     ENTRY : eval -> flag to answer: should input buffer be evaluted
 *                     or passed-thru an uninterpreted mode
 *
 *   RETURNS : 
 *
-*/
BOOLEAN INPUT_TYP::setup( CHAR *prompt, BOOLEAN eval )
{
 #define ibuf     smallbuf
 STREAM_TYP      *stream;
 CHAR            *pcmd;
 INT_2            pcmd_len    = 0;
 INT_4            value       = 0;

   // setup input information
   if ( bypass ) {
      bypass = FALSE;
      return( TRUE );
   }
   ibuf[0] = 0;

   // setup index
   stream = stream_ptr();
   if ( stream == NULL )
      return( FALSE );

   // obtain input from keyboard
   if ( (stream->fp == NULL)  && (stream->data == NULL) ) {
      cgets( ibuf, prompt );
      pcmd = ibuf;
      goto x_done;
   }

   // obtain input from designated macro
   if ( stream->data != NULL ) {
      pcmd = stream->cptr;
      if ( *pcmd == 0 ) {
         if ( stream->inc != NULL ) {
            if ( stream->inc_len >= 0 ) {
               push( stream->inc, stream->inc_len );
               stream->inc_len *= (-1);
               return( FALSE );
            }
            stream->inc_len *= (-1);
         }
         if (
              (stream->cond == NULL)          ||
              (!varsubst(stream->cond, TRUE)) ||
              (!eval_expr(&value, ";{"))      ||
              (value == 0)
            ) {
            pop();
            return( FALSE );
         }
         pcmd         = 
         stream->cptr = stream->data;
      }
      pcmd_len      = strlen( pcmd );
      stream->cptr += (pcmd_len+1);
      goto x_done;
   } /* endif */

   // otherwise obtain input from file
   fgets( ibuf, sizeof(ibuf), stream->fp );
   pcmd     = ibuf;
   pcmd_len = strlen(pcmd);
   if ( pcmd_len >= 1 )
      pcmd[pcmd_len-1] = 0;

   // resolution of general eof problem
   if ( feof(stream->fp) ) {
      pop();
      return( FALSE );
   }

x_done:

   // translate & done 
   return(  varsubst(pcmd,eval)  );

 #undef ibuf
} /* setup */


/*+
 *      NAME : make
 *
 *  FUNCTION : the essential INPUT class constructor
 *
 *     ENTRY : none
 *
 *   RETURNS : none
 *
-*/
VOID INPUT_TYP::make( VOID )
{

   // initial base
   bypass    = FALSE;
   input_len = 0;
   input     = NULL;

   // push the standard input
   push();

   // done
   return;

} /* make */


/*+
 *      NAME : unmake
 *
 *  FUNCTION : the essential INPUT class destructor
 *
 *     ENTRY : none
 *
 *   RETURNS : none
 *
-*/
VOID INPUT_TYP::unmake( VOID )
{
 INT_2          idx;
 STREAM_TYP    *stream;

   // close all input streams
   if ( input != NULL ) {
      for( idx=0; (idx < input_len); idx++ ) {
         stream = (*input)[idx];
         if ( stream == NULL )
            continue;
         if ( stream->fp != NULL ) {
            fclose( stream->fp );
         } else if ( stream->data != NULL ) {
            ::free( stream->data );
         }
         ::free( stream );
         stream = NULL;
      }
   }

   // release info
   if ( input != NULL )
      ::free( input );
   input     = NULL;
   input_len = 0;

   // done
   return;

} /* unmake */


/*-------------------------------------------------------------------------*/


