/*   ext2dir.cc : February 1, 1994    */

/* Copyright (C) 1994-1999  Sekhar Bala, Rama Bala, and
 *                          Alphax System, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "std.h"
#include "wildcard.h"
#include "getopt.h"

#include "dump.h"
#include "diskio.h"
#include "ext2fs.h"

#include "ilib.h"


/*-------------------------------------------------------------------------*/


STATIC INT_2 process_find( VOID )
{
 INT_4   j;
 INT_4   cnt;
 INT_4   total       = 0;
 INT_2   i           = 0;
 INT_2   retval      = 11;

   // setup
   blocksize = ( EXT2_BASE_BLOCK << sb->s_log_block_size );
   if ( verbose )
      outvfp( "\nblocksize is %d\n", blocksize );

   // find max inodes of directories
   for( j=0; (j < groups_len); j++ ) {
      total += groups[j].bg_used_dirs_count;
   }
   if ( total >= MAX_LINKS ) {
      outvfp( "ERROR: Total directories %ld exceeds %ld reserved slots\n",
               total, (INT_4) MAX_LINKS );
      set_errhdl( ERR_SYS_NOINIT );
      goto x_done;
   }
   outvfp( "\nTotal number of directories: %ld\n", total );

   // setup
   pblock = (CHAR *) ::malloc( blocksize );
   if ( pblock == NULL ) {
      set_errhdl( ERR_SYS_NOMEM );
      goto x_done;
   }
   pbitblock = (CHAR *) ::malloc( blocksize );
   if ( pbitblock == NULL ) {
      set_errhdl( ERR_SYS_NOMEM );
      goto x_done;
   }

   // get/resolve dir-info by loading from disk
   if ( verbose )
      outvfp( "\nSetting up links...\n" );
   if ( !setup_links() )
      goto x_done;
   if ( verbose )
      outvfp( "Resloving links...\n" );
   if ( !resolve_links() )
      goto x_done;
   if ( verbose )
      outvfp( "Writing Data File...\n" );
   if ( write_file(fname) != 0 )
      goto x_done;

   // dump summary
   cnt = 0;
   outvfp( "\nThe unlinked inodes are: " );
   for( i=0; (i < links_len); i++ ) {
      if ( links[i].parent != -1 )
         continue;
      if ( i == 0 )
         outvfp( "\n" );
      outvfp( "Inode %8ld ", links[i].inode_num );
      outvfp( "Status %04X", links[i].flags     );
      outvfp( "\n" );
      cnt++;
   }
   if ( cnt == 0 )
      outvfp( "none\n\n" );

   // done
   retval = 0;

x_done:

   return( retval );

} /* process_find */


STATIC INT_2 execute_find( INT_2 disk, INT_2 partition )
{
 INT_2   retval;
 FILE   *vfp;

   // init
   retval = 10;
   vfp    = log;
   if ( log == NULL )
      vfp = stdout;

   // initialize
   if ( (init_LIBIO(disk) != 0) || (errs[0] != 0) )
      goto x_done;

   if ( verbose ) {
      outvfp("\n");
      dumpdisk(vfp);
      outvfp("\n");
   }

   // initialize
   g_p_idx = (partition - 1);
   init_FS();
   if ( errs[0] != 0 )
      goto x_done;

   if ( verbose ) {
      dumppart( TRUE, vfp );
      outvfp("\n");
   }

   // load superblock and group
   loadsb_FS();
   if ( errs[0] != 0 )
      goto x_done;
   loadgroups_FS();
   if ( errs[0] != 0 )
      goto x_done;

   retval = process_find();

x_done:

   // dump any found errors
   if ( errs[0] != 0 )
      errhdl( "ERROR", vfp );

   // cleanup
   clean_FS();
   clean_LIBIO();
   if ( log != NULL ) {
      fclose( log );
      log = NULL;
   }

   // done
   return( retval );

} /* execute_find */


STATIC INT_2 execute_test( VOID )
{
 INT_2     i;
 INT_2     j;
 INT_2     cnt;
 INT_2     retval = 13;

   // load data from file
   if ( read_file(fname) != 0 ) {
    FILE *vfp = stdout;

      if ( log != NULL )
         vfp = log;
      errhdl( "ERROR", vfp );
      goto x_done;
   }

   // search
   j   = -1;
   cnt = 0;
   outvfp( "\n" );
   for( i=0; (i < links_len); i++ ) {
      if ( links[i].inode_num == 2 )
         j = i;
      if ( links[i].parent != -1 )
         continue;
      outvfp( "%4d) ",        i );
      outvfp( "Inode %7ld ",  links[i].inode_num );
      outvfp( "Status %04X ", links[i].flags     );
      outvfp( "\n" );
      cnt++;
   }
   if ( cnt != 0 )
      outvfp( "\n" );
   outvfp("There were %d missing links out of %d\n\n", cnt, links_len);

   // dump root
   if ( j == -1 ) {
      outvfp( "Root (childen) not found!\n" );
   } else {
      outvfp( "Root is:\n" );
      dump_links( j );
      outvfp( "\n" );
   }

   // done
   retval = 0;

x_done:

   if ( log != NULL ) {
      fclose( log );
      log = NULL;
   }
   return( retval );

} /* execute_test */


STATIC VOID usage( VOID )
{

   printf( "\n" );
   printf( "ext2dir -f [opts] disk-# partition-#\n" );
   printf( "  find inodes and create inode data file\n" );
   printf( "    -x          -> skip_bitblock testing\n" );
   printf( "    -v          -> be verbose\n" );
   printf( "    -f fname    -> use 'fname' as inode data file (inode.dat)\n" );
   printf( "    -l logfname -> use 'logfname' as log file\n" );
   printf( "\n" );
   printf( "ext2dir -l [opts]\n" );
   printf( "  list information from inode data file\n" );
   printf( "    -v          -> be verbose\n" );
   printf( "    -f fname    -> use 'fname' as inode data file (inode.dat)\n" );
   printf( "    -l logfname -> use 'logfname' as log file\n" );
   printf( "\n" );
   printf( "ext2dir -t [opts]\n" );
   printf( "  test for missing directory links from inode data file\n" );
   printf( "    -v          -> be verbose\n" );
   printf( "    -f fname    -> use 'fname' as inode data file (inode.dat)\n" );
   printf( "    -l logfname -> use 'logfname' as log file\n" );
   printf( "\n" );
   return;

} /* usage */


main( int argc, char *argv[] )
{
 INT_2   retval;
 INT_2   command;
 CHAR   *arg;
 CHAR   *endp;
 INT_2   disk;
 INT_2   partition;

   printf( "\n              "
           "ext2DIR V1.00 - Copyright (c) 1994-1999, Sekhar Bala,"
           "\n              "
           "                Ramchander Balasubramanian,"
           "\n              "
           "                and Alphax Systems, Inc."
           "\n              "
           "-----------------------------------------------------"
           "\n\n" );

   // startup cmdlin processing with validation of first option
   command = getopt( argc, argv, "flt" );
   if ( (command == -1) || (command == '?') ) {
      usage();
      return( 1 );
   }

   // get options for all commands
   while (TRUE) {
      retval = getopt( argc, argv, "vxfl" );
      if ( retval == -1 )
         break;
      if ( (command == 'f') && (retval == 'x') ) {
         skip_bitblock = TRUE;
         continue;
      }
      if ( retval == 'v' ) {
         verbose = TRUE;
         continue;
      }
      if ( retval == 'f' ) {
         arg = argopt( argc, argv );
        if ( arg == NULL ) {
           printf( "option 'f' requires a inode-data-fname\n" );
           usage();
           return( 2 );
        }
        strcpy( fname, arg );
        continue;
      }
      if ( retval == 'l' ) {
         arg = argopt( argc, argv );
         if ( arg == NULL ) {
            printf( "option 'l' requires a logfname\n" );
            usage();
            return( 3 );
         }
         log = fopen( arg, "w" );
         if ( log == NULL )
            printf( "Logfile '%s' cannot be opened!\n", arg );
         continue;
      }
      printf( "unknown option '%c'\n", (CHAR) retval );
      usage();
      return( 4 );
   }

   // process 'f' command
   if ( command == 'f' ) {
      // get arguments
      if ( optind >= argc ) {
         printf( "missing disk-#\n" );
         usage();
         return( 5 );
      }
      disk = (INT_2) strtoul( argv[optind], &endp, 0 );
      if ( *endp != 0 ) {
         printf( "disk-# can be 0,1 for floppies 128-x for /dev/hda...\n" );
         return( 5 );
      }
      optind++;
      if ( optind >= argc ) {
         printf( "missing partition-#\n" );
         usage();
         return( 5 );
      }
      partition = (INT_2) strtoul( argv[optind], &endp, 0 );
      if ( (*endp != 0) || (partition == 0) ) {
         printf( "bad partition-# disk %d; must be 1..n\n", disk );
         return( 5 );
      }
      return(  execute_find(disk,partition)  );
   }

   // process 'l' command
   if ( command == 'l' ) {
      if ( read_file(fname) != 0 ) {
         errhdl( "ERROR" );
         return( 0 );
      }
      outvfp( "\n" );
      for( int i=0; (i < links_len); i++ ) {
         dump_links( i );
         outvfp( "\n" );
      }
      if ( log != NULL )
         fclose( log );
      return( 0 );
   }

   // process 't' command
   if ( command == 't' ) {
      return(  execute_test()  );
   }

   // done
   printf( "Unknown command option '%c'\n", (CHAR) command );
   usage();
   return( 1 );

} /* main */


/*-------------------------------------------------------------------------*/



