/*   ext2fs.h : February 1, 1994    */

/* Copyright (C) 1994-1999  Sekhar Bala, Rama Bala, and
 *                          Alphax System, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __EXT2FS_H
#define __EXT2FS_H

/*------------------------------------------------------------------------*/

typedef struct superblock_typ {
   INT_U4      s_inodes_count;         /* Inodes count                     */
   INT_U4      s_blocks_count;         /* Blocks count                     */
   INT_U4      s_r_blocks_count;       /* Reserved blocks count            */
   INT_U4      s_free_blocks_count;    /* Free blocks count                */
   INT_U4      s_free_inodes_count;    /* Free inodes count                */
   INT_U4      s_first_data_block;     /* First Data Block                 */
   INT_U4      s_log_block_size;       /* Block size                       */
   INT_4       s_log_frag_size;        /* Fragment size                    */
   INT_U4      s_blocks_per_group;     /* # Blocks per group               */
   INT_U4      s_frags_per_group;      /* # Fragments per group            */
   INT_U4      s_inodes_per_group;     /* # Inodes per group               */
   INT_4       s_mtime;                /* Mount time                       */
   INT_4       s_wtime;                /* Write time                       */
   INT_U2      s_mnt_count;            /* Mount count                      */
   INT_2       s_max_mnt_count;        /* Maximal mount count              */
   INT_U2      s_magic;                /* Magic signature                  */
   INT_U2      s_state;                /* File system state                */
   INT_U2      s_errors;               /* Behaviour when detecting errors  */
   INT_U2      s_pad;
   INT_4       s_lastcheck;            /* time of last check               */
   INT_U4      s_checkinterval;        /* max. time between checks         */
   INT_U4      s_creator_os;           /* OS                               */
   INT_U4      s_rev_level;            /* Revision level                   */
   INT_U2      s_def_resuid;           /* Default uid for reserved blocks  */
   INT_U2      s_def_resgid;           /* Default gid for reserved blocks  */
   INT_U4      s_reserved[235];        /* Padding to the end of the block  */
} SUPERBLOCK_TYP;

typedef struct groupdesc_typ {
   INT_U4      bg_block_bitmap;        /* Blocks bitmap block              */
   INT_U4      bg_inode_bitmap;        /* Inodes bitmap block              */
   INT_U4      bg_inode_table;         /* Inodes table block               */
   INT_U2      bg_free_blocks_count;   /* Free blocks count                */
   INT_U2      bg_free_inodes_count;   /* Free inodes count                */
   INT_U2      bg_used_dirs_count;     /* Directories count                */
   INT_U2      bg_pad;
   INT_U4      bg_reserved[3];
} GROUPDESC_TYP;

typedef struct inode_typ {
   INT_U2      i_mode;                 /* File mode                        */
   INT_U2      i_uid;                  /* Owner Uid                        */
   INT_U4      i_size;                 /* Size in bytes                    */
   INT_4       i_atime;                /* Access time                      */
   INT_4       i_ctime;                /* Creation time                    */
   INT_4       i_mtime;                /* Modification time                */
   INT_4       i_dtime;                /* Deletion Time                    */
   INT_U2      i_gid;                  /* Group Id                         */
   INT_U2      i_links_count;          /* Links count                      */
   INT_U4      i_blocks;               /* Blocks count                     */
   INT_U4      i_flags;                /* File flags                       */
   INT_U4      i_reserved1;            /* Reserved 1                       */
   INT_U4      i_block[15];            /* Pointers to blocks               */
   INT_U4      i_version;              /* File version (for NFS)           */
   INT_U4      i_file_acl;             /* File ACL                         */
   INT_U4      i_dir_acl;              /* Directory ACL                    */
   INT_U4      i_faddr;                /* Fragment address                 */
   UCHAR       i_frag;                 /* Fragment number                  */
   UCHAR       i_fsize;                /* Fragment size                    */
   INT_U4      i_reserved2[2];         /* Reserved 2                       */
} INODE_TYP;

typedef struct badinode_typ {
   INT_4       inode_num;
   INODE_TYP   inode;
} BADINODE_TYP;

typedef struct entry_typ {
   INT_U4       inode_num;
   INT_U2       rec_len;
   INT_U2       name_len;
   CHAR         name[256];
} ENTRY_TYP;

typedef struct dir_typ {
   INT_4              inode_num;    // inode of directory entry
   INODE_TYP          inode;        // acutal inode
   CHAR              *name;         // between 0 and 256 chars
   struct dir_typ    *next;         // next dir ptr
} DIR_TYP;

// version
#define EXT2FS_DATE              "94/10/23"
#define EXT2FS_VERSION           "0.5a"

// EXT2 FS Reserved Inodes (1 thru 10)
#define FSINODE_BADBLOCK         1       // bad block inode #
#define FSINODE_ROOT             2       // root filesystem inode #
#define FSINODE_ACLIDX           3       // access control list inode #
#define FSINODE_ACLDATA          4       // acl data inode #
#define FSINODE_BOOT             5       // boot loader inode #
#define FSINODE_UNDEL            6       // undelete inode #
#define FSINODE_RES1             7
#define FSINODE_RES2             8
#define FSINODE_RES3             9
#define FSINODE_RES4            10
#define FSINODE_FIRST           11

// magic numbers
#define EXT2_PRE_02B_MAGIC       0xEF51
#define EXT2_SUPER_MAGIC         0xEF53

// maximum count of links to a file
#define EXT2_LINK_MAX            32000

// inode flags
#define EXT2_SECRM_FL         0x00000001 /* Secure deletion */
#define EXT2_UNRM_FL          0x00000002 /* Undelete */
#define EXT2_COMPR_FL         0x00000004 /* Compress file */
#define EXT2_SYNC_FL          0x00000008 /* Synchronous updates */
#define EXT2_IMMUTABLE_FL     0x00000010 /* Immutable file */
#define EXT2_APPEND_FL        0x00000020 /* writes to file may only append */
#define EXT2_NODUMP_FL        0x00000040 /* do not dump file */

// mount flags 
#define EXT2_MOUNT_CHECK_NORMAL     0x0001   /* Do some more checks */
#define EXT2_MOUNT_CHECK_STRICT     0x0002   /* Do again more checks */
#define EXT2_MOUNT_CHECK           (EXT2_MOUNT_CHECK_NORMAL | EXT2_MOUNT_CHECK_STRICT)
#define EXT2_MOUNT_GRPID            0x0004   /* Create files with directory's group */
#define EXT2_MOUNT_DEBUG            0x0008   /* Some debugging messages */
#define EXT2_MOUNT_ERRORS_CONT      0x0010   /* Continue on errors */
#define EXT2_MOUNT_ERRORS_RO        0x0020   /* Remount fs ro on errors */
#define EXT2_MOUNT_ERRORS_PANIC     0x0040   /* Panic on errors */
#define EXT2_MOUNT_MINIX_DF         0x0080   /* Mimics the Minix statfs */

#define clear_opt(o, opt)     o &= ~EXT2_MOUNT_##opt
#define set_opt(o, opt)       o |= EXT2_MOUNT_##opt
#define test_opt(sb, opt)     ((sb)->u.ext2_sb.s_mount_opt & EXT2_MOUNT_##opt)

// maximal mount counts between two filesystem checks
#define EXT2_DFL_MAX_MNT_COUNT      20 /* Allow 20 mounts */
#define EXT2_DFL_CHECKINTERVAL      0  /* Don't use interval check */

// behaviour when detecting errors
#define EXT2_ERRORS_CONTINUE        1  /* Continue execution */
#define EXT2_ERRORS_RO              2  /* Remount fs read-only */
#define EXT2_ERRORS_PANIC           3  /* Panic */
#define EXT2_ERRORS_DEFAULT         EXT2_ERRORS_CONTINUE

// i_mode Types
#define IMODE_FIFO            0x01
#define IMODE_CHARDEV         0x02
#define IMODE_DIR             0x04
#define IMODE_BLOCKDEV        0x06
#define IMODE_FILE            0x08
#define IMODE_SLINK           0x0A
#define IMODE_SOCKET          0x0C

// i_mode Macros
#define _MKMODE(m)            (((m) >> 12) & 0x000F)
#define _MKID(m)              ((m) & 0x01FF)
#define _ISDIR(m)             ( _MKMODE(m) == IMODE_DIR )
#define _ISLINK(m)            ( _MKMODE(m) == IMODE_SLINK )

#define S_IFMT       00170000
#define S_IFSOCK     0140000
#define S_IFLNK      0120000
#define S_IFREG      0100000
#define S_IFBLK      0060000
#define S_IFDIR      0040000
#define S_IFCHR      0020000
#define S_IFIFO      0010000
#define S_ISUID      0004000
#define S_ISGID      0002000
#define S_ISVTX      0001000

#define S_ISLNK(m)   (((m) & S_IFMT) == S_IFLNK)
#define S_ISREG(m)   (((m) & S_IFMT) == S_IFREG)
#define S_ISDIR(m)   (((m) & S_IFMT) == S_IFDIR)
#define S_ISCHR(m)   (((m) & S_IFMT) == S_IFCHR)
#define S_ISBLK(m)   (((m) & S_IFMT) == S_IFBLK)
#define S_ISFIFO(m)  (((m) & S_IFMT) == S_IFIFO)
#define S_ISSOCK(m)  (((m) & S_IFMT) == S_IFSOCK)

#define S_IRWXU 00700
#define S_IRUSR 00400
#define S_IWUSR 00200
#define S_IXUSR 00100

#define S_IRWXG 00070
#define S_IRGRP 00040
#define S_IWGRP 00020
#define S_IXGRP 00010

#define S_IRWXO 00007
#define S_IROTH 00004
#define S_IWOTH 00002
#define S_IXOTH 00001

// iblock elements
#define EXT2_IBLOCK_NR        12
#define EXT2_IBLOCK_SINGLE    12
#define EXT2_IBLOCK_DOUBLE    13
#define EXT2_IBLOCK_TRIPLE    14

#define _IBLOCK(w)            (EXT2_IBLOCK ##w - 12)

#define EXT2_BASE_BLOCK       1024
#define EXT2_BASE_FRAGMENT    1024

EXTERN BOOLEAN             g_verbose_ext2fs;

EXTERN SUPERBLOCK_TYP     *sb;

EXTERN INT_4               groups_len;
EXTERN GROUPDESC_TYP      *groups;

EXTERN INT_4               inode_no;
EXTERN INODE_TYP           inode;

EXTERN DIR_TYP            *dir;

#define MAX_BADINODES      128

EXTERN INT_2               badinode_len;
EXTERN BADINODE_TYP        badinode[MAX_BADINODES];

/*------------------------------------------------------------------------*/
                              /* ext2fs.c */

EXTERN INT_2 init_FS          ( VOID );
EXTERN VOID  clean_FS         ( VOID );

EXTERN INT_2 loadsb_FS        ( INT_4 loc=(1024L*1) );
EXTERN INT_2 loadgroups_FS    ( INT_4 len_fake=-1, INT_4 loc=(1024L*2) );

EXTERN INT_2 readinode_FS     ( INT_4 ino, INODE_TYP *i );
EXTERN INT_2 loadinode_FS     ( INT_4 ino );

EXTERN INT_2 readblock_FS     ( INODE_TYP *i, INT_4 *block, INT_4 *length );
EXTERN INT_2 getblock_FS      ( INT_4 *block=NULL, INT_4 *length=NULL );

EXTERN INT_2 readdir_FS       ( INODE_TYP *i, DIR_TYP **pdir );
EXTERN INT_2 loaddir_FS       ( VOID );

/*------------------------------------------------------------------------*/
#endif


