/*   dump.cc : February 1, 1994    */

/* Copyright (C) 1994-1999  Sekhar Bala, Rama Bala, and
 *                          Alphax System, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include <time.h>

#include "std.h"
#include "diskio.h"
#include "ext2fs.h"
#include "dump.h"

/*------------------------------------------------------------------------*/

#ifdef GO32
   #define GETC()      getchar()
#else
   #define GETC()      getc(stdin)
#endif

STATIC TAGDESC_TYP   pdesc[] = {
  { PART_INVALID,      "Invalid"                                     },
  { PART_EXTENDED,     "Media Extended"                              },
  { PART_MSDOS,        "MSDOS"                                       },
  { PART_LINUXSWAP,    "Linux Swap"                                  },
  { PART_EXT2,         "Linux Ext2"                                  },
  { PART_WIN95SWAP,    "WIN95 Swap"                                  },
  { 0,                 NULL                                          }
};
#define pdesc_len ( (sizeof(pdesc)/sizeof(pdesc[0])) - 1 )

STATIC CHAR   *mdesc[] = {         // octal crap
  { "Unknown"          },          // 00
  { "FIFO"             },          // 01
  { "Character Device" },          // 02
  { "unknown"          },          // 03
  { "Directory"        },          // 04
  { "unknown"          },          // 05
  { "Block Device"     },          // 06
  { "unknown"          },          // 07
  { "Regular File"     },          // 10
  { "unknown"          },          // 11
  { "Symbolic Link"    },          // 12
  { "unknown"          },          // 13
  { "Socket"           },          // 14
  { "unknown"          },          // 15
  { "unknown"          },          // 16
  { "unknown"          }           // 17
};
#define mdesc_len ( sizeof(mdesc)/sizeof(mdesc[0]) )

#ifdef GO32
   #define CTIME(x)       ctime( (unsigned int *) x )
#else
   #define CTIME(x)       ctime( x )
#endif

/*------------------------------------------------------------------------*/


GLOBAL CHAR *partdesc( INT_U2 parttype )
{
 INT_2    i;
 CHAR    *ptext;

   ptext = "Unknown";
   for( i=0; (i < (INT_2) pdesc_len); i++ ) {
      if ( parttype == pdesc[i].value ) {
         ptext = pdesc[i].desc;
         break;
      }
   }
   return( ptext );

} /* partdesc */


GLOBAL CHAR *modedesc( INT_U2 i_mode )
{
 INT_2    i;
 CHAR    *ptext;

   ptext = "Unknown";
   i     = _MKMODE( i_mode );
   if ( (i >= 0) && (i < (INT_2) mdesc_len) )
      ptext = mdesc[i];
   return( ptext );

} /* modedesc */


GLOBAL VOID dumpdata(
   VOID    *pbuf,
   INT_4    pbuf_len,
   INT_U4   sector,
   FILE    *fp,
   INT_2    maxlen  )
{
 INT_4      i;
 INT_2      cnt;
 INT_2      xch;
 CHAR      *ptr;
 CHAR       hex[18];

   i   = 0;
   cnt = 0;
   ptr = (CHAR *) pbuf;
   memset( hex, 0, sizeof(hex) );

   fprintf( fp, "--------------------------------------------------\n" );
   if ( sector != 0 )
      fprintf( fp, "Block %ld (0x%lX): \n", sector, sector );

   fprintf( fp, "%04lX: ", i );
   for( i=0; (i < pbuf_len); i++ ) {
      if ( i == maxlen )
         break;
      if ( ((i != 0) && ((i % 256)==0) ) && (fp == stdout) ) {
         printf(" -- Pause (any key) -- ? ");
         xch = GETC();
         printf("\n");
         fprintf( fp, "%04lX: ", i-1 );
         if (
              (xch == 'q') || (xch == 'Q') ||
              (xch == 'e') || (xch == 'E') ||
              (xch == 'x') || (xch == 'X') ||
              (xch == 0x1b)
            ) {
            cnt = 0;
            break;
         }
         if ( xch == 0 )
            GETC();
      }
      hex[cnt] = '.';
      if ( (*ptr >= ' ') && (*ptr <= '~') )
         hex[cnt] = *ptr;
      fprintf( fp, "%02x ", __INT_2(*ptr) );
      ptr++;
      if ( ++cnt == 16 ) {
         fprintf( fp, " - %s\n", hex );
         fprintf( fp, "%04lX: ", i );
         memset( hex, 0, sizeof(hex) );
         cnt = 0;
      }
   }

   if ( cnt != 0 ) {
      for( i=cnt; (i < 16); i++ )
         fprintf( fp, "   " );
      fprintf( fp, " - %s", hex );
   }
   fprintf( fp, "\n" );

   fprintf( fp, "--------------------------------------------------\n" );

   return;

} /* dumpdata */


GLOBAL VOID dumpdisk( FILE *fp )
{
#ifndef GO32
 INT_2  disk;
 INT_2  dev_len;
 CHAR   device[20];

   disk = g_disk;
   strcpy( device, "/dev/fd0" );
   if ( disk & 0x80 ) {
      disk &= 0x07;
      strcpy( device, g_devname );
   }
   dev_len            = strlen( device );
   device[dev_len-1] += (char) disk;
   fprintf( fp, "      Disk: %s (%u)\n",      device, g_disk );
#else
   fprintf( fp, "      Disk: %u\n",           g_disk );
   fprintf( fp, "     Heads: %u (0x%04X)\n",  g_heads, g_heads );
   fprintf( fp, "   Sectors: %u (0x%04X)\n",  g_sectors, g_sectors );
   fprintf( fp, " Cylinders: %u (0x%04X)\n",  g_cylinders, g_cylinders );
#endif
   return;

} /* dumpdisk */


GLOBAL VOID dumppart( BOOLEAN hex, FILE *fp )
{
 INT_2   i;
 CHAR   *ptype;

   fprintf( fp, " IDX  A  TYPE  DESC                  START  LENGTH(MB)\n" );
   fprintf( fp, "------------------------------------------------------\n" );

   for( i=0; (i < g_p_len); i++ ) {

      if ( g_p[i].type == PART_INVALID )
         continue;

      if ( i == g_p_idx )
         fprintf( fp, "*" );
      else
         fprintf( fp, " " );

      fprintf( fp, "%3d  ", i+1 );
      if ( g_p[i].flags & PARTF_ACTIVE )
         fprintf( fp, "A  " );
      else
         fprintf( fp, "   " );

      fprintf( fp, "0x%02X  ", g_p[i].type );
      ptype = partdesc( g_p[i].type );
      fprintf( fp, "%-15.15s  ", ptype );

      if ( hex ) {
         fprintf( fp, "%10lX  ", g_p[i].start );
         fprintf( fp, "%10lX  ", ((g_p[i].length*512L)/1024L) );
      } else {
         fprintf( fp, "%10ld  ", g_p[i].start );
         fprintf( fp, "%10ld  ", ((g_p[i].length*512L)/1024L) );
      }

      fprintf( fp, "\n" );

   } /* end for */

   return;

} /* dumppart */


GLOBAL VOID dumpsb( SUPERBLOCK_TYP *s, BOOLEAN brief, FILE *fp )
{

   if ( s == NULL ) {
      fprintf( fp, "Superblock not loaded!\n" );
      return;
   }
   fprintf( fp, "-----------------------------------------------\n" );
   fprintf( fp, "         Inode Count: %lu\n", s->s_inodes_count );
   fprintf( fp, "         Block Count: %lu\n", s->s_blocks_count );
   fprintf( fp, "Reserved Block Count: %lu\n", s->s_r_blocks_count );
   fprintf( fp, "         Free Blocks: %lu\n", s->s_free_blocks_count );
   fprintf( fp, "         Free Inodes: %lu\n", s->s_free_inodes_count );
   fprintf( fp, "    First Data Block: %lu\n", s->s_first_data_block );
   fprintf( fp, "      Log Block Size: %lu\n", s->s_log_block_size );
   fprintf( fp, "       Log Frag Size: %ld\n", s->s_log_frag_size );
   fprintf( fp, "    Blocks per Group: %lu\n", s->s_blocks_per_group );
   fprintf( fp, " Fragments per Group: %lu\n", s->s_frags_per_group );
   fprintf( fp, "    Inodes per Group: %lu\n", s->s_inodes_per_group );
   if ( !brief ) {
      fprintf( fp, "          Mount Time: %s",    CTIME(&(s->s_mtime)) );
      fprintf( fp, "          Write Time: %s",    CTIME(&(s->s_wtime)) );
   }
   fprintf( fp, "         Mount Count: %u\n",  s->s_mnt_count );
   fprintf( fp, "     Max Mount Count: %d\n",  s->s_max_mnt_count );
   fprintf( fp, "        Magic Number: %X\n",  s->s_magic );
   fprintf( fp, "   File System State: %X\n",  s->s_state );
   fprintf( fp, "     Error Behaviour: %X\n",  s->s_errors );
   fprintf( fp, "           Pad Value: %u\n",  s->s_pad );
   fprintf( fp, "          Last Check: %s",    CTIME(&(s->s_lastcheck)) );
   fprintf( fp, "      Check Interval: %lu\n", s->s_checkinterval );
   fprintf( fp, "          Creator OS: %lu\n", s->s_creator_os );
   fprintf( fp, "      Revision Level: %lu\n", s->s_rev_level );
   if ( !brief ) {
      fprintf( fp, "         Default UID: %u\n",  s->s_def_resuid );
      fprintf( fp, "         Default GID: %u\n",  s->s_def_resgid );
   }
   fprintf( fp, "-----------------------------------------------\n" );
   return;

} /* dumpsb */


GLOBAL VOID dumpgroups(
   GROUPDESC_TYP  *g,
   INT_4           g_len,
   INT_4           g_idx,
   FILE           *fp )
{
 INT_4   i;

   if ( g == NULL ) {
      fprintf( fp, "Group Descriptions are not loaded!\n" );
      return;
   }
   fprintf( fp, "----------------------------\n" );
   fprintf( fp, "groups_len = %ld\n", g_len );
   for( i=0; (i < g_len); i++ ) {
      if ( (g_idx != -1) && (i != g_idx) )
         continue;
      fprintf( fp, "----------------------------\n" );
      fprintf( fp, "       Group Number: %ld\n", i );
      fprintf( fp, "Blocks Bitmap Block: %lu\n", g[i].bg_block_bitmap );
      fprintf( fp, "Inodes Bitmap Block: %lu\n", g[i].bg_inode_bitmap );
      fprintf( fp, " Inodes Table Block: %lu\n", g[i].bg_inode_table );
      fprintf( fp, "        Free Blocks: %u\n",  g[i].bg_free_blocks_count );
      fprintf( fp, "        Free Inodes: %u\n",  g[i].bg_free_inodes_count );
      fprintf( fp, "   Used Directories: %u\n",  g[i].bg_used_dirs_count );
   }
   fprintf( fp, "----------------------------\n" );
   return;

} /* dumpgroups */


GLOBAL VOID dumpinode( INT_4 ino, INODE_TYP *i, FILE *fp )
{
// INT_2   mtype;

   if ( (i == NULL) || (ino <= 0) ) {
      fprintf( fp, "Inode not loaded!\n" );
      return;
   }
   fprintf( fp, "     Inode Number: %ld\n",   ino );
   fprintf( fp, "        File Mode: o%06o",   i->i_mode );
   fprintf( fp,           " (%02o",           _MKMODE(i->i_mode) );
   fprintf( fp,           "- %s)\n",           modedesc(i->i_mode) );
   fprintf( fp, "        Owner UID: %u\n",    i->i_uid );
   fprintf( fp, "     Size (bytes): %lu\n",   i->i_size );
   fprintf( fp, "      Access Time: (%8ld) %s",
                i->i_atime, CTIME(&(i->i_atime)) );
   fprintf( fp, "    Creation Time: (%8ld) %s",
                i->i_ctime, CTIME(&(i->i_ctime)) );
   fprintf( fp, "Modification Time: (%8ld) %s",
                i->i_mtime, CTIME(&(i->i_mtime)) );
   fprintf( fp, "    Deletion Time: (%8ld) %s",
                i->i_dtime, CTIME(&(i->i_dtime)) );
   fprintf( fp, "       Owner GID: %u\n",    i->i_gid );
   fprintf( fp, "     Links Count: %u\n",    i->i_links_count );
   fprintf( fp, "    Blocks Count: %lu",     i->i_blocks );
   fprintf( fp,                       "  (512 bytes)\n" );
   fprintf( fp, "      File Flags: 0x%lX\n", i->i_flags );
   fprintf( fp, "    File Version: %lu\n",   i->i_version );
   fprintf( fp, "        File ACL: %lu\n",   i->i_file_acl );
   fprintf( fp, "   Directory ACL: %lu\n",   i->i_dir_acl );
   fprintf( fp, "Fragment Address: %lu\n",   i->i_faddr );
   fprintf( fp, " Fragment Number: %u\n",    i->i_frag );
   fprintf( fp, "   Fragment Size: %u\n",    i->i_fsize );
   return;

} /* dumpinode */


GLOBAL VOID dumpiblocks( INT_4 ino, INODE_TYP *i, FILE *fp )
{
 INT_2   j;
 INT_2   imode;

   if ( (i == NULL) || (ino <= 0) ) {
      fprintf( fp, "Inode not loaded!\n" );
      return;
   }
   imode = _MKMODE( i->i_mode );
   fprintf( fp, "Inode Number: %ld\n", ino );
   for( j=0; (j < EXT2_IBLOCK_TRIPLE+1); j++ ) {
      fprintf( fp, "%2d) x%08lX - ", j, i->i_block[j] );
      if ( (imode == IMODE_SLINK) && (j < EXT2_IBLOCK_NR) ) {
         fprintf( fp, "  '%-4.4s'\n", (CHAR *) &i->i_block[j] );
      } else {
         fprintf( fp, "%8lu\n", i->i_block[j] );
      }
//    fprintf( fp, "%2d) x%08lX - %8lu\n", j, i->i_block[j], i->i_block[j] );
   }
   if ( imode == IMODE_SLINK ) {
      j = (INT_2) ( EXT2_IBLOCK_NR * sizeof(INT_U4) );
      fprintf( fp, "--> %-*.*s\n", j, j, (CHAR *) &i->i_block[0] );
   }
   return;

} /* dumpiblocks */


/*------------------------------------------------------------------------*/

