/*
 *    ext2hide - a superblock reserved space file stashing utility
 *	by infi/2006
 *
 *    finds and uses reserved space in user's primary and backup superblocks
 *    under ext2/3 to store user-specified data
 *
 *    requires:
 *	libext2fs, getopt (not yet)
 *
 *    to compile:
 * 	gcc -lext2fs -O3 ext2hide.c -o ext2hide
 *
 *    usage:
 *	ext2hide -h
 *
 */

// needed for lseek64
#define _LARGEFILE64_SOURCE

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ext2fs/ext2fs.h>
#include <linux/unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "ext2hide.h"
#include "error.h"

// ew.  globals.
config cfg;

/*************** MAIN *****************/
int main(int argc, char **argv) {
	ext2_filsys fs;
	meta_info meta;
	int ret,fd;
	char *space;

	memset(&meta,0,sizeof(meta_info));
	memset(&cfg,0,sizeof(config));

	// check arguments, setup configuration
	if (((ret = check_args(argc,argv,&cfg)) != E_SBSUCCESS) ||
         (cfg.function == NOFUNC)) {
		usage(USAGELONG);
		exit(E_SBINVAL);
	}

	// print version information
	if (cfg.verbose >= VNORMAL)
		version();

	// check the library version, warn the user if < MINLIB
	cfg.libver = check_lib_ver(MINLIB);

	// open the filesystem
	if ((ret = fs_open(&fs,&cfg)) != E_SBSUCCESS)
		exit(ret);

	// self-explanatory :b
	print_fsinfo(fs,&cfg);

	// find the superblocks
	cfg.sbcount = find_superblocks(fs,cfg.sb);
	
	// get individual superblock information
	cfg.total = get_sbinfo(fs,&cfg);

	if (cfg.verbose >= VNORMAL)
		printf("%lu total bytes usable on %s.\n",cfg.total,fs->device_name);

	// allocate memory buffer and read data from usable space
	space = get_blocks(fs,&cfg);

	if (cfg.verbose >= VDEBUG)
		printf("space = 0x%.8x\n",space);

    ret = E_SBSUCCESS;
	switch (cfg.function) {
	case 'l':
		ret = list_files(fs,&cfg,&meta,space);
        break;
	case 'V':
		version();
		break;
	case 'w':
		ret = write_file(fs,&cfg,&meta,space);
        break;
	case 'x':
		ret = extract_file(fs,&cfg,&meta,space);
        break;
	case 'd':
		ret = delete_file(fs,&cfg,&meta,space);
        break;
	case 'c':
		ret = clear_all(fs,&cfg,&meta,space);
        break;
	default:
		fprintf(stderr,"o rly?  wtf is %d?\n",cfg.function);
        ret = E_SBINVAL;
	};
    if (cfg.verbose == VDEBUG)
        printf("Exiting with exitcode %d\n",ret);
	return ret;
}
/*********** END MAIN ***********/

/*********** FUNCTIONS **********/

// internally used prototypes
int _open_filesystem(ext2_filsys fs, int flags);
int _close_filesystem(int fd);
ulong _read_blockgroup(int fd,ulong bytes,void *buf);
ulong _write_blockgroup(int fd,ulong bytes,void *buf);
ullong _seek_blockgroup(int fd,ext2_filsys fs,ulong bgnum);

// directly called from main: write a new file to resv. space
int write_file(ext2_filsys fs,config *cfg,meta_info *meta,char *space) {
	struct stat s;
	file_entry *file, *tempf;
	FILE *f;
	char *fname, *fdata;
	int free;

	if (cfg->verbose >= VDEBUG)
		printf("In write_file(fs,cfg,meta,space)\n");
	// pull filename out
	fname = calloc(MAXFNAMELEN+1,1);
	if (rindex(cfg->fname,'/') != NULL)
		strncpy(fname,(char *)(rindex(cfg->fname,'/')+1),
						strlen((char *)(rindex(cfg->fname,'/')+1)));
	else
		strncpy(fname,cfg->fname,strlen(cfg->fname));
	fname = realloc(fname,strlen(fname)+1);

	// stat and open file
	if ((stat(cfg->fname,&s) == -1)  || ((f = fopen(cfg->fname,"r")) == NULL)) {
		fprintf(stderr,"\nError reading file '%s': ",cfg->fname);
		perror(NULL);
		return E_SBNOENT;
	}

	// pull meta infoz from resv. space image
	get_meta_info(fs,cfg,meta,space);

	// see if we even have enough free space to store the file
	if ((s.st_size+strlen(fname)+1+(sizeof(ulong)*3)) >
			(free = get_free_space(cfg,meta,space))) {
		fprintf(stderr,"Cannot store %s (%d bytes w/hdr); %d data bytes free.\n",
				fname,(s.st_size+strlen(fname)+1+sizeof(ulong)),free-8);
		return E_SBNOSPC;
	}

	// read in the file
	fdata = calloc(s.st_size,1);
	if (fread(fdata,1,s.st_size,f) < s.st_size) {
		fprintf(stderr,"\nError reading file '%s': ",cfg->fname);
		perror(NULL);
		return E_SBIO;
	}
	fclose(f);

    // prompt the user, if appropriate
    if ((cfg->prompt) && (prompt("Write file to filesystem?") == 0))
        return E_SBCANCEL;

	// fill file_info struct with plundered warez
	file = calloc(sizeof(file_entry),1);
	file->f_name = fname;
	file->f_size = s.st_size;
	file->f_data = fdata;
	file->next = NULL;
	if (meta->file != NULL) {
		tempf = meta->file;
		while (tempf->next != NULL)
			tempf = tempf->next;
		tempf->next = file;
	} else
		meta->file = file;
	meta->total_size += strlen(file->f_name)+1+sizeof(ulong)+file->f_size;

	// now fill the buffer...
	fill_buffer_from_meta(cfg,meta,space);
	// ... and flush to disk.
	writeout_buffer(fs,cfg,meta,space);

	// finally done!
	if (cfg->verbose >= VNORMAL)
		printf("Wrote %d bytes from %s.\n",s.st_size,cfg->fname);
	return E_SBSUCCESS;
}

// directly called from main: delete a file from resv. space
int delete_file(ext2_filsys fs,config *cfg,meta_info *meta,char *space) {
	file_entry *file, *tempf = NULL;
	int found = 0, filenum = 0;

	if (cfg->verbose >= VDEBUG)
		printf("In delete_file(fs,cfg,meta,space)\n");
	// pull the meta information
	get_meta_info(fs,cfg,meta,space);
	file = meta->file;
	if (cfg->verbose >= VDEBUG)
		printf("file = %x\n",file);
	// begin searching for the filename
	while (file != NULL) {
		filenum++;
		if (cfg->verbose >= VDEBUG)
			printf("cfg->fname = '%s', file->f_name = '%s'\n",cfg->fname,file->f_name);
		if (strcmp(file->f_name,cfg->fname) == 0) {
			// found!
		    if (cfg->verbose >= VDEBUG)
			    printf("Found!\n");
			if (filenum == 1) {
				meta->file = file->next;
			} else {
				tempf->next = file->next;
			}
			free(file->f_name);
			free(file->f_data);
			free(file);
			found++;
			break;
		} else {
			tempf = file;
			file = file->next;
		}
	}
	// the filename was in there.  KILL IT!
	if (found) {
        if (cfg->verbose >= VNORMAL)
            printf("\t%-8lu\t%s\n",file->f_size,cfg->fname);
        if ((cfg->prompt) && (prompt("Really delete file?") == 0))
            return E_SBCANCEL;
		memset(space,0,cfg->total);
		fill_buffer_from_meta(cfg,meta,space);
//		meta->total_size = (cfg->total - get_free_space(cfg,meta,space) -
//								(sizeof(ulong) * 2));
		if (cfg->verbose >= VDEBUG)
			printf("meta->total_size = %d\n",meta->total_size);
		writeout_buffer(fs,cfg,meta,space);
		if (cfg->verbose >= VNORMAL)
			printf("File %s deleted.\n",cfg->fname);
		return E_SBSUCCESS;
	} else {
        if (cfg->verbose >= VNORMAL)
            printf("File %s not found in reserved space.\n",cfg->fname);
    	return E_SBNOENT;
    }
}

// directly called from main: clear (zero) out reserved space
int clear_all(ext2_filsys fs,config *cfg,meta_info *meta,char *space) {
	int ret;

	// TODO: confirmation somewhere in here...
    if ((cfg->prompt) && (prompt("Really clear reserved space?") == 0))
        return E_SBCANCEL;
	memset(space,0,cfg->total);

	ret = writeout_buffer(fs,cfg,meta,space);
    if (cfg->verbose >= VNORMAL)
        printf("Reserved space cleared.\n");
	return ret;
}

int writeout_buffer(ext2_filsys fs,config *cfg,meta_info *meta,char *space) {
	int i, fd, ret, memofs=0;

	if (cfg->verbose >= VDEBUG)
		printf("In writeout_buffer(fs,cfg,meta,space)\n");
	fd = _open_filesystem(fs,O_RDWR);
	for (i=0;i<cfg->sbcount;i++) {
		if (cfg->verbose >= VDEBUG)
			printf("clear_all loop: i=%d, memofs=%d, fd=%d\n",i,memofs,fd);
		if ((ret = _seek_blockgroup(fd,fs,cfg->sb[i])) < 0) {
			if (cfg->verbose >= VDEBUG)
				printf("_seek returned %d!\n",ret);
			return E_SBSEEK;
		}
		if (_write_blockgroup(fd,cfg->usable[i],space+memofs) < cfg->usable[i]) {
			if (cfg->verbose >= VDEBUG)
				printf("_write failed!\n");
			return E_SBIO;
		}
		memofs += cfg->usable[i];
	}
	_close_filesystem(fd);
	return E_SBSUCCESS;
}

// directly called from main: extract a file from resved space
int extract_file(ext2_filsys fs,config *cfg,meta_info *meta,char *space) {
	file_entry *file;
	FILE *f;
	extern int errno;
    struct stat buf;

	if (cfg->verbose >= VDEBUG)
		printf("In extract_file(fs,cfg,meta,space)\n");
	if (check_sig(space)) {
		if (cfg->verbose >= VNOISY)
			printf("Found signature!\n");
		get_meta_info(fs,cfg,meta,space);
		file = meta->file;
		while (file != NULL) {
			if (strncmp(file->f_name,cfg->fname,strlen(cfg->fname)) == 0) {
				if (cfg->verbose >= VNORMAL)
					printf("Extracting %s (%lu bytes)...\n",file->f_name,file->f_size);
				umask(0077);
                if ((stat(cfg->fname,&buf)==0) && (cfg->prompt) &&
                    (prompt("File exists, overwrite?") == 0))
                        return E_SBCANCEL;
				if ((f = fopen(cfg->fname,"w")) == NULL)
					return errno;
				if (fwrite(file->f_data,1,file->f_size,f) < file->f_size)
					return E_SBIO;
				fclose(f);
				if (cfg->verbose >= VNORMAL)
					printf("\nWrote %d bytes to %s.\n",file->f_size,cfg->fname);
				return E_SBSUCCESS;
			} else {
				if (cfg->verbose >= VNOISY)
					printf("%s != %s\n",cfg->fname,file->f_name);
			}
			file = file->next;
		}
		printf("File %s not found.\n",cfg->fname);
	} else
		printf("Signature not found.  No files stored.\n");
	return E_SBNOSIG;
}

// directly called from main: print a list of all files in resv. space
int list_files(ext2_filsys fs,config *cfg,meta_info *meta,char *space) {
	file_entry *file;
	int filecount=0;
	ulong size;

	if (cfg->verbose >= VDEBUG)
		printf("In list_files(cfg,meta,space)\n");
	if (check_sig(space)) {
		if (cfg->verbose >= VNOISY)
			printf("Found signature!\n");
		get_meta_info(fs,cfg,meta,space);
        if (cfg->verbose == VQUIET) {
	        size = (meta->total_size == 0) ? -8 : meta->total_size;
            printf("%lu/%lu\n",size+8,cfg->total);
        }
		if (cfg->verbose >= VERBOSE)
			printf("Sig: '%c%c%c%c'\nStored data bytes: %lu\n",
				meta->sig & 0xFF,meta->sig >> 8,
				meta->sig >> 16, meta->sig >> 24, meta->total_size);
        if (cfg->verbose >= VNORMAL) {
            printf("\tVolume: %s (%s)\n\n",cfg->fsdev,
			        strlen(fs->super->s_volume_name) > 0 
						? fs->super->s_volume_name 
						: "no label");
		    printf("\tSize\t\tFilename\n");
        }
		file = meta->file;
		while (file != NULL) {
			filecount++;
            if (cfg->verbose >= VNORMAL)
    			printf("\t%-8lu\t%s\n",file->f_size,file->f_name);
            else
                printf("%lu\t%s\n",file->f_size,file->f_name);
			file = file->next;
		}
	} else {
		if (cfg->verbose >= VNORMAL) {
			printf("\n");
		    printf("Signature not found.  No files stored.\n\n");
        } else {
	        size = (meta->total_size == 0) ? -8 : meta->total_size;
            printf("%lu/%lu\n",size+8,cfg->total);
        }
		meta->total_size = 0;
	}
    if (cfg->verbose >= VNORMAL) {
	    size = (meta->total_size  == 0) ? -8 : meta->total_size;
        printf("%4d file%s found\t%8d bytes used\t%8d bytes free\n\n",
				filecount,(filecount != 1)?"s":"", size+8,
				cfg->total - (size+8));
    }
	return E_SBSUCCESS;
}

// open a direct filesystem file descriptor
int _open_filesystem(ext2_filsys fs, int flags) {
	int fd;

	if (cfg.verbose >= VDEBUG)
		printf("in _open_filesystem(fs,%d): fd=%d\n",flags,fd);
	if ((fd = open(fs->device_name,flags)) == -1) {
		perror("open");
		exit(E_SBIO);
	}
	return fd;
}

// close the direct filesystem file descriptor
int _close_filesystem(int fd) {
	if (cfg.verbose >= VDEBUG)
		printf("in _close_filesystem(%d)\n",fd);
	return (close(fd) == 0) ? E_SBSUCCESS : E_SBIO;
}

// seek to blockgroup reserved space in open file descriptor
ullong _seek_blockgroup(int fd,ext2_filsys fs,ulong bgnum) {
	ullong ofs;
	if (cfg.verbose >= VDEBUG)
		printf("in _seek_blockgroup(%d,fs,%d): ",fd,bgnum);
	ofs = get_offset(bgnum,fs->blocksize,fs->super->s_blocks_per_group) +
				(SBOFFSET-sizeof(fs->super->s_reserved));
	if (cfg.verbose >= VDEBUG)
		printf("seeking to fd(%d):%lu\n",fd,ofs);
	return (lseek64(fd,ofs,SEEK_SET) == ofs) ? ofs : -1;
}

// read reserved space bytes from BG bgnum from fd into *buf
ulong _read_blockgroup(int fd,ulong bytes,void *buf) {
	if (cfg.verbose >= VDEBUG)
		printf("in _read_blockgroup(%d,%lu,%X)\n",fd,bytes,buf);
	return read(fd,buf,bytes);
}

// write bytes bytes from buf to fd
ulong _write_blockgroup(int fd,ulong bytes,void *buf) {
	if (cfg.verbose >= VDEBUG)
		printf("in _write_blockgroup(%d,%lu,0x%x)\n",fd,bytes,buf);
	return write(fd,buf,bytes);
}

// rewrite free space buffer in memory from stored meta information
void fill_buffer_from_meta(config *cfg,meta_info *meta,char *space) {
	file_entry *file;
	int offset = 0;

	if (cfg->verbose >= VDEBUG)
		printf("In fill_buffer_from_meta(cfg,meta,space)\n");
	memset(space,0,cfg->total);
	// populate from meta_info
	memcpy(space+offset,&meta->sig,sizeof(ulong));
	offset += sizeof(ulong);
	// recalculate free space
	meta->total_size = (cfg->total - get_free_space(cfg,meta,space) -
							(sizeof(ulong) * 2));
	memcpy(space+offset,&meta->total_size,sizeof(ulong));
	offset += sizeof(ulong);
	// start populating from file entries
	file = meta->file;
	while (file != NULL) {
		memcpy(space+offset,file->f_name,strlen(file->f_name)+1);
		offset += strlen(file->f_name) + 1;
		memcpy(space+offset,&file->f_size,sizeof(ulong));
		offset += sizeof(ulong);
		memcpy(space+offset,file->f_data,file->f_size);
		offset += file->f_size;
		file = file->next;
	}
}

// return amount of bytes free in resv. space, after reading meta info
int get_free_space(config *cfg,meta_info *meta,char *space) {
	int free;
	file_entry *file;

	if (cfg->verbose >= VDEBUG)
		printf("In get_free_space(cfg,meta,space)\n");
	free = cfg->total;
	if (!check_sig(space))
		return free;

	free -= sizeof(ulong) * 2;
	file = meta->file;
	while (file != NULL) {
		free -= (file->f_size + strlen(file->f_name) + 1 + sizeof(ulong));
		file = file->next;
	}
	if (cfg->verbose >= VNORMAL)
		printf("%d bytes remaining in reserved space.\n",free);
	
	return free;
}

// return true if signature found in buffer, false if not
int check_sig(char *buf) {
	return strncmp(buf,sig,strlen(sig)) == 0 ? 1 : 0;
}

// get all of the reserved space into a contiguous area of memory
char *get_blocks(ext2_filsys fs,config *cfg) {
	char *space;
	int i, fd, memofs=0;

	space = calloc(cfg->total,1);

	if (cfg->verbose >= VDEBUG)
		printf("In get_blocks(fs,cfg)\n");
	fd = _open_filesystem(fs,O_RDONLY);
	for (i=0;i<cfg->sbcount;i++) {
		if (_seek_blockgroup(fd,fs,cfg->sb[i]) < 0) return NULL;
		if (_read_blockgroup(fd,cfg->usable[i],space+memofs) < 
			cfg->usable[i]) return NULL;
		memofs += cfg->usable[i];
	}
	_close_filesystem(fd);

	return space;
}

// put file_entry in meta->file pointer, and return byte size of file + fname
int get_file(meta_info *meta,char *buf,ulong bufsize,ulong filenum) {
	char *fname, *fdata;
	ulong fsize;
	file_entry *file, *tempf;
	int i, ofs=0, bufofs;

	if (cfg.verbose >= VDEBUG)
		printf("In get_file(meta,buf,%d,%lu)\n",bufsize,filenum);
	// sanity checking
	if ((filenum < 0) || (meta == NULL) || (buf == NULL) || (bufsize < 1))
		return 0;							// bytes

	// position the buffer offset pointer correctly
	bufofs = sizeof(ulong) * 2;				// default here
	file = meta->file;
	if (filenum > 1) {
		i = 1;
		while (file != NULL) {
			bufofs += strlen(file->f_name) + 1 + sizeof(ulong) + file->f_size;
			file = file->next;
			i++;
		}
		if (i < filenum)
			return 0;						// bytes
	}

    // bugfix: segfaulted randomly due to buffer overrun here.  my bad.
    if (bufofs >= cfg.total)
        return 0;

	fname = calloc(MAXFNAMELEN+1,1);
	do {
		memcpy(fname+ofs,buf+bufofs+ofs,1);
	} while ((*(buf+bufofs+ofs++) != 0) && (strlen(fname) < MAXFNAMELEN));
	fname = realloc(fname,ofs);

	if (strlen(fname) < 1) {
		free(fname);
		return 0;							// bytes
	}

	memcpy(&fsize,buf+bufofs+ofs,sizeof(ulong));
	ofs += sizeof(ulong);

	fdata=calloc(fsize+1,1);
	for (i=0;i < fsize;i++)
		memcpy(fdata+i,buf+bufofs+ofs+i,1);

	// fill file pointer with plundered warez
	file = calloc(sizeof(file_entry),1);
	file->f_name = fname;
	file->f_size = fsize;
	file->f_data = fdata;
	file->next = NULL;

	// store the pointer
	if (filenum == 1)
		meta->file = file;
	else {
		tempf = meta->file;
		while (tempf->next != NULL)
			tempf = tempf->next;
		tempf->next = file;
	}

	// return byte count
	return strlen(file->f_name) + 1 + sizeof(ulong) + file->f_size;
}

// pull the ext2hide meta information from the filesystem buffer
int get_meta_info(ext2_filsys fs,config *cfg, meta_info *meta, char *buf) {
	int currptr=0, filenum = 0, ret;

	if (cfg->total < 1) return -1;
	if (cfg->verbose >= VDEBUG)
		printf("In get_meta_info(fs,cfg,meta,buf)\n");
	if (check_sig(buf)) {
		memcpy(&meta->sig,buf,sizeof(ulong));
		memcpy(&meta->total_size,buf+sizeof(ulong),sizeof(ulong));
		meta->file = NULL;
		if (meta->total_size > 0) {
			currptr = 8;
			do {
				if ((ret = get_file(meta,buf,cfg->total,++filenum)) > 0)
					currptr += ret;
				else {
					fprintf(stderr,"ERROR!  Inconsistency detected!\n");
					fprintf(stderr,"Size of data incorrectly reported as %d bytes!\n");
					fprintf(stderr,"Please restart with -c option to clear data.\n");
					exit(E_SBIO);
				}
				currptr += get_file(meta,buf,cfg->total,++filenum);
			} while (((meta->total_size+8) - currptr) > 0);
		}
	} else {
		memcpy(&meta->sig,sig,4);
		meta->total_size = 0;
	}
	return E_SBSUCCESS;
}

// determine usable byte count in this superblock
ulong get_usable(ulong bgnum, ulong blocksize, ulong reserved) {
	ulong usable=0;
	if (cfg.verbose > VDEBUG)
		printf("In get_usable(%lu,%lu,%lu)\n",bgnum,blocksize,reserved);
	usable = bgnum
		? (blocksize-(SBOFFSET-reserved))
		: (blocksize == SBOFFSET)
		? (blocksize-(blocksize-reserved))
		: (blocksize-SBOFFSET-(SBOFFSET-reserved));
	return usable;
}

// determine the offset of a particular block group
ullong get_offset(ullong bgnum, ullong blocksize, ullong blockspergroup) {
	return bgnum ? (bgnum * blocksize * blockspergroup) : SBOFFSET;
}

// scan the filesystem for all superblocks, fill array and return quantity
int find_superblocks(ext2_filsys fs,ulong sb[]) {
	int i;
	ulong sbcount=0;
	blk_t bn;

	if (cfg.verbose > VDEBUG)
		printf("In find_superblocks(fs,sb)\n");
	for (i=0;i<fs->group_desc_count;i++)
		if ((bn = has_superblock(fs,i)) != -1)
			sb[sbcount++] = i;
	return sbcount;
}

// parse command-line arguments
int check_args(int argc,char **argv,config *cfg) {
	int option;
	extern char *optarg;
	extern int optind, optopt, opterr;

	if (argc < 2) {
		usage(USAGESHORT);
		exit(E_SBINVAL);
	}

	cfg->verbose = DEFVERBOSITY;	// DEFAULT: verbosity level = 1
	cfg->prompt  = DEFPROMPT;	// DEFAULT: prompt for user confirmation
	optopt = 0;
	opterr = 0;

	// parse arguments w/ getopt
	while ((option = getopt(argc,argv,"lw:x:d:cVqvfh?")) != -1) {
		if (optopt) {
			fprintf(stderr,"Invalid argument: -%c\n",optopt);
			usage(USAGELONG);
			exit(E_SBINVAL);
		}
		else
			switch (option) {
			case 'v':	// increase verbosity
				cfg->verbose++;
				break;
			case 'V':	// print version and exit
				version();
				exit(E_SBSUCCESS);
			case 'q':	// be quiet
				cfg->verbose = 0;
				break;
			case 'f':
				cfg->prompt = 0;
				break;
			case 'h':	// help
			case '?':
				usage(USAGELONG);
				exit(E_SBSUCCESS);
			case 'w':	// write file to reserved space
			case 'x':	// extract file from resv. space
			case 'd':	// delete file from resv. space
				if (optarg == 0) {
					fprintf(stderr,"Error: You must pass a filename with -%c.\n",option);
					usage(USAGELONG);
					exit(E_SBINVAL);
				} else
					strncpy(cfg->fname,optarg,strlen(optarg));
				// fall through here
			case 'c':
			case 'l':
				cfg->function = option;
				break;
			default:
				fprintf(stderr,"Invalid argument: -%c\n",
								option);
				usage(USAGELONG);
				exit(E_SBINVAL);
			};
	}
	if (argv[optind] == NULL) {
		fprintf(stderr,"Error: Filesystem device not specified!\n");
		usage(USAGELONG);
		exit(E_SBINVAL);
	}
	strncpy((char *)cfg->fsdev,argv[optind],strlen(argv[optind]));
	return E_SBSUCCESS;
}

// print version information
void version(void) {
	printf("%s v%s by %s, Copyleft %s\n",PROGNAME,VERSION,AUTHOR,COPYYEAR);
}

// actually open the filesystem itself
int fs_open(ext2_filsys *fs,config *cfg) {
	errcode_t ret;

	if (cfg->verbose >= VDEBUG)
		printf("In fs_open(fs,cfg)\n");
	if (cfg->verbose >= VNORMAL) {
		printf("Opening '%s'...",cfg->fsdev);
		fflush(stdout);
	}
	if ((ret=ext2fs_open(cfg->fsdev,0,0,0,unix_io_manager,fs)) != 0) {
		fprintf(stderr,"Cannot read filesystem, sorry!\n",ret);
		return E_SBACCES;
	}
	if (cfg->verbose >= VNORMAL)
		printf("success.\n");
	return E_SBSUCCESS;
}

// print usage information
void usage(int longargs) {
	printf("Usage: %s command [options] device\n",PROGNAME);
	if (longargs) {
		printf(" Commands\n");
		printf("  -l\t\tlist contents of reserved space\n");
		printf("  -w fname\tsave from fname and store in reserved space\n");
		printf("  -x fname\textract fname from reserved space, write to fname\n");
		printf("  -d fname\tdelete fname from reserved space\n");
		printf("  -c\t\tclear (zero) all reserved space\n");
		printf("  -V\t\tprint version information and exit\n");
		printf("  -h | -?\tshow usage information (you're looking at it)\n");
		printf(" Options\n");
		printf("  -q\t\tbe quiet; no screen output\n");
		printf("  -v\t\tbe verbose; use multiple times to increase verbosity\n");
		printf("  -f\t\tdo not prompt for any confirmation\n");
		printf(" Device\n");
		printf("  device can be any valid ext2 filesystem device (even a file!)\n\n");
	} else {
		printf("  %s -h for more options.\n",PROGNAME);
	}
}

// print superblock count and offset information
ulong get_sbinfo(ext2_filsys fs, config *cfg) {
	int i;
	ulong total=0;

	if (cfg->verbose >= VERBOSE)
		printf("%lu superblocks: (",cfg->sbcount);
	for (i=0;i<cfg->sbcount;i++) {
		if (cfg->verbose >= VERBOSE)
			printf("%d%s",cfg->sb[i],i<(cfg->sbcount-1)?",":"");
		cfg->usable[i] = get_usable(cfg->sb[i],fs->blocksize,
					sizeof(fs->super->s_reserved));
		total += cfg->usable[i];
	}
	if (cfg->verbose >= VERBOSE)
		printf(")\n");
	if (cfg->verbose >= VNOISY) {
		for (i=0;i<cfg->sbcount;i++) {
			printf("SB %d (BG %2d) @ byte ofs %-10llu (%lu usable bytes)\n",
				i, cfg->sb[i], get_offset(cfg->sb[i],
						fs->blocksize,
						fs->super->s_blocks_per_group),
				cfg->usable[i]);
		}
	}
	return total;
}

// print filesystem information
void print_fsinfo(ext2_filsys fs,config *cfg) {
	if (cfg->verbose >= VNOISY) {
		printf("      Device name: %s\n",fs->device_name);
		printf("      Volume name: %s\n",fs->super->s_volume_name);
		printf("       Block size: %ld bytes\n",fs->blocksize);
		printf("  Superblock size: %ld bytes\n",
					sizeof(struct ext2_super_block));
		printf("   Reserved space: %ld bytes\n",
					sizeof(fs->super->s_reserved));
		printf("    Journal inode: %ld\n",fs->super->s_journal_inum);
		printf("      Block count: %ld\n",fs->super->s_blocks_count);
		printf("     Blocks/group: %ld\n",fs->super->s_blocks_per_group);
		printf("      Desc blocks: %ld\n",fs->desc_blocks);
		printf("Superblock number: %ld\n",fs->super->s_block_group_nr);
		printf("     Block groups: %ld\n",fs->group_desc_count);
	} else if (cfg->verbose >= VERBOSE) {
		printf("Stats: %s (\"%s\"): bs:%ld rsv:%ld bks:%ld b/g:%ld gps:%ld\n",
			fs->device_name, fs->super->s_volume_name,
			fs->blocksize, sizeof(fs->super->s_reserved),
			fs->super->s_blocks_count,fs->super->s_blocks_per_group,
			fs->group_desc_count);
	}
}

// return block number if blockgroup contains a superblock, else -1
// (0 is a valid superblock, too!)
blk_t has_superblock(ext2_filsys fs, ulong grpdesc) {
/***********************************************************************
 * Commented out section due to ext2fs_super_and_bgd_loc not being
 * implemented until e2fsprogs v1.35.  Besides, I previously thought
 * that ext2fs_super_and_bgd_loc actually checked the filesystem, but
 * it just does a root calculation for sparse filesystems.  Following
 * Theo's lead here :)
 ***********************************************************************/
/*	blk_t super_blk, old_desc_blk, new_desc_blk;
	ext2fs_super_and_bgd_loc(fs,grpdesc,&super_blk,
					&old_desc_blk,&new_desc_blk,0); */
	if ((grpdesc == 0) || (ext2fs_bg_has_super(fs,grpdesc)))
		return grpdesc;
/*	if ((grpdesc == 0) || super_blk)
		return super_blk; */
	return -1;
}

// check library version, and warn user if less than minimum (pre-1.35)
int check_lib_ver(int minlib) {
	const char *ver_string,*ver_date;
	int lib_ver;
	lib_ver = ext2fs_get_library_version(&ver_string,&ver_date);
	if ((lib_ver < minlib) && (cfg.verbose >= VNORMAL))
		printf("Using old library version; please see WARNING file.\n");
	if (cfg.verbose >= VERBOSE)
		printf("Ext2 Library Version: %s (%s)\n",ver_string,ver_date);
	return lib_ver;
}

// prompt the user with a yes/no question, clear buffer, return true or false
int prompt(char *PS1) {
    int c, retval;
    if (strlen(PS1) == 0)
        printf("\nAre you sure? [y/n] ");
    else
        printf("\n%s [y/n] ",PS1);
    c = getchar();
    switch (c) {
        case 'y': case 'Y': retval = 1; break;
        case 'n': case 'N': retval = 0; break;
        default: printf("I'll take that as a no.\n"); retval = 0;
    }
    do
        c = getchar();
    while (c != '\n' && c != EOF);
    return retval;
}

/* END ext2hide.c */
