#!/usr/bin/perl -w
#
#	Simple test engine for ext2hide
#	  ti/2006
#
#   ASSUMPTIONS:
#    *  You have 3 files in a subdirectory of $CWD called "work", named
#       "fs1024", "fs2048", "fs4096".  These files are valid ext2/3
#       filesystems, created with mke2fs [-j] -b $block_size.  The block
#       sizes on these filesystems should be the same as the filename.
#    *  You are running this test script from the same directory as a
#       compiled copy of ext2hide.
#	 *  You have a working diff(1) and md5sum(1) (and obviously perl)
#
#    You can create the filesystems using "dd" and "mke2fs".

use strict;
use Test::Simple tests => 58;
use File::Temp qw/tempfile/;

$|=1;

################### VARS ####################

my $prog   = "ext2hide";
my $wdir   = "work";
my $fs1024 = "$wdir/fs1024";
my $fs2048 = "$wdir/fs2048";
my $fs4096 = "$wdir/fs4096";
my %cmds = ( 
			'list'    => '-l',
			'write'   => '-w',
			'extract' => '-x',
			'delete'  => '-d',
			'clear'   => '-c',
			'version' => '-V',
		    );
my %args = (
			'quiet'   => '-q',
			'verbose' => '-v',
			'force'   => '-f',
			'help'    => '-h',
			'help2'   => '-?',
		   );
my %fses = (
			$fs1024   => 0,
		    $fs2048   => 0,
            $fs4096   => 0,
		   );
my %bins = (
                    'diff'   => '',
                    'md5sum' => '',
                    'dd'     => '',
                    'mke2fs' => '',
                    'yes'    => '',
           );
my $hdrsize    = 8;
my $tfnamesize = 12;
my $tfhdrsize  = 4;
my $offset     = $hdrsize + $tfnamesize + $tfhdrsize;
my $failed	   = 0;
my $BSD_STYLE  = 1;

################### MAIN ####################

print uc($prog)." testing utility, by Jason McManus (infi)\n\n";

# find our needed sister programs
find_binfiles();
# make sure the program is compiled, and the filesystems exist
test_filesexist();
# perform the tests
test_fileslist();
test_maxfilesizewrites();
test_multiwrites();

# print exit messages
print "\nDone with all tests.  You had $failed failures.";
print "  You rock!" unless $failed;
print "\n\nDo you wish to delete the temporary working files (114MB)? [y/n] ";
if (lc(readkey()) eq 'y') {
    unlink($fs1024,$fs2048,$fs4096);
    rmdir($wdir);
}
print "Testing complete.  Thank you, drive through.\n\n";

exit(0);

################### SUBS ####################

# find our needed sister programs
sub find_binfiles {
    for my $bin (keys %bins) {
        my $path = locate($bin,split(/:/,$ENV{'PATH'}),"/sbin","/usr/sbin",
                          "/usr/local/sbin","/bin","/usr/bin","/usr/local/bin");
        $path = ask_dir($bin) unless ($path);
        $bins{$bin} = $path;
    }
}

sub ask_dir {
    my ($fname) = shift;
    my ($fpath,$i);
    do {
        print "$fpath nonexistent.\n" if $fpath;
        print "Cannot locate $fname; please enter full path: ";
        $fpath = <>;
        chomp $fpath;
        die "Error locating $fname.\n" if (++$i >= 3);
    } while (($fpath ne "") and (! -x $fpath));
    return $fpath;
}

### Test necessary file existence ###
sub test_filesexist {
	print "Checking existence of files... ";
    unless (-e $prog) {
        print "\n**** $prog not found.  Run './Build all' first. ****\n\n";
        exit(1);
    };
    genfiles() unless ((-e $fs1024) && (-e $fs2048) && (-e $fs4096));
	print "\nAll set.  Starting tests...\n\n";
	return 1;
}

### Test -l option ###
sub test_fileslist {
	print "Testing list (-l)...\n";
	# check real filesystems
	for my $fs (keys %fses) {
		ok($fses{$fs} = fsusable($fs),"$prog -l $fs") 
	}
	# check non-existent
	my @cmdline = ("./$prog",$args{'quiet'},$cmds{'list'},"NONEXISTENT");
	ok(system(@cmdline),join(' ',@cmdline)) or $failed++;
	print "Done.\n\n";
	return 1;
}
### END Test -l option ###

### Test multiple small file writes ###
sub test_multiwrites {
	print "Testing multiple small file writes...\n";
	for my $fs (keys %fses) {
		my ($numfiles,$t,%rndfiles) = (4,0,());
		for (1..$numfiles) {
			my $i = int rand(($fses{$fs} / $numfiles)-16);
			$t += $i;
			my $fname = makerandfile($i);
			$rndfiles{$fname} = $i;
		}
		for my $fn (keys %rndfiles) {
			my @cmdline = ("./$prog",$args{'force'},$args{'quiet'},$cmds{'write'},$fn,$fs);
			ok(mysystem(@cmdline),"$numfiles writes: $fn ($rndfiles{$fn})")
				or $failed++;
			rename $fn,"$fn.bak";
		}
		for my $fn (keys %rndfiles) {
			my $fn2 = (split '/',$fn)[1];
			my @cmdline = ("./$prog",$args{'force'},$args{'quiet'},$cmds{'extract'},$fn2,$fs);
			ok(mysystem(@cmdline),"$numfiles extracts: $fn2 ($rndfiles{$fn})")
				or $failed++;
			rename $fn2,$fn;
			@cmdline = ($bins{'diff'},$fn,"$fn.bak");
			ok(mysystem(@cmdline),"$fn2 pre/post equivalency check")
				or $failed++;
		}
		unless ($failed) {
 			unlink ($_,"$_.bak") for (keys %rndfiles);
			my @cmdline = ("./$prog",$args{'force'},$args{'quiet'},$cmds{'clear'},$fs);
			ok(mysystem(@cmdline),"$fs clearing of data") or $failed++;
		} else {
			die "Something broke.  Early abortion!\n";
		}
	}
	print "Done.\n\n";
	return 1;
}
### END Test multiple small file writes ###

### Test -w/-x/-c/diff/MD5 options ###
sub test_maxfilesizewrites {
    print "Testing maximum file size write (-w), extract (-x), clear (-c)...\n";
    for my $fs (keys %fses) {
		my $fname = makerandfile($fses{$fs} - $offset);
        my ($fname2) = $fname =~ m#.*/(.*)$#;
        my $md5pre = `$bins{'md5sum'} $fs`;
        my @cmdline = ("./$prog",$args{'force'},$args{'quiet'},$cmds{'write'},$fname,$fs);
        ok(mysystem(@cmdline),"$fs single file write ($fname2)") or die;#$failed++;
        rename($fname,"$fname.bak");
        @cmdline = ("./$prog",$args{'quiet'},$cmds{'extract'},$fname2,$fs);
        ok(mysystem(@cmdline),"$fs file extraction ($fname2)") or die;#$failed++;
        rename($fname2,$fname);
        @cmdline = ($bins{'diff'},$fname,"$fname.bak");
        ok(mysystem(@cmdline),"$fname2 pre/post equivalency check") or $failed++;
        unlink($fname,"$fname.bak") unless $failed;
        @cmdline = ("./$prog",$args{'force'},$args{'quiet'},$cmds{'clear'},$fs);
        ok(mysystem(@cmdline),"$fs clearing of data") or $failed++;
        my $md5post = `$bins{'md5sum'} $fs`;
        ok($md5post eq $md5pre,"$fs pre/post equivalency check") or $failed++;
    }
    print "Done.\n\n";
	return 1;
}
### END Test -w/-x/-c/diff/MD5 options ###

# reverse the output so a 0 exitcode = true
sub mysystem {
	return system(@_) ? 0 : 1;
}

# return usable space in filesystem, using the ext2hide program itself.
sub fsusable {
	my $fs = shift;
	my $output = `./$prog $cmds{'list'} $fs`;
	my ($space) = $output =~ /^(\d+)\stotal bytes.*/m;
	return $space ? $space : 0;
}

# dump random bytes from urandom into test storage file
sub makerandfile {
    my $size = shift;
    my $buf;
    open(my $fh,"</dev/urandom") or die "Huh?  /dev/urandom: $!\n";
    die "Short read on /dev/urandom: $!\n"
        unless (read($fh,$buf,$size) == $size);
    close($fh);
        my ($tempfh,$tempfname) = tempfile("Test.XXXXXX",DIR=>$wdir);
    print $tempfh $buf;
    close($tempfh);
    return $tempfname;
}

# generate test files for first run.
sub genfiles {
    print "\nIt looks like this is your first run!\n\n";
    print "We have to set up a few files before running the test.  These files\n";
    print "will allow testing on non-live filesystem files of the 3 block sizes\n";
    print "that ext2fs allows, in order to test the reading and writing of this\n";
    print "program.  These files will take up over 110MB, so make sure that you\n";
    print "have that much space available before continuing.\n\n";
    print "Would you like to continue? [y/n] ";
    unless (lc(readkey()) eq 'y') {
        print "Ok, thanks anyway.\n";
        exit(1);
    }
    mkdir($wdir);
    print "Generating 1024-byte b/s filesystem-in-a-box...\n";
    my @cmdline = ($bins{'dd'},"if=/dev/zero","of=$fs1024","bs=1024","count=16K");
    system(@cmdline);
    my $output=`$bins{'yes'} | $bins{'mke2fs'} -b 1024 $fs1024`;
    print "Generating 2048-byte blocksize filesystem-in-a-box...\n";
    @cmdline = ($bins{'dd'},"if=/dev/zero","of=$fs2048","bs=1024","count=32K");
    system(@cmdline);
    $output=`$bins{'yes'} | $bins{'mke2fs'} -b 2048 $fs2048`;
    print "Generating 4096-byte blocksize filesystem-in-a-box...\n";
    @cmdline = ($bins{'dd'},"if=/dev/zero","of=$fs4096","bs=1024","count=64K");
    system(@cmdline);
    $output=`$bins{'yes'} | $bins{'mke2fs'} -b 4096 $fs4096`;
}

# read a key from the user.
sub readkey {
    if ($BSD_STYLE) {
        system "stty cbreak </dev/tty >/dev/tty 2>&1";
    } else {
        system "stty", '-icanon', 'eol', "\001";
    }
    my $key = getc(STDIN);
    if ($BSD_STYLE) {
        system "stty -cbreak </dev/tty >/dev/tty 2>&1";
    } else {
        system "stty", 'icanon', 'eol', '^@'; # ASCII null
    }
    print "\n";
    return $key;
}

# find executables, return full path
sub locate {
    my ($fname,@path) = @_;
    my $path = "";
    for my $dir (@path) {
        return "$dir/$fname" if (-x "$dir/$fname");
    }
    return "";
}
