/* symlink.c
 *
 * Symbolic link handling for EFS
 *
 * (C)1995,96 Christian Vogelgsang
 *
 * Based on the symlink.c from minix-fs by Linus
 */

#include "efs.h"

static int efs_readlink(struct inode *, char *, int);
static int efs_follow_link(struct inode *, struct inode *, int, int, 
			   struct inode **);

struct inode_operations efs_symlink_in_ops = {
	NULL,			/* no file-operations */
	NULL,			/* create */
	NULL,			/* lookup */
	NULL,			/* link */
	NULL,			/* unlink */
	NULL,			/* symlink */
	NULL,			/* mkdir */
	NULL,			/* rmdir */
	NULL,			/* mknod */
	NULL,			/* rename */
	efs_readlink,		/* readlink */
	efs_follow_link,	/* follow_link */
	NULL,
	NULL,
	NULL,			/* bmap */
	NULL,			/* truncate */
	NULL			/* permission */
};


/* ----- efs_getlinktarget -----
   read the target of the link from the data zone of the file
*/
static char *efs_getlinktarget(struct inode *in)
{
  struct buffer_head * bh;
  char *name;
  LONG size = in->i_size;
  LONG block;
  
  /* link data longer than 1024 not supported */
  if(size>2*EFS_BLOCK_SIZE) {
    printk("efs_getlinktarget: name too long: %lu\n",in->i_size);
    return NULL;
  }
  
  /* get some memory from the kernel to store the name */
  name = kmalloc(size+1,GFP_KERNEL);
  if(!name) return NULL;
  
  /* read first 512 bytes of target */
  block = efs_bmap(in,0);
  bh = bread(in->i_dev,block,EFS_BLOCK_SIZE);
  if(!bh) {
    kfree(name);
    return NULL;
  }
  memcpy(name,bh->b_data,(size>EFS_BLOCK_SIZE)?EFS_BLOCK_SIZE:size);
  brelse(bh);
  
  /* if the linktarget is long, read the next block */
  if(size>EFS_BLOCK_SIZE) {
    bh = bread(in->i_dev,block+1,EFS_BLOCK_SIZE);
    if(!bh) {
      kfree(name);
      return NULL;
    }
    memcpy(name+EFS_BLOCK_SIZE,bh->b_data,size-EFS_BLOCK_SIZE);
    brelse(bh);
  }
  
  /* terminate string and return it */
  name[size]=0;
  return name;
}


/* ----- efs_follow_link -----
   get the inode of the link target
*/
static int efs_follow_link(struct inode * dir, struct inode * inode,
	int flag, int mode, struct inode ** res_inode)
{
  int error;
  char *name;
  
  /* check for errors */
  *res_inode = NULL;
  if (!dir) {
    dir = current->fs->root;
    dir->i_count++;
  }
  if (!inode) {
    iput(dir);
    return -ENOENT;
  }
  if (!S_ISLNK(inode->i_mode)) {
    iput(dir);
    *res_inode = inode;
    return 0;
  }
  if (current->link_count > 5) {
    iput(inode);
    iput(dir);
    return -ELOOP;
  }
  
  /* read the linkdata=target */
  name = efs_getlinktarget(inode);
  if(!name) {
    iput(inode);
    iput(dir);
    return -EIO;
  }

  /* try to get inode for link target */
  iput(inode);
  current->link_count++;
  error = open_namei(name,flag,mode,res_inode,dir);
  current->link_count--;
  kfree(name);

  return error;
}


/* ----- efs_readlink -----
   read the target of a link and return the name
*/
static int efs_readlink(struct inode * inode, char * buffer, int buflen)
{
  int i;
  char c;
  char *name;
  
  if (!S_ISLNK(inode->i_mode)) {
    iput(inode);
    return -EINVAL;
  }
  if (buflen > 1023)
    buflen = 1023;

  name = efs_getlinktarget(inode);
  iput(inode);
  if (!name)
    return 0;

  /* copy the link target to the given buffer */
  i = 0;
  while (i<buflen && (c = name[i])) {
    i++;
    put_user(c,buffer++);
  }

  kfree(name);
  return i;
}
