/* namei.c
   
   name lookup for EFS filesystem
   
   (C)95,96 Christian Vogelgsang
*/

#include "efs.h"


/* ----- efs_findInodeByName -----
   search a raw efs dir entry for the given name
   
   dir     - inode of directory
   oname   - name to search for
   onamelen- length of name
   
   return  - inode number of the found entry or 0 on error
*/
static LONG efs_findInodeByName(struct inode *dir,const char *oname,
				int onamelen)
{
  struct efs_in_info *ini = (struct efs_in_info *)&dir->u.generic_ip;
  struct buffer_head *bh;
  LONG inode,in;
  BYTE *raw;
  SHORT max,i,off;
  BYTE *name;
  SHORT namelen;
  LONG blknum,b;
 
  /* Warnings */ 
  if(ini->tot!=1)
    printk("efs_find: More than one extent!\n");
  if(dir->i_size & (EFS_BLOCK_SIZE-1))
    printk("efs_find: dirsize != blocklen * n\n");
 
  /* Search in every dirblock */
  inode = 0;
  blknum = dir->i_size >> EFS_BLOCK_SIZE_BITS;
  for(b=0;b<blknum;b++) {

    /* Read a raw dirblock */
    bh = bread(dir->i_dev,efs_bmap(dir,b),EFS_BLOCK_SIZE);
    if(!bh) return 0;
    
    raw = bh->b_data;
    /* number of direntries in this block */
    max = (SHORT)raw[EFS_DB_ENTRIES];
    for(i=0;i<max;i++) {
      
      /* read offset of entry */
      off = (SHORT)raw[EFS_DB_FIRST+i]<<1;

      /* inode, namelen and name of direntry */
      in = ConvertLong(raw,off); 
      namelen = (SHORT)raw[off+EFS_DI_NAMELEN]; 
      name = raw + off + EFS_DI_NAME;
    
      /* we found the name! */
      if((namelen==onamelen)&&(!memcmp(oname,name,onamelen))) {
	inode = in;
	break;
      }
    }

    brelse(bh);

    /* found a name -> ready */
    if(inode) break;
  }
  
  return inode;
}


/* ----- efs_lookup -----
   lookup inode operation:
   check if a given name is in the dir directory
   
   dir       - pointer to inode of directory
   name      - name we must search for
   len       - length of name
   result    - pointer to inode struct if we found it or NULL on error
   
   return    - 0 everything is ok or <0 on error
*/
int efs_lookup(struct inode *dir,const char *name,int len,
	       struct inode **result)
{
  int   inode;
  
  *result = NULL;
  
  /* check if it's a valid dir */
  if(!dir) 
    return -ENOENT;
  if(!S_ISDIR(dir->i_mode)) {
    iput(dir);
    return -ENOENT;
  }
  
  /* search for the inode by name */
  inode = efs_findInodeByName(dir,name,len);
  if(!inode) {
    iput(dir);
    return -ENOENT;
  }

  /* try to get the inode */
  *result = iget(dir->i_sb,inode);
  if(!*result) {
    iput(dir);
    return -EACCES;
  }
    
  iput(dir);
  return 0;
}
