/* file.c
   
   read files on EFS filesystems
   now replaced by generic functions of the kernel:
   leaves only mapping of file block number -> disk block number in this file
   
   (C)95,96 Christian Vogelgsang
*/

#include "efs.h"

static struct file_operations efs_file_ops = {
  NULL,
  generic_file_read,
  NULL,
  NULL,
  NULL,
  NULL,
  generic_file_mmap,
  NULL,
  NULL,
  NULL
};

struct inode_operations efs_file_in_ops = {
  &efs_file_ops,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  generic_readpage,
  NULL,
  efs_bmap,
  NULL,
  NULL
};
 
#define MIN(a,b) ((a)<(b)?(a):(b))

#define CHECK(num) \
  eblk = ini->extends[num][0]; \
  epos = ini->extends[num][1] & 0xffffff; \
  elen = ini->extends[num][1] >> 24; \
  if((blk >= epos)&&(blk < (epos+elen))) \
    result = (blk - epos) + eblk + sbi->fs_start;


/* ----- efs_getblknum -----
   find the disc block number for a given logical file block number
   
   in       - inode of file
   blk      - logical file block number
   
   return   - 0 on error, or unmapped block number
*/
static LONG efs_getblk(struct inode *in,LONG blk)
{
  struct efs_sb_info *sbi = (struct efs_sb_info *)&in->i_sb->u.generic_sbp;
  struct efs_in_info *ini = (struct efs_in_info *)&in->u.generic_ip;
  struct buffer_head *bh;

  LONG result = 0;
  LONG eblk,epos,elen;
  int num,extnum,readahead;
  LONG extblk;
  SHORT extoff,pos,cur,tot;
  BYTE *buf;


  /* first check the current extend */
  cur = ini->cur;
  tot = ini->tot;
  CHECK(cur)
  if(result) 
    return result;
    
  /* if only one extent exsits and we are here the test failed */
  if(tot==1) {
    printk("efs: bmap failed on one extent!\n");
    return 0;
  }

  /* check the stored extends in the inode */
  num = MIN(tot,EFS_MAX_EXTENTS);
  for(pos=0;pos<num;pos++) {
    /* don't check the current again! */
    if(pos==cur) 
      continue;

    CHECK(pos)
    if(result) {
      ini->cur = pos;
      return result;
    }
  }

  /* If the inode has only direct extents, 
     the above tests must have found the block's extend! */
  if(tot<=EFS_MAX_EXTENTS) {
    printk("efs: bmap failed for direct extents!\n");
    return 0;
  }  

  /* --- search in the indirect extensions list blocks --- */
#ifdef DEBUG
  printk("efs - indirect search for %lu\n",blk);
#endif

  /* calculate block and offset for begin of extent descr and read it */
  extblk = ini->extblk;
  extoff = 0;
  bh = bread(in->i_dev,extblk,EFS_BLOCK_SIZE);
  if(!bh) {
    printk("efs: read error in indirect extends\n");
    return 0;
  }
  buf = (BYTE *)bh->b_data;

  pos = 0; /* number of extend store in the inode */
  extnum = 0; /* count the extends in the indirect blocks */ 
  readahead = 10; /* number of extends to read ahead */
  while(1) {

    /* skip last current extent store in the inode */
    if(pos==cur) pos = (pos+1)%EFS_MAX_EXTENTS;

    /* read new extent in inode buffer */
    ini->extends[pos][0] = ConvertLong(buf,extoff);
    ini->extends[pos][1] = ConvertLong(buf,extoff+4);

    /* we must still search */ 
    if(!result) {
      CHECK(pos)
      if(result)
	ini->cur = pos;
    }
    /* we found it already and read ahead */
    else {
      readahead--;
      if(!readahead)
	break;
    }

    /* next storage place */
    pos = (pos+1)%EFS_MAX_EXTENTS;
    extnum++;

    /* last extent checked -> finished */
    if(extnum==tot) {
      if(!result)
	printk("efs: bmap on indirect failed!\n");
      break;
    }

    extoff += 8;
    /* need new block */
    if(extoff==EFS_BLOCK_SIZE) {
      extoff = 0;
      extblk++;
	
      brelse(bh);
      bh = bread(in->i_dev,extblk,EFS_BLOCK_SIZE);
      if(!bh) {
	printk("efs: read error in indirect extends\n");
	return 0;
      }
      buf = (BYTE *)bh->b_data;
    }
  }
  brelse(bh);

  return result;
}  


/* ----- efs_bmap ----- 
   bmap: map a file block number to a device block number
   
   in    - inode owning the block
   block - block number
   
   return - disk block
*/
int efs_bmap(struct inode *in, int block)
{
  /* quickly reject invalid block numbers */
  if(block<0) {
#ifdef DEBUG
    printk("efs_bmap: block < 0\n");
#endif
    return 0;
  }
  /* since the kernel wants to read a full page of data, i.e. 8 blocks
     we must check if the block number is not too large */
  if(block>((in->i_size-1)>>EFS_BLOCK_SIZE_BITS)) {
#ifdef DEBUG
    printk("efs_bmap: block %d > max %d == %d\n",
	   block,in->i_size>>EFS_BLOCK_SIZE_BITS,in->i_blocks);
#endif
    return 0;
  }

  return efs_getblk(in,block);
}
