/* dir.c
   
   directory inode operations for EFS filesystem
   
   (C)95,96 Christian Vogelgsang
*/

#include "efs.h"

static int efs_readdir(struct inode *,struct file *,void *,filldir_t);

static struct file_operations efs_dir_ops = {
  NULL, 
  NULL, 
  NULL,
  efs_readdir,
  NULL, 
  NULL, 
  NULL, 
  NULL, 
  NULL
};

struct inode_operations efs_dir_in_ops = {
  &efs_dir_ops,
  NULL,
  efs_lookup,
  NULL, 
  NULL, 
  NULL, 
  NULL, 
  NULL, 
  NULL, 
  NULL, 
  NULL, 
  NULL, 
  NULL, 
  NULL,
  efs_bmap,
  NULL,
  NULL
};


/* ----- efs_readdir -----
   readdir inode operation:
   read the next directory entry of a given dir file
   
   inode     - pointer to inode struct of directory
   filp      - pointer to file struct of directory inode
   dirent    - pointer to dirent struct that has to be filled
   filldir   - function to store values in the directory

   return    - 0 ok, <0 error
*/
static int efs_readdir(struct inode *inode, struct file *filp,
		void *dirent, filldir_t filldir)
{
  struct efs_in_info *ini = (struct efs_in_info *)&inode->u.generic_ip;
  struct buffer_head *bh;
  BYTE  *rawdirblk;
  LONG  iteminode;
  SHORT namelen;
  BYTE  *nameptr;
  LONG  numitems;
  SHORT itemnum;
  LONG  block;
  SHORT rawdepos;  

  /* some checks */
  if(!inode || !inode->i_sb || !S_ISDIR(inode->i_mode)) 
    return -EBADF;
  
  /* Warnings */
  if(ini->tot!=1)
    printk("efs_readdir: More than one extent!\n");
  if(inode->i_size & (EFS_BLOCK_SIZE-1))
    printk("efs_readdir: dirsize != blocksize*n\n");

  /* f_pos contains: dirblock<<BLOCK_SIZE | # of item in dirblock */
  block   = filp->f_pos >> EFS_BLOCK_SIZE_BITS;
  itemnum = filp->f_pos & 0xff; 
  
  /* We found the last entry -> ready */
  if(block == (inode->i_size>>EFS_BLOCK_SIZE_BITS))
    return 0;

  /* get disc block number from dir block num: 0..i_size/BLOCK_SIZE */ 
  bh = bread(inode->i_dev,efs_bmap(inode,block),EFS_BLOCK_SIZE);
  if(!bh) return 0;

  /* dirblock */
  rawdirblk = (BYTE *)bh->b_data; 
  /* number of entries stored in this dirblock */
  numitems = rawdirblk[EFS_DB_ENTRIES]; 
  /* offset in block of #off diritem */
  rawdepos = (SHORT)rawdirblk[EFS_DB_FIRST+itemnum]<<1;

  /* diritem first contains the inode number, the namelen and the name */
  iteminode = ConvertLong(rawdirblk,rawdepos);
  namelen = (SHORT)rawdirblk[rawdepos+EFS_DI_NAMELEN];
  nameptr = rawdirblk + rawdepos + EFS_DI_NAME;  

#ifdef DEBUG
  printk("efs: dir #%d - inode %lx namelen %u\n",itemnum,iteminode,namelen);
#endif

  /* copy filename and data in direntry */
  filldir(dirent,nameptr,namelen,filp->f_pos,iteminode);

  brelse(bh);

  /* store pos of next item */
  itemnum++;
  if(itemnum==numitems) {
    itemnum = 0;
    block++;
  }
  filp->f_pos = (block<<EFS_BLOCK_SIZE_BITS) | itemnum;
  
  return 0;
}
