/*
    xgui.cpp user interface for Xlib


    flexemu, an MC6809 emulator running FLEX
    Copyright (C) 1997-2000  W. Schwotzer

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <misc1.h>

#include "xgui.h"

#ifdef X11

#include <stdio.h>
#include "pia1.h"

void XGui::initialize(struct sGuiOptions *pOptions)
{
	XAbstractGui::initialize(pOptions);
	message = NULL;
	display = XOpenDisplay((char *)NULL);
	if (display == NULL) {
		fprintf(stderr, "can't open %s\n", XDisplayName((char *)NULL));
		exit(EXIT_FAILURE);
	};
	cpu->initialize_byte_conv_table(ImageByteOrder(display) == LSBFirst);
	initialize_word_conv_table(ImageByteOrder(display) == LSBFirst);
	screen = DefaultScreenOfDisplay(display);
	initialize_e2window(pOptions);
	create_cpuview();
} 

Display *XGui::getDisplay(void)
{
	return display;
}

Window XGui::getWindow(void)
{
	return e2window;
}

void XGui::process_resize(XEvent *pevent)
{
	guiXSize = pevent->xconfigure.width / WINDOWWIDTH;
	guiYSize = pevent->xconfigure.height / WINDOWHEIGHT;
	e2video->init_blocks_to_update();
}

void XGui::event_queue(void)
{
	Window	root, child;
	int	newX, newY;
	int	winX, winY;
	unsigned int newButtonMask;
	XEvent	event;
	KeySym	keysym;
	long	event_mask;
	Byte	i;
	SWord	key = 0;
	int	revert;

	event_mask = ButtonPressMask | ExposureMask |
			KeyPressMask;
	if (XCheckWindowEvent(display, cpu_window, event_mask, &event))
		switch (event.type) {
		   case NoExpose:
		        break;
		   case ButtonPress:
		   	   process_button_input(&event);
			   break;
		   case KeyPress: 
			keysym = XLookupKeysym(&event.xkey,
				(event.xkey.state & ShiftMask) ?
				 ShiftMapIndex : 0);
			for (i=0; i < 2; i++) {
				if (bp_input[i])
					process_bp_input(keysym, i);
			}
			update_1_second(NO_CHANGE);
			break;
		   case Expose:
			update_cpuview(NO_CHANGE);
			break;
		} // switch
	event_mask = ExposureMask | KeyPressMask | StructureNotifyMask;
	if (XCheckWindowEvent(display, e2window, event_mask, &event))
		switch (event.type) {
		   case KeyPress:
		       if (!io->key_buffer_full() &&
			   (key = translate_to_ascii(&event.xkey)) >= 0)
				io->put_ch(key);
			break;
		   case ConfigureNotify:
		   	process_resize(&event);
		   	break;
		   case Expose:
			e2video->init_blocks_to_update();
			break;
		} // switch
	
	XGetInputFocus(display, &child, &revert);
	if (child == e2window && (XQueryPointer(display, e2window,
		&root, &child, &newX, &newY,
		&winX, &winY, &newButtonMask) == True)) {
		io->deltaX = newX - oldX;
		io->deltaY = newY - oldY;
		io->buttonMask = convert_buttonmask(newButtonMask);
		oldX = newX;
		oldY = newY;
	} // if
	XAbstractGui::event_queue();
}


void XGui::initialize_e2window(struct sGuiOptions *pOptions)
{
	unsigned int		i, j;
	Visual		       *visual;
	XGCValues		gcv;
	XEvent			event;
	XSetWindowAttributes	attrs;
	XColor			xcolor, exact_color;
	XSizeHints		*pxsh;
	XClassHint		*pxch;

	oldX       = 0;
	oldY       = 0;
	exit_flag  = 0;

	// handle some default values
	if (pOptions->color.empty())
		pOptions->color = "green";

	attrs.event_mask		= KeyPressMask | ExposureMask;

	e2window = XCreateWindow(display, RootWindowOfScreen(screen), 0, 0,
		WINDOWWIDTH * guiXSize, WINDOWHEIGHT * guiYSize, 0,
		CopyFromParent, InputOutput, (Visual *)CopyFromParent,
		CWEventMask, &attrs);

	if (pOptions->synchronized)
		XSynchronize(display, True);
	if (!XAllocNamedColor(display, DefaultColormapOfScreen(screen),
	   pOptions->color, &xcolor, &exact_color))
		xcolor.pixel = WhitePixelOfScreen(screen);
	if (!pOptions->inverse) {
		gcv.foreground = xcolor.pixel;
		gcv.background = BlackPixelOfScreen(screen);
	} else {
		gcv.foreground = BlackPixelOfScreen(screen);
		gcv.background = xcolor.pixel;
	};
	e2gc = XCreateGC(display, e2window, GCForeground | GCBackground, &gcv);

	pxsh = XAllocSizeHints();
	if (pxsh != NULL) {
		pxsh->min_width   = WINDOWWIDTH;
		pxsh->max_width   = WINDOWWIDTH * MAX_GUIXSIZE;
		pxsh->min_height  = WINDOWHEIGHT;
		pxsh->max_height  = WINDOWHEIGHT * MAX_GUIYSIZE;
		pxsh->width_inc   = WINDOWWIDTH;
		pxsh->height_inc  = WINDOWHEIGHT;
		pxsh->flags = PMinSize | PMaxSize | PResizeInc;
		XSetStandardProperties(display, e2window,
			PROGRAMNAME, PROGRAMNAME, None, NULL, 0, pxsh);
		XFree(pxsh);
	}
	pxch = XAllocClassHint();
	if (pxch != NULL) {
		pxch->res_name  = "e2video";
		pxch->res_class = "E2video";
		XSetClassHint(display, e2window, pxch);
		XFree(pxch);
	}
	visual = DefaultVisualOfScreen(screen);
	for (j = 0; j < YBLOCKS; j++)
		for (i = 0; i < 2; i++) {
			image[i][j] = XCreateImage(display, visual,
			1, XYBitmap, 0,
			(char *)(cpu->video_ram + j * YBLOCK_SIZE +
				i * VIDEORAM_SIZE * 6),
				BLOCKWIDTH, BLOCKHEIGHT, 32, 0);
		} // for
	for (i = 0; i < MAX_GUIXSIZE; i++) {
		for (j = 0; j < MAX_GUIYSIZE; j++) {
			alt_image[i][j] = XCreateImage(display, visual, 1,
				XYBitmap, 0, (char *)copy_block,
				BLOCKWIDTH*(i+1), BLOCKHEIGHT*(j+1), 32, 0);
		} // for
	} // for

	// show window on display
	XSelectInput(display, e2window, ExposureMask | KeyPressMask |
		StructureNotifyMask);
	XMapWindow(display, e2window);
	// wait, until window is visible
	XWindowEvent(display, e2window, ExposureMask, &event);
	e2video->init_blocks_to_update();
}


void XGui::create_message_dialog(Widget parent)
{
} // create_message_dialog

int XGui::gui_type(void)
{
	return GUI_X11;
}

XGui::XGui(
	Mc6809* x_cpu = NULL,
	Inout* x_io = NULL,
	E2video* x_video = NULL,
	struct sGuiOptions *pOptions = NULL) :
		XAbstractGui(x_cpu, x_io, x_video, pOptions)
{
	initialize(pOptions);
}

XGui::~XGui()
{
	int i, j;

        for (j = 0; j < YBLOCKS; j++) {
                for (i = 0; i < 2; i++) {
                        image[i][j]->data = NULL;
                        XDestroyImage(image[i][j]);
        	} // for
        } // for
        // remove data pointer otherwise it would be freed
        for (i = 0; i < MAX_GUIXSIZE; i++) {
        	for (j = 0; j < MAX_GUIYSIZE; j++) {
               		alt_image[i][j]->data = NULL;
                	XDestroyImage(alt_image[i][j]);
        	} // for
        } // for
	XFreeGC(display, e2gc);
	XFreeGC(display, cpu_gc);
	XFreeGC(display, inv_gc);
	XCloseDisplay(display);
}

#endif // X11

