/*
    win32gui.cpp  user interface for Win32


    flexemu, an MC6809 emulator running FLEX
    Copyright (C) 1997-2000  W. Schwotzer

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <misc1.h>

#ifdef WIN32

#include <ctype.h>
#include <stdlib.h>
#include <new>
#include "resource.h"
#include "win32gui.h"
#ifndef PATH_MAX
#define PATH_MAX MAX_PATH
#endif

Win32Gui *wgui = NULL;


extern LRESULT CALLBACK cpuWindowWndProc(
	HWND hwnd,
	UINT message,
	WPARAM wParam,
	LPARAM lParam);

// process window functions
// ATTENTION: all functions must be reentrant!

LRESULT CALLBACK e2windowWndProc(
	HWND hwnd,
	UINT message,
	WPARAM wParam,
	LPARAM lParam)
{
	switch(message) {
		case WM_PAINT:   wgui->onPaint(hwnd);
			// must call DefWindowProc here, don't know why?
			return DefWindowProc(hwnd, message,
				wParam, lParam);
		case WM_COMMAND: wgui->onCommand(hwnd,LOWORD(wParam));
			break;
		case WM_CHAR:	wgui->onChar(hwnd,
			(SWord)LOBYTE(wParam), (int)LOWORD(lParam));
			break;
		case WM_KEYDOWN: wgui->onKeyDown(hwnd,
				wParam, (int)LOWORD(lParam));
			break;
		case WM_TIMER:   wgui->onTimer(hwnd, (UINT)wParam);
			break;
		case WM_SIZE:    return wgui->onSize(hwnd, (int)wParam,
					(int)LOWORD(lParam), (int)HIWORD(lParam));
		case WM_DESTROY: wgui->onDestroy();
			break;
		case WM_CLOSE: wgui->onClose();
			break;
		case WM_GETMINMAXINFO: return wgui->onMinMaxInfo((MINMAXINFO *)lParam);
		default: return DefWindowProc(hwnd, message, wParam, lParam);
	} // switch
	return TRUE;
} // e2windowWndProc 

void Win32Gui::onTimer(HWND hwnd, UINT id)
{
} // onTimer

void Win32Gui::onCommand(HWND hwnd, int cmd)
{
	switch (cmd) {
		// Main Window menu
#ifdef NAFS
		case IDM_UPDATE_FS:     updateNafs();           break;
#else
		case IDM_UPDATE_FS:                             break;
#endif
		case IDM_EXIT:          cpu->set_new_state(EXIT);break;
		case IDM_RUN:           set_new_state(RUN);     break;
		case IDM_STOP:          set_new_state(STOP);    break;
		case IDM_RESET:         set_new_state(RESET_RUN);break;
		case IDM_VIEW:          popup_cpu();            break;
		case IDM_ABOUT:         popup_about(hwnd);      break;
		case IDM_COPYRIGHT:     popup_copyright(hwnd);  break;
		case IDM_INTRODUCTION: popup_help(hwnd);        break;
	}
	return;
} // onCommand

void Win32Gui::onChar(HWND hwnd, SWord ch, int repeat)
{
	SWord key;
	
	if ((key = translate_to_ascii(ch)) >= 0)
	io->put_ch((Byte)key);

} // onChar

void Win32Gui::onKeyDown(HWND hwnd, SWord ch, int repeat)
{
	SWord key;
	
	if ((key = translate_to_ascii1(ch)) >= 0)
	io->put_ch((Byte)key);

} // onKeyDown

void Win32Gui::onPaint(HWND hwnd)
{
	e2video->init_blocks_to_update();
} // onPaint

int Win32Gui::onSize(HWND hwnd, int sizeType, int width, int height)
{
	RECT    oldRect, rect;

	if (sizeType == SIZE_RESTORED) {
		if (width % WINDOWWIDTH != 0 || height % WINDOWHEIGHT != 0) {
			guiXSize = (width + (WINDOWWIDTH / 2)) /  WINDOWWIDTH;
			guiYSize = (height + (WINDOWHEIGHT / 2)) /  WINDOWHEIGHT;
			GetWindowRect(e2screen, &oldRect);
			rect.left       = (long)0;
			rect.top        = (long)0;
			rect.right      = (long)WINDOWWIDTH * guiXSize;
			rect.bottom     = (long)WINDOWHEIGHT * guiYSize;
			if (AdjustWindowRectEx(&rect, WS_OVERLAPPEDWINDOW, TRUE, 0)) {
				width  = rect.right - rect.left;
				height = rect.bottom - rect.top;
			};
			MoveWindow(e2screen, oldRect.left, oldRect.top,
				width, height, TRUE);
		}
	}
	return  0;
} // onSize



void Win32Gui::onClose()
{
	cpu->set_new_state(EXIT);
} // onClose

void Win32Gui::onDestroy()
{
	PostQuitMessage(0);
} // onDestroy

int Win32Gui::onMinMaxInfo(MINMAXINFO *lpmmi)
{
	RECT    rect;

	rect.left       = (long)0;
	rect.top        = (long)0;
	rect.right      = (long)WINDOWWIDTH;
	rect.bottom     = (long)WINDOWHEIGHT;
	if (AdjustWindowRectEx(&rect, WS_OVERLAPPEDWINDOW, TRUE, 0)) {
		lpmmi->ptMinTrackSize.x = rect.right - rect.left;
		lpmmi->ptMinTrackSize.y = rect.bottom - rect.top;
	};
	rect.right      = (long)WINDOWWIDTH * MAX_GUIXSIZE;
	rect.bottom     = (long)WINDOWHEIGHT * MAX_GUIYSIZE;
	if (AdjustWindowRectEx(&rect, WS_OVERLAPPEDWINDOW, TRUE, 0)) {
		lpmmi->ptMaxTrackSize.x = rect.right - rect.left;
		lpmmi->ptMaxTrackSize.y = rect.bottom - rect.top;
	};
	return  0;
} // onMinMaxInfo

SWord Win32Gui::translate_to_ascii(SWord key)
{
	switch (key) {
		case 0xdf: return '^';  // german sharp s
		case 0xa7: return '#';  // paragraph
		case 0xb4: return '\'';
		case 0xb5: return 'u';  // my
		case 0xb0: return '0';  // degree
		case 0xb2: return '2';  // index 2
		case 0xb3: return '3';  // index 3
		case 0xf6: return '[';  // german umlaut oe
		case 0xe4: return ']';  // german umlaut ae
		case 0xfc: return '\\'; // german umlaut ue
		case 0xd6: return '{';  // german umlaut OE
		case 0xc4: return '}';  // german umlaut AE
		case 0xdc: return '|';  // german umlaut UE
	}
	return key;
} // translate_to_ascii

SWord Win32Gui::translate_to_ascii1(SWord key)
{               
	if (GetKeyState(VK_CONTROL) < 0)
		switch(key) {
			case VK_PAUSE:
			case VK_CANCEL: if (cpu) cpu->set_nmi(); return -1;
			case VK_HOME:   return 0xb7;
			case VK_UP:     return 0xb8;
			case VK_PRIOR:  return 0xb9;
			case VK_LEFT:   return 0xb4;
			case VK_CLEAR:  return 0xb5;
			case VK_RIGHT:  return 0xb6;
			case VK_END:    return 0xb1;
			case VK_DOWN:   return 0xb2;
			case VK_NEXT:   return 0xb3;
			default: return -1;
		}
	if (GetKeyState(VK_SHIFT) < 0)
		switch(key) {
			case VK_F1:     return 0xca;
			case VK_F2:     return 0xcb;
			case VK_F3:     return 0xcc;
			case VK_F4:     return 0xcd;
			case VK_F5:     return 0xce;
			case VK_F6:     return 0xcf;
			case VK_F7:     return 0xd0;
			case VK_F8:     return 0xd1;
			case VK_F9:     return 0xd2;
			case VK_PAUSE:
			case VK_CANCEL: if (cpu) cpu->set_nmi(); return -1;
			case VK_HOME:   return 0xe7;
			case VK_UP:     return 0xe8;
			case VK_PRIOR:  return 0xe9;
			case VK_LEFT:   return 0xe4;
			case VK_CLEAR:  return 0xe5;
			case VK_RIGHT:  return 0xe6;
			case VK_END:    return 0xe1;
			case VK_DOWN:   return 0xe2;
			case VK_NEXT:   return 0xe3;
			default: return -1;
		}
	switch(key) {
		case VK_F1:     return 0xc0;
		case VK_F2:     return 0xc1;
		case VK_F3:     return 0xc2;
		case VK_F4:     return 0xc3;
		case VK_F5:     return 0xc4;
		case VK_F6:     return 0xc5;
		case VK_F7:     return 0xc6;
		case VK_F8:     return 0xc7;
		case VK_F9:     return 0xc8;
		case VK_F10:    return 0xc9;
		case VK_F11:    return 0xca;
		case VK_F12:    return 0xcb;

		case VK_HOME:   return 0xf7;
		case VK_UP:     return 0xf8;
		case VK_PRIOR:  return 0xf9;
		case VK_LEFT:   return 0xf4;
		case VK_CLEAR:  return 0xf5;
		case VK_RIGHT:  return 0xf6;
		case VK_END:    return 0xf1;
		case VK_DOWN:   return 0xf2;
		case VK_NEXT:   return 0xf3;
		default: return -1;
	}
	return -1;
}

void Win32Gui::initialize(struct sGuiOptions *pOptions)
{
	AbstractGui::initialize(pOptions);
	wgui        = this;
	hAccel      = NULL;
	hAccelHwnd  = NULL;
	bp_input[0] = 0;
	bp_input[1] = 0;
	initialize_word_conv_table(0);
	initialize_e2window(pOptions);
} // initialize

#ifdef NAFS
void Win32Gui::updateNafs(void)
{
	HCURSOR origCursor;

	if (io->fdc != NULL) {
		origCursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
		if (!io->fdc->update_all_drives())
			popup_message("can't update file system\n\
There are open files\n");
		SetCursor(origCursor);
	}
}
#endif // ifdef NAFS

// return 0 on success
int Win32Gui::popup_help(HWND hwnd)
{
	char helpfile[PATH_MAX];
	char curdir[PATH_MAX];
	int res;

	GetCurrentDirectory(PATH_MAX, curdir);
	if (strlen(pOptions->doc_dir) != 0)
		strcpy(helpfile, pOptions->doc_dir);
	else {
		strcpy(helpfile, curdir);
		strcat(helpfile, "\\doc\\");
	}
	if (helpfile[strlen(helpfile)-1] != PATHSEPARATOR)
		strcat(helpfile, PATHSEPARATORSTRING);
	strcat(helpfile, "flexemu.htm");
	if ((res = (int)ShellExecute(
		hwnd,           // handle of parent window
		"open",         // open file
		helpfile,       // file to open
		NULL,           // parameters
		curdir,         // directory
		SW_SHOWNORMAL   // openmode for new window
	)) > 32)
		return 0; // success
	return -1; // no success
}

void Win32Gui::popup_message(char *pmessage)
{
	if (*pmessage)
		MessageBox(e2screen, pmessage, PROGRAMNAME " error",
			MB_OK | MB_ICONERROR);
}


void Win32Gui::event_queue(void)
{
	MSG msg;
	//int newX, newY;
	//int winX, winY;
	int count;
	int newButtonMask;

	count = 0;
	// process all queued events
	while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE)) {
		if (hAccel == NULL || hAccelHwnd != msg.hwnd ||
		     !TranslateAccelerator(msg.hwnd, hAccel, &msg)) {
			TranslateMessage(&msg);
			DispatchMessage(&msg);
			count++;
		}
	} // while

	if (GetActiveWindow() == e2screen) {
		newButtonMask = 0;
		if (IsLButtonDown())
			newButtonMask |= L_MB;
		if (IsMButtonDown())
			newButtonMask |= M_MB;
		if (IsRButtonDown())
			newButtonMask |= R_MB;
		if (GetKeyState(VK_CONTROL) < 0)
			newButtonMask |= CONTROL_KEY;
		if (GetKeyState(VK_SHIFT) < 0)
			newButtonMask |= SHIFT_KEY;
		io->buttonMask = newButtonMask;
	}

	/*      io->deltaX = newX - oldX;
		io->deltaY = newY - oldY;
		oldX = newX;
		oldY = newY;
	*/
	AbstractGui::event_queue(); // Poll parallel port
}


void Win32Gui::set_bell(int percent)
{
	// volume in percent is ignored
	Beep(400, 200);
} // set_bell

void Win32Gui::update_block(int block_number)
{
	HDC     hdc;
	HBITMAP hBitmapOrig;
	int     firstpartHeight; // Height of first part of divided block
	int     startLine;       // start scanline of block
	int     i;
	Byte    *src, *block;

if (e2video->are_blocks_to_update() || cpu->changed[block_number]) {
	hdc = GetDC(e2screen);
	if (!(e2video->vico1 & 0x02)) {
		// copy block from video ram into device independant bitmap
		src = (cpu->video_ram + block_number *
			YBLOCK_SIZE + e2video->vico1 * VIDEORAM_SIZE * 6);
		block = copy_block;
		switch ((guiXSize << 4) | guiYSize) {
		int     j;
		Byte    *btrg;
		Word    *trg;
			  case 0x11: // single width, single height
				// if standard gui Size directly copy from video_ram:
				block = src;
				break;
				// if not standard gui Size first calculate contents of
				// block to display:
			  case 0x21: // double width, single heigth
				trg = (Word *)copy_block;
				for (i = 0; i < BLOCKHEIGHT; i++) {
					for (j = 0; j < RASTERLINE_SIZE; j++) {
						*(trg++) = word_conv_tab[*(src++)];
					}
				}
				break;
			  case 0x22: // double width, double height
				trg = (Word *)copy_block;
				for (i = 0; i < BLOCKHEIGHT; i++) {
					for (j = 0; j < RASTERLINE_SIZE; j++) {
						*trg = *(trg+RASTERLINE_SIZE) =
						word_conv_tab[*(src++)];
						trg++;
					}
					trg += RASTERLINE_SIZE;
				}
				break;
			  case 0x23: // double width, tripple height
				trg = (Word *)copy_block;
				for (i = 0; i < BLOCKHEIGHT; i++) {
					for (j = 0; j < RASTERLINE_SIZE; j++) {
						*trg = *(trg+RASTERLINE_SIZE) =
						  *(trg+(2*RASTERLINE_SIZE)) =
							word_conv_tab[*(src++)];
						trg++;
					}
					trg += 2*RASTERLINE_SIZE;
				}
				break;
			  case 0x12: // single width, double height
			  case 0x13: // single width, tripple height
				btrg = copy_block;
				for (i = 0; i < BLOCKHEIGHT; i++) {
					for (j = 0; j < guiYSize; j++) {
						memcpy(btrg, src, RASTERLINE_SIZE);
						btrg += RASTERLINE_SIZE;
					}
					src += RASTERLINE_SIZE;
				}
				break;
		} // switch
		SetDIBits(
			hdc, image[guiXSize-1][guiYSize-1],
			0, BLOCKHEIGHT * guiYSize, 
			block, bmi[guiXSize-1][guiYSize-1], DIB_RGB_COLORS);
		hBitmapOrig = SelectBitmap(hMemoryDC, image[guiXSize-1][guiYSize-1]);
		startLine = ((WINDOWHEIGHT - e2video->vico2 +
			block_number * BLOCKHEIGHT) % WINDOWHEIGHT) * guiYSize;
		if (block_number == e2video->divided_block) {
			firstpartHeight = e2video->vico2 % BLOCKHEIGHT;
			// first half display on the bottom of the window
			BitBlt(hdc, 0, startLine,
				BLOCKWIDTH * guiXSize,
				firstpartHeight * guiYSize,
				hMemoryDC, 0, 0,
				SRCCOPY);
			// second half display on the top of window
			BitBlt(hdc, 0, 0,
				BLOCKWIDTH * guiXSize,
				(BLOCKHEIGHT - firstpartHeight) * guiYSize,
				hMemoryDC, 0, firstpartHeight * guiYSize,
				SRCCOPY);
			// first half display on the bottom of the window
		} else {
			BitBlt(hdc, 0, startLine,
				BLOCKWIDTH * guiXSize, BLOCKHEIGHT * guiYSize,
				hMemoryDC, 0, 0, SRCCOPY);
		} // else
		SelectBitmap(hMemoryDC, hBitmapOrig);
	} else {
		// display an "empty" screen:
		for (i = 0; i < RASTERLINE_SIZE * guiXSize * BLOCKHEIGHT * guiYSize; i++) {
			copy_block[i] = 0xff;
		}
		SetDIBits(
			hdc, image[guiXSize-1][guiYSize-1],
			0, BLOCKHEIGHT * guiYSize, 
			copy_block, bmi[guiXSize-1][guiYSize-1], DIB_RGB_COLORS);
		hBitmapOrig = SelectBitmap(hMemoryDC, image[guiXSize-1][guiYSize-1]);
		BitBlt(hdc, 0,
			(block_number * BLOCKHEIGHT) * guiYSize,
			BLOCKWIDTH * guiXSize,
			BLOCKHEIGHT * guiYSize,
			hMemoryDC, 0, 0, SRCCOPY);
		SelectBitmap(hMemoryDC, hBitmapOrig);
	} // else
	cpu->changed[block_number] = 0;
	ReleaseDC(e2screen, hdc);
} // if
} // update_block

void Win32Gui::initialize_e2window(struct sGuiOptions *pOptions)
{
	HWND w;

	if (!registerWindowClasses(pOptions->hInstance, 0)) {
		MessageBox(NULL, "RegisterClassEx failed\n" \
			"Unable to create Mainwindow of " PROGRAMNAME,
			PROGRAMNAME " error",
			MB_OK | MB_ICONERROR);
		exit(EXIT_FAILURE);
	}
	if ((w = create_main_view(pOptions)) == NULL) {
		MessageBox(NULL, "CreateWindowEx failed\n" \
			"Unable to create Mainwindow of " PROGRAMNAME,
			PROGRAMNAME " error",
			MB_OK | MB_ICONERROR);
		exit(EXIT_FAILURE);
	}       
	create_cpuview(w, pOptions);
	initialize_after_create(w, pOptions);
	manage_widget(w, pOptions);
	initialize_after_open(w, pOptions);
	e2screen = w;
} // initialize_e2window


BOOL Win32Gui::registerWindowClasses (HINSTANCE hinst, UINT ResPoolID)
{
	WNDCLASSEX      wcex ;

	// Fill in window class structure with parameters that describe
	// the Main window.
	wcex.cbSize        = sizeof (WNDCLASSEX) ;
	wcex.style         = CS_BYTEALIGNCLIENT | CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS ;
	wcex.lpfnWndProc   = e2windowWndProc ;
	wcex.cbClsExtra    = 0 ;
	wcex.cbWndExtra    = 0 ;
	wcex.hInstance     = hinst ;
	wcex.hIcon         = LoadIcon (hinst, MAKEINTRESOURCE (IDI_AFLEXMAIN)) ;
	wcex.hCursor       = LoadCursor (NULL, IDC_ARROW);
	wcex.hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH);
//      wcex.lpszMenuName  = MAKEINTRESOURCE(IDR_MAINMENU);
	wcex.lpszMenuName  = NULL;
	wcex.lpszClassName = "Flexemu";
	wcex.hIconSm       = NULL ;
	if (!RegisterClassEx (&wcex))
		return FALSE;

	// CPU Window
	wcex.lpfnWndProc   = cpuWindowWndProc ;
	wcex.hIcon         = LoadIcon (hinst, MAKEINTRESOURCE (IDI_FLEXCPU)) ;
	wcex.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
	wcex.lpszClassName = "Mc6809";
	if (!RegisterClassEx (&wcex))
		return FALSE;
	return TRUE;
}

HWND Win32Gui::create_main_view(struct sGuiOptions *pOptions)
{
   HWND hwnd;
   RECT rect;
   int  width, height;
   MENUITEMINFO menuItem = {
	sizeof(MENUITEMINFO),           // size of structure
	MIIM_ID | MIIM_STATE | MIIM_TYPE, // set ID, state and type
	MFT_STRING,                     // menutype (String, Bitmap, Sep. ...)
	MFS_ENABLED,                    // menustate (enabled, checked ...)
	0,                              // ID of menuitem
	NULL,                           // handle of submenu
	NULL,                           // bitmap for checked
	NULL,                           // bitmap for unchecked
	0,                              // user defined data
	"",                             // item string ...
	0};                             // stringlength if returned

   hInstance = pOptions->hInstance;
   menubar = CreateMenu();
//   menubar = NULL;
   if (menubar != NULL) {
	menu1 = CreateMenu();
	AppendMenu(menubar, MF_POPUP | MF_STRING, (UINT)menu1, "&File");
	menu2 = CreateMenu();
	AppendMenu(menubar, MF_POPUP | MF_STRING, (UINT)menu2, "&Processor");
	menu3 = CreateMenu();
	AppendMenu(menubar, MF_POPUP | MF_STRING, (UINT)menu3, "&Help");
#ifdef NAFS
	menuItem.dwTypeData = "&Update file system";
	menuItem.wID = IDM_UPDATE_FS;
	InsertMenuItem(menu1, 1, TRUE, &menuItem);
#endif
	menuItem.dwTypeData = "&Exit";
	menuItem.wID = IDM_EXIT;
	InsertMenuItem(menu1, 2, TRUE, &menuItem);

	menuItem.dwTypeData = "&Run";
	menuItem.wID = IDM_RUN;
	InsertMenuItem(menu2, 1, TRUE, &menuItem);
	menuItem.dwTypeData = "&Stop";
	menuItem.wID = IDM_STOP;
	InsertMenuItem(menu2, 2, TRUE, &menuItem);
	menuItem.dwTypeData = "R&eset";
	menuItem.wID = IDM_RESET;
	InsertMenuItem(menu2, 3, TRUE, &menuItem);
	menuItem.fType = MFT_SEPARATOR;
	InsertMenuItem(menu2, 4, TRUE, &menuItem);
	menuItem.dwTypeData = "&View";
	menuItem.fType = MFT_STRING;
	menuItem.wID = IDM_VIEW;
	InsertMenuItem(menu2, 5, TRUE, &menuItem);

	menuItem.dwTypeData = "&Introduction";
	menuItem.wID = IDM_INTRODUCTION;
	InsertMenuItem(menu3, 1, TRUE, &menuItem);      
	menuItem.dwTypeData = "&About " PROGRAMNAME;
	menuItem.wID = IDM_ABOUT;
	InsertMenuItem(menu3, 2, TRUE, &menuItem);
	menuItem.dwTypeData = "&Licence";
	menuItem.wID = IDM_COPYRIGHT;
	InsertMenuItem(menu3, 3, TRUE, &menuItem);
   }

   rect.left    = (long)0;
   rect.right   = (long)WINDOWWIDTH * guiXSize;
   rect.top     = (long)0;
   rect.bottom  = (long)WINDOWHEIGHT * guiYSize;
   if (AdjustWindowRectEx(&rect, WS_OVERLAPPEDWINDOW, TRUE, 0)) {
	width  = rect.right - rect.left;
	height = rect.bottom - rect.top;
   } else {
	width  = WINDOWWIDTH+8;
	height = WINDOWHEIGHT+24;
   }
   hwnd = CreateWindowEx (0,            // Extended window styles
		    PROGRAMNAME,        // Address of registered class name
		    PROGRAMNAME,          // Address of window name
		    WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU |
		       WS_THICKFRAME | WS_MINIMIZEBOX,  // Window style
		    CW_USEDEFAULT,      // Horizontal position of window
		    0,                  // Vertical position of window
		    width,              // Window width
		    height,             // Window height
		    NULL,               // Handle of parent or owner window
		    menubar,            // Handle of menu for this window
		    pOptions->hInstance,// Handle of application instance
		    NULL) ;             // Address of window-creation data

    //ASSERT (NULL != hwnd) ;

    if (!hwnd)
       return NULL ;
    return hwnd ;
}  // create_main_view


BOOL CALLBACK aboutDialogProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch(message) {
		case WM_INITDIALOG:
			return wgui->onAboutInit(hwnd);
		case WM_COMMAND:
			return wgui->onAboutCommand(hwnd,LOWORD(wParam));
		case WM_CLOSE:
			return wgui->onAboutClose(hwnd);
	} // switch
	return FALSE;
} // aboutDialogProc


void Win32Gui::popup_about(HWND hwnd)
{
	DialogBox(hInstance, MAKEINTRESOURCE(IDD_AB_DIALOG), hwnd, CALLBACKCAST aboutDialogProc);
} // popup_about

void Win32Gui::popdown_about(HWND hwnd)
{
	EndDialog(hwnd, 0);
}

void Win32Gui::stripBlanks(char *str)
{
	char *p1, *p2;

	p1 = str;
	p2 = str;
	while (*p2) {
		// skip all leading blanks
		while (*p2 == ' ')
			p2++;
		// copy line
		while (*p2 && *p2 != '\n')
			*(p1++) = *(p2++);
		*(p1++) = *p2;
		if (p2)
			p2++;
	} // while
} // stripBlanks

BOOL Win32Gui::onAboutInit(HWND hwnd)
{
	char    aboutstring[512];


	strcpy(aboutstring, "About ");
	strcat(aboutstring, PROGRAMNAME);
	SetWindowText(hwnd, aboutstring);
	strcpy(aboutstring, HEADER1);
	strcat(aboutstring, "V " PROGRAM_VERSION);
	strcat(aboutstring, HEADER2);
	stripBlanks(aboutstring);
	SetDlgItemText(hwnd, IDC_AB_TEXT, aboutstring);
	return TRUE;
}

BOOL Win32Gui::onAboutClose(HWND hwnd)
{
	popdown_about(hwnd);
	return TRUE;
}

BOOL Win32Gui::onAboutCommand(HWND hwnd, int cmd)
{
	switch (cmd) {
		// BP-Dialog button controls
		case IDC_AB_OK:         popdown_about(hwnd); break;
		case IDC_AB_COPYRIGHT:  popup_copyright(hwnd);
					popdown_about(hwnd); break;
		default:                return FALSE;
	}
	return TRUE;
} // onAboutCommand

void Win32Gui::popup_copyright(HWND hwnd)
{
	char str[PATH_MAX];
	char *copyrightFile = "COPYING.TXT";

	GetCurrentDirectory(PATH_MAX, str);
	if ((int)ShellExecute(
		hwnd,           // handle of parent window
		"open",         // open file
		copyrightFile,  // file to open
		NULL,           // parameters
		str,            // directory
		SW_SHOWNORMAL   // openmode for new window
		) <= 32) {
		sprintf(str, "Unable to display file %s", copyrightFile);
		MessageBox(e2screen, str, PROGRAMNAME " error",
			MB_OK | MB_ICONERROR);
	}
} // popup_copyright

void Win32Gui::initialize_after_create(HWND w, struct sGuiOptions *pOptions)
{
	HDC             hdc;
	int             bpp;
	struct sRGBDef  *pc;

	oldX          = 0;
	oldY          = 0;
	Word    red   = 255;
	Word    green = 255;
	Word    blue  = 255;
	int     foregroundIdx;
	int     backgroundIdx;
	int     i, j;
	BITMAPINFO *pbmi;

	cpu->initialize_byte_conv_table(0);
	pc = (struct sRGBDef *)&colors;
	while(pc->colorName != NULL) {
		if (strcmp(pOptions->color, pc->colorName) == 0)
			break;
		pc++;
	}
	if (pc->colorName != NULL) {
		red   = pc->red;
		green = pc->green;
		blue  = pc->blue;
	}
	if (!pOptions->inverse) {
		foregroundIdx = 1;
		backgroundIdx = 0;
	} else {
		foregroundIdx = 0;
		backgroundIdx = 1;
	}

	bpp = 1; // specify bits per pixel
	for (i = 0; i < MAX_GUIXSIZE; i++) {
		for (j = 0; j < MAX_GUIYSIZE; j++) {
			pbmi = (BITMAPINFO *)new char[sizeof(BITMAPINFOHEADER) +
				(1 << bpp) * sizeof(RGBQUAD)];
			hdc = GetDC(w);
			hMemoryDC = CreateCompatibleDC(hdc);

			pbmi->bmiHeader.biSize           = sizeof(BITMAPINFOHEADER);
			pbmi->bmiHeader.biPlanes         = 1;
			pbmi->bmiHeader.biWidth          = BLOCKWIDTH * (i+1);
			pbmi->bmiHeader.biHeight         = -BLOCKHEIGHT * (j+1);
			pbmi->bmiHeader.biBitCount       = bpp;
			pbmi->bmiHeader.biCompression    = BI_RGB;
			pbmi->bmiHeader.biSizeImage      = 0;
			pbmi->bmiHeader.biXPelsPerMeter  = 0;
			pbmi->bmiHeader.biYPelsPerMeter  = 0;
			pbmi->bmiHeader.biClrUsed        = 1 << bpp;
			pbmi->bmiHeader.biClrImportant   = 1 << bpp;
			pbmi->bmiColors[backgroundIdx].rgbBlue   = 0;
			pbmi->bmiColors[backgroundIdx].rgbGreen  = 0;
			pbmi->bmiColors[backgroundIdx].rgbRed    = 0;
			pbmi->bmiColors[foregroundIdx].rgbBlue   = (BYTE)blue;
			pbmi->bmiColors[foregroundIdx].rgbGreen  = (BYTE)green;
			pbmi->bmiColors[foregroundIdx].rgbRed    = (BYTE)red;
			pbmi->bmiColors[0].rgbReserved   = 0;
			pbmi->bmiColors[1].rgbReserved   = 0;
			image[i][j]  = CreateDIBitmap(hdc, &pbmi->bmiHeader, 0, NULL,
				pbmi, DIB_RGB_COLORS);
			bmi[i][j] = pbmi;
		} // for
	} // for
	// initialize Bitmap with all ones
	PatBlt(hdc, 0, 0, BLOCKWIDTH*guiXSize, BLOCKHEIGHT*guiYSize, WHITENESS);
	ReleaseDC(w, hdc);
}

void Win32Gui::manage_widget(HWND w, struct sGuiOptions *pOptions)
{
	ShowWindow(w, pOptions->nCmdShow);
}  // manage widget


void Win32Gui::initialize_after_open(HWND w, struct sGuiOptions *pOptions)
{
	UpdateWindow(w);
}

int Win32Gui::gui_type(void)
{
	return GUI_WINDOWS;
}

Win32Gui::Win32Gui(
	Mc6809* x_cpu = NULL,
	Inout*  x_io = NULL,
	E2video* x_video = NULL,
	struct sGuiOptions *pOptions = NULL) :
		AbstractGui(x_cpu, x_io, x_video, pOptions)
{
	initialize(pOptions);
}

Win32Gui::~Win32Gui()
{
	HDC hdc;
	int i, j;


	hdc = GetDC(e2screen);
	hAccel     = NULL;
	hAccelHwnd = NULL;
	// release all bitmaps
	for (i = 0; i < MAX_GUIXSIZE; i++) {
		for (j = 0; j < MAX_GUIXSIZE; j++) {
			DeleteObject(image[i][j]);
			if (bmi[i][j] != NULL)
				delete [] bmi[i][j];
		} // for
	} // for
	// delete memory device context
	DeleteDC(hMemoryDC);
	ReleaseDC(e2screen, hdc);
	DestroyWindow(e2screen);
	wgui = NULL;
}

#endif // #ifdef WIN32
