/*
    rfilecnt.cpp


    FLEXplorer, An explorer for any FLEX file or disk container
    Copyright (C) 1998-2000  W. Schwotzer

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <string.h>		// needed for memcpy
#include "rfilecnt.h"
#include "flexerr.h"


FlexRamFileContainer::FlexRamFileContainer() : FlexFileContainer(), pfb(NULL)
{
}

int FlexRamFileContainer::OpenForFile(const char *aPath)
{
	unsigned int sectors;

	// open file like implemented in upper class
	FlexFileContainer::OpenForFile(aPath);
	if (fp == NULL)
		return 0;
	sectors = ByteOffset(param.max_track+1, 1) / param.byte_p_sector ;
	delete pfb;
	pfb = new Byte[sectors * param.byte_p_sector];
	if (fseek(fp, 0, SEEK_SET)) {
		ex.setString(FERR_READING_FROM, *path);
		throw ex;
	}
	// read total disk into memory
	if (fread(&pfb[0], param.byte_p_sector, sectors, fp) != sectors) {
		ex.setString(FERR_READING_FROM, *path);
		throw ex;
	}
	return 1;
}


int FlexRamFileContainer::Close(void)
{
	int error;

	error = 0;
	if (fp != NULL) {
		unsigned int sectors;

		sectors = ByteOffset(param.max_track+1, 1) / param.byte_p_sector;
		if (fseek(fp, 0, SEEK_SET)) {
			ex.setString(FERR_WRITING_TO, *path);
			error = 1;
		}
		if (fwrite(pfb, param.byte_p_sector, sectors, fp) != sectors) {
			ex.setString(FERR_WRITING_TO, *path);
			error = 1;
		}
		fclose(fp);
		fp = NULL;
	}
	delete path;
	path = NULL;
	delete pfb;
	pfb = NULL;
	if (error)
		throw ex;
	return 1;
}

int FlexRamFileContainer::ReadSector(Byte *pbuffer, int trk, int sec)
{
	int pos;

	if (pfb == NULL) {
		return 0;
	}
	pos = ByteOffset(trk, sec);
	if (pos < 0)
		return 0;
	memcpy(pbuffer, pfb + pos, param.byte_p_sector);
	return 1;
}

int FlexRamFileContainer::WriteSector(const Byte *pbuffer, int trk, int sec)
{
	int pos;

	if (pfb == NULL) {
		return 0;
	}
	pos = ByteOffset(trk, sec);
	if (pos < 0)
		return 0;
	memcpy(pfb + pos, pbuffer, param.byte_p_sector);
	return 1;
}

void FlexRamFileContainer::Initialize_for_flx_format(
		s_floppy		*pfloppy,
		s_flex_header	*pheader,
		Byte			wp)
{
	FlexFileContainer::Initialize_for_flx_format(pfloppy, pheader, wp);
	pfloppy->type |= TYPE_RAM_CONTAINER;
}

void FlexRamFileContainer::Initialize_for_dsk_format(
		s_floppy	*pfloppy,
		s_formats	*pformat,
		Byte		wp)
{
	FlexFileContainer::Initialize_for_dsk_format(pfloppy, pformat, wp);
	pfloppy->type |= TYPE_RAM_CONTAINER;
}
