/*
    ndircont.cpp


    FLEXplorer, An explorer for any FLEX file or disk container
    Copyright (C) 1998-2000  W. Schwotzer

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <misc1.h>

#ifdef NAFS
#include <stdio.h>
#include <sys/stat.h>
#include <new>          // needed for (nothrow)
#ifdef _MSC_VER
#include <io.h>                 // needed for access
#endif
#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>

#include "bdir.h"
#include "ndircont.h"
#include "fdirent.h"
#include "fcinfo.h"
#include "flexerr.h"

// detailed debug messegas can be written to a debug file:
// #define DEBUG_FILE "nafs.log"
#define DEBUG_FILE "debug.txt"
#ifdef DEBUG_FILE
#include "debug.h"
#endif

int initializeWithZero(char *ptr, int count)
{ 
	char *p = ptr; 
	for (int i = 0; i < count; i++) 
		p[i] = '\0';
	return count;
}

NafsDirectoryContainer::NafsDirectoryContainer() : dir(NULL)
{
}

NafsDirectoryContainer::~NafsDirectoryContainer()
{
	// final cleanup: close if not already done
	try {
		Close();
	} catch (...) {
		// ignore exceptions
		// exceptions in destructors cause much problems
		// usually the file should be closed already
	}
}

int NafsDirectoryContainer::Open(const char *path)
{
	struct stat             sbuf;
	static Word             number = 0;

	if (path == NULL || stat(path, &sbuf) || !S_ISDIR(sbuf.st_mode)) {
		ex.setString(FERR_UNABLE_TO_OPEN, path);
		throw ex;
	}
	if (dir != NULL)
		delete dir;
	dir = new BString(path);
	attributes = 0;
	if (access(path, W_OK))
		attributes |= FLX_READONLY;
	mount(number);
	number++;
	return 1;       // satisfy compiler
}

// type, track and sectors parameter will be ignored
int NafsDirectoryContainer::Create(const char *pdir,
					const char *name,
					int track /* = 0 */, int sectors /* = 0 */,
					int type /* = TYPE_DSK_FORMAT */)
{
	struct stat             sbuf;
	static Word             number = 0;
	BString         totalPath;

	if (pdir == NULL || stat(pdir, &sbuf) || !S_ISDIR(sbuf.st_mode)) {
		ex.setString(FERR_UNABLE_TO_CREATE, name);
		throw ex;
	}
	totalPath = pdir;
	totalPath += PATHSEPARATORSTRING;
	totalPath += name;
	if (!BDirectory::Create(totalPath, 0755)) {
		ex.setString(FERR_UNABLE_TO_CREATE, name);
		throw ex;
	}
	mount(number);
	number++;
	return 1;       // satisfy compiler
};

int NafsDirectoryContainer::IsContainerOpened(void)
{
	return (dir != NULL);
};

int NafsDirectoryContainer::Close(void)
{
	if (dir != NULL) {
		close_new_files();
		free_memory();
		delete dir;
		dir = NULL;
	}
	return 1;
};

int     NafsDirectoryContainer::GetInfo(FlexContainerInfo& info)
{
	struct s_sys_info_sector buffer;

	if (dir == NULL)
		return 0;
	if (!ReadSector((Byte *)&buffer, 0, 3)) {
		ex.setString(FERR_READING_TRKSEC, 0, 3, *dir);
		throw ex;
	}
	info.SetDate(buffer.day, buffer.month, buffer.year);
	info.SetTrackSector(buffer.last_trk+1, buffer.last_sec);
	info.SetFree((((buffer.free[0] << 8) | buffer.free[1]) * param.byte_p_sector) >> 10);
	info.SetTotalSize(((buffer.last_sec * (buffer.last_trk+1)) * param.byte_p_sector) >> 10);
	info.SetName(buffer.disk_name);
	info.SetPath(*dir);
	info.SetType(param.type);
	info.SetAttributes(attributes);
	return 1;
}

int NafsDirectoryContainer::GetBytesPerSector(void)
{
	return param.byte_p_sector;
}

int NafsDirectoryContainer::IsWriteProtected(void)
{
	return param.write_protect;
}

int NafsDirectoryContainer::IsTrackValid(int track)
{
	return (track <= param.max_track);
}

int NafsDirectoryContainer::IsSectorValid(int track, int sector)
{
	if (track) 
		return (sector != 0 && sector <= (param.max_sector * 2)); 
	else
		return (sector != 0 && sector <= (param.max_sector0 * 2)); 
}

int     NafsDirectoryContainer::GetContainerType(void)
{
	return param.type;
}

///////////////////////////////////////////////////////
// enhanced interface only has a dummy implementation !
// (class can't be used within flexdisk)
///////////////////////////////////////////////////////

int     NafsDirectoryContainer::OpenDirectory(const char *pattern)
{return 0; };
int     NafsDirectoryContainer::NextDirEntry(FlexDirEntry& entry)
{return 0; };   
int     NafsDirectoryContainer::CloseDirectory(void)
{return 0; };
int     NafsDirectoryContainer::FindFile(const char *fileName, FlexDirEntry& entry)
{return 0; };   
int     NafsDirectoryContainer::DeleteFile(const char *fileName)
{return 0; };
int     NafsDirectoryContainer::RenameFile(const char *oldName, const char *newName)
{return 0; };
int     NafsDirectoryContainer::FileCopy(const char *sourceName, const char *destName,
		FileContainerIf& destination)
{return 0; };
int     NafsDirectoryContainer::SetAttributes(const char *fileName, int setMask, int clearMask)
{return 0; };


///////////////////////////////////////////////////////
// private interface
///////////////////////////////////////////////////////

void NafsDirectoryContainer::initialize_header(Byte wp) {
	param.offset        = 0;
	param.write_protect = wp;
	param.max_sector    = MAX_SECTOR >> 1;
	param.max_sector0   = (INIT_DIR_SECTORS + 4) >> 1;
	param.max_track     = MAX_TRACK;
	param.byte_p_sector = SECTOR_SIZE;
	param.byte_p_track0 = param.max_sector0 << 9; 
	param.byte_p_track  = param.max_sector << 9; 
	param.type                      = TYPE_DIRECTORY | TYPE_NAFS_DIRECTORY;
	pflex_links     = new struct s_link_table[LINK_TABLE_SIZE];
	initializeWithZero((char *)pflex_links, LINK_TABLE_SIZE * sizeof(s_link_table));
	pflex_sys_info  = new struct s_sys_info_sector[2];
	initializeWithZero((char *)pflex_sys_info, 2 * sizeof(s_sys_info_sector));
	pflex_unused    = new struct s_unused_sector;
	initializeWithZero((char *)pflex_unused, sizeof(s_unused_sector));
	pflex_directory = new struct s_dir_sector[INIT_DIR_SECTORS];
	initializeWithZero((char *)pflex_directory, INIT_DIR_SECTORS * sizeof(s_dir_sector));
	dir_sectors      = INIT_DIR_SECTORS;
	pnew_file       = new struct s_new_file[INIT_NEW_FILES];
	initializeWithZero((char *)pnew_file, INIT_NEW_FILES * sizeof(s_new_file));
	new_files        = INIT_NEW_FILES;
	dir_extend.sec_trk = 0;
	if (pflex_links == NULL || pflex_sys_info == NULL ||
	    pflex_unused == NULL || pflex_directory == NULL ||
	    pnew_file == NULL) {
		char *pmsg = "Not enough memory, can't continue\n";
#ifdef WIN32
		MessageBox(NULL, pmsg, PROGRAMNAME " error",
			MB_OK | MB_ICONERROR);
#else
		fprintf(stderr, pmsg);
#endif
		exit(EXIT_FAILURE);
	} // if
} // initialize_header

// if valid flex filename return is > 0 otherwise = 0
/*
	some examples:

	allowed:        x.a xx.a xxxxxxxx.a x xx xxxxxxxx
	not allowed:    x. .a xx. xxxxxxxxx.a X.a xxxxxxxxX.a
*/
bool NafsDirectoryContainer::IsFlexFilename(const char *pfilename,
						char *pname /* = NULL */,
						char *pext  /* = NULL */)
{
	int     result; // result from sscanf should be int
	char    dot;
	char	name[9];
	char	ext[4];

	dot    = '\0';
	result = sscanf(pfilename, "%1[a-z]%7[a-z0-9_-]",
		(char *)&name, (char *)&name + 1);
	if (!result || result == EOF)
		return false;
	if (result == 1)
		result = sscanf(pfilename, "%*1[a-z]%c%1[a-z]%2[a-z0-9_-]",
		&dot, (char *)&ext, (char *)&ext + 1);
	else
		result = sscanf(pfilename, "%*1[a-z]%*7[a-z0-9_-]%c%1[a-z]%2[a-z0-9_-]",
		&dot, (char *)&ext, (char *)&ext + 1);
	if (!result || result == 1 || result == EOF)
		return false;
	if (strlen(name) + strlen(ext) + (dot == '.' ? 1 : 0) != strlen(pfilename))
		return false;
	strupper(name);
	strupper(ext);
	if (pname)
		strcpy(pname, name);
	if (pext)
		strcpy(pext, ext);
	return true;
} // IsFlexFilename


void NafsDirectoryContainer::initialize_flex_directory()
{
	Word i, j;
	char *p;

	for (i = 0; i < dir_sectors; i++) {
		p = (char *)(pflex_directory + i);
		for (j=0; j < SECTOR_SIZE; j++)
			*(p++) = 0x00;
		if (i < dir_sectors - 1) {
			(pflex_directory+i)->next_trk = (i+5) / MAX_SECTOR;
			(pflex_directory+i)->next_sec =
				((i+5) % MAX_SECTOR) + 1;
		} // if
	} // for
} // initialize_flex_directory


void NafsDirectoryContainer::scan_unix_filename(char *pfn, char *pfilename)
{
	char *p;
	Word i, j;

	j = 0;
	p = (char *)pfn;
	if (*p != -1) {
		// get file name
		for (i=0; i < 8 && *p; i++) {
			*(pfilename+i) = tolower(*p);
			p++;
		}
		if (i > 0) {
			p += 8 - i;
			*(pfilename + (i++)) = '.';
			// get file extension
			for (j=i; j < i+3 && *p; j++) {
				*(pfilename+j) = tolower(*p);
				p++;
			} // for
		} // if
	} // if
	*(pfilename+j) = '\0';
} // scan_unix_filename


// return unix filename from a pointer to a flex directory entry 
char *NafsDirectoryContainer::to_unix_filename(s_dir_entry *pde)
{
	static char filename[12];

	scan_unix_filename((char *)pde, (char *)&filename);
	return (char *)&filename;
} // to_unix_filename


// return unix filename from a given file_id
char *NafsDirectoryContainer::unix_filename(SWord file_id)
{
	static char filename[12];

	if (file_id < 0)
		sprintf((char *)&filename, "tmp%02d",  NEW_FILE1 - file_id);
	else
		scan_unix_filename(
		   (char *)&((pflex_directory + (file_id/10))->dir_entry[file_id%10].filename),
		   (char *)&filename);
	return (char *)&filename;
} // unix_filename

// return the record nr (beginning from 0) of a new file which first
// sector has index 'index' into flex link table
Word NafsDirectoryContainer::record_nr_of_new_file(SWord new_file_index, Word index)
{
	SWord   i;
	Word    record_nr;

	record_nr = 0;
	i = (pnew_file + new_file_index)->first.st.trk * MAX_SECTOR +
		(pnew_file + new_file_index)->first.st.sec - 1;
	while (i != index && i >= 0) {
		record_nr++;
		i = (pflex_links+i)->next.st.trk * MAX_SECTOR +
			(pflex_links+i)->next.st.sec - 1;
	} // while
	return record_nr;
}

SWord NafsDirectoryContainer::index_of_new_file(Byte track, Byte sector)
{
	SWord           i;
	char            path[PATH_MAX+1];
	s_new_file      *pnf;

	i = 0;
	while ((pnew_file+i)->first.sec_trk && i < new_files) {
		if (track == (pnew_file+i)->next.st.trk &&
		    sector == (pnew_file+i)->next.st.sec)
			return i;
		i++;
	} // while
	if (i >= new_files) {
		// new file table is full, increase new file table by 2
		pnf = new(std::nothrow) struct s_new_file[new_files + 2];
		if (pnf == NULL)
			return -1; // can't allocate memory
		memcpy(pnf, pnew_file, new_files * sizeof(struct s_new_file));
		delete [] pnew_file;
		pnew_file = pnf;
		new_files += 2;
		(pnew_file+(new_files-2))->first.sec_trk = 0;
		(pnew_file+(new_files-1))->first.sec_trk = 0;
	} // if
	strcpy((char *)&(pnew_file+i)->filename,
		unix_filename(NEW_FILE1 - i));
	strcpy((char *)&path, *dir);
	strcat((char *)&path, PATHSEPARATORSTRING);
	strcat((char *)&path, (pnew_file+i)->filename);
#ifdef sun
	(pnew_file+i)->fp = fopen(path, "w+");
#else
	(pnew_file+i)->fp = fopen(path, "wb+");
#endif
	if ((pnew_file+i)->fp == NULL)
		return -1;
	(pnew_file+i)->first.st.trk = track;
	(pnew_file+i)->first.st.sec = sector;
	(pnew_file+i)->next.sec_trk = 0;
	(pnew_file+i)->f_record     = 0;
	return i;
} // index_of_new_file


Byte NafsDirectoryContainer::extend_directory(
	SWord                   index,
	s_dir_sector    *pdb)
{
	struct s_dir_sector *pfd;

	// increase flex_directory by one sector
	pfd = new(std::nothrow) struct s_dir_sector[dir_sectors + 1];
	if (pfd == NULL)
		return 0xff; // can't allocate memory
	memcpy(pfd, pflex_directory, dir_sectors * sizeof(struct s_dir_sector));
	memcpy(pfd+dir_sectors, pdb, SECTOR_SIZE);
	(pflex_links+index)->f_record = dir_sectors;
	(pflex_links+index)->file_id  = DIRECTORY;
	delete [] pflex_directory;
	pflex_directory = pfd;
	dir_sectors++;
	dir_extend.sec_trk = 0;// reset directory extend track/sector
	return 0;
} // extend_directory


t_st *NafsDirectoryContainer::link_address(void)
{
	static t_st     link;
	s_dir_entry     *pd;
	Word            i;

	link.sec_trk = 0;
	for (i = 0; i < dir_sectors * 10; i++) {
		pd = &((pflex_directory + (i / 10))->dir_entry[i % 10]);
		if (!strncmp(pd->filename, "FLEX\0\0\0\0", 8) &&
			!strncmp(pd->file_ext, "SYS", 3)) {
			link.st.trk = pd->start_trk;            
			link.st.sec = pd->start_sec;
			break;
		}               
	} // for
	return &link;
}  // link_address


//if directory can't be extended return -1
SWord NafsDirectoryContainer::next_free_dir_entry()
{
	SWord                   i;
	Word                    j, index;
	s_sys_info_sector       *psis;
	char                    sector_buffer[SECTOR_SIZE];
	Word                    trk, sec;

	for (i = 0; i < dir_sectors * 10; i++) {
		if (!((pflex_directory + (i / 10))->dir_entry[i % 10].filename[0]))
			return i;
	} // for
	psis = pflex_sys_info;
	for (j = 0; j < SECTOR_SIZE; j++)
		sector_buffer[j] = 0x00;
	sector_buffer[3] = dir_sectors - INIT_DIR_SECTORS + 1;
	trk = psis->fc_start_trk;
	sec = psis->fc_start_sec;
	index = trk * MAX_SECTOR + sec - 1;
	if (!extend_directory(index, (s_dir_sector *)sector_buffer)) {
		(pflex_directory+dir_sectors-2)->next_trk =
			(Byte)trk;
		(pflex_directory+dir_sectors-2)->next_sec =
			(Byte)sec;
		if (++psis->fc_start_sec > MAX_SECTOR) {
			psis->fc_start_sec = 1;
			psis->fc_start_trk++;
		}
		if (--psis->free[1] == 0xff)
			psis->free[0]--;
		return i;
	} else
		return -1;
}  // next_free_dir_entry


void NafsDirectoryContainer::initialize_flex_link_table()
{
	Word                                            i, fc_start, free;
	struct s_link_table                     *plink;
	struct s_sys_info_sector        *psis;

	fc_start = MAX_SECTOR;
	// entries for system and directory sectors
	for (i = 0; i < fc_start; i++) {
		plink = pflex_links + i;
		plink->next.sec_trk     = 0;
		plink->record_nr[0]     = 0;
		plink->record_nr[1]     = 0;
		plink->f_record         = i < 4 ? 0 : i - 4;
		plink->file_id          = i < 4 ? SYSTEM : DIRECTORY;
	} // for
	for (i = fc_start; i < LINK_TABLE_SIZE; i++) {
		plink = pflex_links + i;
		if (i == LINK_TABLE_SIZE-1)
			plink->next.sec_trk = 0;
		else {
			plink->next.st.trk = (i+1) / MAX_SECTOR;
			plink->next.st.sec = ((i+1) % MAX_SECTOR) + 1;
		} // else
		plink->record_nr[0]     = 0;
		plink->record_nr[1]     = 0;
		plink->f_record         = 0;
		plink->file_id          = FREE_CHAIN;
	} // for
	free = LINK_TABLE_SIZE - fc_start;
	// and now update system info sectors
	for (i = 0; i < 2; i++) {
		psis = pflex_sys_info + i;
		psis->fc_start_trk      = fc_start / MAX_SECTOR;
		psis->fc_start_sec      = (fc_start % MAX_SECTOR) + 1;
		psis->fc_end_trk        = MAX_TRACK;
		psis->fc_end_sec        = MAX_SECTOR;
		psis->free[0]           = free >> 8;
		psis->free[1]           = free & 0xff;
	} // for
} // initialize_flex_link_table


void NafsDirectoryContainer::initialize_new_file_table()
{
	Word i;

	for (i = 0; i < new_files; i++) {
		(pnew_file+i)->first.sec_trk = 0;
		(pnew_file+i)->next.sec_trk = 0;
	} // for
} // initialize_new_file_table

// check for any open file
Byte NafsDirectoryContainer::open_files()
{
	for (Word i = 0; i < new_files; i++) {
		if ((pnew_file+i)->first.sec_trk != 0)
			return 1;
	} // for
	return 0;
} // open_files


void NafsDirectoryContainer::close_new_files()
{
	Word            i;
	Byte            first = 1;
	char            msg[ERR_SIZE];

	msg[0] = '\0';
	for (i = 0; i < new_files; i++) {
		if ((pnew_file+i)->first.sec_trk != 0) {
			if (first) { 
				strcpy(msg, "There are still open files\ntemporarily stored as:\n");
				first = 0;
			}
			fclose((pnew_file+i)->fp);
			if (strlen(msg) < (ERR_SIZE - 18)) {
				strcat(msg, "   ");
				strcat(msg, (pnew_file+i)->filename);
				strcat(msg, "\n");
			} // if
		} // if
	} // for
	if (strlen(msg) > 0)
#ifdef WIN32
		MessageBox(NULL, msg, PROGRAMNAME " warning",
			MB_OK | MB_ICONEXCLAMATION);
#else
		fprintf(stderr, msg);
#endif
} // close_new_files


void NafsDirectoryContainer::free_memory()
{
	delete [] pflex_links;
	delete [] pflex_directory;
	delete [] pflex_sys_info;
	delete    pflex_unused;
	delete [] pnew_file;
} // free_memory

// if file won't fit return 0 otherwise return 1
Byte NafsDirectoryContainer::add_to_link_table(
	SWord           dir_index,
	off_t           size,
	Byte            random,
	t_st            *pbegin,
	t_st            *pend)
{
	Word                            i, free, begin, records;
	struct s_link_table             *plink;
	struct s_sys_info_sector        *psis;

	psis = pflex_sys_info;  
	free = (psis->free[0] << 8) + psis->free[1];
	if (size > (off_t)(free * 252L))
		return 0;
	records = (size + 251L) / 252;
	pbegin->st.sec = psis->fc_start_sec;
	pbegin->st.trk = psis->fc_start_trk;
	begin = (pbegin->st.trk * MAX_SECTOR) + pbegin->st.sec - 1;
	for (i = 1; i <= records; i++) {
		plink = pflex_links + i + begin - 1;
		if (i == records) 
			plink->next.sec_trk = 0;
		if (random) {
			plink->record_nr[0]     = i > 2 ? (i-2) >> 8 : 0;
			plink->record_nr[1]     = i > 2 ? (i-2) & 0xff : 0;
		} else {
			plink->record_nr[0]     = i >> 8;
			plink->record_nr[1]     = i & 0xff;
		}
		plink->f_record         = i-1;
		plink->file_id          = dir_index;
	} // for
	pend->st.sec = ((i + begin - 2) % MAX_SECTOR) + 1;
	pend->st.trk = (i + begin - 2) / MAX_SECTOR;
	// update sys info sector
	psis->fc_start_sec = ((i + begin - 1) % MAX_SECTOR) + 1;
	psis->fc_start_trk = (i + begin -1) / MAX_SECTOR;
	psis->free[0] = (free - records) >> 8;
	psis->free[1] = (free - records) & 0xff;
	return 1;
} // add_to_link_table


void NafsDirectoryContainer::add_to_directory(
	char            *name,
	char            *ext,
	SWord           dir_index,
	Byte            random,
	struct          stat *pstat,
	t_st            *pbegin,
	t_st            *pend,
	Byte            wp)
{
	s_dir_entry     *pd;
	struct tm       *lt;
	SWord           records;

	lt = localtime(&(pstat->st_mtime));
	records = (pstat->st_size + 251L) / 252;
	pd = &((pflex_directory + (dir_index / 10))->dir_entry[dir_index % 10]);
	strcpy(pd->filename, name);
	strcpy(pd->file_ext, ext);
	pd->file_attr   = wp ? 0xc0 : 0x00;
	pd->start_trk   = pbegin->st.trk;
	pd->start_sec   = pbegin->st.sec;
	pd->end_trk     = pend->st.trk;
	pd->end_sec     = pend->st.sec;
	pd->records[0]  = records >> 8;
	pd->records[1]  = records & 0xff;
	pd->sector_map  = random ? 0x02 : 0x00;
	pd->month       = lt->tm_mon + 1;
	pd->day         = lt->tm_mday;
	pd->year        = lt->tm_year;
} // add_to_directory

Word NafsDirectoryContainer::is_in_file_random(const char *ppath, const char *pfilename)
{
	FILE    *fp;
	char    str[PATH_MAX+1];

	strcpy(str, ppath);
	strcat(str, PATHSEPARATORSTRING "random");
	if ((fp = fopen(str, "r")) != NULL) {
		while (!feof(fp)) {
			fgets(str, PATH_MAX, fp);
			if (strchr(str, '\n'))
				*strchr(str, '\n') = '\0';
			if (strcmp(pfilename, str) == 0) {
				fclose(fp);
				return 1;
			}
		}
		fclose(fp);
	} // if
	return 0;

} // is_in_file_random

void NafsDirectoryContainer::modify_random_file(char *path, struct stat *pstat, t_st *pbegin)
{
	Byte    file_sector_map[252 * 2];
	SDWord  data_size;
	Word    i, n, index;
	FILE    *fp;

	data_size = pstat->st_size - (252L * 2);
	if (data_size >= 252L * 2) {
		index = (pbegin->st.trk * MAX_SECTOR + pbegin->st.sec) - 1 + 2;
		for (i = 0; i < 252 * 2; i++)
			file_sector_map[i] = 0;
		n = 0;
		for (n = 0; n < (data_size / (252L * 255)) ; n++) {
			file_sector_map[3*n]   = index / MAX_SECTOR;
			file_sector_map[3*n+1] = (index % MAX_SECTOR) + 1;
			file_sector_map[3*n+2] = 255;
			index += 255;
		} // for
		i = (Word)(data_size % (252L * 255));
		if (i != 0) {
			file_sector_map[3*n]   = index / MAX_SECTOR;
			file_sector_map[3*n+1] = (index % MAX_SECTOR) + 1;
			file_sector_map[3*n+2] = (i+251) / 252;
		} // if
#ifdef sun
		fp = fopen(path, "r+");
#else
		fp = fopen(path, "rb+");
#endif
		if (fp != NULL)
			fwrite(&file_sector_map, 252, 2, fp);
		fclose(fp);
	} // if
} // modify_random_file

// dwp = write protect for drive

void NafsDirectoryContainer::fill_flex_directory(Byte dwp)
{
#ifdef WIN32
	HANDLE          hdl;
	WIN32_FIND_DATA pentry;
#else
	DIR             *pd;
	struct dirent   *pentry;
#endif
	char            name[9], ext[4], path[PATH_MAX+1], *pfilename;
	SWord           dir_index;
	Word            random;
	t_st            begin, end;
	struct stat     sbuf;

	initialize_flex_directory();
	initialize_flex_link_table();
	dir_index = 0;
#ifdef WIN32
	strcpy(path, *dir);
	strcat(path, PATHSEPARATORSTRING "*.*");
	if ((hdl = FindFirstFile(path, &pentry)) != INVALID_HANDLE_VALUE) {
		do {
#else
	if ((pd = opendir(*dir)) != NULL) {
		while ((pentry = readdir(pd)) != NULL) {
#endif
#ifdef WIN32
			pfilename = pentry.cFileName;
			strlower(pfilename);
#else
			pfilename = (char *)&pentry->d_name;
#endif
			random = 0;
			strcpy((char *)&path, *dir);
			strcat((char *)&path, PATHSEPARATORSTRING);
			strcat((char *)&path, pfilename);
			if (IsFlexFilename(pfilename, (char *)&name, (char *)&ext) &&
			   !stat(path, &sbuf) && (S_ISREG(sbuf.st_mode)) &&
			   sbuf.st_size > 0 &&
			   (dir_index = next_free_dir_entry()) >= 0) {
#ifdef WIN32
				random = (pentry.dwFileAttributes & FILE_ATTRIBUTE_HIDDEN) ? 1 : 0;
#else
				random = sbuf.st_mode & S_IXUSR;
#endif
				// CDFS-Support: look for file name in file 'random'
				if (strcmp(pfilename, "errors.sys") == 0)
					random = random;
				if (dwp)
					random |= is_in_file_random(*dir, pfilename);
				if (add_to_link_table(dir_index,
					sbuf.st_size, (Byte)random, &begin, &end)) {
					add_to_directory(name, ext,
					   dir_index, (Byte)random, &sbuf,
					   &begin, &end, dwp || access(path, W_OK));
					// Unfinished: don't write sector map if write protected
					if (random && !dwp)
						modify_random_file(path,
							&sbuf, &begin);
				}
			}
#ifdef WIN32
		} while (FindNextFile(hdl, &pentry) != 0);
		FindClose(hdl);
#else
		} // while
		closedir(pd);
#endif
	} //if 
} // initialize_flex_directory


void NafsDirectoryContainer::initialize_flex_sys_info_sectors(Word number)
{
	Word                            i;
	char                            name[9], ext[4];
	const char                      *pName;
	struct stat                     sbuf;
	struct tm                       *lt;
	struct s_sys_info_sector        *psis;

	if(!stat(*dir, &sbuf)) {
		psis = pflex_sys_info;
		lt = localtime(&(sbuf.st_mtime));
		name[0] = '\0';
		ext[0]  = '\0';
		pName = dir->chars() + strlen(*dir) - 1;
		if (*pName == PATHSEPARATOR)
			pName--;
		while (pName != dir->chars() && *pName != PATHSEPARATOR)
			pName--;
		if (*pName == PATHSEPARATOR)
			pName++;
		if (sscanf(pName, "%8[a-zA-Z0-9_-]", name) != 1)
			strcpy(name, "UNIX");
		else {
			strupper(name);
			strupper(ext);
		} // else
		for (i=0; i < 16; i++)
			psis->unused1[i] = 0;
		strncpy((char *)&(psis->disk_name), name, 8);
		strncpy((char *)&(psis->disk_ext), ext, 3);
		psis->disk_number[0] = number >> 8;
		psis->disk_number[1] = number & 0xff;
		psis->fc_start_trk   = 0;
		psis->fc_start_sec   = 0;
		psis->fc_end_trk     = 0;
		psis->fc_end_sec     = 0;
		psis->free[0]        = 0;
		psis->free[1]        = 0;
		psis->month          = lt->tm_mon + 1;
		psis->day            = lt->tm_mday;
		psis->year           = lt->tm_year;
		psis->last_trk       = MAX_TRACK;
		psis->last_sec       = MAX_SECTOR;
	
		for (i=0; i < 216; i++)
			psis->unused2[i] = 0;
		memcpy(pflex_sys_info + 1, pflex_sys_info, sizeof(struct s_sys_info_sector));
	} // if
} // initialize_flex_sys_info_sectors


// I don't know what this sector should be used for but
// emulate it anyway
void NafsDirectoryContainer::initialize_flex_unused_sector()
{
	Word i;

	pflex_unused->next_trk = 0x00;
	pflex_unused->next_sec = 0x03;
	for (i=0; i < 254; i++)
		pflex_unused->unused[i] = 0;
} // initialize_flex_unused_sector


// change the file id in flex_links
void NafsDirectoryContainer::change_file_id(SWord index, SWord old_id, SWord new_id)
{
	SWord i = index;
	while (i >= 0 && (pflex_links+i)->file_id == old_id) {
		(pflex_links+i)->file_id = new_id;
		i = (pflex_links+i)->next.st.trk * MAX_SECTOR +
			(pflex_links+i)->next.st.sec - 1;
	} // while
} // change_file_id


void NafsDirectoryContainer::check_for_delete(SWord dir_index, s_dir_sector *pdb)
{
	Word            i;
	SWord           index;
	char            path[PATH_MAX+1], *pfilename;
	s_dir_sector    *pd;

	pd = pflex_directory + dir_index;
	for (i = 0; i < 10; i++) {
		if (pdb->dir_entry[i].filename[0] == -1 &&
		   pd->dir_entry[i].filename[0] != -1) {
			pfilename = unix_filename(dir_index * 10 + i);
			index = pd->dir_entry[i].start_trk * MAX_SECTOR +
				pd->dir_entry[i].start_sec - 1;
			strcpy((char *)&path, *dir);
			strcat((char *)&path, PATHSEPARATORSTRING);
			strcat((char *)&path, pfilename);
			unlink(path);
			change_file_id(index, dir_index * 10 + i, FREE_CHAIN);
#ifdef DEBUG_FILE
			LOG_X("     delete %s\n", pfilename);
#endif
			break;
		} // if
	} // for        

} // check_for_delete


void NafsDirectoryContainer::check_for_rename(SWord dir_index, s_dir_sector *pdb)
{
	Word    i;
	char    old_path[PATH_MAX+1], new_path[PATH_MAX+1];
	char    *pfilename1, *pfilename2;

	for (i = 0; i < 10; i++) {
		pfilename1 = unix_filename(dir_index * 10 + i);
		pfilename2 = to_unix_filename(&pdb->dir_entry[i]);
		if (*pfilename1 && *pfilename2 &&
		   strcmp(pfilename1, pfilename2) != 0) {
			strcpy((char *)&old_path, *dir);
			strcat((char *)&old_path, PATHSEPARATORSTRING);
			strcat((char *)&old_path, pfilename1);
			strcpy((char *)&new_path, *dir);
			strcat((char *)&new_path, PATHSEPARATORSTRING);
			strcat((char *)&new_path, pfilename2);
			rename(old_path, new_path);
#ifdef DEBUG_FILE
			LOG_XX("     rename %s to %s\n",
				pfilename1, pfilename2);
#endif
			break;
		} // if
	} // for        
} // check_for_rename


void NafsDirectoryContainer::check_for_extend(SWord dir_index, s_dir_sector *pdb)
{
	s_dir_sector *pd;

	pd = pflex_directory + dir_index;
	if (!pd->next_sec && !pd->next_trk && (pdb->next_trk || pdb->next_sec)) {
		   dir_extend.st.trk = pdb->next_trk;
		   dir_extend.st.sec = pdb->next_sec;
	} // if
} // check_for_extend


// return -1 if not successful otherwise return 0
// years < 70 will be represented as >= 2000
SWord NafsDirectoryContainer::set_file_time(char *ppath, Byte month, Byte day, Byte year)
{
	struct stat    statbuf;
	struct utimbuf timebuf;
	struct tm      file_time;

	if (stat(ppath, &statbuf) >= 0) {
		timebuf.actime = statbuf.st_atime;
		file_time.tm_sec   = 0;
		file_time.tm_min   = 0;
		file_time.tm_hour  = 0;
		file_time.tm_mon   = month-1;
		file_time.tm_mday  = day;
		file_time.tm_year  = year < 70 ? year + 100 : year;
		file_time.tm_isdst = 0;
		timebuf.modtime    = mktime(&file_time);
		if (timebuf.modtime >= 0 && utime(ppath, &timebuf) >= 0)
				return 0;
			else
				return -1;
	} // if
	return -1;
} // set_file_time


Byte NafsDirectoryContainer::check_for_new_file(SWord dir_index, s_dir_sector *pd)
{
	Word            i, j;
	SWord           index;
	char            old_path[PATH_MAX+1], new_path[PATH_MAX+1];
#ifndef WIN32
	struct stat     sbuf;
#endif

	j = 0;
	while ((pnew_file+j)->first.sec_trk && j < new_files) {
		for (i = 0; i < 10; i++) {
			if ((pnew_file+j)->first.st.sec == pd->dir_entry[i].start_sec &&
			    (pnew_file+j)->first.st.trk == pd->dir_entry[i].start_trk) {
				index = (pnew_file+j)->first.st.trk * MAX_SECTOR +
				   (pnew_file+j)->first.st.sec - 1; 
				change_file_id(index, NEW_FILE1-j,
					10 * dir_index + i);
				fclose((pnew_file+j)->fp);
				strcpy((char *)&old_path, *dir);
				strcat((char *)&old_path, PATHSEPARATORSTRING);
				strcat((char *)&old_path, (pnew_file+j)->filename);
				// check for random file, if true set user execute bit
				if (pd->dir_entry[i].sector_map & 0x02)
#ifdef WIN32
					SetFileAttributes(old_path, FILE_ATTRIBUTE_HIDDEN);
#else
					if (!stat(old_path, &sbuf))
						chmod(old_path, sbuf.st_mode | S_IXUSR);
#endif
				strcpy((char *)&new_path, *dir);
				strcat((char *)&new_path, PATHSEPARATORSTRING);
				strcat((char *)&new_path,
					unix_filename((pflex_links+index)->file_id));
				rename(old_path, new_path); 
#ifdef DEBUG_FILE
			LOG_XX("     new file %s, was %s\n",
				unix_filename((pflex_links+index)->file_id),
				(pnew_file+j)->filename);
#endif
				set_file_time(new_path, pd->dir_entry[i].month,
				   pd->dir_entry[i].day, pd->dir_entry[i].year);
				// remove entry in new file table and
				// move following entries
				while ((pnew_file+j+1)->first.sec_trk && j < new_files-1) {
					strcpy((char *)&(pnew_file+j)->filename, (pnew_file+j+1)->filename);
					(pnew_file+j)->first.sec_trk = (pnew_file+j+1)->first.sec_trk;
					(pnew_file+j)->next.sec_trk  = (pnew_file+j+1)->next.sec_trk;
					(pnew_file+j)->f_record      = (pnew_file+j+1)->f_record;
					(pnew_file+j)->fp            = (pnew_file+j+1)->fp;
					index = (pnew_file+j)->first.st.trk * MAX_SECTOR +
					   (pnew_file+j)->first.st.sec - 1; 
					change_file_id(index, NEW_FILE1-j-1, NEW_FILE1-j);
					j++;
				} // while
				(pnew_file+j)->first.sec_trk = 0;
				return 0;
			} // if
		} // for
		j++;
	} // while
	return 0;
} // check_for_new_file


Byte NafsDirectoryContainer::last_of_free_chain(Byte trk, Byte sec)
{
	return (pflex_sys_info->fc_end_trk == trk &&    
		pflex_sys_info->fc_end_sec == sec);
} // last_of_free_chain


// return != 0 on success
int     NafsDirectoryContainer::ReadSector(Byte *buffer, int trk, int sec)
{
	char                    *p, path[PATH_MAX+1], *mode;
	Word                    i, di;
	size_t                  bytes;
	FILE                    *fp;
	t_st                    *link;
	struct s_link_table     *pfl;
	int                             result;

#ifdef DEBUG_FILE
	LOG_XX("read: %02X/%02X ", trk, sec);
#endif
#ifdef sun
	mode = "r";
#else
	mode = "rb";
#endif
	result = 1;
	i = trk * MAX_SECTOR + (sec - 1);
	pfl = pflex_links+i;
	switch (pfl->file_id) {
		case SYSTEM :
#ifdef DEBUG_FILE
			LOG("systemsector\n");
#endif
			p = NULL; // satisfy compiler
			if (sec >= 3)
				p = (char *)(pflex_sys_info + sec - 3);
			else if (sec == 2)
				p = (char *)pflex_unused;
			else if (sec == 1) {
				bytes = SECTOR_SIZE;
				for (i = 0; i < SECTOR_SIZE; i++)
					buffer[i] = 0;
				buffer[0] = 0x39; // means RTS
				link = link_address();
				strcpy((char *)&path, *dir);
				strcat((char *)&path, PATHSEPARATORSTRING "boot");
				if ((fp = fopen(path, mode)) != NULL) {
					bytes = fread(&buffer, 1, SECTOR_SIZE, fp);
					fclose(fp);
				}
				if (bytes != SECTOR_SIZE) {
					result = 0;
				}
				buffer[3] = link->st.trk;
				buffer[4] = link->st.sec;
				break;
			} else
				break;
			memcpy(buffer, p, param.byte_p_sector);
			break;
		case DIRECTORY:
#ifdef DEBUG_FILE
			LOG("directorysector\n");
#endif
			di = pfl->f_record;
			p = (char *)(pflex_directory + di);
			memcpy(buffer, p, param.byte_p_sector);
			break;
		case FREE_CHAIN :
#ifdef DEBUG_FILE
			LOG("free chain\n");
#endif
			buffer[0] = pfl->next.st.trk;
			buffer[1] = pfl->next.st.sec;
			buffer[2] = pfl->record_nr[0];
			buffer[3] = pfl->record_nr[1];
			// free chain sector reads always
			// filled with zero
			for (i = 4; i < SECTOR_SIZE; i++)
				buffer[i] = 0;
			break;
		default : 
			if (pfl->file_id >= 0) {
#ifdef DEBUG_FILE
				LOG_X("sector of file %s\n", unix_filename(pfl->file_id));
#endif
				strcpy((char *)&path, *dir);
				strcat((char *)&path, PATHSEPARATORSTRING);
				strcat((char *)&path, unix_filename(pfl->file_id));
				if ((fp = fopen(path, mode)) != NULL &&
					!fseek(fp, (long)(pfl->f_record * 252L),
					   SEEK_SET)) {
						bytes = fread(&buffer[4], 1, 252, fp);
					fclose(fp);
				// stuff last sector of file with 0
					for (i = 4+bytes; i < SECTOR_SIZE; i++)
						buffer[i] = 0;
				} else { // unable to read sector
					result = 0;
				} // else
			} else {
			// new file with name tmpXX
				fp = (pnew_file+(NEW_FILE1 - pfl->file_id))->fp;
				if (!fseek(fp, (long)(pfl->f_record * 252L), SEEK_SET)) {
					bytes = fread(&buffer[4], 1, 252, fp);
					// stuff last sector of file with 0
					for (i = 4+bytes; i < SECTOR_SIZE; i++)
						buffer[i] = 0;
					fseek(fp, 0L, SEEK_END); // position end of file
				} else { // unable to read sector
					result = 0;
				} // else
			} // else
			buffer[0] = pfl->next.st.trk;
			buffer[1] = pfl->next.st.sec;
			buffer[2] = pfl->record_nr[0];
			buffer[3] = pfl->record_nr[1];
			break;
	} // switch
	return result;
} //read_nafs_sector


int     NafsDirectoryContainer::WriteSector(const Byte *buffer, int trk, int sec)
{
	char                    *p, path[PATH_MAX+1], *mode;
	Word                    i, di;
	size_t                  bytes;
	SWord                   new_file_index;
	FILE                    *fp;
	struct s_link_table     *pfl;
	int                             result;

#ifdef DEBUG_FILE
	LOG_XX("write: %02X/%02X ", trk, sec);
#endif
	fp    = NULL;
#ifdef sun
	mode  = "r+";
#else
	mode  = "rb+";
#endif
	result = 1;
	bytes = 0;      // satisfy compiler
	i     = trk * MAX_SECTOR + (sec - 1);
	pfl   = pflex_links + i;
	switch (pfl->file_id) {
		case SYSTEM :
#ifdef DEBUG_FILE
			LOG("systemsector\n");
#endif
			p = NULL; // satisfy compiler
			if (sec >= 3)
				p = (char *)(pflex_sys_info + sec - 3);
			else if (sec == 2)
				p = (char *)pflex_unused;
			else if (sec == 1) {
#ifdef sun
				mode = "w";
#else
				mode = "wb";
#endif
				strcpy((char *)&path, *dir);
				strcat((char *)&path, PATHSEPARATORSTRING "boot");
				if ((fp = fopen(path, mode)) != NULL) {
					bytes = fwrite(&buffer, 1, SECTOR_SIZE, fp);
					fclose(fp);
				}
				if (bytes != SECTOR_SIZE)
					result = 0;
				break;
			} else
				break;
			memcpy(p, buffer, param.byte_p_sector);
			break;
		case DIRECTORY:
#ifdef DEBUG_FILE
			LOG("directorysector\n");
#endif
			di = pfl->f_record;
			p = (char *)(pflex_directory + di);
			check_for_delete(di, (s_dir_sector *)buffer);
			check_for_new_file(di, (s_dir_sector *)buffer);
			check_for_rename(di, (s_dir_sector *)buffer);
			check_for_extend(di, (s_dir_sector *)buffer);
			memcpy(p, buffer, param.byte_p_sector);
			break;
		case FREE_CHAIN :
#ifdef DEBUG_FILE
			LOG("free chain\n");
#endif
			if (dir_extend.st.trk == trk && dir_extend.st.sec == sec) {
				extend_directory(i, (s_dir_sector *)buffer);
#ifdef DEBUG_FILE
				LOG("      extend directory\n");
#endif
				break;
			} // if
			if (last_of_free_chain(trk, sec) && (buffer[1] || buffer[0])) {
				pfl->next.st.trk  = buffer[0];
				pfl->next.st.sec  = buffer[1];
				pfl->record_nr[0] = buffer[2];
				pfl->record_nr[1] = buffer[3];
#ifdef DEBUG_FILE
				LOG("      file deleted\n");
#endif
				break;
			}
			if ((new_file_index = index_of_new_file(trk, sec)) < 0) {
#ifdef DEBUG_FILE
				LOG("   ** error: unable to create new file\n");
#endif
				result = 0; // no more new files can be created
				break;
			}
#ifdef DEBUG_FILE
			LOG_X("      file %s\n",
				(pnew_file+new_file_index)->filename);
#endif
			fp = (pnew_file+new_file_index)->fp;
			(pnew_file+new_file_index)->next.st.trk = buffer[0];
			(pnew_file+new_file_index)->next.st.sec = buffer[1];
			pfl->file_id  = NEW_FILE1 - new_file_index;
			// pfl->f_record = (pfs->pnew_file+new_file_index)->f_record++;
			//pfl->f_record = ((sector_buffer[2] << 8) | sector_buffer[3]) - 1;
			pfl->next.st.trk  = buffer[0];
			pfl->next.st.sec  = buffer[1];
			pfl->record_nr[0] = buffer[2];
			pfl->record_nr[1] = buffer[3];
			pfl->f_record = record_nr_of_new_file(new_file_index, i);
			if (ftell(fp) != (pfl->f_record * 252L) && 
			   fseek(fp, (long)(pfl->f_record * 252L), SEEK_SET) != 0)
				result = 0;
			else if ((bytes = fwrite(&buffer[4], 1, 252, fp)) != 252)
				result = 0;
			// (pfs->pnew_file+new_file_index)->f_record++;
			break;
		default : 
			if (pfl->file_id >=  0) {
#ifdef DEBUG_FILE
				LOG_X("sector of file %s\n", unix_filename(pfl->file_id));
#endif
				strcpy((char *)&path, *dir);
				strcat((char *)&path, PATHSEPARATORSTRING);
				strcat((char *)&path, unix_filename(pfl->file_id));
				pfl->next.st.trk = buffer[0];
				pfl->next.st.sec = buffer[1];
				pfl->record_nr[0]= buffer[2];
				pfl->record_nr[1]= buffer[3];
				if ((fp = fopen(path, mode)) == NULL)
					result = 0;
				else {
					if (ftell(fp) != pfl->f_record && 
					   fseek(fp, (long)(pfl->f_record * 252L), SEEK_SET) != 0)
						result = 0;
					else if ((bytes = fwrite(&buffer[4], 1, 252, fp)) != 252)
						result = 0;
					fclose(fp);
				} // else
			} else {
#ifdef DEBUG_FILE
				LOG_X("sector of new file %s\n",
					unix_filename(pfl->file_id));
#endif
				fp = (pnew_file+(NEW_FILE1 - pfl->file_id))->fp;
				if (ftell(fp) != pfl->f_record && 
				   fseek(fp, (long)(pfl->f_record * 252L), SEEK_SET) != 0)
					result = 0;
				else if ((bytes = fwrite(&buffer[4], 1, 252, fp)) != 252)
					result = 0;
			} //else
			pfl->next.st.trk = buffer[0];
			pfl->next.st.sec = buffer[1];
			pfl->record_nr[0]= buffer[2];
			pfl->record_nr[1]= buffer[3];
	} // switch
	return result;
} // write_nafs_sector


void NafsDirectoryContainer::mount(Word number)
{
	int wp; // should be int because of call 'access'

	wp = access(*dir, W_OK);
	initialize_header(wp);
	initialize_new_file_table();
	initialize_flex_sys_info_sectors(number);
	initialize_flex_unused_sector();
	fill_flex_directory(wp);
  } // mount

void NafsDirectoryContainer::ReadToBuffer(const char *fileName, FlexFileBuffer &buffer)
{
	return;
}

int NafsDirectoryContainer::WriteFromBuffer(const char *fileName, const FlexFileBuffer &buffer)
{
	return 0;
}

#endif // #ifdef NAFS
