/*
    mc6809st.cpp

    flexemu, an MC6809 emulator running FLEX
    Copyright (C) 1997-2000  W. Schwotzer
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.
			
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
			
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <misc1.h>
#include <stdlib.h>
#include <limits.h>

#include "mc6809.h"
#include "memory.h"
#include "inout.h"
#include "absgui.h"
#include "da6809.h"
#include "intmem.h"

// uncomment this for debug output for frequency control
//#define	DEBUG_FILE	"freq.txt"


#ifdef FASTFLEX
#define PC ipcreg
#include "engine.h"
#else
#define PC pc
#endif

#ifdef FASTFLEX
t_cycles Mc6809::exec_irqs(void)
{
    if (exceptions & (IRQ | FIRQ | NMI)) {
	    if (sync_flg) {
	    	if (sync_flg == SYNC_INSTR)	
	    		PC++;		// jump over SYNC or CWAI instruction
		else
			PC += 2;
    	   	sync_flg = 0;	// clear sync state
	    }
	    if ((exceptions & NMI) && !nmi_armed) {
		    DO_NMI;
		    exceptions &= ~NMI;
	    } else if ((exceptions & FIRQ) && !(iccreg & 0x40)) {
		    DO_FIRQ;
		    exceptions &= ~FIRQ;
	    } else if ((exceptions & IRQ) && !(iccreg & 0x10)) {
		    DO_IRQ;
		    exceptions &= ~IRQ;
	    } // else
    }  // if
    return 5; // rounded
}  // exec_irqs
#else
t_cycles Mc6809::exec_irqs(void)
{
    if (exceptions & (IRQ | FIRQ | NMI)) {
	    if (sync_flg) {
	    	if (sync_flg == SYNC_INSTR)	
	    		PC++;		// jump over SYNC or CWAI instruction
		else
			PC += 2;
    	   	sync_flg = 0;	// clear sync state
	    }
	    if ((exceptions & NMI) && !nmi_armed) {
		    nmi();
		    exceptions &= ~NMI;
	    } else if ((exceptions & FIRQ) && !cc.bit.f) {
		    firq();
		    exceptions &= ~FIRQ;
	    } else if ((exceptions & IRQ) && !cc.bit.i) {
		    irq();
		    exceptions &= ~IRQ;
	    } // else
    }  // if
    return 5; // rounded
}  // exec_irqs
#endif

void Mc6809::get_status(struct s_cpu_status *stat)
{
#ifdef FASTFLEX
    stat->a     = iareg;
    stat->b     = ibreg;
    stat->cc    = iccreg;
    stat->dp    = idpreg;
    stat->pc    = ipcreg;
    stat->x     = ixreg;
    stat->y     = iyreg;
    stat->u     = iureg;
    stat->s     = isreg;
    stat->state = state;
    stat->freq = frequency;
    stat->cycles = (total_cycles + cycles) / 10;
#else
    stat->a     = a;
    stat->b     = b;
    stat->cc    = cc.all;
    stat->dp    = dp;
    stat->pc    = pc;
    stat->x     = x;
    stat->y     = y;
    stat->u     = u;
    stat->s     = s;
    stat->state = state;
    stat->freq = frequency;
    stat->cycles = total_cycles + cycles;
#endif
}  // get_status


void Mc6809::set_new_state(Byte x_user_input)
{
    user_input = x_user_input;
}

// within the statemachine2 the state RUN, (and partially STEP) will be
// executed.
// it executes the States STOP, STEP and NEXT.
// changes to any other state will return to statemachine1.
//  RUN is static and runs forever
//  STOP sets SINGLESTEP and runs in the singlestep loop until state change
//  STEP will be handled in the singlestep loop and changes to STOP
//  NEXT is intermediate, it sets the BREAKPOINT (bp[2]) exception, executes
//    any instruction up to the breakpoint and changes to STOP

Byte Mc6809::statemachine2(void)
{
   if (state == RUN) {
   	exceptions &= ~SINGLESTEP;
   	next_update = cycles + CYCLES_TO_UPDATE;
   }
   while (1) {
	if (exceptions) {
		// check if invalid instr. occured
		if (exceptions & (IRQ | FIRQ | NMI)) {
			cycles += exec_irqs();
		}
		if (exceptions & BREAKPOINT) {
			if (PC == bp[0] || PC == bp[1] || PC == bp[2]) {
				if (PC == bp[2]) {
					bp[2] = 0x10000L;
					if (bp[0] > 0xffff &&
					   bp[1] > 0xffff &&
					   bp[2] > 0xffff)
					     exceptions &= ~BREAKPOINT;
				} // if
				state = STOP;
				exceptions |= SINGLESTEP;
			} // if
		} // if BREAKPOINT
		if (exceptions & INVALID) {
			exceptions &= ~INVALID;
			state = STOP;
			exceptions |= SINGLESTEP;
		} // if INVALID
		// check for SINGLESTEP flag must be executed AFTER
		// BREAKPOINT and INVALID
		if (exceptions & SINGLESTEP) {
			if (state == STEP)
				state = STOP;
			if (gui != NULL)
				gui->update_cpuview(state);
			user_input = singlestep();
			if (user_input != STEP && user_input != RUN) {
				return user_input;
			}
			state = user_input;
		} // if SINGLESTEP
	} // if

// optionally make a logfile containing all executed instructions
// uncomment define for DEBUG_FILE on top of this file

#ifdef DEBUG_FILE
	{
		char *pa, *pb;
		Byte step;

		if (disassembler != NULL && logfp != NULL) {
			disassembler->disassemble(PC, &step, &pa, &pb);
			fprintf(logfp, "%04X %s\n", PC, pb);
		}
	}
#endif

#ifdef FASTFLEX
#include "engine.cpp"
#else
#include "mc6809ex.cpp"
#endif

	if ((cycles >= next_update)) {
		if (update_flag != 0) {
			// update_frequency must come before calc. next_update
			update_frequency();
			if (inout != NULL)
				inout->update_1_second(state);
			update_flag = 0;
		}
		next_update = cycles + CYCLES_TO_UPDATE;
		if (gui != NULL)
			gui->update();
		if (delay > 0 && state == RUN) {
			int dummy;
			char cdummy[17];

			for (dummy = delay; dummy >= 0; dummy--)
				strcpy((char *)cdummy, "1234567890ABCDEF");
		} // if
		// a user_input can also be set asynchronously !
		switch (user_input) {
			   case RUN: if (state != RUN) {
			   		state = user_input;
				      	if (gui != NULL)
			   	         	gui->update_cpuview(NO_CHANGE);
			              	exceptions &= ~SINGLESTEP;
				      }
				      break;
			   case STOP: state = user_input;
			   	      frequency = (float)0.0;
				      if (gui != NULL)
			   	         gui->update_cpuview(NO_CHANGE);
			   	      exceptions |= SINGLESTEP; break;
			   case RESET:
			   case EXIT:
			   case RESET_RUN: return user_input;
		} // switch
		user_input = NO_CHANGE;
	} // if
   } // while
} // statemachine2


void Mc6809::do_reset(void)
{
	reset();
	reset_io();
}

// return with state RUN, STEP, RESET, RESET_RUN, EXIT from this function

Byte Mc6809::singlestep(void)
{
	char *pa, *pb;
	Byte step;

	user_input = NO_CHANGE;
	while (1) {
		if (gui != NULL)
			gui->update();	// update video display
		switch (user_input) {
			case NEXT:
				// if no disassembler present execute as STEP
				if (disassembler != NULL) {
					if (gui != NULL)
						gui->update_cpuview(user_input);
					bp[2] =
				 	   PC + disassembler->disassemble(
						   (unsigned int)PC, &step,
						   &pa, &pb);
					if (step) {
						user_input = STEP;
						exceptions |= SINGLESTEP;
						exceptions &= ~BREAKPOINT;
						bp[2] = 0x10000;
						if (gui != NULL)
							gui->update_cpuview(user_input);
					} else {
						user_input = RUN;
						exceptions &= ~SINGLESTEP;
						exceptions |= BREAKPOINT;
					}
					break;
				}
			case STOP:
				state = user_input;
				break;
			case STEP:
				exceptions |= SINGLESTEP;
				if (gui != NULL)
					gui->update_cpuview(user_input);
				break;
			case RUN:
				cycles = 0;
   				next_update = CYCLES_TO_UPDATE;
				exceptions &= ~SINGLESTEP;
				break;
		} // switch
		if (update_flag != 0) {
			if (inout != NULL)
				inout->update_1_second(NO_CPUVIEW_UPDATE);
			update_flag = 0;
		}
		if (user_input != STOP && user_input != NO_CHANGE)
			return user_input;
	} // while
} // singlestep


Byte Mc6809::statemachine1(void)
{
	Byte tmp, prev_state; 

	if (user_input == EXIT)
		state = user_input;
	prev_state = state;
	while(state != EXIT) {
		if (gui != NULL)
			gui->update_cpuview(NO_CHANGE);
		user_input = NO_CHANGE;
		switch (state) {
			case STOP:
			case RUN:
				tmp = statemachine2();
				prev_state = state;
				state = tmp;
				break;
			case RESET:
				do_reset(); state = prev_state;
				break;
			case RESET_RUN:
				do_reset();
				state = RUN;
				break;
			case EXIT:
				break;
		} // switch
	} // while
	return state;
} // statemachine1

void Mc6809::set_frequency(float target_freq)
{
	if (target_freq <= 0) {
		target_frequency = (float)0.0;
		delay		= 0;
		delta_delay	= 0;
	} else {
		target_frequency = target_freq;
		delta_delay	= 1000;
		delay		= delta_delay;
	}
} // set_frequency

void Mc6809::update_frequency(void)
{
#ifdef DEBUG_FILE
	FILE *fp;
#endif

	// calculate frequency in MHz
	if (state == RUN) 
#ifdef FASTFLEX
		frequency = (float)(cycles / 10000000.0);
#else
		frequency = (float)(cycles / 1000000.0);
#endif
	total_cycles += cycles;
	cycles = 0;
	delta_frequency = frequency - target_frequency;
	// control frequency
	if (target_frequency > 0.0) {
		delta_delay = (int)(DELTA_DELAY_CONST * delta_frequency);
		delay += delta_delay;
	}
#ifdef DEBUG_FILE
	if ((fp = fopen(DEBUG_FILE, "a")) != NULL) {
		fprintf(fp, "f=%5.2f df=%5.2f d=%8d dd=%8d\n",
			frequency, delta_frequency, delay, delta_delay);
		fclose(fp);
	}
#endif
} // update_frequency


