/*
    mc6809mm.cpp


    flexemu, an MC6809 emulator running FLEX
    Copyright (C) 1997-2000  W. Schwotzer

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/



#include <misc1.h>
#include <stdio.h>
#include <stdlib.h>

#include "memory.h"
#include "mc6809.h"


// memory must be initialized AFTER all memory mapped I/O is created
void Mc6809::init_memory(void)
{
	Word i;

	for (i=0; i < YBLOCKS; i++)
		changed[i] = 0;
	for (i=0; i < MAX_IO_DEVICES; i++)
		ioDevices[i] = NULL;
	io_initialized	= 0;
	io_base_addr	= 10000L;

	// initialize default pointer for mmu configuration
	for (i = 0; i < MAX_VRAM; i++)
		vram_ptrs[i] = NULL;
	vram_ptrs[0x04] = &video_ram[VIDEORAM_SIZE * 3];
	vram_ptrs[0x05] = &video_ram[VIDEORAM_SIZE * 4];
	vram_ptrs[0x06] = &video_ram[VIDEORAM_SIZE * 5];
	vram_ptrs[0x0c] = &video_ram[VIDEORAM_SIZE * 0];
	vram_ptrs[0x0d] = &video_ram[VIDEORAM_SIZE * 1];
	vram_ptrs[0x0e] = &video_ram[VIDEORAM_SIZE * 2];
	vram_ptrs[0x00] = &video_ram[VIDEORAM_SIZE * 9];
	vram_ptrs[0x01] = &video_ram[VIDEORAM_SIZE * 10];
	vram_ptrs[0x02] = &video_ram[VIDEORAM_SIZE * 11];
	vram_ptrs[0x08] = &video_ram[VIDEORAM_SIZE * 6];
	vram_ptrs[0x09] = &video_ram[VIDEORAM_SIZE * 7];
	vram_ptrs[0x0a] = &video_ram[VIDEORAM_SIZE * 8];

	// initialize mmu pointers
	for (i = 0; i < MAX_VRAM; i++)
		ppage[i] = NULL;
} // init_memory

void Mc6809::uninit_memory(void)
{
	Word i;

	delete [] ppIo;
	ppIo = NULL;
	for (i = 0; i < MAX_IO_DEVICES; i++)
		delete ioDevices[i]; 
} // uninit_memory

void Mc6809::initialize_io_page(Word base_addr)
{
	size_t	i, range;

	if (base_addr > 0xffff)
		return;
	range = 0x10000L - base_addr;
	ppIo  = new struct sIoSelect[range];
	for (i = 0; i < range; i++) {
		(ppIo + i)->device  = (IoDevice *)this;
		(ppIo + i)->offset  = base_addr + i;
	}
	io_initialized = 1;
	io_base_addr   = base_addr;
}

// return 0 if not successful
Byte Mc6809::add_io_device(IoDevice *device,
	Word base_addr1, Byte range1,
	Word base_addr2, Byte range2)
{
	Word i = 0;
	Word offset;

	if (!io_initialized)
		return 0;	// first initialize I/O
	while(i < MAX_IO_DEVICES && ioDevices[i] != NULL)
		i++; 
	if (i == MAX_IO_DEVICES)
		return 0;	// all io-devices already in use
	if (base_addr1 < io_base_addr ||
		(range2 > 0 && base_addr2 < io_base_addr))
		return 0;	// base addr out of io address space
	ioDevices[i]    = device;
	offset = base_addr1 - io_base_addr;
	for (i=0; i < range1; i++) {
		(ppIo + offset + i)->device = device;
		(ppIo + offset + i)->offset = i;
	}
	offset = base_addr2 - io_base_addr;
	for (i=0; i < range2; i++) {
		(ppIo + offset + i)->device = device;
		(ppIo + offset + i)->offset = base_addr2 - base_addr1 + i;
	}
	return 1;
}

void Mc6809::initialize_byte_conv_table(int lsb_first)
{
	Byte new_val;
	Word i, bitpos, new_bitpos;

	for (i=0; i < 256; i++) {
		if (lsb_first) {
			new_val = 0;
			new_bitpos = 0x80;
			for (bitpos=1; bitpos <= 0x80; bitpos <<= 1) {
				if (i & bitpos)
					new_val |= new_bitpos;
				new_bitpos >>= 1;
			} // for
		} else
			new_val = (Byte)i;
		byte_conversion_table[i] = new_val;
	}  // for 
}


void Mc6809::reset_io(void)
{
	Word i;

	for (i = 0; i < MAX_IO_DEVICES; i++)
		if (ioDevices[i] != NULL && ioDevices[i] != this)
			ioDevices[i]->resetIo(); 
} // reset

Byte Mc6809::readFromIo(Word addr)
{
	struct sIoSelect *pIo;
	SDWord 		  diff;

	diff = addr - io_base_addr;
	pIo = ppIo + diff;
	return (pIo->device)->readIo(pIo->offset);
}

// ATTENTION:
// the file mc6809st.cc has it's own memory interface
// the read byte function will be executed as inline code
Byte Mc6809::read(Word addr)
{
	Byte 		 *p;
	struct sIoSelect *pIo;
	SDWord 		  diff;

	// read from memory mapped I/O
	if ((diff = (addr - io_base_addr)) >= 0) {
		pIo = ppIo + diff;
		return (pIo->device)->readIo(pIo->offset);
	}
	// otherwise read from memory
	if ((p = ppage[addr >> 12]) != NULL)
		return byte_conversion_table[(Byte)*(p + (addr & 0x3fff))];
	return memory[addr];

} // read

Byte Mc6809::readIo(Word addr)
{
	Byte *p;

	// this part is implemented twice for performance reasons
	p = ppage[addr >> 12];
	if (p != NULL)
		return byte_conversion_table[(Byte)*(p + (addr & 0x3fff))];
	return memory[addr]; 
} // readIo

Word Mc6809::read_word(Word addr)
{
	Byte 		 *p;
	struct sIoSelect *pIo;
	SDWord 		  diff;
	Word		  tmp;

	// read from memory mapped I/O
	if ((diff = (addr - io_base_addr)) >= 0) {
		pIo = ppIo + diff;
		tmp = (pIo->device)->readIo(pIo->offset) << 8;
		pIo = ppIo + diff + 1;
		return tmp | (pIo->device)->readIo(pIo->offset);
	}
	// otherwise read from memory
	p = ppage[addr >> 12];
	if (p != NULL)
		tmp = byte_conversion_table[
			(Byte)*(p + (addr & 0x3fff))] << 8;
	else
		tmp = memory[addr] << 8;
	addr++;
	p = ppage[addr >> 12];
	if (p != NULL)
		return tmp | byte_conversion_table[
			(Byte)*(p + (addr & 0x3fff))];
	return tmp | memory[addr];
} // read_word


void Mc6809::write(Word addr, Byte val)
{
	Word vr_offset;  // offset into video ram (modulo 0x4000)
	struct sIoSelect *pIo;
	Byte		 *p;
	SDWord 		  diff;

	if ((diff = (addr - io_base_addr)) >= 0) {
		pIo = ppIo + diff;
		(pIo->device)->writeIo(pIo->offset, val);
	} else {
		p = ppage[addr >> 12];
		if (p != NULL) {
			vr_offset = addr & 0x3fff;
			changed[vr_offset / YBLOCK_SIZE] = 1;
			*(p + vr_offset) = byte_conversion_table[val];
		} else
			if (addr < ROM_BASE)
				memory[addr] = val;
	} // else
} // write

void Mc6809::writeIo(Word addr, Byte val)
{
	Word vr_offset;  // offset into video ram (modulo 0x4000)
	Byte *p;

	// this part is implemented twice for performance reasons
	p = ppage[addr >> 12];
	if (p != NULL) {
		vr_offset = addr & 0x3fff;
		changed[vr_offset / YBLOCK_SIZE] = 1;
		*(p + vr_offset) = byte_conversion_table[val];
	} else
		if (addr < ROM_BASE)
			memory[addr] = val;
} // writeIo

// write Byte into ROM
void Mc6809::write_rom(Word offset, Byte val)
{
	memory[offset] = val;
} // write_rom


void Mc6809::switch_mmu(Word offset, Byte val)
{
	if (offset < 12)
		ppage[offset] = vram_ptrs[val & 0x0f]; 
} // switch_mmu


void Mc6809::write_word(Word addr, Word val)
{
	write(addr++, (Byte)(val >> 8));
	write(addr, (Byte)val);
} // write_word

