/*
     mc6809in.cc

     flexemu, an MC6809 emulator running FLEX
     Copyright (C) 1997  W. Schwotzer

     This file is based on usim-0.91 which is
     Copyright (C) 1994 by R. B. Bellis
*/


#include <stdio.h>
#include "misc1.h"
#include "memory.h"
#include "mc6809.h"
#include "intmem.h"


#ifndef FASTFLEX
void Mc6809::abx(void)
{
	x += b;
}

void Mc6809::help_adc(Byte& x)
{
	Byte	m, t;
	Word    tw;
	
	m = fetch_operand();

	t = (x & 0x0f) + (m & 0x0f) + cc.bit.c;
	cc.bit.h = BTST4(t);		// Half carry
	t = (x & 0x7f) + (m & 0x7f) + cc.bit.c;
	cc.bit.v = BTST7(t);		// Bit 7 carry in
	tw = x + m + cc.bit.c;
	cc.bit.c = BTST8(tw);		// Bit 7 carry out
	x = tw & 0xff;

	cc.bit.v ^= cc.bit.c;
	cc.bit.n = BTST7(x);
	cc.bit.z = !x;
}

void Mc6809::adca(void)
{
	help_adc(a);
}

void Mc6809::adcb(void)
{
	help_adc(b);
}

t_cycles Mc6809::help_adc_i(Byte& x)
{
	t_cycles cycl;

	Byte	m, t;
	Word	tw;
	
	m = fetch_indexed(&cycl);

	t = (x & 0x0f) + (m & 0x0f) + cc.bit.c;
	cc.bit.h = BTST4(t);		// Half carry
	t = (x & 0x7f) + (m & 0x7f) + cc.bit.c;
	cc.bit.v = BTST7(t);		// Bit 7 carry in
	tw = x + m + cc.bit.c;
	cc.bit.c = BTST8(tw);		// Bit 7 carry out
	x = tw & 0xff;

	cc.bit.v ^= cc.bit.c;
	cc.bit.n = BTST7(x);
	cc.bit.z = !x;
	return cycl;
}

t_cycles Mc6809::adca_i(void)
{
	return help_adc_i(a);
}

t_cycles Mc6809::adcb_i(void)
{
	return help_adc_i(b);
}

void Mc6809::help_add(Byte& x)
{
	Byte	m, t;
	Word	tw;
	
	m = fetch_operand();

	t = (x & 0x0f) + (m & 0x0f);
	cc.bit.h = BTST4(t);		// Half carry
	t = (x & 0x7f) + (m & 0x7f);
	cc.bit.v = BTST7(t);		// Bit 7 carry in
	tw = x + m;
	cc.bit.c = BTST8(tw);		// Bit 7 carry out
	x = tw & 0xff;

	cc.bit.v ^= cc.bit.c;
	cc.bit.n = BTST7(x);
	cc.bit.z = !x;
}

void Mc6809::adda(void)
{
	help_add(a);
}

void Mc6809::addb(void)
{
	help_add(b);
}

t_cycles Mc6809::help_add_i(Byte& x)
{
	t_cycles cycl;

	Byte	m, t;
	Word	tw;
	
	m = fetch_indexed(&cycl);

	t = (x & 0x0f) + (m & 0x0f);
	cc.bit.h = BTST4(t);		// Half carry
	t = (x & 0x7f) + (m & 0x7f);
	cc.bit.v = BTST7(t);		// Bit 7 carry in
	tw = x + m;
	cc.bit.c = BTST8(tw);		// Bit 7 carry out
	x = tw & 0xff;

	cc.bit.v ^= cc.bit.c;
	cc.bit.n = BTST7(x);
	cc.bit.z = !x;
	return cycl;
}

t_cycles Mc6809::adda_i(void)
{
	return help_add_i(a);
}

t_cycles Mc6809::addb_i(void)
{
	return help_add_i(b);
}

void Mc6809::addd(void)
{
	Word	m, t;
	DWord	tl;
	
	m = fetch_word_operand();

	t = (d & 0x7fff) + (m & 0x7fff);
	cc.bit.v = BTST15(t);
	tl = (DWord)d + m;
	cc.bit.c = BTST16(tl);
	d = (Word)(tl & 0xffff);

	cc.bit.v ^= cc.bit.c;
	cc.bit.n = BTST15(d);
	cc.bit.z = !d;
}

t_cycles Mc6809::addd_i(void)
{
	t_cycles cycl;
	Word	m, t;
	DWord	tl;

	m = fetch_indexed_word(&cycl);

	t = (d & 0x7fff) + (m & 0x7fff);
	cc.bit.v = BTST15(t);
	tl = (DWord)d + m;
	cc.bit.c = BTST16(tl);
	d = (Word)(tl & 0xffff);

	cc.bit.v ^= cc.bit.c;
	cc.bit.n = BTST15(d);
	cc.bit.z = !d;
	return cycl;
}

void Mc6809::help_and(Byte& x)
{
	x = x & fetch_operand();
	cc.bit.n = BTST7(x);
	cc.bit.z = !x;
	cc.bit.v = 0;
}

void Mc6809::anda(void)
{
	help_and(a);
}

void Mc6809::andb(void)
{
	help_and(b);
}

void Mc6809::andcc(void)
{
	cc.all &= READ_PI(pc);
}

t_cycles Mc6809::help_and_i(Byte& x)
{
	t_cycles cycl;

	x = x & fetch_indexed(&cycl);
	cc.bit.n = BTST7(x);
	cc.bit.z = !x;
	cc.bit.v = 0;
	return cycl;
}

t_cycles Mc6809::anda_i(void)
{
	return help_and_i(a);
}

t_cycles Mc6809::andb_i(void)
{
	return help_and_i(b);
}

void Mc6809::help_asr(Byte& x)
{
	cc.bit.c = BTST0(x);
	x >>= 1;	/* Shift word right */
	if ((cc.bit.n = BTST6(x)) != 0) {
		BSET7(x);
	}
	cc.bit.z = !x;
}

void Mc6809::asra(void)
{
	help_asr(a);
}

void Mc6809::asrb(void)
{
	help_asr(b);
}

void Mc6809::asr(void)
{
	Word	addr = fetch_effective_address();
	Byte	m = READ(addr);

	help_asr(m);
	WRITE(addr, m);
}

t_cycles Mc6809::asr_i(void)
{
	Word addr;
	Byte m;
	t_cycles cycl;
	
	addr = fetch_indexed_address(&cycl);
	m = READ(addr);
	help_asr(m);
	WRITE(addr, m);
	return cycl;
}

void Mc6809::bcc(void)
{
	Word offset;
	
	if (!cc.bit.c) {
		offset = (Word)READ(pc);
		if (offset & 0x80)
			offset |= 0xff00;
		pc += offset + 1;
	} else
		pc++;
}

t_cycles Mc6809::lbcc(void)
{
	return do_lbr(!cc.bit.c);
}

void Mc6809::bcs(void)
{
	Word offset;
	
	if (cc.bit.c) {
		offset = (Word)READ(pc);
		if (offset & 0x80)
			offset |= 0xff00;
		pc += offset + 1;
	} else
		pc++;
}

t_cycles Mc6809::lbcs(void)
{
	return do_lbr(cc.bit.c);
}

void Mc6809::beq(void)
{
	Word offset;
	
	if (cc.bit.z) {
		offset = (Word)READ(pc);
		if (offset & 0x80)
			offset |= 0xff00;
		pc += offset + 1;
	} else
		pc++;
}

t_cycles Mc6809::lbeq(void)
{
	return do_lbr(cc.bit.z);
}

void Mc6809::bge(void)
{
	do_br(!(cc.bit.n ^ cc.bit.v));
}

t_cycles Mc6809::lbge(void)
{
	return do_lbr(!(cc.bit.n ^ cc.bit.v));
}

void Mc6809::bgt(void)
{
	do_br(!(cc.bit.z | (cc.bit.n ^ cc.bit.v)));
}

t_cycles Mc6809::lbgt(void)
{
	return do_lbr(!(cc.bit.z | (cc.bit.n ^ cc.bit.v)));
}

void Mc6809::bhi(void)
{
	do_br(!(cc.bit.c | cc.bit.z));
}

t_cycles Mc6809::lbhi(void)
{
	return do_lbr(!(cc.bit.c | cc.bit.z));
}

void Mc6809::bita(void)
{
	help_bit(a);
}

void Mc6809::bitb(void)
{
	help_bit(b);
}

void Mc6809::help_bit(Byte x)
{
	Byte t = x & fetch_operand();
	cc.bit.n = BTST7(t);
	cc.bit.v = 0;
	cc.bit.z = !t;
}

t_cycles Mc6809::bita_i(void)
{
	return help_bit_i(a);
}

t_cycles Mc6809::bitb_i(void)
{
	return help_bit_i(b);
}

t_cycles Mc6809::help_bit_i(Byte x)
{
	t_cycles cycl;

	Byte t = x & fetch_indexed(&cycl);
	cc.bit.n = BTST7(t);
	cc.bit.v = 0;
	cc.bit.z = !t;
	return cycl;
}

void Mc6809::ble(void)
{
	do_br(cc.bit.z | (cc.bit.n ^ cc.bit.v));
}

t_cycles Mc6809::lble(void)
{
	return do_lbr(cc.bit.z | (cc.bit.n ^ cc.bit.v));
}

void Mc6809::bls(void)
{
	do_br(cc.bit.c | cc.bit.z);
}

t_cycles Mc6809::lbls(void)
{
	return do_lbr(cc.bit.c | cc.bit.z);
}

void Mc6809::blt(void)
{
	do_br(cc.bit.n ^ cc.bit.v);
}

t_cycles Mc6809::lblt(void)
{
	return do_lbr(cc.bit.n ^ cc.bit.v);
}

void Mc6809::bmi(void)
{
	do_br(cc.bit.n);
}

t_cycles Mc6809::lbmi(void)
{
	return do_lbr(cc.bit.n);
}

void Mc6809::bne(void)
{
	Word offset;
	
	if (!cc.bit.z) {
		offset = (Word)READ(pc);
		if (offset & 0x80)
			offset |= 0xff00;
		pc += offset + 1;
	} else
		pc++;
}

t_cycles Mc6809::lbne(void)
{
	return do_lbr(!cc.bit.z);
}

void Mc6809::bpl(void)
{
	do_br(!cc.bit.n);
}

t_cycles Mc6809::lbpl(void)
{
	return do_lbr(!cc.bit.n);
}

void Mc6809::bra(void)
{
	Word offset;
	
	offset = (Word)READ(pc);
	if (offset & 0x80)
		offset |= 0xff00;
	pc += offset + 1;
}

t_cycles  Mc6809::lbra(void)
{
	do_lbr(1);
	return 0;
}

void Mc6809::brn(void)
{
	do_br(0);
}

t_cycles Mc6809::lbrn(void)
{
	do_lbr(0);
	return 0;
}

void Mc6809::bsr(void)
{
	Byte	x = READ_PI(pc);
	WRITE(--s, (Byte)pc);
	WRITE(--s, (Byte)(pc >> 8));
	pc += extend8(x);
}

t_cycles Mc6809::lbsr(void)
{
	Word x;
	
	x = READ_WORD(pc);
	pc += 2;
	WRITE(--s, (Byte)pc);
	WRITE(--s, (Byte)(pc >> 8));
	pc += x;
	return 0;
}

void Mc6809::bvc(void)
{
	do_br(!cc.bit.v);
}

t_cycles Mc6809::lbvc(void)
{
	return do_lbr(!cc.bit.v);
}

void Mc6809::bvs(void)
{
	do_br(cc.bit.v);
}

t_cycles Mc6809::lbvs(void)
{
	return do_lbr(cc.bit.v);
}

void Mc6809::clra(void)
{
	help_clr(a);
}

void Mc6809::clrb(void)
{
	help_clr(b);
}

void Mc6809::clr(void)
{
	Word	addr = fetch_effective_address();
	Byte	m = READ(addr);
	help_clr(m);
	WRITE(addr, m);
}

t_cycles Mc6809::clr_i(void)
{
	Word addr;
	Byte m;
	t_cycles cycl;
	
	addr = fetch_indexed_address(&cycl);
	m = READ(addr);
	help_clr(m);
	WRITE(addr, m);
	return cycl;
}

void Mc6809::help_clr(Byte& x)
{
	cc.all &= 0xf0;
	cc.all |= 0x04;
	x = 0;
}

void Mc6809::cmpa(void)
{
	help_cmp(a);
}

void Mc6809::cmpb(void)
{
	help_cmp(b);
}

void Mc6809::help_cmp(Byte d)
{
// Sw: fixed overflow and carry bug
	Byte s, e, r;

	s = fetch_operand();

	r = d - s;
	e = (d ^ s) & 0x80 ? 1 : 0;
	cc.bit.n = BTST7(r);
	cc.bit.z = r ? 0 : 1 ;
	cc.bit.v = e & (((s ^ r) & 0x80) ? 0 : 1);
	cc.bit.c = ((((~d) &  s) & 0x80) ? 1 : 0) | (!e & cc.bit.n);
}

t_cycles Mc6809::cmpa_i(void)
{
	return help_cmp_i(a);
}

t_cycles Mc6809::cmpb_i(void)
{
	return help_cmp_i(b);
}

t_cycles Mc6809::help_cmp_i(Byte d)
{
	Byte s, e, r;
	t_cycles cycl;

	s = fetch_indexed(&cycl);

	r = d - s;
	e = (d ^ s) & 0x80 ? 1 : 0;
	cc.bit.n = BTST7(r);
	cc.bit.z = r ? 0 : 1 ;
	cc.bit.v = e & (((s ^ r) & 0x80) ? 0 : 1);
	cc.bit.c = ((((~d) &  s) & 0x80) ? 1 : 0) | (!e & cc.bit.n);
	return cycl;
}

void Mc6809::cmpd(void)
{
	help_cmp(d);
}

void Mc6809::cmpx(void)
{
	help_cmp(x);
}

void Mc6809::cmpy(void)
{
	help_cmp(y);
}

void Mc6809::cmpu(void)
{
	help_cmp(u);
}

void Mc6809::cmps(void)
{
	help_cmp(s);
}

void Mc6809::help_cmp(Word d)
{
// Sw: fixed overflow and carry bug
	Word s, e, r;

	s = fetch_word_operand();

	r = d - s;
	e = (d ^ s) & 0x8000 ? 1 : 0;
	cc.bit.n = BTST15(r); 
	cc.bit.z = r ? 0 : 1 ;
	cc.bit.v = e & (((s ^ r) & 0x8000) ? 0 : 1);
	cc.bit.c = ((((~d) &  s) & 0x8000) ? 1 : 0) | (!e & cc.bit.n);
}

t_cycles Mc6809::cmpd_i(void)
{
	Word o, s, e, r;
	t_cycles cycl;

	o = d;
	s = fetch_indexed_word(&cycl);
	r = o - s;
	e = (o ^ s) & 0x8000 ? 1 : 0;
	cc.bit.n = BTST15(r); 
	cc.bit.z = r ? 0 : 1 ;
	cc.bit.v = e & (((s ^ r) & 0x8000) ? 0 : 1);
	cc.bit.c = ((((~o) &  s) & 0x8000) ? 1 : 0) | (!e & cc.bit.n);
	return cycl;
}

t_cycles Mc6809::cmpx_i(void)
{
	Word s, e, r;
	t_cycles cycl;

	s = fetch_indexed_word(&cycl);
	r = x - s;
	e = (x ^ s) & 0x8000 ? 1 : 0;
	cc.bit.n = BTST15(r); 
	cc.bit.z = r ? 0 : 1 ;
	cc.bit.v = e & (((s ^ r) & 0x8000) ? 0 : 1);
	cc.bit.c = ((((~x) &  s) & 0x8000) ? 1 : 0) | (!e & cc.bit.n);
	return cycl;
}

t_cycles Mc6809::cmpy_i(void)
{
	Word s, e, r;
	t_cycles cycl;

	s = fetch_indexed_word(&cycl);
	r = y - s;
	e = (y ^ s) & 0x8000 ? 1 : 0;
	cc.bit.n = BTST15(r); 
	cc.bit.z = r ? 0 : 1 ;
	cc.bit.v = e & (((s ^ r) & 0x8000) ? 0 : 1);
	cc.bit.c = ((((~y) &  s) & 0x8000) ? 1 : 0) | (!e & cc.bit.n);
	return cycl;
}

t_cycles Mc6809::cmpu_i(void)
{
	Word s, e, r;
	t_cycles cycl;

	s = fetch_indexed_word(&cycl);
	r = u - s;
	e = (u ^ s) & 0x8000 ? 1 : 0;
	cc.bit.n = BTST15(r); 
	cc.bit.z = r ? 0 : 1 ;
	cc.bit.v = e & (((s ^ r) & 0x8000) ? 0 : 1);
	cc.bit.c = ((((~u) &  s) & 0x8000) ? 1 : 0) | (!e & cc.bit.n);
	return cycl;
}

t_cycles Mc6809::cmps_i(void)
{
	Word xs, e, r;
	t_cycles cycl;

	xs = fetch_indexed_word(&cycl);
	r = s - xs;
	e = (s ^ xs) & 0x8000 ? 1 : 0;
	cc.bit.n = BTST15(r); 
	cc.bit.z = r ? 0 : 1 ;
	cc.bit.v = e & (((xs ^ r) & 0x8000) ? 0 : 1);
	cc.bit.c = ((((~s) &  xs) & 0x8000) ? 1 : 0) | (!e & cc.bit.n);
	return cycl;
}

void Mc6809::coma(void)
{
	help_com(a);
}

void Mc6809::comb(void)
{
	help_com(b);
}

void Mc6809::com(void)
{
	Word	addr = fetch_effective_address();
	Byte	m = READ(addr);
	help_com(m);
	WRITE(addr, m);
}

t_cycles Mc6809::com_i(void)
{
	Word addr;
	Byte m;
	t_cycles cycl;

	addr = fetch_indexed_address(&cycl);
	m = READ(addr);
	help_com(m);
	WRITE(addr, m);
	return cycl;
}

void Mc6809::help_com(Byte& x)
{
	x = ~x;
	cc.bit.c = 1;
	cc.bit.v = 0;
	cc.bit.n = BTST7(x);
	cc.bit.z = !x;
}

void Mc6809::daa(void)
{
	Byte	c = 0;
	Byte	lsn = (a & 0x0f);
	Byte	msn = (a & 0xf0) >> 4;

	if (cc.bit.h || (lsn > 9)) {
		c |= 0x06;
	}

	if (cc.bit.c ||
	    (msn > 9) ||
	    ((msn > 8) && (lsn > 9))) {
		c |= 0x60;
	}

	{
		Word	t = (Word)a + c;
		cc.bit.c = BTST8(t);
		a = (Byte)t;
	}

	cc.bit.n = BTST7(a);
	cc.bit.z = !a;
}

void Mc6809::deca(void)
{
	help_dec(a);
}

void Mc6809::decb(void)
{
	help_dec(b);
}

void Mc6809::dec(void)
{
	Word	addr = fetch_effective_address();
	Byte	m = READ(addr);
	help_dec(m);
	WRITE(addr, m);
}

t_cycles Mc6809::dec_i(void)
{
	Word addr;
	Byte m;
	t_cycles cycl;
	
	addr = fetch_indexed_address(&cycl);
	m = READ(addr);
	help_dec(m);
	WRITE(addr, m);
	return cycl;
}

void Mc6809::help_dec(Byte& x)
{
	cc.bit.v = (x == 0x80);
	x = x - 1;
	cc.bit.n = BTST7(x);
	cc.bit.z = !x;
}

void Mc6809::eora(void)
{
	help_eor(a);
}

void Mc6809::eorb(void)
{
	help_eor(b);
}

void Mc6809::help_eor(Byte& x)
{
	x = x ^ fetch_operand();
	cc.bit.v = 0;
	cc.bit.n = BTST7(x);
	cc.bit.z = !x;
}

t_cycles Mc6809::eora_i(void)
{
	return help_eor_i(a);
}

t_cycles Mc6809::eorb_i(void)
{
	return help_eor_i(b);
}

t_cycles Mc6809::help_eor_i(Byte& x)
{
	t_cycles cycl;

	x = x ^ fetch_indexed(&cycl);
	cc.bit.v = 0;
	cc.bit.n = BTST7(x);
	cc.bit.z = !x;
	return cycl;
}


void Mc6809::exg(void)
{
	int	r1, r2;
	Byte	w, tb;
	Word	tw;
	
	w = READ_PI(pc);
	r1 = (w & 0xf0) >> 4;
	r2 = (w & 0x0f);
	if (r1 <= 5) {
		if (r2 > 5) {
			--pc;
			invalid("exchange register");
			return;
		}
		switch(w) {
			case 0x01: 
			case 0x10: tw = d; d = x; x = tw; break;
			case 0x02: 
			case 0x20: tw = d; d = y; y = tw; break;
			case 0x03: 
			case 0x30: tw = d; d = u; u = tw; break;
			case 0x04: 
			case 0x40: tw = d; d = s; s = tw; break;
			case 0x05: 
			case 0x50: tw = d; d = pc; pc = tw; break;
			case 0x12: 
			case 0x21: tw = x; x = y; y = tw; break;
			case 0x13: 
			case 0x31: tw = x; x = u; u = tw; break;
			case 0x14: 
			case 0x41: tw = x; x = s; s = tw; break;
			case 0x15: 
			case 0x51: tw = x; x = pc; pc = tw; break;
			case 0x23: 
			case 0x32: tw = y; y = u; u = tw; break;
			case 0x24: 
			case 0x42: tw = y; y = s; s = tw; break;
			case 0x25: 
			case 0x52: tw = y; y = pc; pc = tw; break;
			case 0x34: 
			case 0x43: tw = u; u = s; s = tw; break;
			case 0x35: 
			case 0x53: tw = u; u = pc; pc = tw; break;
			case 0x45: 
			case 0x54: tw = s; s = pc; pc = tw; break;
		} // switch
	} else if (r1 >= 8 && r2 <= 11) {
		if (r2 < 8 || r2 > 11) {
			--pc;
			invalid("exchange register");
			return;
		}
		switch(w) {
			case 0x98: 
			case 0x89: tb = a; a = b; b = tb; break;
			case 0xa8: 
			case 0x8a: tb = a; a = cc.all; cc.all = tb; break;
			case 0xb8: 
			case 0x8b: tb = a; a = dp; dp = tb; dp16 = (Word)(dp << 8); break;
			case 0xa9: 
			case 0x9a: tb = b; b = cc.all; cc.all = tb; break;
			case 0xb9: 
			case 0x9b: tb = b; b = dp; dp = tb; dp16 = (Word)(dp << 8); break;
			case 0xba: 
			case 0xab: tb = dp; dp = cc.all; cc.all = tb; dp16 = (Word)(dp << 8); break;
		} // switch
	} else  {
		--pc;
		invalid("exchange register");
		return;
	}
}

void Mc6809::inca(void)
{
	help_inc(a);
}

void Mc6809::incb(void)
{
	help_inc(b);
}

void Mc6809::inc(void)
{
	Word	addr = fetch_effective_address();
	Byte	m = READ(addr);
	help_inc(m);
	WRITE(addr, m);
}

t_cycles Mc6809::inc_i(void)
{
	Word addr;
	Byte m;
	t_cycles cycl;
	
	addr = fetch_indexed_address(&cycl);
	m = READ(addr);
	help_inc(m);
	WRITE(addr, m);
	return cycl;
}

void Mc6809::help_inc(Byte& x)
{
	cc.bit.v = (x == 0x7f);
	x = x + 1;
	cc.bit.n = BTST7(x);
	cc.bit.z = !x;
}

void Mc6809::jmp(void)
{
	pc = fetch_effective_address();
}

t_cycles Mc6809::jmp_i(void)
{
	t_cycles cycl;
	
	pc = fetch_indexed_address(&cycl);
	return cycl;
}

void Mc6809::jsr(void)
{
	Word	addr = fetch_effective_address();
	WRITE(--s, (Byte)pc);
	WRITE(--s, pc >> 8);
	pc = addr;
}

t_cycles Mc6809::jsr_i(void)
{
	Word addr;
	t_cycles cycl;

	addr = fetch_indexed_address(&cycl);
	WRITE(--s, (Byte)pc);
	WRITE(--s, pc >> 8);
	pc = addr;
	return cycl;
}

void Mc6809::lda(void)
{
	a = fetch_operand();
	cc.bit.n = BTST7(a);
	cc.bit.v = 0;
	cc.bit.z = !a;
}

void Mc6809::ldb(void)
{
	b = fetch_operand();
	cc.bit.n = BTST7(b);
	cc.bit.v = 0;
	cc.bit.z = !b;
}

t_cycles Mc6809::lda_i(void)
{
	t_cycles cycl;

	a = fetch_indexed(&cycl);
	cc.bit.n = BTST7(a);
	cc.bit.v = 0;
	cc.bit.z = !a;
	return cycl;
}

t_cycles Mc6809::ldb_i(void)
{
	t_cycles cycl;

	b = fetch_indexed(&cycl);
	cc.bit.n = BTST7(b);
	cc.bit.v = 0;
	cc.bit.z = !b;
	return cycl;
}


void Mc6809::ldd(void)
{
	d = fetch_word_operand();
	cc.bit.n = d & 0x8000 ? 1 : 0;
	cc.bit.v = 0;
	cc.bit.z = !d;
}

void Mc6809::ldx(void)
{
	x = fetch_word_operand();
	cc.bit.n = x & 0x8000 ? 1 : 0;
	cc.bit.v = 0;
	cc.bit.z = !x;
}

void Mc6809::ldy(void)
{
	y = fetch_word_operand();
	cc.bit.n = y & 0x8000 ? 1 : 0;
	cc.bit.v = 0;
	cc.bit.z = !y;
}

void Mc6809::lds(void)
{
	s = fetch_word_operand();
	cc.bit.n = s & 0x8000 ? 1 : 0;
	cc.bit.v = 0;
	cc.bit.z = !s;
}

void Mc6809::ldu(void)
{
	u = fetch_word_operand();
	cc.bit.n = u & 0x8000 ? 1 : 0;
	cc.bit.v = 0;
	cc.bit.z = !u;
}

void Mc6809::help_ld(Word& x)
{
	x = fetch_word_operand();
	cc.bit.n = x & 0x8000 ? 1 : 0;
	cc.bit.v = 0;
	cc.bit.z = !x;
}

t_cycles Mc6809::ldd_i(void)
{
	t_cycles cycl;

	d = fetch_indexed_word(&cycl);
	cc.bit.n = d & 0x8000 ? 1 : 0;
	cc.bit.v = 0;
	cc.bit.z = !d;
	return cycl;
}

t_cycles Mc6809::ldx_i(void)
{
	t_cycles cycl;

	x = fetch_indexed_word(&cycl);
	cc.bit.n = x & 0x8000 ? 1 : 0;
	cc.bit.v = 0;
	cc.bit.z = !x;
	return cycl;
}

t_cycles Mc6809::ldy_i(void)
{
	t_cycles cycl;

	y = fetch_indexed_word(&cycl);
	cc.bit.n = y & 0x8000 ? 1 : 0;
	cc.bit.v = 0;
	cc.bit.z = !y;
	return cycl;
}

t_cycles Mc6809::ldu_i(void)
{
	t_cycles cycl;

	u = fetch_indexed_word(&cycl);
	cc.bit.n = u & 0x8000 ? 1 : 0;
	cc.bit.v = 0;
	cc.bit.z = !u;
	return cycl;
}

t_cycles Mc6809::lds_i(void)
{
	t_cycles cycl;

	s = fetch_indexed_word(&cycl);
	cc.bit.n = s & 0x8000 ? 1 : 0;
	cc.bit.v = 0;
	cc.bit.z = !s;
	return cycl;
}

t_cycles Mc6809::leax(void)
{
	t_cycles cycl;

	x = fetch_indexed_address(&cycl);
	cc.bit.z = !x;
	return cycl;
}

t_cycles Mc6809::leay(void)
{
	t_cycles cycl;

	y = fetch_indexed_address(&cycl);
	cc.bit.z = !y;
	return cycl;
}

t_cycles Mc6809::leas(void)
{
	t_cycles cycl;

	s = fetch_indexed_address(&cycl);
	return cycl;
}

t_cycles Mc6809::leau(void)
{
	t_cycles cycl;

	u = fetch_indexed_address(&cycl);
	return cycl;
}

void Mc6809::lsla(void)
{
	help_lsl(a);
}

void Mc6809::lslb(void)
{
	help_lsl(b);
}

void Mc6809::lsl(void)
{
	Word	addr = fetch_effective_address();
	Byte	m = READ(addr);
	help_lsl(m);
	WRITE(addr, m);
}

t_cycles Mc6809::lsl_i(void)
{
	Word addr;
	Byte m;
	t_cycles cycl;
	
	addr = fetch_indexed_address(&cycl);
	m = READ(addr);
	help_lsl(m);
	WRITE(addr, m);
	return cycl;
}

void Mc6809::help_lsl(Byte& x)
{
	cc.bit.c = BTST7(x);
	cc.bit.v = BTST7(x) ^ BTST6(x);
	x <<= 1;
	cc.bit.n = BTST7(x);
	cc.bit.z = !x;
}

void Mc6809::lsra(void)
{
	help_lsr(a);
}

void Mc6809::lsrb(void)
{
	help_lsr(b);
}

void Mc6809::lsr(void)
{
	Word	addr = fetch_effective_address();
	Byte	m = READ(addr);
	help_lsr(m);
	WRITE(addr, m);
}

t_cycles Mc6809::lsr_i(void)
{
	Word addr;
	Byte m;
	t_cycles cycl;
	
	addr = fetch_indexed_address(&cycl);
	m = READ(addr);
	help_lsr(m);
	WRITE(addr, m);
	return cycl;
}

void Mc6809::help_lsr(Byte& x)
{
	cc.bit.c = BTST0(x);
	x >>= 1;	/* Shift word right */
	cc.bit.n = 0;
	cc.bit.z = !x;
}

void Mc6809::mul(void)
{
	d = a * b;
	cc.bit.c = BTST7(b);
	cc.bit.z = !d;
}

void Mc6809::nega(void)
{
	help_neg(a);
}

void Mc6809::negb(void)
{
	help_neg(b);
}

void Mc6809::neg(void)
{
	Word 	addr = fetch_effective_address();
	Byte	m = READ(addr);
	help_neg(m);
	WRITE(addr, m);
}

t_cycles Mc6809::neg_i(void)
{
	Word addr;
	Byte m;
	t_cycles cycl;

	addr = fetch_indexed_address(&cycl);
	m = READ(addr);
	help_neg(m);
	WRITE(addr, m);
	return cycl;
}

void Mc6809::help_neg(Byte& x)
{
// Sw: fixed carry bug
	cc.bit.v = (x == 0x80);
	cc.bit.c = (x != 0);
	{
		x = ((~x) + 1) & 0xff;
	}

	cc.bit.n = BTST7(x);
	cc.bit.z = !x;
}

void Mc6809::nop(void)
{
}

void Mc6809::ora(void)
{
	help_or(a);
}

void Mc6809::orb(void)
{
	help_or(b);
}

void Mc6809::help_or(Byte& x)
{
	x = x | fetch_operand();
	cc.bit.v = 0;
	cc.bit.n = BTST7(x);
	cc.bit.z = !x;
}

t_cycles Mc6809::ora_i(void)
{
	return help_or_i(a);
}

t_cycles Mc6809::orb_i(void)
{
	return help_or_i(b);
}

t_cycles Mc6809::help_or_i(Byte& x)
{
	t_cycles cycl;

	x = x | fetch_indexed(&cycl);
	cc.bit.v = 0;
	cc.bit.n = BTST7(x);
	cc.bit.z = !x;
	return cycl;
}

void Mc6809::orcc(void)
{
	cc.all |= fetch_operand();
}

t_cycles Mc6809::pshs(void)
{
	Byte imm;

	imm = READ_PI(pc);
	return help_psh(imm, s, u);
}

t_cycles Mc6809::pshu(void)
{
	Byte imm;

	imm = READ_PI(pc);
	return help_psh(imm, u, s);
}

t_cycles Mc6809::help_psh(Byte w, Word& s, Word& u)
{
	switch ((Byte)(w & 0xf0)) {
		case 0xf0:	WRITE(--s, (Byte)pc);
				WRITE(--s, (Byte)(pc >> 8));
		case 0x70:	WRITE(--s, (Byte)u);
				WRITE(--s, (Byte)(u >> 8));
		case 0x30:	WRITE(--s, (Byte)y);
				WRITE(--s, (Byte)(y >> 8));
		case 0x10:	WRITE(--s, (Byte)x);
				WRITE(--s, (Byte)(x >> 8));
		case 0x00:	break;
		case 0xe0:	WRITE(--s, (Byte)pc);
				WRITE(--s, (Byte)(pc >> 8));
		case 0x60:	WRITE(--s, (Byte)u);
				WRITE(--s, (Byte)(u >> 8));
		case 0x20:	WRITE(--s, (Byte)y);
				WRITE(--s, (Byte)(y >> 8));
				break;
		case 0xd0:	WRITE(--s, (Byte)pc);
				WRITE(--s, (Byte)(pc >> 8));
		case 0x50:	WRITE(--s, (Byte)u);
				WRITE(--s, (Byte)(u >> 8));
				WRITE(--s, (Byte)x);
				WRITE(--s, (Byte)(x >> 8));
				break;
		case 0xc0:	WRITE(--s, (Byte)pc);
				WRITE(--s, (Byte)(pc >> 8));
		case 0x40:	WRITE(--s, (Byte)u);
				WRITE(--s, (Byte)(u >> 8));
				break;
		case 0xb0:	WRITE(--s, (Byte)pc);
				WRITE(--s, (Byte)(pc >> 8));
				WRITE(--s, (Byte)y);
				WRITE(--s, (Byte)(y >> 8));
				WRITE(--s, (Byte)x);
				WRITE(--s, (Byte)(x >> 8));
				break;
		case 0xa0:	WRITE(--s, (Byte)pc);
				WRITE(--s, (Byte)(pc >> 8));
				WRITE(--s, (Byte)y);
				WRITE(--s, (Byte)(y >> 8));
				break;
		case 0x90:	WRITE(--s, (Byte)pc);
				WRITE(--s, (Byte)(pc >> 8));
				WRITE(--s, (Byte)x);
				WRITE(--s, (Byte)(x >> 8));
				break;
		case 0x80:	WRITE(--s, (Byte)pc);
				WRITE(--s, (Byte)(pc >> 8));
				break;
	} // switch
	switch ((Byte)(w & 0x0f)) {
		case 0x0f:	WRITE(--s, dp);
		case 0x07:	WRITE(--s, b);
		case 0x03:	WRITE(--s, a);
		case 0x01:	WRITE(--s, cc.all);
		case 0x00:	break;
		case 0x0e:	WRITE(--s, dp);
		case 0x06:	WRITE(--s, b);
		case 0x02:	WRITE(--s, a);
				break;
		case 0x0d:	WRITE(--s, dp);
		case 0x05:	WRITE(--s, b);
				WRITE(--s, cc.all);
				break;
		case 0x0c:	WRITE(--s, dp);
		case 0x04:	WRITE(--s, b);
				break;
		case 0x0b:	WRITE(--s, dp);
				WRITE(--s, a);
				WRITE(--s, cc.all);
				break;
		case 0x09:	WRITE(--s, dp);
				WRITE(--s, cc.all);
				break;
		case 0x0a:	WRITE(--s, dp);
				WRITE(--s, a);
				break;
		case 0x08:	WRITE(--s, dp);
				break;
	} // switch
	return psh_pul_cycles[w];
}

t_cycles Mc6809::puls(void)
{
	Byte	w = READ_PI(pc);
	return help_pul(w, s, u);
}

t_cycles Mc6809::pulu(void)
{
	Byte	w = READ_PI(pc);
	return help_pul(w, u, s);
}

t_cycles Mc6809::help_pul(Byte w, Word& s, Word& u)
{
	switch ((Byte)(w & 0x0f)) {
		case 0x0f:	cc.all = READ_PI(s);
		case 0x0e:	a      = READ_PI(s);
		case 0x0c:	b      = READ_PI(s);
		case 0x08:	dp     = READ_PI(s);
				dp16   = (Word)(dp << 8);
		case 0x00:	break;
		case 0x07:	cc.all = READ_PI(s);
		case 0x06:	a      = READ_PI(s);
		case 0x04:	b      = READ_PI(s);
				break;
		case 0x0b:	cc.all = READ_PI(s);
		case 0x0a:	a      = READ_PI(s);
				dp     = READ_PI(s);
				dp16   = (Word)(dp << 8);
				break;
		case 0x03:	cc.all = READ_PI(s);
		case 0x02:	a      = READ_PI(s);
				break;
		case 0x0d:	cc.all = READ_PI(s);
				b      = READ_PI(s);
				dp     = READ_PI(s);
				dp16   = (Word)(dp << 8);
				break;
		case 0x09:	cc.all = READ_PI(s);
				dp     = READ_PI(s);
				dp16   = (Word)(dp << 8);
				break;
		case 0x05:	cc.all = READ_PI(s);
				b      = READ_PI(s);
				break;
		case 0x01:	cc.all = READ_PI(s);
				break;
	} // switch
	switch ((Byte)(w & 0xf0)) {
		case 0xf0:	x  = READ_WORD(s); s += 2;
		case 0xe0:	y  = READ_WORD(s); s += 2;
		case 0xc0:	u  = READ_WORD(s); s += 2;
		case 0x80:	pc = READ_WORD(s); s += 2;
		case 0x00:	break;
		case 0x70:	x  = READ_WORD(s); s += 2;
		case 0x60:	y  = READ_WORD(s); s += 2;
		case 0x40:	u  = READ_WORD(s); s += 2;
				break;
		case 0xb0:	x  = READ_WORD(s); s += 2;
		case 0xa0:	y  = READ_WORD(s); s += 2;
				pc = READ_WORD(s); s += 2;
				break;
		case 0x30:	x  = READ_WORD(s); s += 2;
		case 0x20:	y  = READ_WORD(s); s += 2;
				break;
		case 0xd0:	x  = READ_WORD(s); s += 2;
				u  = READ_WORD(s); s += 2;
				pc = READ_WORD(s); s += 2;
				break;
		case 0x90:	x  = READ_WORD(s); s += 2;
				pc = READ_WORD(s); s += 2;
				break;
		case 0x50:	x  = READ_WORD(s); s += 2;
				u  = READ_WORD(s); s += 2;
				break;
		case 0x10:	x  = READ_WORD(s); s += 2;
				break;
	} // switch
	return psh_pul_cycles[w];
}

void Mc6809::rola(void)
{
	help_rol(a);
}

void Mc6809::rolb(void)
{
	help_rol(b);
}

void Mc6809::rol(void)
{
	Word	addr = fetch_effective_address();
	Byte	m = READ(addr);
	help_rol(m);
	WRITE(addr, m);
}

t_cycles Mc6809::rol_i(void)
{
	Word addr;
	Byte m;
	t_cycles cycl;
	
	addr = fetch_indexed_address(&cycl);
	m = READ(addr);
	help_rol(m);
	WRITE(addr, m);
	return cycl;
}

void Mc6809::help_rol(Byte& x)
{
	int	oc;

	oc = cc.bit.c;
	cc.bit.c = BTST7(x);
	cc.bit.n = BTST6(x);
	cc.bit.v = cc.bit.c ^ cc.bit.n;
	x = x << 1;
	if (oc) BSET0(x);
	cc.bit.z = !x;
}

void Mc6809::rora(void)
{
	help_ror(a);
}

void Mc6809::rorb(void)
{
	help_ror(b);
}

void Mc6809::ror(void)
{
	Word	addr = fetch_effective_address();
	Byte	m = READ(addr);
	help_ror(m);
	WRITE(addr, m);
}

t_cycles Mc6809::ror_i(void)
{
	Word addr;
	Byte m;
	t_cycles cycl;
	
	addr = fetch_indexed_address(&cycl);
	m = READ(addr);
	help_ror(m);
	WRITE(addr, m);
	return cycl;
}

void Mc6809::help_ror(Byte& x)
{
	int	oc = cc.bit.c;
	cc.bit.c = BTST0(x);
	x = x >> 1;
	if (oc) BSET7(x);
	cc.bit.n = BTST7(x);
	cc.bit.z = !x;
}

t_cycles Mc6809::rti(void)
{
	cc.all = READ_PI(s);
	if (cc.bit.e) {
		help_pul(0xfe, s, u);
		return 15;
	} else {
		pc = READ_WORD(s);
		s += 2;
		return 6;
	}
}

void Mc6809::rts(void)
{
	pc = READ_WORD(s);
	s += 2;
}

void Mc6809::sbca(void)
{
	help_sbc(a);
}

void Mc6809::sbcb(void)
{
	help_sbc(b);
}

void Mc6809::help_sbc(Byte& d)
{
// Sw: fixed overflow and carry bug
	Byte s, e, r;

	s = fetch_operand();

	r = d - s - (cc.all & 0x01);
	e = (d ^ s) & 0x80 ? 1 : 0;
	cc.bit.n = (r & 0x80) ? 1 : 0;
	cc.bit.z = r ? 0 : 1 ;
	cc.bit.v = e & (((s ^ r) & 0x80) ? 0 : 1);
	cc.bit.c = ((((~d) &  s) & 0x80) ? 1 : 0) | (!e & cc.bit.n);
	d = r;
}

t_cycles Mc6809::sbca_i(void)
{
	return help_sbc_i(a);
}

t_cycles Mc6809::sbcb_i(void)
{
	return help_sbc_i(b);
}

t_cycles Mc6809::help_sbc_i(Byte& d)
{
// Sw: fixed overflow and carry bug
	Byte s, e, r;
	t_cycles cycl;

	s = fetch_indexed(&cycl);

	r = d - s - (cc.all & 0x01);
	e = (d ^ s) & 0x80 ? 1 : 0;
	cc.bit.n = (r & 0x80) ? 1 : 0;
	cc.bit.z = r ? 0 : 1 ;
	cc.bit.v = e & (((s ^ r) & 0x80) ? 0 : 1);
	cc.bit.c = ((((~d) &  s) & 0x80) ? 1 : 0) | (!e & cc.bit.n);
	d = r;
	return cycl;
}

void Mc6809::sex(void)
{
	cc.bit.n = BTST7(b);
	cc.bit.z = !b;
	a = cc.bit.n ? 255 : 0;
}

void Mc6809::sta(void)
{
	Word	addr;
	
	addr = fetch_effective_address();
	WRITE(addr, a);
	cc.bit.n = BTST7(a);
	cc.bit.v = 0;
	cc.bit.z = !a;
}

void Mc6809::stb(void)
{
	Word	addr;
	
	addr = fetch_effective_address();
	WRITE(addr, b);
	cc.bit.n = BTST7(b);
	cc.bit.v = 0;
	cc.bit.z = !b;
}

t_cycles Mc6809::sta_i(void)
{
	Word	addr;
	t_cycles cycl;

	addr = fetch_indexed_address(&cycl);
	WRITE(addr, a);
	cc.bit.n = BTST7(a);
	cc.bit.v = 0;
	cc.bit.z = !a;
	return cycl;
}

t_cycles Mc6809::stb_i(void)
{
	Word	addr;
	t_cycles cycl;
	
	addr = fetch_indexed_address(&cycl);
	WRITE(addr, b);
	cc.bit.n = BTST7(b);
	cc.bit.v = 0;
	cc.bit.z = !b;
	return cycl;
}


void Mc6809::std(void)
{
	help_st(d);
}

void Mc6809::stx(void)
{
	help_st(x);
}

void Mc6809::sty(void)
{
	help_st(y);
}

void Mc6809::sts(void)
{
	help_st(s);
}

void Mc6809::stu(void)
{
	help_st(u);
}

void Mc6809::help_st(Word x)
{
	Word	addr = fetch_effective_address();
	WRITE_WORD(addr, x);
	cc.bit.v = 0;
	cc.bit.n = BTST15(x);
	cc.bit.z = !x;
}

t_cycles Mc6809::stx_i(void)
{
	Word addr;
	t_cycles cycl;

	addr = fetch_indexed_address(&cycl);
	WRITE_WORD(addr, x);
	cc.bit.v = 0;
	cc.bit.n = x & 0x8000 ? 1 : 0;
	cc.bit.z = !x;
	return cycl;
}

t_cycles Mc6809::sty_i(void)
{
	Word addr;
	t_cycles cycl;

	addr = fetch_indexed_address(&cycl);
	WRITE_WORD(addr, y);
	cc.bit.v = 0;
	cc.bit.n = y & 0x8000 ? 1 : 0;
	cc.bit.z = !y;
	return cycl;
}

t_cycles Mc6809::stu_i(void)
{
	Word addr;
	t_cycles cycl;

	addr = fetch_indexed_address(&cycl);
	WRITE_WORD(addr, u);
	cc.bit.v = 0;
	cc.bit.n = u & 0x8000 ? 1 : 0;
	cc.bit.z = !u;
	return cycl;
}

t_cycles Mc6809::sts_i(void)
{
	Word addr;
	t_cycles cycl;

	addr = fetch_indexed_address(&cycl);
	WRITE_WORD(addr, s);
	cc.bit.v = 0;
	cc.bit.n = s & 0x8000 ? 1 : 0;
	cc.bit.z = !s;
	return cycl;
}

t_cycles Mc6809::std_i(void)
{
	Word addr;
	t_cycles cycl;

	addr = fetch_indexed_address(&cycl);
	WRITE_WORD(addr, d);
	cc.bit.v = 0;
	cc.bit.n = d & 0x8000 ? 1 : 0;
	cc.bit.z = !d;
	return cycl;
}

void Mc6809::suba(void)
{
	help_sub(a);
}

void Mc6809::subb(void)
{
	help_sub(b);
}

void Mc6809::help_sub(Byte& d)
{
// Sw: fixed overflow and carry bug
	Byte s, e, r;

	s = fetch_operand();

	r = d - s;
	e = (d ^ s) & 0x80 ? 1 : 0;
	cc.bit.n = (r & 0x80) ? 1 : 0;
	cc.bit.z = r ? 0 : 1 ;
	cc.bit.v = e & (((s ^ r) & 0x80) ? 0 : 1);
	cc.bit.c = ((((~d) &  s) & 0x80) ? 1 : 0) | (!e & cc.bit.n);
	d = r;
}

t_cycles Mc6809::suba_i(void)
{
	return help_sub_i(a);
}

t_cycles Mc6809::subb_i(void)
{
	return help_sub_i(b);
}

t_cycles Mc6809::help_sub_i(Byte& d)
{
	Byte s, e, r;
	t_cycles cycl;

	s = fetch_indexed(&cycl);
	r = d - s;
	e = (d ^ s) & 0x80 ? 1 : 0;
	cc.bit.n = (r & 0x80) ? 1 : 0;
	cc.bit.z = r ? 0 : 1 ;
	cc.bit.v = e & (((s ^ r) & 0x80) ? 0 : 1);
	cc.bit.c = ((((~d) &  s) & 0x80) ? 1 : 0) | (!e & cc.bit.n);
	d = r;
	return cycl;
}

void Mc6809::subd(void)
{
// Sw: fixed overflow and carry bug
	Word o, s, e, r;

	s = fetch_word_operand();

	o = d;
	r = o - s;
	e = (o ^ s) & 0x8000 ? 1 : 0;
	cc.bit.n = (r & 0x8000) ? 1 : 0;
	cc.bit.z = r ? 0 : 1 ;
	cc.bit.v = e & (((s ^ r) & 0x8000) ? 0 : 1);
	cc.bit.c = ((((~o) &  s) & 0x8000) ? 1 : 0) | (!e & cc.bit.n);
	d = r;
}

t_cycles Mc6809::subd_i(void)
{
	Word o, s, e, r;
	t_cycles cycl;

	s = fetch_indexed_word(&cycl);

	o = d;
	r = o - s;
	e = (o ^ s) & 0x8000 ? 1 : 0;
	cc.bit.n = (r & 0x8000) ? 1 : 0;
	cc.bit.z = r ? 0 : 1 ;
	cc.bit.v = e & (((s ^ r) & 0x8000) ? 0 : 1);
	cc.bit.c = ((((~o) &  s) & 0x8000) ? 1 : 0) | (!e & cc.bit.n);
	d = r;
	return cycl;
}

void Mc6809::sync(void)
{
	pc--; // processor loops in sync instruction until interrupt
	sync_flg = SYNC_INSTR;
}

void Mc6809::cwai(void)
{
	if (!sync_flg) {
		cc.all &= READ_PI(pc);
		cc.bit.e = 1;
		help_psh(0xff, s, u);
		sync_flg = CWAI_INSTR;
		pc--;
	}
	pc--; // processor loops in cwai instruction until interrupt
}

void Mc6809::swi(void)
{
	cc.bit.e = 1;
	help_psh(0xff, s, u);
	cc.bit.f = cc.bit.i = 1;
	pc = READ_WORD(0xfffa);
}

void Mc6809::swi2(void)
{
	cc.bit.e = 1;
	help_psh(0xff, s, u);
	pc = READ_WORD(0xfff4);
}

void Mc6809::swi3(void)
{
	cc.bit.e = 1;
	help_psh(0xff, s, u);
	pc = READ_WORD(0xfff2);
}

void Mc6809::nmi(void)
{
	if (sync_flg != CWAI_INSTR) {
		cc.bit.e = 1;
		help_psh(0xff, s, u);
	}
	cc.bit.f = cc.bit.i = 1;
	pc = READ_WORD(0xfffc);
}

void Mc6809::firq(void)
{
	if (sync_flg != CWAI_INSTR) {
		cc.bit.e = 0;
		help_psh(0x81, s, u);
	}
	cc.bit.f = cc.bit.i = 1;
	pc = READ_WORD(0xfff6);
}

void Mc6809::irq(void)
{
	if (sync_flg != CWAI_INSTR) {
		cc.bit.e = 1;
		help_psh(0xff, s, u);
	}
	cc.bit.i = 1;			// Sw: don't set flag f !!
	pc = READ_WORD(0xfff8);
}

void Mc6809::tfr(void)
{
	int	r1, r2;
	Byte	w = READ_PI(pc);
	r1 = (w & 0xf0) >> 4;
	r2 = (w & 0x0f);
	if (r1 <= 5) {
		if (r2 > 5) {
			--pc;
			invalid("transfer register");
			return;
		}
		switch (w) {
			case 0x10: d = x; break;
			case 0x20: d = y; break;
			case 0x30: d = u; break;
			case 0x40: d = s; break;
			case 0x50: d = pc; break;
			case 0x01: x = d; break;
			case 0x21: x = y; break;
			case 0x31: x = u; break;
			case 0x41: x = s; break;
			case 0x51: x = pc; break;
			case 0x02: y = d; break;
			case 0x12: y = x; break;
			case 0x32: y = u; break;
			case 0x42: y = s; break;
			case 0x52: y = pc; break;
			case 0x03: u = d; break;
			case 0x13: u = x; break;
			case 0x23: u = y; break;
			case 0x43: u = s; break;
			case 0x53: u = pc; break;
			case 0x04: s = d; break;
			case 0x14: s = x; break;
			case 0x24: s = y; break;
			case 0x34: s = u; break;
			case 0x54: s = pc; break;
			case 0x05: pc = d; break;
			case 0x15: pc = x; break;
			case 0x25: pc = y; break;
			case 0x35: pc = u; break;
			case 0x45: pc = s; break;
		} // switch
	} else if (r1 >= 8 && r2 <= 11) {
		if (r2 < 8 || r2 > 11) {
			--pc;
			invalid("transfer register");
			return;
		}
		switch (w) {
			case 0x98: a = b; break;
			case 0x89: b = a; break;
			case 0xa8: a = cc.all; break;
			case 0x8a: cc.all = a; break;
			case 0xb8: a = dp; break;
			case 0x8b: dp = a; dp16 = (Word)(dp << 8); break;
			case 0xa9: b = cc.all; break;
			case 0x9a: cc.all = b; break;
			case 0xb9: b = dp; break;
			case 0x9b: dp = b; dp16 = (Word)(dp << 8); break;
			case 0xba: cc.all = dp; break;
			case 0xab: dp = cc.all; dp16 = (Word)(dp << 8); break;
		} // switch
	} else  {
		--pc;
		invalid("transfer register");
		return;
	}
}

void Mc6809::tsta(void)
{
	help_tst(a);
}

void Mc6809::tstb(void)
{
	help_tst(b);
}

void Mc6809::tst(void)
{
	Word	addr = fetch_effective_address();
	Byte	m = READ(addr);
	help_tst(m);
}

t_cycles Mc6809::tst_i(void)
{
	Word addr;
	Byte m;
	t_cycles cycl;
	
	addr = fetch_indexed_address(&cycl);
	m = READ(addr);
	help_tst(m);
	return cycl;
}

void Mc6809::help_tst(Byte x)
{
	cc.bit.v = 0;
	cc.bit.n = x & 0x80 ? 1 : 0;
	cc.bit.z = !x;
}

void Mc6809::do_br(int test)
{
	Word offset;
	
	if (test) {
		offset = (Word)READ_PI(pc);
		if (offset & 0x80)
			offset |= 0xff00;
		pc += offset;
	} else
		pc++;
}

t_cycles Mc6809::do_lbr(int test)
{
	Word offset;
	
	if (test) {
		offset = READ_WORD(pc);
		pc += offset + 2;
		return 1;
	} else {
		pc += 2;
		return 0;
	}
}

Word& Mc6809::refreg(Byte post)
{
// Sw: enhanced
	switch ((Byte)(post & 0x60)) {
		case 0x00 : return x;
		case 0x20 : return y;
		case 0x40 : return u;
		case 0x60 : return s;
		case 0x80 : return s; // one more case for performance
		default   : return s; // should never happen
	} // switch
} // refreg


Byte Mc6809::fetch_operand(void)
{
	Byte		ret = 0;
	Word		addr;

	switch (mode) {
		case immediate :
		case relative  : return READ_PI(pc);// is identical to case relative
		case extended  : addr = READ_WORD(pc);
				 pc += 2;
			         return READ(addr);
		case direct    : addr = dp16 | READ_PI(pc);
		                 return READ(addr);
		default        : invalid("addressing mode");
				 return ret;
	}  // switch
}  // fetch_operand


Word Mc6809::fetch_word_operand(void)
{
	Word addr = 0;

	switch(mode) {
		case immediate :
		case relative  : addr = READ_WORD(pc);
				 pc += 2;
				 return addr;
		case extended  : addr = READ_WORD(pc);
				 pc += 2;
	 			 return READ_WORD(addr);
		case direct    : addr = dp16 | READ_PI(pc);
				 return READ_WORD(addr);
		default        : invalid("addressing mode");
				 return addr;
	} // switch
}  // fetch_word_operand

Word Mc6809::fetch_effective_address(void)
{
	Word		addr = 0;

	if (mode == extended) {
		addr = READ_WORD(pc);
		pc += 2;
		return addr;
	} else 
	if (mode == direct) {
		addr = dp16 | READ_PI(pc);
		return addr;
	} else
		invalid("addressing mode");
	return addr;
} // fetch_effective_address

Word Mc6809::fetch_indexed_address(t_cycles *c)
{
	Word		addr;
	Byte		post;

	post = READ_PI(pc);
	if ((post & 0x8e) == 0x82)
		do_predecrement(post);
	addr = do_effective_address(post);
	if ((post & 0x8e) == 0x80)
		do_postincrement(post);
	*c = indexed_cycles[post];
	return addr;
} // fetch_indexed_address

Word Mc6809::fetch_indexed_word(t_cycles *c)
{
	Word		addr;
	Byte		post;

	post = READ_PI(pc);
	if ((post & 0x8e) == 0x82)
		do_predecrement(post);
	addr = do_effective_address(post);
	if ((post & 0x8e) == 0x80)
		do_postincrement(post);
	*c = indexed_cycles[post];
	return READ_WORD(addr);
}

Byte Mc6809::fetch_indexed(t_cycles *c)
{
	Word		addr;
	Byte		post;
	Byte		ret;

	post = READ_PI(pc);
	if ((post & 0x8e) == 0x82)
		do_predecrement(post);
        addr = do_effective_address(post);
        ret = READ(addr);
	if ((post & 0x8e) == 0x80)
        	do_postincrement(post);
	*c = indexed_cycles[post];
        return ret;
}

Word Mc6809::do_effective_address(Byte post)
{
	Word addr = 0;
	Word offset;

	if ((post & 0x80) == 0x00) {
		offset = post & 0x1f;
		if (offset & 0x10)
			offset |= 0xffe0;
		addr = refreg(post) + offset;
	} else {
		switch ((Byte)(post & 0x7f)) {
			case 0x00: case 0x01: case 0x02: case 0x03: // X
			case 0x04: case 0x14: case 0x11: case 0x13:
				addr = x;
				break;
			case 0x20: case 0x21: case 0x22: case 0x23: // Y
			case 0x24: case 0x31: case 0x33: case 0x34:
				addr = y;
				break;
			case 0x40: case 0x41: case 0x42: case 0x43: // U
			case 0x44: case 0x51: case 0x53: case 0x54:
				addr = u;
				break;
			case 0x60: case 0x61: case 0x62: case 0x63: // S
			case 0x64: case 0x71: case 0x73: case 0x74:
				addr = s;
				break;
			case 0x05: case 0x15:
				addr = extend8(b) + x;
				break;
			case 0x25: case 0x35:
				addr = extend8(b) + y;
				break;
			case 0x45: case 0x55:
				addr = extend8(b) + u;
				break;
			case 0x65: case 0x75:
				addr = extend8(b) + s;
				break;
			case 0x06: case 0x16:
				addr = extend8(a) + x;
				break;
			case 0x26: case 0x36:
				addr = extend8(a) + y;
				break;
			case 0x46: case 0x56:
				addr = extend8(a) + u;
				break;
			case 0x66: case 0x76:
				addr = extend8(a) + s;
				break;
			case 0x08: case 0x18:
				addr = x + extend8(READ_PI(pc));
				break;
			case 0x28: case 0x38:
				addr = y + extend8(READ_PI(pc));
				break;
			case 0x48: case 0x58:
				addr = u + extend8(READ_PI(pc));
				break;
			case 0x68: case 0x78:
				addr = s + extend8(READ_PI(pc));
				break;
			case 0x09: case 0x19:
				addr = x + READ_WORD(pc);
				pc += 2;
				break;
			case 0x29: case 0x39:
				addr = y + READ_WORD(pc);
				pc += 2;
				break;
			case 0x49: case 0x59:
				addr = u + READ_WORD(pc);
				pc += 2;
				break;
			case 0x69: case 0x79:
				addr = s + READ_WORD(pc);
				pc += 2;
				break;
			case 0x0b: case 0x1b:
				addr = d + x;
				break;
			case 0x2b: case 0x3b:
				addr = d + y;
				break;
			case 0x4b: case 0x5b:
				addr = d + u;
				break;
			case 0x6b: case 0x7b:
				addr = d + s;
				break;
			case 0x0c: case 0x1c: case 0x2c: case 0x3c:
			case 0x4c: case 0x5c: case 0x6c: case 0x7c:
				addr = extend8(READ_PI(pc));
				addr += pc;
				break;
			case 0x0d: case 0x1d: case 0x2d: case 0x3d:
			case 0x4d: case 0x5d: case 0x6d: case 0x7d:
				addr = READ_WORD(pc);
				pc += 2;
				addr += pc;
				break;
			case 0x1f: case 0x3f: case 0x5f: case 0x7f:
				addr = READ_WORD(pc);
				pc += 2;
				break;
			default: --pc;
				invalid("indirect addressing postbyte");
				break;
		}

		/* Do extra indirection */
		if (post & 0x10) {
			addr = READ_WORD(addr);
		}
	}

	return addr;
}

void Mc6809::do_postincrement(Byte post)
{
	switch (post) {
		case 0x80: x++; break;
		case 0xa0: y++; break;
		case 0xc0: u++; break;
		case 0xe0: s++; break;
		case 0x90: case 0xb0: case 0xd0: case 0xf0:
			--pc;
			invalid("postincrement");
			break;
		case 0x91:
		case 0x81: x += 2; break;
		case 0xb1:
		case 0xa1: y += 2; break;
		case 0xd1:
		case 0xc1: u += 2; break;
		case 0xf1:
		case 0xe1: s += 2; break;
	} // switch
} // do_postincrement

void Mc6809::do_predecrement(Byte post)
{
	switch (post) {
		case 0x82: x--; break;
		case 0xa2: y--; break;
		case 0xc2: u--; break;
		case 0xe2: s--; break;
		case 0x92: case 0xb2: case 0xd2: case 0xf2:
			--pc;
			invalid("predecrement");
			break;
			case 0x91:
		case 0x83: x -= 2; break;
		case 0xb3:
		case 0xa3: y -= 2; break;
		case 0xd3:
		case 0xc3: u -= 2; break;
		case 0xf3:
		case 0xe3: s -= 2; break;
	} // switch
} // do_predecrement
#endif // ifdef FASTFLEX

