/*
    mc6809.h

    flexemu, an MC6809 emulator running FLEX
    Copyright (C) 1997-2000  W. Schwotzer

    This file is based on usim-0.91 which is
    Copyright (C) 1994 by R. B. Bellis
*/



#ifndef __mc6809_h__
#define __mc6809_h__

#include <stdio.h>
#include <signal.h>
#include <time.h>
#include "classes.h"
#include "typedefs.h"
#include "memory.h"

#ifdef __osf__
extern "C" unsigned int alarm(unsigned int);
#endif


#define NMI			0x01
#define FIRQ			0x02
#define IRQ			0x04
#define INVALID			0x08
#define BREAKPOINT		0x10
#define SINGLESTEP		0x20

#define	NO_CHANGE		0
#define	RUN			1
#define	STOP			2
#define	STEP			3
#define	EXIT			4
#define	RESET			5
#define	NEXT			6
#define	RESET_RUN		7
#define	NO_CPUVIEW_UPDATE	20	

#define SYNC_INSTR		1
#define CWAI_INSTR		2

// nr. of processor cycles before duing a window update 
#ifdef FASTFLEX
#define CYCLES_TO_UPDATE	20000
#else
#define CYCLES_TO_UPDATE	2000
#endif

#define DELTA_DELAY_CONST	1000

typedef unsigned long int t_cycles; // must be unsigned long because of sprintf

struct s_cpu_status {
	float		freq;
	t_cycles	cycles;
	Byte		a, b, cc, dp;
	Word		pc, s, u, x, y;
	Byte		state;
};

class Mc6809 : public IoDevice {

// only needed for debugging
protected:
	FILE *logfp;

// Processor state, user input for new state:
protected:
	Byte state, user_input;

// Processor addressing modes
protected:

	enum e_mode {
				immediate,
				relative,
				inherent,
				extended,
				direct,
				indexed
	} mode;

// Processor registers
protected:

	Byte			indexed_cycles[256];	// add. cycles for
							// indexed addr.
	Byte			psh_pul_cycles[256];	// add. cycles for psh
							// and pull-instr. 
	Byte			nmi_armed, exceptions;	// for handling
							// interrupts
	Byte			sync_flg;		// for SYNC instruction
#ifdef FASTFLEX
	Word			ipcreg, iureg, isreg, ixreg, iyreg;
	Byte			iareg, ibreg, iccreg, idpreg;
	Word			eaddr;
	Byte			ireg;
	Byte			iflag;
	Byte			tb;
	Word			tw;
	Byte			k;
	Byte			*pMem;		// needed for memory access
#else
	Word			pc;
	Word			u, s;		// Stack pointers
	Word			x, y;		// Index registers
	Byte			dp;		// Direct Page register
	Word			dp16;		// dp converted to 16 Bit
	union {
		Word			d;	// Combined accumulator
		struct {
#ifdef WORDS_BIGENDIAN
			Byte		a;	// Accumulator a
			Byte		b;	// Accumulator b
#else
			Byte		b;	// Accumulator b
			Byte		a;	// Accumulator a
#endif
		} byte;
	} acc;
	Byte&			a;
	Byte&			b;
	Word&			d;
	union {
		Byte			all;	// Condition code register
		struct {
#ifdef BITFIELDS_LSB_FIRST
			Byte		c : 1;	// Carry
			Byte		v : 1;	// Overflow
			Byte		z : 1;	// Zero
			Byte		n : 1;	// Negative
			Byte		i : 1;	// IRQ disable
			Byte		h : 1;	// Half carry
			Byte		f : 1;	// FIRQ disable
			Byte		e : 1;	// Entire
#else
			Byte		e : 1;	// Entire
			Byte		f : 1;	// FIRQ disable
			Byte		h : 1;	// Half carry
			Byte		i : 1;	// IRQ disable
			Byte		n : 1;	// Negative
			Byte		z : 1;	// Zero
			Byte		v : 1;	// Overflow
			Byte		c : 1;	// Carry
#endif
		} bit;
	} cc;
#endif // FASTFLEX

// support for target frequency
protected:

	int		delay, delta_delay;
	float		target_frequency;
	float		delta_frequency;
	float		frequency;	// actual frequency

protected:

	Da6809*		disassembler;
	AbstractGui*	gui;
	Inout*		inout;

// funcitons for instruction execution:

private:


	void			init(void);
	void			illegal();
#ifndef FASTFLEX
	Word&			refreg(Byte);
	Byte			fetch_operand(void);
	Word			fetch_word_operand(void);
	Word			fetch_effective_address(void);
	Word			fetch_indexed_address(t_cycles *cycles);
	Word			fetch_indexed_word(t_cycles *cycles);
	Byte			fetch_indexed(t_cycles *cycles);
	Word			do_effective_address(Byte);
	void			do_predecrement(Byte);
	void			do_postincrement(Byte);

	void			abx();
	t_cycles		page_10();
	t_cycles		page_11();
	void			adca(), adcb();
	t_cycles		adca_i(), adcb_i();
	void			adda(), addb(), addd();
	t_cycles		adda_i(), addb_i(), addd_i();
	void			anda(), andb();
	t_cycles		anda_i(), andb_i();
	void			andcc();
	void			asra(), asrb(), asr();
	t_cycles		asr_i();
	void			bcc();
	t_cycles		lbcc();
	void			bcs();
	t_cycles		lbcs();
	void			beq();
	t_cycles		lbeq();
	void			bge();
	t_cycles		lbge();
	void			bgt();
	t_cycles		lbgt();
	void			bhi();
	t_cycles		lbhi();
	void			bita(), bitb();
	t_cycles		bita_i(), bitb_i();
	void			ble();
	t_cycles		lble();
	void			bls();
	t_cycles		lbls();
	void			blt();
	t_cycles		lblt();
	void			bmi();
	t_cycles		lbmi();
	void			bne();
	t_cycles		lbne();
	void			bpl();
	t_cycles		lbpl();
	void			bra();
	t_cycles		lbra();
	void			brn();
	t_cycles		lbrn();
	void			bsr();
	t_cycles		lbsr();
	void			bvc();
	t_cycles		lbvc();
	void			bvs();
	t_cycles		lbvs();
	void			clra(), clrb(), clr();
	t_cycles		clr_i();
	void			cmpa(), cmpb();
	t_cycles		cmpa_i(), cmpb_i();
	void			cmpd(), cmpx(), cmpy(), cmpu(), cmps();
	t_cycles		cmpd_i(), cmpx_i(), cmpy_i(), cmpu_i(), cmps_i();
	void			coma(), comb(), com();
	t_cycles		com_i();
	void			daa();
	void			deca(), decb(), dec();
	t_cycles		dec_i();
	void			eora(), eorb();
	t_cycles		eora_i(), eorb_i();
	void			exg();
	void			inca(), incb(), inc();
	t_cycles		inc_i();
	void			jmp();
	t_cycles		jmp_i();
	void			jsr();
	t_cycles		jsr_i();
	void			lda(), ldb();
	t_cycles		lda_i(), ldb_i();
	void			ldd(), ldx(), ldy(), lds(), ldu();
	t_cycles		ldd_i(), ldx_i(), ldy_i(), lds_i(), ldu_i();
	t_cycles		leax(), leay(), leas(), leau(); 
	void			lsla(), lslb(), lsl();
	void			lsra(), lsrb(), lsr();
	t_cycles		lsr_i(), lsl_i();
	void			mul();
	void			nega(), negb(), neg();
	t_cycles		neg_i();
	void			nop();
	void			ora(), orb();
	t_cycles		ora_i(), orb_i();
	void			orcc();
	t_cycles		pshs(), pshu();
	t_cycles		puls(), pulu();
	void			rola(), rolb(), rol();
	void			rora(), rorb(), ror();
	t_cycles		rol_i(), ror_i();
	t_cycles		rti();
	void			rts();
	void			sbca(), sbcb();
	t_cycles		sbca_i(), sbcb_i();
	void			sex();
	void			sta(), stb();
	t_cycles		sta_i(), stb_i();
	t_cycles		std_i(), stx_i(), sty_i(), sts_i(), stu_i();
	void			std(), stx(), sty(), sts(), stu();
	void			suba(), subb();
	t_cycles		suba_i(), subb_i();
	void 			subd();
	t_cycles		subd_i();
	void			swi(), swi2(), swi3();
	void			cwai(), sync();
	void			tfr();
	void			tsta(), tstb(), tst();
	t_cycles		tst_i();

	void			do_br(int);
	t_cycles		do_lbr(int);

	void			help_adc(Byte&);
	t_cycles		help_adc_i(Byte&);
	void			help_add(Byte&);
	t_cycles		help_add_i(Byte&);
	void			help_and(Byte&);
	t_cycles		help_and_i(Byte&);
	void			help_asr(Byte&);
	void			help_bit(Byte);
	t_cycles		help_bit_i(Byte);
	void			help_clr(Byte&);
	void			help_cmp(Byte);
	t_cycles		help_cmp_i(Byte);
	void			help_cmp(Word);
	t_cycles		help_cmp_i(Word);
	void			help_com(Byte&);
	void			help_dec(Byte&);
	void			help_eor(Byte&);
	t_cycles		help_eor_i(Byte&);
	void			help_inc(Byte&);
	void			help_ld(Word&);
	void			help_lsr(Byte&);
	void			help_lsl(Byte&);
	void			help_neg(Byte&);
	void			help_or(Byte&);
	t_cycles		help_or_i(Byte&);
	t_cycles		help_psh(Byte, Word&, Word&);
	t_cycles		help_pul(Byte, Word&, Word&);
	void			help_ror(Byte&);
	void			help_rol(Byte&);
	void			help_sbc(Byte&);
	t_cycles		help_sbc_i(Byte&);
	void			help_st(Word);
	void			help_sub(Byte&);
	t_cycles		help_sub_i(Byte&);
	void			help_sub(Word&);
	void			help_tst(Byte);
#endif
	void			nmi(), firq(), irq();

	void			invalid(const char *pmessage);

// Functions to handle processor state
public:

	Byte			singlestep(void);
	Byte			statemachine1(void);
	Byte			statemachine2(void);
	void			do_reset(void);
	void			set_new_state(Byte user_input);

// interrupt handling:
public:
	void			reset(void);		// CPU reset
	t_cycles		exec_irqs(void);
	void			set_nmi(void);
	void			set_firq(void);
	void			set_irq(void);

// functions to load hexfile

private:

	void			load_intelhex(FILE *fp);
	void			load_motorola_srec(FILE *fp);
public:
	bool			load_hexfile(const char *filename,
					bool ignore_errors = false);

// breakpoint support
protected:
		unsigned int	bp[3];
		t_cycles 	next_update;
		t_cycles	total_cycles;	// actual proc-cycle after reset
		t_cycles	cycles;		// proc-cycle for one second

public:
		void		set_bp(int which, Word address);
		unsigned int	get_bp(int which);
		int		is_bp_set(int which);
		void		reset_bp(int which);

// interface to other classes
public:
		void		set_gui(AbstractGui* x_gui);
		void		set_inout(Inout* x_inout);
		void		set_disassembler(Da6809 *x_da);
		int		disassemble(Word address, Byte *pstep,
					char **pb1, char **pb2);
		void		get_status(struct s_cpu_status *pstatus);
		void		set_frequency(float target_frequency);

// errorhandling
private:
		char		err_msg[128];

// interface to gui
public:
		void		popup_message(char *msg);
		static Word	update_flag;
protected:
		void		update_frequency(void);

//****************************************************
// the memory interface is beginning from here
// it is included into the cpu class because of 
// performance 
//****************************************************

// Internal registers

private:

	Byte		*ppage[16];
	Byte		 memory[0x10000L];
	Byte		 byte_conversion_table[256];
	Byte		*vram_ptrs[MAX_VRAM];
	int		 io_base_addr;
	int		 io_initialized;
	IoDevice	*ioDevices[MAX_IO_DEVICES];
	struct sIoSelect *ppIo;

public:
	// interface to video display
	Byte video_ram[VIDEORAM_SIZE * 12];
	char changed[YBLOCKS];

private:

	void	init_memory(void);
	void	uninit_memory(void);

// Initialisation functions

public:

	void	initialize_byte_conv_table(int lsb_first);
	void	initialize_io_page(Word base_addr);
	Byte	add_io_device(
			IoDevice *device,
			Word base_addr1, Byte range1,
			Word base_addr2, Byte range2);

// public interface
public:

	void	reset_io(void);
	Byte	read(Word addr);
	Byte	readFromIo(Word addr);
	Word	read_word(Word addr);
	void	write(Word addr, Byte val);
	void	write_word(Word addr, Word val);
	void	write_rom(Word addr, Byte val);
	void	switch_mmu(Word offset, Byte val);

// interface for io device protocol

public:

	virtual Byte	readIo(Word addr);
	virtual void	writeIo(Word addr, Byte val);

// Public constructor and destructor
public:
				Mc6809(void);
	virtual			~Mc6809();

		
};

#ifdef __osf__
extern "C" unsigned int alarm(unsigned int);
#endif


#endif // __mc6809_h__

