/*
    mc6809.cpp

    flexemu, an MC6809 emulator running FLEX
    Copyright (C) 1997-2000  W. Schwotzer

    This file is based on usim-0.91 which is 
    Copyright (C) 1994 by R. B. Bellis
*/


#include <misc1.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <ctype.h>

#include "memory.h"
#include "mc6809.h"
#include "da6809.h"
#include "absgui.h"
#include "intmem.h"


#ifdef FASTFLEX
#define PC ipcreg
#else
#define PC pc
#endif

Word Mc6809::update_flag;

Mc6809::Mc6809(void)
#ifndef FASTFLEX
	: a(acc.byte.a), b(acc.byte.b), d(acc.d)
#endif
{
	disassembler	= NULL;
	gui	    	= NULL;
	inout		= NULL;
	ppIo		= NULL;
	update_flag     = 0;
	delta_frequency	= (float)0.0;
	frequency	= (float)0.0;
	set_frequency((float)0.0);		// run as fast as possible
	init();
	init_memory();

#ifdef DEBUG_FILE
	logfp = fopen(DEBUG_FILE, "w");
#endif
}

Mc6809::~Mc6809()
{
	uninit_memory();
#ifdef DEBUG_FILE
	if (logfp != NULL)
		fclose(logfp);
#endif
}

void Mc6809::set_disassembler(Da6809 *x_disassembler)
{
	disassembler = x_disassembler;
}

void Mc6809::init(void)
{
	Word i;
	Byte cycles;

	// all breakpoints are reset
	for (i=0; i < 3; i++)
		reset_bp(i);
	exceptions = 0;
	state	   = RUN;
	user_input = NO_CHANGE;
	for (i = 0; i < 128; i++)
			indexed_cycles[i] = 1;		
	for (i = 128; i < 256; i++)
		switch ((Byte)(i & 0x1f)) {
			case 0x05: case 0x06: case 0x08: case 0x0c:
				indexed_cycles[i] = 1; break;
			case 0x00: case 0x02:
				indexed_cycles[i] = 2; break;
			case 0x01: case 0x03: case 0x14:
				indexed_cycles[i] = 3; break;
			case 0x09: case 0x0b: case 0x18: case 0x15:
			case 0x16: case 0x1c:
				indexed_cycles[i] = 4; break;
			case 0x0d: case 0x1f:
				indexed_cycles[i] = 5; break;
			case 0x11: case 0x13:
				indexed_cycles[i] = 6; break;
			case 0x19: case 0x1b:
				indexed_cycles[i] = 7; break;
			case 0x1d:
				indexed_cycles[i] = 8; break;
			default:
				indexed_cycles[i] = 0; break;
		} // switch

	for (i = 0; i < 256; i++) {
		cycles = 0;
		if (i & 0x01) cycles++;
		if (i & 0x02) cycles++;
		if (i & 0x04) cycles++;
		if (i & 0x08) cycles++;
		if (i & 0x10) cycles += 2;
		if (i & 0x20) cycles += 2;
		if (i & 0x40) cycles += 2;
		if (i & 0x80) cycles += 2;
		psh_pul_cycles[i] = cycles;		
	} // for
} // init

void Mc6809::reset(void)
{
	cycles		= 0;
	next_update	= CYCLES_TO_UPDATE;
	nmi_armed	= 0;
	sync_flg	= 0;
	/* no interrupts yet */
	exceptions	&= ~(IRQ | FIRQ | NMI);
	reset_bp(2);	// remove next-breakpoint
	if (bp[0] > 0xffff && bp[1] > 0xffff && bp[2] > 0xffff)
		exceptions &= ~BREAKPOINT;
#ifdef FASTFLEX
	ipcreg		= READ_WORD(0xfffe);
	idpreg		= 0x00;		/* Direct page register = 0x00 */
	iccreg		= 0x50;		/* set i and f bit		*/
	eaddr		= 0;
	iflag		= 0;
	tb		= 0;
	tw		= 0;
	k		= 0;
#else
	pc		= READ_WORD(0xfffe);
	dp		= 0x00;		/* Direct page register = 0x00 */
	dp16		= (Word)(dp << 8);
	cc.all		= 0x00;		/* Clear all flags */
	cc.bit.i	= 1;		/* IRQ disabled */
	cc.bit.f	= 1;		/* FIRQ disabled */
#endif
}

int Mc6809::disassemble(Word address, Byte *pstep,
	char **pb1, char **pb2)
{
	if (disassembler == NULL)
		return 0;
	return disassembler->disassemble(address, pstep, pb1, pb2);
}

void Mc6809::set_nmi(void)
{
	exceptions |= NMI;
} // set_nmi

void Mc6809::set_firq(void)
{
	exceptions |= FIRQ;
} // set_firq

void Mc6809::set_irq(void)
{
	exceptions |= IRQ;
} // set_irq


void Mc6809::invalid(const char *msg)
{
	sprintf((char *)&err_msg, "\
invalid %s\n\
pc=%04x opcode=%02x\n\
processor stopped\n", msg ? msg : "",
	PC - 1, READ(PC - 1));
	popup_message(err_msg);
	exceptions |= INVALID;
}

void Mc6809::popup_message(char *pmsg)
{
	if (gui != NULL)
		gui->popup_message(pmsg);
	else
#ifdef WIN32
		MessageBox(NULL, pmsg, PROGRAMNAME " error",
			MB_OK | MB_ICONERROR);

#else
		fprintf(stderr, pmsg);
#endif
}

//----------------------------------------------------------------------------
// Processor loading routines
//----------------------------------------------------------------------------
static Byte fread_byte(FILE *fp)
{
	char			str[3];
	DWord			l;

	str[0] = fgetc(fp);
	str[1] = fgetc(fp);
	str[2] = '\0';

	l = strtol(str, NULL, 16);
	return (Byte)(l & 0xff);
}

static Word fread_word(FILE *fp)
{
	Word		ret;

	ret = fread_byte(fp);
	ret <<= 8;
	ret |= fread_byte(fp);

	return ret;
}

void Mc6809::load_intelhex(FILE *fp)
{
	Byte	done = 0;
	Byte	n, t, b;
	Word	addr;

	while (!done) {
		(void)fgetc(fp);
		n = fread_byte(fp);
		addr = fread_word(fp);
		t = fread_byte(fp);
		if (t == 0x00) {
			while (n--) {
				b = fread_byte(fp);
				write_rom(addr++, b);
			}
		} else if (t == 0x01) {
			PC = addr;
			done = 1;
		}
		// Read and discard checksum byte
		(void)fread_byte(fp);
		if (fgetc(fp) == '\r') (void)fgetc(fp);
	}
}

void Mc6809::load_motorola_srec(FILE *fp)
{
	Byte	done = 0;
	Byte	n, t, b;
	Word	addr;


	while (!done) {
		(void)fgetc(fp); /* read 'S' */
		t = fgetc(fp);   /* read type of line */
		n = fread_byte(fp);
		switch (t) {
		case '0': n -= 1;
			   while (n--) {
				b = fread_byte(fp);
			   }; 
			   break;
		case '1': n -= 3;
			   addr = fread_word(fp);
		           while (n--) {
				b = fread_byte(fp);
				write_rom(addr++, b);
			   }; 
			   break;
		case '9': addr = fread_word(fp);
			   PC = addr;
			   done = 1;
			   break;
		}
		// Read and discard checksum byte
		(void)fread_byte(fp);
		if (fgetc(fp) == '\r') (void)fgetc(fp);
	}
	fclose(fp);
}

bool Mc6809::load_hexfile(const char *filename, bool ignore_errors)
{
	FILE	*fp;
	Word	ch;

	fp = fopen(filename, "r");
	if (fp == NULL) {
		if  (!ignore_errors) {
			BString pmsg;

			pmsg.printf("Unable to locate or read \"%s\"\n", filename);
#ifdef WIN32
			MessageBox(NULL, (const char *)pmsg,
				PROGRAMNAME " error",
				MB_OK | MB_ICONERROR);
#else
			fprintf(stderr, pmsg);
#endif
		}
		return false;
	} // if
	ch = fgetc(fp);
	ungetc(ch, fp);
	if (ch == ':')
		load_intelhex(fp);
	else if (toupper(ch) == 'S')
		load_motorola_srec(fp);
	else {
		BString pmsg;

		pmsg.printf("File \"%s\" has unknown fileformat\n", filename);
#ifdef WIN32
		MessageBox(NULL, (const char *)pmsg, PROGRAMNAME " error",
			MB_OK | MB_ICONERROR);
#else
		fprintf(stderr, pmsg, filename);	
#endif
		return false;
	}
	return true;
}  // load_hexfile


void Mc6809::set_bp(int which, Word address)
{
	bp[which] = address;
	exceptions |= BREAKPOINT;
}

unsigned int Mc6809::get_bp(int which)
{
	return bp[which];
}

int Mc6809::is_bp_set(int which)
{
	return bp[which] < 0x10000;
}

void Mc6809::reset_bp(int which)
{
	bp[which] = 0x10000;
	if (bp[0] > 0xffff && bp[1] > 0xffff && bp[2] > 0xffff)
		exceptions &= ~BREAKPOINT;
}

void Mc6809::set_gui(AbstractGui* x_gui)
{
	gui = x_gui;
}  // set_gui

void Mc6809::set_inout(Inout* x_inout)
{
	inout = x_inout;
}  // set_inout

