/*
    main.cpp


    flexemu, an MC6809 emulator running FLEX
    Copyright (C) 1997-2000  W. Schwotzer

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <misc1.h>
#include <signal.h>
#include <limits.h>
#include <new>
#include <new.h>

#include "classes.h"
#include "e2.h"
#include "mc6809.h"
#include "inout.h"
#include "e2video.h"
#include "pia2.h"
#include "da6809.h"
#include "acia1.h"
#include "mc146818.h"
#include "e2floppy.h"
#include "command.h"
#include "mmu.h"
#include "pia1.h"
#include "win32gui.h"
#include "foptman.h"


// class memory is now included in cpu:
#define MEMORY cpu
#define PMEMORY (*MEMORY)

#ifndef WIN32
#ifdef __osf__
extern "C" unsigned int alarm(unsigned int);
#endif
#endif

#ifdef WIN32
UINT idTimer;   // id of global system timer
#endif

// define an exception handler when new fails

#ifdef _MSC_VER
// with MSC it's possible to retry memory allocation
int std_new_handler(size_t n)
{
	int result;

	result = MessageBox(NULL, gMemoryAllocationErrorString,
			PROGRAMNAME " error", MB_RETRYCANCEL | MB_ICONWARNING);
	if (result == IDRETRY)
		return 1;       // retry once more
	throw STD_NAME_SPACE::bad_alloc();
	return 0;
}
#else
void std_new_handler(void)
{
	throw STD_NAME_SPACE::bad_alloc();
}
#endif

#ifndef WIN32
#ifdef SIGALRM
extern "C" RETSIGTYPE update(int)
{
	Mc6809::update_flag = 1;
	(void)signal(SIGALRM, update);
	alarm(1);
}
#endif // SIGALRM
#else
VOID CALLBACK OneSecTimerProc(HWND hwnd, UINT uMsg, UINT idEvent, DWORD dwTime)
{
	Mc6809::update_flag = 1;
} // OneSecTimerProc
#endif // WIN32

bool startup(
	Mc6809             **cpu,       
	Inout              **io,
	Da6809             **disassembler,
	struct sGuiOptions *pGuiOptions,
	struct sOptions    *pOptions)
{
	E2video         *video;
	IoDevice        *device;
	Command         *comm;

	*cpu              = new Mc6809();
	(*cpu)->initialize_io_page(GENIO_BASE);
	*disassembler     = new Da6809(*cpu);
	*io               = new Inout(*cpu, pGuiOptions);
	(*cpu)->set_disassembler(*disassembler);
	(*cpu)->set_inout(*io);

	// instanciate all memory mapped I/O devices

	device            = new Mmu(*io, *cpu);
	PMEMORY->add_io_device(device, MMU_BASE, MMU_MASK, 0, 0);
	device            = new Acia1(*io, *cpu);
	PMEMORY->add_io_device(device, ACIA1_BASE, ACIA1_MASK, 0, 0);
	device            = new Pia1(*io, *cpu);
	PMEMORY->add_io_device(device, PIA1_BASE, PIA1_MASK, 0, 0);
	(*io)->set_pia1((Mc6821 *)device);
	device            = new Pia2(*io, *cpu);
	PMEMORY->add_io_device(device, PIA2_BASE, PIA2_MASK, 0, 0);
	(*io)->set_pia2((Mc6821 *)device);
	E2floppy *fdc     = new E2floppy(*io, *cpu);
	fdc->disk_directory(pOptions->disk_dir);
	fdc->mount_all_drives(pOptions->drive);
	PMEMORY->add_io_device(fdc, FDCA_BASE, FDCA_MASK,
				FDCB_BASE, FDCB_MASK);
	(*io)->set_fdc(fdc);
	comm              = new Command(*io, *cpu);
	PMEMORY->add_io_device(comm, COMM_BASE, COMM_MASK, 0, 0);
	comm->set_fdc(fdc);
	video             = new E2video(*io, *cpu);
	PMEMORY->add_io_device(video, VICO_BASE, VICO_MASK, 0, 0);
	(*io)->set_video(video);

	(*io)->init(pOptions->reset_key);
	if (!(pOptions->term_mode && (*io)->is_terminal_supported()))
#ifdef WIN32
		(*io)->create_gui(GUI_WINDOWS);
#else
#ifdef X11
#ifdef XTK
		(*io)->create_gui(GUI_XTOOLKIT);
#else
		(*io)->create_gui(GUI_X11);
#endif // XTK
#endif // X11
#endif // WIN32

	// instanciate real time clock right before initialize alarm
	device   = new Mc146818(*io, *cpu);
	PMEMORY->add_io_device(device, RTC_LOW, RTC_HIGH-RTC_LOW+1, 0, 0);
	(*io)->set_rtc((Mc146818 *)device);

#ifndef WIN32
#ifdef SIGALRM
	(void)signal(SIGALRM, update);
	alarm(1);
#endif
#else
	idTimer = SetTimer(NULL, 0, 1000, CALLBACKCAST OneSecTimerProc);
#endif // WIN32

	if (!(*cpu)->load_hexfile(pOptions->hex_file, true) &&
		pOptions->hex_file.index(PATHSEPARATOR) < 0) {
		BString hexFilePath;

		hexFilePath = pOptions->disk_dir + PATHSEPARATORSTRING + 
			pOptions->hex_file;
		if (!(*cpu)->load_hexfile(hexFilePath))
			return false;
	}

	(*cpu)->reset_io();
	(*cpu)->reset();
	return true;
} // startup

void shutdown (
	Mc6809      *cpu,       
	Inout       *io,
	Da6809      *disassembler)
{
#ifdef WIN32
	if (idTimer)
		KillTimer(NULL, idTimer);
#endif
	// gui is an instance of io
	if (io->gui != NULL)
		delete io->gui;
	delete io;
	delete disassembler;
	delete cpu;
} // shutdown

#ifndef WIN32
int main(int argc, char *argv[])
{
	struct           sOptions    options;
	struct           sGuiOptions guiOptions;
	Mc6809          *cpu;
	Inout           *io;
	Da6809          *disassembler;
	FlexOptionManager optionMan;

	set_new_handler(std_new_handler);
	try {
		optionMan.InitOptions(&guiOptions, &options, argc, argv);
		optionMan.GetOptions(&guiOptions, &options);
		optionMan.GetEnvironmentOptions(&guiOptions, &options);
		optionMan.GetCommandlineOptions(&guiOptions, &options,
			argc, argv);

		if (startup(&cpu, &io, &disassembler, &guiOptions, &options)) {
			cpu->statemachine1();
			optionMan.WriteOptions(&guiOptions, &options);
		}
		shutdown(cpu, io, disassembler);
	} catch (exception &e) {
		shutdown(cpu, io, disassembler);
		fprintf(stdout, PROGRAMNAME ": An error has occured: %s\n",
			e.what());
		exit(1);
	}
	exit(0);
	return 0; // satisfy compiler
}

#else

void scanCmdLine(LPSTR lpCmdLine, int *argc, char **argv) {
	*argc = 1;
	*(argv + 0) = "flexemu";
	while (*lpCmdLine) {
		*(argv + *argc) = lpCmdLine;
		while (*lpCmdLine && *lpCmdLine != ' ' && *lpCmdLine != '\t')
			lpCmdLine++;
		if (*lpCmdLine)
			*(lpCmdLine++) = '\0';
		while (*lpCmdLine && (*lpCmdLine == ' ' || *lpCmdLine == '\t'))
			lpCmdLine++;
		(*argc)++;
	}
} // scanCmdLine

int WINAPI WinMain (
	HINSTANCE hInstance,
	HINSTANCE hPrevInstance,
	LPSTR lpCmdLine,
	int nCmdShow)
{
	struct           sOptions    options;
	struct           sGuiOptions guiOptions;
	Mc6809          *cpu;
	Inout           *io;
	Da6809          *disassembler;
	int             argc;
	char            *argv[50];
	FlexOptionManager optionMan;

	guiOptions.hInstance = hInstance;
	guiOptions.nCmdShow  = nCmdShow;

#ifdef _MSC_VER        
_PNH oldHandler = set_new_handler(std_new_handler);
#endif        
try {
		scanCmdLine(lpCmdLine, &argc, (char **)argv);
		optionMan.InitOptions(&guiOptions, &options, argc, argv);
		optionMan.GetOptions(&guiOptions, &options);
		optionMan.GetEnvironmentOptions(&guiOptions, &options);
		optionMan.GetCommandlineOptions(&guiOptions, &options,
			argc, argv);

		if (startup(&cpu, &io, &disassembler, &guiOptions, &options)) {
			cpu->statemachine1();
			optionMan.WriteOptions(&guiOptions, &options);
		}
		shutdown(cpu, io, disassembler);
	} catch (STD_NAME_SPACE::bad_alloc UNUSED(&e)) {
		MessageBox(NULL, gMemoryAllocationErrorString,
			PROGRAMNAME " error", MB_OK | MB_ICONERROR);
		exit(EXIT_FAILURE);
	}
	exit(EXIT_SUCCESS);
	return 0; // satisfy compiler
} // WinMain

#endif // #ifdef WIN32

