/*
    inout.cpp


    flexemu, an MC6809 emulator running FLEX
    Copyright (C) 1997-2000  W. Schwotzer

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <misc1.h>
#ifdef ultrix
#include <sys/ioctl.h>
#endif

#include "classes.h"
#include "inout.h"

#ifdef X11
#include "xgui.h"
#endif

#ifdef XTK
#include "xtgui.h"
#endif

#ifdef WIN32
#include "win32gui.h"
#endif

#ifdef ultrix
extern "C" int ioctl(...);
#endif

extern int errno;

// pointer to this instance for signal handling
static Inout *gio = NULL;

// signal handlers

void sig_reset(int sig_no)
{
	if (gio != NULL)
		gio->signal_reset(sig_no);
}


Inout::Inout(Mc6809 *x_cpu, struct sGuiOptions *x_options)
{
	cpu  = x_cpu;
	options = x_options;
	fdc  = NULL;
	rtc  = NULL;
	pia1 = NULL;
	pia2 = NULL;
	gui  = NULL;
	reset();
	reset_serial();
	gio = this;
} 

Inout::~Inout(void)
{
#ifdef HAVE_TERMIOS_H
	if (isatty(fileno(stdin))) {
		tcsetattr(0, TCSAFLUSH, &save_termios);
		fprintf(stdout, "\n");
	}
#endif // #ifdef HAVE_TERMIOS_H
}


void Inout::reset()
{
	in	         = 0;
	out	         = 0;
	deltaX	         = 0;
	deltaY	         = 0;
	buttonMask       = 0;
}

void Inout::init(Word reset_key)
{
#ifdef HAVE_TERMIOS_H
	struct termios	buf;
	tcflag_t	mask;
	long		disable;

	if (isatty(fileno(stdin))) {
		if (tcgetattr(fileno(stdin), &save_termios) < 0) {
			fprintf(stderr, "unable to initialize terminal\n");
			if (cpu != NULL)
				cpu->set_new_state(EXIT);
		} else {
			buf = save_termios;

			// c_lflag:
			mask = 0
#ifdef ICANON
				| ICANON
#endif
#ifdef ECHO
				| ECHO
#endif
#ifdef IEXTEN
				| IEXTEN
#endif
			;
			buf.c_lflag &= ~mask;
#ifdef ISIG
			buf.c_lflag |= ISIG;
#endif

			// c_iflag:
			mask = 0
#ifdef BRKINT
				| BRKINT
#endif
#ifdef ISTRIP
				| ISTRIP
#endif
#ifdef IXON
				| IXON
#endif
#ifdef ICRNL
				| ICRNL
#endif
			;
			buf.c_iflag &= ~mask;

// test: c_oflag not needed to be changed
			// c_oflag:
//			mask = 0
#ifdef OPOST
//				| OPOST
#endif
//			;
//			buf.c_oflag |= mask;
			buf.c_cc[VMIN]  = 0;
			buf.c_cc[VTIME] = 0;
			disable = fpathconf(fileno(stdin), _PC_VDISABLE);
			if (disable < 0) {
				disable = reset_key;
			}
#if defined(VINTR) && defined(SIGINT)
			buf.c_cc[VINTR] = reset_key;
			signal(SIGINT, sig_reset);
#endif
#if defined(VQUIT) && defined(SIGQUIT)
			buf.c_cc[VQUIT] = disable;
			signal(SIGQUIT, sig_reset);
#endif
#ifdef VSUSP
			buf.c_cc[VSUSP] = disable;
#endif
#if defined(VSUSP) && defined(SIGQUIT)
			buf.c_cc[VSUSP] = disable;
#ifdef VDSUSP
			buf.c_cc[VDSUSP] = disable;
#endif
			signal(SIGTSTP, sig_reset);
#endif
		}
		if (tcsetattr(fileno(stdin), TCSAFLUSH, &buf) < 0) {
			// on error try to switch back,
			// otherwise terminal is damaged
			tcsetattr(fileno(stdin), TCSAFLUSH, &save_termios);
			fprintf(stderr, "unable to initialize terminal\n");
			if (cpu != NULL)
				cpu->set_new_state(EXIT);
		}
	}
#endif // #ifdef HAVE_TERMIOS_H
}

void Inout::set_gui(AbstractGui *x_gui)
{
	gui = x_gui;
}

void Inout::set_fdc(E2floppy *x_device)
{
	fdc = x_device;
}

void Inout::set_rtc(Mc146818 *x_device)
{
	rtc = x_device;
}

void Inout::set_pia1(Mc6821 *x_device)
{
	pia1 = x_device;
}

void Inout::set_pia2(Mc6821 *x_device)
{
	pia2 = x_device;
}

void Inout::set_video(E2video *x_video)
{
	video = x_video;
}

AbstractGui *Inout::create_gui(int type)
{
	if (video != NULL) {
		if (gui != NULL && gui->gui_type() != type && (
#ifdef X11
			type == GUI_X11 ||
#endif
#ifdef XTK
			type == GUI_XTOOLKIT ||
#endif
#ifdef WIN32
			type == GUI_WINDOWS ||
#endif
			type == -9999)) { // dummy
			delete gui;
			gui = NULL;
		}
		if (gui == NULL) {
		   switch(type) {
#ifdef X11
			case GUI_X11:
				gui = new XGui(cpu, this, video, options);
				break;
#endif
#ifdef XTK
			case GUI_XTOOLKIT: 
				gui = new XtGui(cpu, this, video, options);
				break;
#endif
#ifdef WIN32
			case GUI_WINDOWS: 
				gui = new Win32Gui(cpu, this, video, options);
				break;
#endif
		   } // switch
		   cpu->set_gui(gui);
		} // if
	} // if
	return gui;
}

// one second updates are generated by the cpu
// in this method they will be transmitted to all objects
// which need it
void Inout::update_1_second(Byte state)
{
	if (rtc != NULL)
		rtc->update_1_second();
	if (gui != NULL)
		gui->update_1_second(state);
}

void Inout::signal_reset(int sig_no)
{
	signal(sig_no, sig_reset); // set handler again
	if (sig_no == SIGINT && cpu != NULL)
		cpu->set_new_state(RESET_RUN);
}

void Inout::put_ch(Byte key)
{
	if (!key_buffer_full()) {
		key_buffer[in++] = key;
		if (in >= KEY_BUFFER_SIZE)
			in = 0;
	}  // if
} // put_ch

Byte Inout::key_buffer_full(void)
{
	return ((in == out - 1) || (out == 0 && (in == KEY_BUFFER_SIZE-1)));
}

Byte Inout::poll(void)
{
	if (in != out)
		return 1;
	else
		return 0;	
}

// input should always be polled before read_ch

Byte Inout::read_ch(void)
{
	Byte temp;

	while (in == out)    // check for empty buffer
		return 0x00; // should never happen, return a dummy value
	temp = key_buffer[out++];
	if (out >= KEY_BUFFER_SIZE)
		out = 0;
	return temp;
}

// read character, but leave it in the queue
Byte Inout::read_queued_ch(void)
{
	while (in == out)    // check for empty buffer
		return 0x00; // should never happen, return a dummy value
	return key_buffer[out];
}

void Inout::reset_serial()
{
	in_ser		= 0;
	out_ser		= 0;
}

void Inout::put_ch_serial(Byte key)
{
	// convert back space character
#ifdef HAVE_TERMIOS_H
#ifdef VERASE
	if (key == save_termios.c_cc[VERASE] || key == 0x7f) {
		key = BACK_SPACE;
	}
#endif
#endif // #ifdef HAVE_TERMIOS_H
	if (!key_buffer_full_serial()) {
		key_buffer_ser[in_ser++] = key;
		if (in_ser >= KEY_BUFFER_SIZE)
			in_ser = 0;
	}  // if
} // put_ch_serial

Byte Inout::key_buffer_full_serial(void)
{
	return ((in_ser == out_ser - 1) ||
		(out_ser == 0 && (in_ser == KEY_BUFFER_SIZE-1)));
}

// poll serial port for input
Byte Inout::poll_serial(void)
{
#ifdef HAVE_TERMIOS_H
	char	buf[1];
	static Word count = 0;

	if (++count >= 100) {
		count = 0;
		fflush(stdout);
		if (read(fileno(stdin), &buf, 1) > 0) {
			put_ch_serial(buf[0]);
		}
	}
	if (in_ser != out_ser)
		return 1;
	else
		return 0;
#else
	return 0;
#endif // #ifdef HAVE_TERMIOS_H
}

// read a serial byte from cpu
// ATTENTION: input should always be polled before read_ch_ser
Byte Inout::read_ch_serial(void)
{
	Byte temp;

	while (in_ser == out_ser)    // check for empty buffer
		return 0x00; // should never happen, return a dummy value
	temp = key_buffer_ser[out_ser++];
	if (out_ser >= KEY_BUFFER_SIZE)
		out_ser = 0;
	return temp;
}

// read character, but leave it in the queue
Byte Inout::read_queued_ch_serial(void)
{
	while (in_ser == out_ser)    // check for empty buffer
		return 0x00; // should never happen, return a dummy value
	return key_buffer_ser[out];
}


void Inout::write_ch_serial(Byte val)
{
#ifdef HAVE_TERMIOS_H
#ifdef VERASE
	if (val == BACK_SPACE) {
		char *str = "\b \b";
		write(fileno(stdout), str, strlen(str));
//		putc('\b', stdout);
//		putc(' ', stdout);
//		putc('\b', stdout);
	} else
#endif
		write(fileno(stdout), &val, 1);
//		putc(val, stdout);
//	if (val < ' ' || val > 0x7e)
//		fflush(stdout);
#endif // #ifdef HAVE_TERMIOS_H
}

void Inout::set_bell(Word x_percent)
{
#ifdef WIN32
	Beep(400, 100);
#else
	static char bell = BELL;

	write(fileno(stdout), &bell, 1);
#endif
}

Word Inout::is_terminal_supported(void)
{
#ifdef HAVE_TERMIOS_H
	return 1;
#else
	return 0;
#endif
}

Word Inout::output_to_terminal(void)
{
#ifdef HAVE_TERMIOS_H
	if (gui != NULL)
		gui->output_to_terminal();
	return 1;
#else
	return 0;
#endif // #ifdef HAVE_TERMIOS_H
}

Word Inout::output_to_graphic(void)
{
	if (gui != NULL) {
		gui->output_to_graphic();
		return 1;
	}
	return 0;
}

