/*
    ffilecnt.cpp


    FLEXplorer, An explorer for any FLEX file or disk container
    Copyright (C) 1998-2000  W. Schwotzer

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <misc1.h>
#include <sys/stat.h>
#include <string.h>
#include <time.h>

#include "fcinfo.h"
#include "flexerr.h"
#include "ffilecnt.h"
#include "fdirent.h"
#include "bdate.h"
#include "ffilecnt.h"
#include "fcopyman.h"
#include "ffilebuf.h"


#ifdef _MSC_VER
#include <io.h>		// needed for access
#endif
/***********************************************/
/* Initialization of a s_flex_header structure */
/***********************************************/

void s_flex_header::initialize(int secsize, int trk, int sec0, int sec, int aSides)
{
	int i, size, noSides;

	noSides = aSides;
	if (aSides < 1)
		noSides = 1;
	if (aSides > 2)
		noSides = 2;
	size = 1; /* default */
	for (i = 15; i >= 7; i--) {
		if (secsize & (1 << i)) {
			size = i - 7;
			break;
		}
	}
	magic_number	= MAGIC_NUMBER;
	write_protect	= 0;
	sizecode		= size;
	sides0			= noSides;
	sectors0		= sec0;
	sides			= noSides;
	sectors			= sec;
	tracks			= trk;
	dummy1			= 0;
	dummy2			= 0;
	dummy3			= 0;
	dummy4			= 0;
	dummy5			= 0;
}


/****************************************/
/* Constructor                          */
/****************************************/

FlexFileContainer::FlexFileContainer(void) :
	fp(NULL),
	path(NULL),
	attributes(0),
	dirIndex(-1)
{
	filePattern = "                ";
}

/****************************************/
/* Destructor                           */
/****************************************/

FlexFileContainer::~FlexFileContainer(void)
{
	// final cleanup: close if not already done
	try {
		Close();
	} catch (...) {
		// ignore exceptions
		// exceptions in destructors cause much problems
		// usually the file should be closed already
	}
}

/****************************************/
/* Public interface                     */
/****************************************/

int FlexFileContainer::GetBytesPerSector(void)
{
	return param.byte_p_sector;
}

int FlexFileContainer::IsWriteProtected(void)
{
	return param.write_protect;
}

int FlexFileContainer::IsTrackValid(int track)
{
	return (track <= param.max_track);
}

int FlexFileContainer::IsSectorValid(int track, int sector)
{
	if (track) 
		return (sector != 0 && sector <= (param.max_sector * 2)); 
	else
		return (sector != 0 && sector <= (param.max_sector0 * 2)); 
}

int FlexFileContainer::Open(const char *filePath)
{
	path = new BString(filePath);
	return OpenForFile(*path);
}

int FlexFileContainer::Create(const char *dir, const char *name, int t, int s, int fmt)
{
	if (fmt != TYPE_DSK_CONTAINER && fmt != TYPE_FLX_CONTAINER) {
		ex.setString(FERR_INVALID_FORMAT, fmt);
		throw ex;
	}

	Format_disk(t, s, (char *)dir, (char *)name, fmt);
	path = new BString(dir);
	if (path->lastchar() != PATHSEPARATOR)
		*path += PATHSEPARATORSTRING;
	*path += name;
	return OpenForFile(*path);
}

// return != 0 on success
// try to close all files still open
// this may cause exeptions!
// usually files should be closed explicitly
// with CloseFile()
int FlexFileContainer::Close(void)
{
	if (fp != NULL) {
		fclose(fp);
		fp = NULL;
	}
	delete path;
	path = NULL;
	return 1;
}

int	FlexFileContainer::OpenDirectory(const char *pattern)
{
	CHECK_NO_CONTAINER_OPEN;	
	CHECK_DIRECTORY_ALREADY_OPENED;	
	dirIndex = 0;
	filePattern = pattern;
	dirSector.next_trk = 0;
	dirSector.next_sec = 5;
	return 1;
}

// return 0 if no more entries found otherwise != 0
int	FlexFileContainer::NextDirEntry(FlexDirEntry& entry)
{
	s_dir_entry	*pd;
	BString	fileName;
	int			matchAlways;

	CHECK_NO_CONTAINER_OPEN;
	CHECK_DIRECTORY_NOT_OPENED;
	entry.SetEmpty();
	matchAlways = (strcmp(filePattern, "*.*") == 0);
	while (entry.IsEmpty()) {
		// check if directory search not at end
		if (dirIndex < 0)
			return 0;
		if ((dirIndex % 10) == 0) {
			if (dirSector.next_trk == 0 && dirSector.next_sec == 0)
				return 0;
			dirSectorTrk = dirSector.next_trk;
			dirSectorSec = dirSector.next_sec;
			if (!ReadSector((Byte *)&dirSector, dirSector.next_trk, dirSector.next_sec)) {
				CloseDirectory();
				ex.setString(FERR_READING_TRKSEC, dirSector.next_trk, dirSector.next_sec, *path);
				throw ex;
			}
		}
		pd = &dirSector.dir_entry[dirIndex % 10];
		// look for the next used directory entry
		if (pd->filename[0] != 0 &&
		(pd->filename[0] != -1)) {
			// ok, found a valid directory entry
			fileName = BString(pd->filename, FLEX_FILENAME_LENGTH);
			fileName += ".";
			fileName += BString(pd->file_ext, FLEX_FILEEXT_LENGTH);
			if (matchAlways ||
			   stricmp(filePattern, fileName) == 0) {
				entry.SetDate(pd->day, pd->month, pd->year);
				entry.SetTotalFileName(fileName);
				entry.SetAttributes(pd->file_attr);
				entry.SetSectorMap(pd->sector_map);
				entry.SetStartTrkSec(pd->start_trk, pd->start_sec);
				entry.SetEndTrkSec(pd->end_trk, pd->end_sec);
				entry.SetSize((pd->records[0] << 8 | pd->records[1]) * param.byte_p_sector);
				entry.SetSectorMap(pd->sector_map);
				entry.ClearEmpty();
			}
		}
		dirIndex++;
	}
	return 1;
}

int	FlexFileContainer::CloseDirectory(void)
{
	CHECK_DIRECTORY_NOT_OPENED;
	filePattern = "";
	dirIndex = -1;
	return 1;
}

// return != 0 if file found
// if file found can also be checked by
// !entry.isEmpty
int FlexFileContainer::FindFile(const char *fileName, FlexDirEntry& entry)
{
	int ret;

	CHECK_NO_CONTAINER_OPEN;
	CHECK_DIRECTORY_ALREADY_OPENED;
	ret = 0;
	if (OpenDirectory(fileName)) {
		try {
			ret = NextDirEntry(entry);
		} catch (FlexException UNUSED(&e)) {
			CloseDirectory();
			throw;
		}
		CloseDirectory();
	}
	return ret;
}

int	FlexFileContainer::DeleteFile(const char *fileName)
{
	FlexDirEntry de;

	CHECK_NO_CONTAINER_OPEN;
	CHECK_DIRECTORY_ALREADY_OPENED;
	if (OpenDirectory(fileName)) {
		try {
			if (NextDirEntry(de)) {
				DeleteDirEntry();
			} else {
				ex.setString(FERR_NO_FILE_IN_CONTAINER, fileName, *path);
				throw ex;
			}
		} catch (FlexException UNUSED(&e)) {
			CloseDirectory();
			throw;
		}
		CloseDirectory();
	}
	return 1;
}

int	FlexFileContainer::RenameFile(const char *oldName, const char *newName)
{
	FlexDirEntry de;

	CHECK_NO_CONTAINER_OPEN;
	CHECK_DIRECTORY_ALREADY_OPENED;
	if (OpenDirectory(oldName)) {
		try {
			if (NextDirEntry(de)) {
				RenameDirEntry(newName);
			} else {
				ex.setString(FERR_NO_FILE_IN_CONTAINER, oldName, *path);
				throw ex;
			}
		} catch (FlexException UNUSED(&e)) {
			CloseDirectory();
			throw;
		}
		CloseDirectory();
	}
	return 1;
}

int FlexFileContainer::FileCopy(const char *sourceName, const char *destName,
			FileContainerIf& destination)
{
	FlexCopyManager copyMan;

	CHECK_NO_CONTAINER_OPEN;
	CHECK_DIRECTORY_ALREADY_OPENED;
	return copyMan.FileCopy(sourceName, destName, *this, destination);
}

int	FlexFileContainer::GetInfo(FlexContainerInfo& info)
{
	struct s_sys_info_sector buffer;

	CHECK_NO_CONTAINER_OPEN;
	if (!ReadSector((Byte *)&buffer, 0, 3)) {
		ex.setString(FERR_READING_TRKSEC, 0, 3, *path);
		throw ex;
	}
	info.SetDate(buffer.day, buffer.month, buffer.year);
	info.SetTrackSector(buffer.last_trk+1, buffer.last_sec);
	info.SetFree((((buffer.free[0] << 8) | buffer.free[1]) * param.byte_p_sector) >> 10);
	info.SetTotalSize(((buffer.last_sec * (buffer.last_trk+1)) * param.byte_p_sector) >> 10);
	info.SetName(buffer.disk_name);
	info.SetPath(*path);
	info.SetType(param.type);
	info.SetAttributes(attributes);
	return 1;
}

// check if an container is opened
// If so return != 0
int FlexFileContainer::IsContainerOpened(void)
{
	if (fp == NULL)
		return 0;
	return 1;
}


int FlexFileContainer::GetContainerType(void)
{
	return param.type;
}

/******************************/
/* Nonpublic interface		  */
/******************************/


int FlexFileContainer::OpenForFile(const char *aPath)
{
	char	*flags;
	struct	stat sbuf;
	struct	s_flex_header header;
	int		wp;

	if (!stat(aPath, &sbuf) && !S_ISREG(sbuf.st_mode)) {
		ex.setString(FERR_UNABLE_TO_OPEN, aPath);
		throw ex;
	}
	wp = access(aPath, W_OK);
	if (wp) {
		attributes |= FLX_READONLY;
		flags = "rb";
	} else
		flags = "rb+";
	fp = fopen(aPath, flags);
	if (fp == NULL) {
		ex.setString(FERR_UNABLE_TO_OPEN, aPath);
		throw ex;
	}
	if (fseek(fp, 0, SEEK_SET)) {
		ex.setString(FERR_UNABLE_TO_OPEN, aPath);
		throw ex;
	}
	// try to read the FLX header
	// to check if it is a FLX formated disk
	if (fread(&header, sizeof(header), 1, fp) == 1 &&
		header.magic_number == MAGIC_NUMBER) {
		// ok it's a FLX format
		Initialize_for_flx_format(&param, &header, wp);
		return 1;
	} else {
		s_formats	format;
		struct s_sys_info_sector buffer;

		// check if it is a DSK formated disk
		// read system info sector
		if (fseek(fp, 2 * SECTOR_SIZE, SEEK_SET)) {
			ex.setString(FERR_UNABLE_TO_OPEN, aPath);
			throw ex;
		}
		if (fread(&buffer, sizeof(buffer), 1, fp) == 1) {
			format.tracks	= buffer.last_trk + 1;
			format.sectors	= buffer.last_sec;
			format.size		= format.tracks * format.sectors * SECTOR_SIZE;
			// do a plausibility check with the size of the DSK file
			if (format.size == sbuf.st_size) {
				// ok it's a DSK format
				Initialize_for_dsk_format(&param, &format, wp);
				return 1;
			}
		}
	}
	ex.setString(FERR_IS_NO_FILECONTAINER, aPath);
	throw ex;
	return 1; // satisfy compiler
}

// if successfull return size of file written. If error return 0
int FlexFileContainer::WriteFromBuffer(const char *fileName, const FlexFileBuffer &buffer)
{
	int		trk = 0, sec = 0;
	int		startTrk, startSec;
	int		nextTrk, nextSec;
	int		i, recordNr;
	FlexDirEntry	de;
	s_sys_info_sector sysInfo;
	// sectorBuffer[2] and [1] are used for the Sector Map
	Byte sectorBuffer[3][SECTOR_SIZE];

	CHECK_NO_CONTAINER_OPEN;
	FindFile(fileName, de);
	if (!de.IsEmpty()) {
		ex.setString(FERR_FILE_ALREADY_EXISTS, fileName);
		throw ex;
	}
	// read sys info sector
	if (!ReadSector((Byte *)&sysInfo, 0, 3)) {
		ex.setString(FERR_READING_TRKSEC, 0, 3, *path);
		throw ex;
	} // get start trk/sec of free chain
	startTrk = nextTrk = sysInfo.fc_start_trk;
	startSec = nextSec = sysInfo.fc_start_sec;
	{	// write each sector to buffer
		int repeat;
		unsigned int smIndex,  smSector;
		int nextPTrk, nextPSec;	// contains next physical trk/sec
		smIndex = 1;
		smSector = 2;
		recordNr = repeat = nextPTrk = nextPSec = 0;
		// at the begin of a random file reserve two sectors for the sector map
		if (recordNr == 0 && buffer.IsRandom())
			repeat = 2;
		do {
			for (i = repeat; i >= 0; i--) {
				trk = nextTrk;
				sec = nextSec;
				if (trk == 0 && sec == 0)
					return 0; // disk full
				if (!ReadSector(&sectorBuffer[i][0], trk, sec)) {
					ex.setString(FERR_READING_TRKSEC, trk, sec, *path);
					throw ex;
				} else if (i) {
					memset(&sectorBuffer[i][2], 0, SECTOR_SIZE - 2);
				}
				nextTrk = sectorBuffer[i][0];
				nextSec = sectorBuffer[i][1];
			}
			if(!buffer.CopyTo(&sectorBuffer[0][4], SECTOR_SIZE - 4,
						recordNr * (SECTOR_SIZE - 4), 0x00)) {
				ex.setString(FERR_WRITING_TRKSEC, trk, sec, *path);
				throw ex;
			}
			recordNr++;
			// if random file update sector map
			if (buffer.IsRandom()) {
				if (trk != nextPTrk || sec != nextPSec || sectorBuffer[smSector][smIndex+2] == 255) {
					smIndex += 3;
					if (smIndex >= SECTOR_SIZE) {
						smSector--;
						if (smSector == 0) {
							ex.setString(FERR_RECORDMAP_FULL, fileName, *path);
							throw ex;
						}
						smIndex = 4;
					}
					sectorBuffer[smSector][smIndex]   = trk;
					sectorBuffer[smSector][smIndex+1] = sec;
				}
				sectorBuffer[smSector][smIndex+2]++;
				nextPTrk = trk;
				if ((nextPSec = sec + 1) > (param.byte_p_track / param.byte_p_sector)) {
					nextPTrk++;
					nextPSec = 1;
				}
			}
			// set record nr and if last sector set link to 0. Write sector
			sectorBuffer[0][2] = recordNr >> 8;
			sectorBuffer[0][3] = recordNr & 0xFF;
			if (recordNr * (SECTOR_SIZE - 4) >= buffer.GetSize())
				sectorBuffer[0][0] = sectorBuffer[0][1] = 0;
			if (!WriteSector(&sectorBuffer[0][0], trk, sec)) {
				ex.setString(FERR_WRITING_TRKSEC, trk, sec, *path);
				throw ex;
			}
			repeat = 0;
		} while (recordNr * (SECTOR_SIZE - 4) < buffer.GetSize());
	}
	sysInfo.fc_start_trk = nextTrk;
	sysInfo.fc_start_sec = nextSec;
	// if free chain full, set end trk/sec of free chain also to 0
	if (!nextTrk && !nextSec) {
		sysInfo.fc_end_trk = nextTrk;
		sysInfo.fc_end_sec = nextSec;
	}
	// if random file, write the sector map buffers back
	nextTrk = startTrk;
	nextSec = startSec;
	if (buffer.IsRandom()) {
		for (i = 2; i >= 1; i--) {
			if (!WriteSector(&sectorBuffer[i][0], nextTrk, nextSec)) {
				ex.setString(FERR_WRITING_TRKSEC, nextTrk, nextSec, *path);
				throw ex;
			}
			nextTrk = sectorBuffer[i][0];
			nextSec = sectorBuffer[i][1];
		}
	}

	// update sys info sector
	int free = sysInfo.free[0] << 8 | sysInfo.free[1];
	free -= recordNr;
	sysInfo.free[0] = free >> 8;
	sysInfo.free[1] = free & 0xFF;
	if (!WriteSector((Byte *)&sysInfo, 0, 3)) {
		ex.setString(FERR_WRITING_TRKSEC, 0, 3, *path);
		throw ex;
	}

	// make new directory entry
	de.SetDate(buffer.GetDate());
	de.SetStartTrkSec(startTrk, startSec);
	de.SetEndTrkSec(trk, sec);
	de.SetTotalFileName(fileName);
	de.SetSize(recordNr * SECTOR_SIZE);
	de.SetAttributes(buffer.GetAttributes());
	de.SetSectorMap(buffer.GetSectorMap());
	CreateDirEntry(de);
	return 1;
}

void FlexFileContainer::ReadToBuffer(const char *fileName, FlexFileBuffer &buffer)
{
	FlexDirEntry	de;
	int				trk, sec;
	int				recordNr, repeat;
	Byte			sectorBuffer[SECTOR_SIZE];
	int				size;

	CHECK_NO_CONTAINER_OPEN;
	FindFile(fileName, de);
	if (de.IsEmpty()) {
		ex.setString(FERR_UNABLE_TO_OPEN, fileName);
		throw ex;
	}
	buffer.SetAttributes(de.GetAttributes());
	buffer.SetSectorMap(de.GetSectorMap());
	buffer.SetDate(de.GetDate());
	size = de.GetSize();
	if (de.IsRandom())
		size -= 2 * SECTOR_SIZE;
	if (size <= 0) {
		ex.setString(FERR_WRONG_PARAMETER);
		throw ex;
	}
	size = size * 252 / 256;
	buffer.Realloc(size);
	de.GetStartTrkSec(&trk, &sec);
	recordNr = 0;
	repeat = 1;
	while (true)  {
		// if random file skip the two sector map sectors
		if (recordNr == 0 && de.IsRandom())
			repeat = 3;
		for (int i = 0; i < repeat; i++) {
			if (trk == 0 && sec == 0)
				return;
			if (!ReadSector(&sectorBuffer[0], trk, sec)) {
				ex.setString(FERR_READING_TRKSEC, trk, sec, fileName);
				throw ex;
			}
			trk = sectorBuffer[0];
			sec = sectorBuffer[1];
		} // for
		if(!buffer.CopyFrom(&sectorBuffer[4], SECTOR_SIZE - 4, recordNr * (SECTOR_SIZE - 4))) {
			ex.setString(FERR_READING_TRKSEC, trk, sec, fileName);
			throw ex;
		}
		recordNr++;
		repeat = 1;
	}
}


// set the file attributes of a file
int	FlexFileContainer::SetAttributes(const char *fileName, int setMask, int clearMask)
{
	FlexDirEntry de;
	int attribs;

	CHECK_NO_CONTAINER_OPEN;
	CHECK_DIRECTORY_ALREADY_OPENED;
	if (OpenDirectory(fileName)) {
		try {
			if (NextDirEntry(de)) {
				attribs = (de.GetAttributes() & ~clearMask) | setMask;
				SetAttributesDirEntry(attribs);
			} else {
				ex.setString(FERR_NO_FILE_IN_CONTAINER,
					fileName, *path);
				throw ex;
			}
		} catch (FlexException UNUSED(&e)) {
			CloseDirectory();
			throw;
		}
		CloseDirectory();
	}
	return 1;
}

int FlexFileContainer::CreateDirEntry(FlexDirEntry& entry) {

	int				i;
	s_dir_sector	ds;
	s_dir_entry		*pde;
	int				nextTrk, nextSec;
	int				tmp1, tmp2;
	BDate		date;

	// first directory sector is trk/sec 0/5
	nextTrk = 0;
	nextSec = 5;
	// loop until all directory sectors read
	while (!(nextTrk == 0 && nextSec == 0)) {
		// read next directory sector
		if (!ReadSector((Byte *)&ds, nextTrk, nextSec)) {
			ex.setString(FERR_READING_TRKSEC, nextTrk, nextSec, *path);
			throw ex;
		}
		for (i = 0; i < 10; i++) {
			// look for the next free directory entry
			pde = &ds.dir_entry[i];
			if (pde->filename[0] == '\0' || pde->filename[0] == -1) {
				int records;

				records = (entry.GetSize() / param.byte_p_sector) +
					(entry.IsRandom() ? 2 : 0);
				strncpy(pde->filename, entry.GetFileName(), FLEX_FILENAME_LENGTH);
				strncpy(pde->file_ext, entry.GetFileExt(), FLEX_FILEEXT_LENGTH);
				pde->file_attr = entry.GetAttributes();
				pde->reserved1 = 0;
				entry.GetStartTrkSec(&tmp1, &tmp2);
				pde->start_trk = tmp1;
				pde->start_sec = tmp2;
				entry.GetEndTrkSec(&tmp1, &tmp2);
				pde->end_trk = tmp1;
				pde->end_sec = tmp2;
				pde->records[0] = records >> 8;
				pde->records[1] = records & 0xFF;
				pde->sector_map = (entry.IsRandom() ? 0x02 : 0x00);
				pde->reserved2 = 0;
				date = entry.GetDate();
				pde->day = date.GetDay();
				pde->month = date.GetMonth();
				pde->year  = date.GetYear() % 100;
				if (!WriteSector((Byte *)&ds, nextTrk, nextSec)) {
					ex.setString(FERR_WRITING_TRKSEC,
						nextTrk, nextSec, *path);
					throw ex;
				}
				return 1;
			}
		}
		nextTrk = ds.next_trk;
		nextSec = ds.next_sec;
	}
	ex.setString(FERR_DIRECTORY_FULL);
	throw ex;
	return 1; // satisfy compiler
}


// deletes the file in the actual selected directory entry
// should only be used after an successful
// openDirectory and nextDirEntry
int FlexFileContainer::DeleteDirEntry(void)
{
	int start_trk, start_sec, end_trk, end_sec;
	int fc_end_trk, fc_end_sec;
	int records, free;
	Byte buffer[SECTOR_SIZE];
	s_sys_info_sector *psis;
	s_dir_entry	*pd;
	
	pd = &dirSector.dir_entry[(dirIndex-1) % 10];
	start_trk = pd->start_trk;
	start_sec = pd->start_sec;
	end_sec = pd->end_sec;
	end_trk = pd->end_trk;
	records = (pd->records[0] << 8) | pd->records[1];
	psis = (s_sys_info_sector *)buffer;

	// deleted file is signed by 0xFF as first byte of filename
	pd->filename[0] = (char)0xFF;
	if (!WriteSector((const Byte *)&dirSector, dirSectorTrk, dirSectorSec))  {
		ex.setString(FERR_WRITING_TRKSEC, dirSectorTrk,dirSectorSec,
				*path);
		throw ex;
	}

	if (!ReadSector(&buffer[0], 0, 3)) { /* read sys info sector */
		ex.setString(FERR_READING_TRKSEC, 0, 3, *path);
		throw ex;
	}
	fc_end_trk = psis->fc_end_trk;
	fc_end_sec = psis->fc_end_sec;

	if (fc_end_trk || fc_end_sec) {
		// add deleted file to free chain if free chain not empty
		if (!ReadSector(&buffer[0], fc_end_trk, fc_end_sec)) {
			ex.setString(FERR_READING_TRKSEC, fc_end_trk,fc_end_sec, *path);
			throw ex;
		}
		buffer[0] = start_trk;
		buffer[1] = start_sec;
		if (!WriteSector(&buffer[0], fc_end_trk, fc_end_sec)) {
			ex.setString(FERR_WRITING_TRKSEC, fc_end_trk,fc_end_sec, *path);
			throw ex;
		}
		if (!ReadSector(&buffer[0], 0, 3)) {
			ex.setString(FERR_READING_TRKSEC, 0, 3, *path);
			throw ex;
		}
		psis->fc_end_trk = end_trk;
		psis->fc_end_sec = end_sec;
	} else {
		// create a new free chain if free chain is empty
		if (!ReadSector(&buffer[0], 0, 3)) {
			ex.setString(FERR_READING_TRKSEC, 0, 3, *path);
			throw ex;
		}
		psis->fc_start_trk = start_trk;
		psis->fc_start_sec = start_sec;
		psis->fc_end_trk = end_trk;
		psis->fc_end_sec = end_sec;
	}
	// update sys info sector
	// update number of free sectors
	// and end of free chain trk/sec
	free = psis->free[0] << 8 | psis->free[1];
	free += records;
	psis->free[0] = free >> 8;
	psis->free[1] = free & 0xff;
	if (!WriteSector(&buffer[0], 0, 3)) {
		ex.setString(FERR_WRITING_TRKSEC, 0, 3, *path);
		throw ex;
	}
	return 1;
}

// renames the file in the actual selected directory entry
// should only be used after an successful
// openDirectory and nextDirEntry
int FlexFileContainer::RenameDirEntry(const char *newName)
{
	int i;
	s_dir_entry	*pd;
	BString totalName, name, ext;

	pd = &dirSector.dir_entry[(dirIndex-1) % 10];
	totalName = newName;
	totalName.upcase();
	i = totalName.index('.');
	if (i >= 0) {
		totalName.at(0, i, name);
		totalName.at(i+1, FLEX_FILEEXT_LENGTH, ext); 
	} else
		name = totalName;
	/* update directory entry */
	for (i = 0; i < FLEX_FILENAME_LENGTH; i++)
		pd->filename[i] = '\0';
	for (i = 0; i < FLEX_FILEEXT_LENGTH; i++)
		pd->file_ext[i] = '\0';
	strncpy(&pd->filename[0], name, FLEX_FILENAME_LENGTH);
	strncpy(&pd->file_ext[0], ext, FLEX_FILEEXT_LENGTH);

	if (!WriteSector((const Byte *)&dirSector,
		dirSectorTrk, dirSectorSec)) {
			ex.setString(FERR_WRITING_TRKSEC,
				dirSectorTrk, dirSectorSec, *path);
			throw ex;
	}
	return 1;
}

// set the date in the actual selected directory entry
// should only be used after an successful
// openDirectory and nextDirEntry
int FlexFileContainer::SetDateDirEntry(const BDate& date)
{
	s_dir_entry	*pd;

	pd = &dirSector.dir_entry[(dirIndex-1) % 10];
	pd->day = date.GetDay();
	pd->month = date.GetMonth();
	pd->year = date.GetYear() % 100;
	if (!WriteSector((const Byte *)&dirSector, dirSectorTrk, dirSectorSec)) {
			ex.setString(FERR_WRITING_TRKSEC,
				dirSectorTrk, dirSectorSec, *path);
			throw ex;
	}
	return 1;		
}

// set the date in the actual selected directory entry
// should only be used after an successful
// openDirectory and nextDirEntry
int FlexFileContainer::SetAttributesDirEntry(int attributes)
{
	s_dir_entry	*pd;

	pd = &dirSector.dir_entry[(dirIndex-1) % 10];
	pd->file_attr = attributes;
	if (!WriteSector((const Byte *)&dirSector, dirSectorTrk, dirSectorSec)) {
			ex.setString(FERR_WRITING_TRKSEC,
				dirSectorTrk, dirSectorSec, *path);
			throw ex;
	}
	return 1;	
}

/****************************************/
/* low level routines                   */
/****************************************/

int FlexFileContainer::ByteOffset(const int trk, const int sec)
{
	int byteOffs;

	byteOffs = param.offset;
	if (trk > 0) {
		byteOffs += param.byte_p_track0;
		byteOffs += param.byte_p_track * (trk - 1);
	}
	byteOffs += param.byte_p_sector * (sec - 1);
	return byteOffs;
}

// low level routine to read a single sector
// should be used with care
// Does not throw any exception !
// returns 0 on failure
int FlexFileContainer::ReadSector(Byte *pbuffer, int trk, int sec)
{
	int pos;

	if (fp == NULL)
		return 0;
	pos = ByteOffset(trk, sec);
	if (pos < 0)
		return 0;
	if (fseek(fp, pos, SEEK_SET)) {
		return 0;
	}
	if (fread(pbuffer, param.byte_p_sector, 1, fp) != 1) {
		return 0;
	}
	return 1;
}

// low level routine to write a single sector
// should be used with care
// Does not throw any exception !
// returns 0 on failure
int FlexFileContainer::WriteSector(const Byte *pbuffer, int trk, int sec)
{
	int pos;

	if (fp == NULL)
		return 0;
	pos = ByteOffset(trk, sec);
	if (pos < 0)
		return 0;
	if (fseek(fp, pos, SEEK_SET)) {
		return 0;
	}
	if (fwrite(pbuffer, param.byte_p_sector, 1, fp) != 1) {
		return 0;
	}
	return 1;
}

void FlexFileContainer::Initialize_for_flx_format(
	s_floppy	*pfloppy,
	s_flex_header	*pheader,
	Byte		wp)
{
	pfloppy->offset        = sizeof(struct s_flex_header);
	pfloppy->write_protect =
		(wp | pheader->write_protect) ? 0x40 : 0;
	pfloppy->max_sector    = pheader->sectors;
	pfloppy->max_sector0   = pheader->sectors0;
	pfloppy->max_track     = pheader->tracks - 1;
	pfloppy->byte_p_sector = 128 << pheader->sizecode;
	pfloppy->byte_p_track0 =
		pheader->sides0 * pheader->sectors0 * pfloppy->byte_p_sector;
	pfloppy->byte_p_track  =
		pheader->sides  * pheader->sectors  * pfloppy->byte_p_sector;
	pfloppy->type		   = TYPE_CONTAINER | TYPE_FLX_CONTAINER;

} // initialize_for_flx_format

void FlexFileContainer::Initialize_for_dsk_format(
	s_floppy	*pfloppy,
	s_formats	*pformat,
	Byte		wp)
{
	pfloppy->offset        = 0;
	pfloppy->write_protect = wp;
	pfloppy->max_sector    = pformat->sectors >> 1;
	pfloppy->max_sector0   = pformat->sectors >> 1;
	pfloppy->max_track     = pformat->tracks - 1;
	pfloppy->byte_p_sector = SECTOR_SIZE;
	pfloppy->byte_p_track0 = pformat->sectors * SECTOR_SIZE; 
	pfloppy->byte_p_track  = pformat->sectors * SECTOR_SIZE; 
	pfloppy->type		   = TYPE_CONTAINER | TYPE_DSK_CONTAINER;
} // initialize_for_dsk_format

void FlexFileContainer::Create_boot_sector(Byte sec_buf[])
{
	FILE		*boot;
	unsigned int	i;

	// first write boot sector if present as file "boot"
	sec_buf[0] = 0x39;	// means RTS
	for (i = 1; i < SECTOR_SIZE; i++)
		sec_buf[i] = 0;
#ifdef sun
	if((boot = fopen("boot", "r")) != NULL)
#else
	if((boot = fopen("boot", "rb")) != NULL)
#endif
		fread(sec_buf, SECTOR_SIZE, 1, boot);
} // create_boot_sector

/* ARGSUSED */
void FlexFileContainer::Create_sector2(Byte sec_buf[], struct s_formats *fmt)
{
	// create unused (???) sector 2
	for (unsigned int i = 0; i < SECTOR_SIZE; i++)
		sec_buf[i] = 0;
	sec_buf[1] = 3;	// link to next sector
} // create_sector2

void FlexFileContainer::Create_sys_info_sector(Byte sec_buf[], char *name,
	struct s_formats *fmt)
{
	struct s_sys_info_sector *sys;
	int		i, start, free;
	time_t		time_now;
	struct tm	*lt;

	memset(sec_buf, 0, SECTOR_SIZE);
	sys = (struct s_sys_info_sector *)sec_buf;
	for (i = 0; i < 8; i++) {
		if (*(name+i) == '.' || *(name+i) == '\0')
			break;
		sys->disk_name[i] = *(name+i);
	} // for
	start			= fmt->dir_sectors + 4;
	free			= (fmt->sectors * fmt->tracks) - start;
	time_now		= time(NULL);
	lt			= localtime(&time_now);
	sys->disk_number[0]	= 0;
	sys->disk_number[1]	= 1;
	sys->fc_start_trk	= start / fmt->sectors;
	sys->fc_start_sec	= (start % fmt->sectors) + 1;
	sys->fc_end_trk		= fmt->tracks - 1;
	sys->fc_end_sec		= (Byte)fmt->sectors;
	sys->free[0]		= free >> 8;
	sys->free[1]		= free & 0xff;
	sys->month		= lt->tm_mon + 1;
	sys->day		= lt->tm_mday;
	sys->year		= lt->tm_year;
	sys->last_trk		= fmt->tracks - 1;
	sys->last_sec		= (Byte)fmt->sectors;
} // create_sys_info_sectors

// on success return != 0
int FlexFileContainer::Write_dir_sectors(FILE *fp, struct s_formats *fmt)
{
	Byte		sec_buf[SECTOR_SIZE];
	int		i;

	memset(sec_buf, 0, SECTOR_SIZE);
	for (i = 0; i < fmt->dir_sectors; i++) {
		sec_buf[0] = 0;
		sec_buf[1] = 0;
		if (i < fmt->dir_sectors - 1) {
			sec_buf[0] = (i + 5) / fmt->sectors;
			sec_buf[1] = ((i + 5) % fmt->sectors) + 1;
		}
		if (fwrite(sec_buf, SECTOR_SIZE, 1, fp) != 1)
			return 0;
	}
	return 1;
} // write_dir_sectors

// on success return != 0
int FlexFileContainer::Write_sectors(FILE *fp, struct s_formats *fmt)
{
	Byte		sec_buf[SECTOR_SIZE];
	int		i;

	memset(sec_buf, 0, SECTOR_SIZE);
	for (i = fmt->dir_sectors + 5;
		i <= fmt->sectors * fmt->tracks; i++) {
		sec_buf[0] = i / fmt->sectors;
		sec_buf[1] = (i % fmt->sectors) + 1;
// use for tests to correctly save random files:
// (the link always jumps over one sector)
//		sec_buf[0] = (i+1) / fmt->sectors;
//		sec_buf[1] = ((i+1) % fmt->sectors) + 1;
		if (i == fmt->sectors * fmt->tracks) {
			sec_buf[0] = sec_buf[1] = 0;
		}
		if (fwrite(sec_buf, SECTOR_SIZE, 1, fp) != 1)
			return 0;
	}
	return 1;
} // write_sectors

void FlexFileContainer::Create_format_table(SWord trk, SWord sec, struct s_formats *pformat)
{
	Word tmp;

	pformat->tracks = trk;
	if (trk < 2)
		pformat->tracks = 2;
	if (trk > 255)
		pformat->tracks = 255;
	pformat->sectors = sec;
	if (sec < 5)
		pformat->sectors = 5;
	if (sec > 255)
		pformat->sectors = 255;
	pformat->size	= pformat->tracks * pformat->sectors * SECTOR_SIZE;
	tmp = pformat->size / DIRSECTOR_PER_KB;
	// calculate number of directory sectors
	// at least track 0 only contains directory sectors
	pformat->dir_sectors = (tmp < sec - 4) ? sec - 4 : tmp;
} // create_format_table

// return != 0 on success
// format FLX or DSK format. FLX format always with sector_size 256
// and same nr of sectors on track 0 and != 0.
// type:
//	use TYPE_DSK_CONTAINER for DSK format
//	use TYPE_FLX_CONTAINER for FLX format

void FlexFileContainer::Format_disk(
				SWord trk,
				SWord sec,
				char *disk_dir,
				char *name,
				int  type /* = TYPE_DSK_CONTAINER */)
{
	FILE		*fp;
	char		*path;
	struct s_formats format;
	Byte		sector_buffer[SECTOR_SIZE];
	struct s_flex_header hdr;
	int			noErr;

	if (disk_dir == NULL || strlen(disk_dir) == 0 ||
		name == NULL || strlen(name) == 0 ||
		trk < 2 || sec < 6) {
		ex.setString(FERR_WRONG_PARAMETER);
		throw ex;
	}
	Create_format_table(trk, sec, &format);
	path = new char[strlen(disk_dir) + strlen(name) + 1];
	strcpy(path, disk_dir);
	if (strlen(path) > 0 && path[strlen(path)-1] != PATHSEPARATOR)
		strcat(path, PATHSEPARATORSTRING);
	strcat(path, name);
#ifdef sun
	fp = fopen(path, "w");
#else
	fp = fopen(path, "wb");
#endif
	delete [] path;
	path = NULL;
	noErr = 1;
	if (fp != NULL) {
		if (type == TYPE_FLX_CONTAINER) {
			hdr.initialize(SECTOR_SIZE, format.tracks, format.sectors, format.sectors, 1);
			if(fwrite(&hdr, sizeof(hdr), 1, fp) != 1)
				noErr = 0;
		}
		Create_boot_sector(sector_buffer);
		if (fwrite(sector_buffer, SECTOR_SIZE, 1, fp) != 1)
				noErr = 0;
		Create_sector2(sector_buffer, &format);
		if (fwrite(sector_buffer, SECTOR_SIZE, 1, fp) != 1)
				noErr = 0;
		Create_sys_info_sector(sector_buffer, name, &format);
		if (fwrite(sector_buffer, SECTOR_SIZE, 1, fp) != 1)
				noErr = 0;
		if (fwrite(sector_buffer, SECTOR_SIZE, 1, fp) != 1)
				noErr = 0;
		if (!Write_dir_sectors(fp, &format))
			noErr = 0;
		if (!Write_sectors(fp, &format))
			noErr = 0;
		fclose(fp);
	} else {
		ex.setString(FERR_UNABLE_TO_FORMAT, name);
		throw ex;
	}
} // format_disk
