/*
    ffilebuf.cpp


    FLEXplorer, An explorer for any FLEX file or disk container
    Copyright (C) 1998-2000  W. Schwotzer

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <misc1.h>
//#include <memory.h>
#include <string.h>	// needed for NULL
#include <stdio.h>
#include <sys/stat.h>
#include "bdate.h"
#include "ffilebuf.h"


FlexFileBuffer::FlexFileBuffer(int n /* = 0 */) :
	pBuffer(NULL), pDate(NULL), size(0), attributes(0), sectorMap(0)
{
	Realloc(n);
}

FlexFileBuffer::~FlexFileBuffer()
{
	delete [] pBuffer;
	delete pDate;
}

const Byte *FlexFileBuffer::GetBuffer(unsigned int offset /* = 0*/,
				unsigned int bytes /* = 1 */) const
{
	if (offset + bytes > size)
		return NULL;
	return pBuffer + offset;
}

// reallocate the buffer with diffrent size
// buffer will be initialized to zero or
// optionally with a copy of the contents of the old buffer
void FlexFileBuffer::Realloc(unsigned int newSize, bool restoreContents /* = false*/)
{
	Byte *newBuffer;

	if (newSize <= 0)
		return;
	if (newSize <= size) {
		// dont allocate memory if buffer size decreases
		size = newSize;
		return;
	}
	newBuffer = new Byte[newSize];
	memset(newBuffer, 0, newSize);
	if (pBuffer && restoreContents)
		memcpy(newBuffer, pBuffer, size);
	delete [] pBuffer;
	pBuffer = newBuffer;
	size = newSize;
}

unsigned int FlexFileBuffer::SizeOfFlexFile(void)
{
	unsigned int count = 0;
	for (unsigned int i = 0; i < size; i++) {
		switch(pBuffer[i]) {
			case 0x0d:
#ifndef WIN32
				count += 1;
#else
				count += 2;
#endif
				break;
			case 0x09: if (i < size-1)
						   count += pBuffer[++i];
					   else
						   count++;
					   break;
			case 0x0a:
			case 0x00: break;
			default: count++; break;
		}
	}
	return count;
}

int FlexFileBuffer::ConvertFromFlex(void)
{
	Byte c;
	Byte *newBuffer;
	unsigned int newIndex, newSize;
	unsigned int count;

	if (!pBuffer || size == 0)
		return 0;

	newSize = SizeOfFlexFile();
	newBuffer = new Byte[ newSize ];
	newIndex = 0;
	for (unsigned int i = 0; i < size; i++) {
		c = pBuffer[i];
		if (c > 0x0d)
			newBuffer[newIndex++] = c;
		else
			if (c == 0x0d) {
#ifndef WIN32
				newBuffer[newIndex++] = 0x0a;
#else
				newBuffer[newIndex++] = 0x0d;
				newBuffer[newIndex++] = 0x0a;
#endif
			} else
				if (c == 0x09) {
					/* expand space compression */
					if (i < size-1)
						count = pBuffer[++i];
					else
						count = 1;
					while (count--)
						newBuffer[newIndex++] = ' ';
				} else
					if (c != 0x00 && c != 0x0a)
						newBuffer[newIndex++] = c;
	} // for
	delete [] pBuffer;
	pBuffer = newBuffer;
	size = newSize;
	return size;
}

int FlexFileBuffer::ConvertToFlex(void)
{
	Byte			c;
	Byte			*newBuffer;
	int				newIndex, newSize;
	unsigned int	spaces;

	if (!pBuffer || size == 0)
		return 0;

	newSize = SizeOfFile();
	newBuffer = new Byte[ newSize ];
	newIndex = 0;
	spaces = 0;
/*	for (int i = 0; i < size; i++) {
		c = pBuffer[i];
		if (c >= ' ')
			newBuffer[newIndex++] = c;
		else {
			if (c != ' ' && c != 0x09 && spaces) {
				if (spaces > 1) {
					newBuffer[newIndex++] = 0x09;
					newBuffer[newIndex++] = spaces;
				} else
					newBuffer[newIndex++] = ' ';
				spaces = 0;
			}
			if (c == ' ') {
				// do space compression
				if (++spaces == 127) {
					newBuffer[newIndex++] = 0x09;
					newBuffer[newIndex++] = spaces;
					spaces = 0;
				}
			} else
			if (c == 0x09) {
				// tab will be converted to 8 spaces
				if (spaces >= 127 - 8) {
					newBuffer[newIndex++] = 0x09;
					newBuffer[newIndex++] = spaces;
					spaces -= 127 - 8;
				} else
					spaces += 8;
			} else
			if (c == 0x0a) {
				newBuffer[newIndex++] = 0x0d;
			} else
			if (c != 0x0d) {
				newBuffer[newIndex++] = c;
			}
		}
	} // while
*/
	for (unsigned int i = 0; i < size; i++) {
		c = pBuffer[i];
		if (c == ' ' && (++spaces == 127)) {
			/* do space compression */
			newBuffer[newIndex++] = 0x09;
			newBuffer[newIndex++] = spaces;
			spaces = 0;
		} else {
			if (spaces) {
				newBuffer[newIndex++] = 0x09;
				newBuffer[newIndex++] = spaces;
				spaces = 0;
			}
			if (c == 0x0d || c > ' ') {
				newBuffer[newIndex++] = c;
			} else {
				if (c == 0x09) {
					newBuffer[newIndex++] = 0x09;
					newBuffer[newIndex++] = 8;
				}
				/* other control chars than 0x09 and 0x0d will be ignored */
			}
		}
	} /* for */
	delete [] pBuffer;
	pBuffer = newBuffer;
	size = newSize;
	return size;
}

unsigned int FlexFileBuffer::SizeOfFile(void)
{
	Byte			c;
	unsigned int	count, spaces;

	if (!pBuffer || size == 0)
		return 0;
	count = spaces = 0;
	for (unsigned int i = 0; i < size; i++) {
		c = pBuffer[i];
		if (c == ' ' && (++spaces == 127)) {
			count += 2;
			spaces = 0;
		} else {
			if (spaces) {
				count += 2;
				spaces = 0;
			}
			if (c == 0x0d || c > ' ') {
				count++;
			} else {
				if (c == 0x09) {
					count += 2;
				}
				/* other control chars than 0x09 and 0x0d will be ignored */
			}
		}
	} /* for */
	return count;
}

bool FlexFileBuffer::IsTextFile(void) const
{
	Byte c;

	for (unsigned int i = 0; i < size; i++) {
		c = pBuffer[i];
		if (c >= ' ' || c == 0x0a || c == 0x0d || c == 0x09)
			continue;
		return false;
	}
	return true;
}

bool FlexFileBuffer::IsFlexTextFile(void) const
{
	Byte c;

	for (unsigned int i = 0; i < size; i++) {
		c = pBuffer[i];
		if (c >= ' ' || c == 0x0a || c == 0x0d || c == 0x00)
			continue;
		if (c == 0x09 && i < size-1) {
			i++;
			continue;
		}
		return false;
	}
	return true;
}

bool FlexFileBuffer::IsExecutableFile(void) const
{
	return false;
}

bool FlexFileBuffer::WriteToFile(const char *path) const
{
	FILE *fp;
	int blocks;

#ifdef sun
	if ((fp = fopen(path, "w")) != NULL) {
#else
	if ((fp = fopen(path, "wb")) != NULL) {
#endif
		blocks = fwrite(pBuffer, GetSize(), 1, fp);
		fclose(fp);
		return (blocks == 1 ? true : false);
	}
	return false;
}

bool FlexFileBuffer::ReadFromFile(const char *path)
{
	struct stat	sbuf;

	if (!stat(path, &sbuf) && S_ISREG(sbuf.st_mode) && sbuf.st_size > 0) {
		FILE *fp;
		int blocks;

		Realloc(sbuf.st_size);
#ifdef sun
		if ((fp = fopen(path, "r")) != NULL) {
#else
		if ((fp = fopen(path, "rb")) != NULL) {
#endif
			blocks = fread(pBuffer, GetSize(), 1, fp);
			fclose(fp);
			return (blocks == 1 ? true : false);
		}
	}
	return false;
}

bool FlexFileBuffer::CopyFrom(const Byte *from, unsigned int aSize,
	unsigned int offset /* = 0 */)
{
	if (offset + aSize > size)
		return false;
	memcpy(pBuffer + offset, from, aSize);
	return true;
}

bool FlexFileBuffer::CopyTo(Byte *to, unsigned int aSize,
	unsigned int offset /* = 0 */, int stuffByte /* = -1 */) const
{
	if (offset + aSize > size) {
		if (stuffByte < 0 || offset >= size)
			return false;
		else {
			memset(to, stuffByte, aSize);
			memcpy(to, pBuffer + offset, size - offset);
			return true;
		}
	}
	memcpy(to, pBuffer + offset, aSize);
	return true;
}

void FlexFileBuffer::FillWith(const Byte pattern /* = 0 */)
{
	for(unsigned int i = 0; i < GetSize(); i++)
		pBuffer[i] = pattern;
}

void  FlexFileBuffer::SetDate(const BDate& date) const
{
	delete pDate;
	pDate = new BDate(date);
}

void  FlexFileBuffer::SetDate(int d, int m, int y) const
{
	delete pDate;
	pDate = new BDate(d, m, y);
}

const BDate& FlexFileBuffer::GetDate(void) const
{
	if (!pDate)
		pDate = new BDate(BDate::Now()); 
	return *pDate;
}
