/*
    fdlist.cpp


    FLEXplorer, An explorer for any FLEX file or disk container
    Copyright (C) 1998-2000  W. Schwotzer

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <misc1.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

#ifndef WX_PRECOMP
#endif

#include <wx/menu.h>
#include <wx/msgdlg.h>
#include <wx/textdlg.h>
#include <wx/utils.h>
#include <wx/clipbrd.h>

#include "fdlist.h"
#include "fdirent.h"
#include "filecont.h"
#include "flexerr.h"
#include "fdcframe.h"
#include "ffilebuf.h"
#include "fcinfo.h"
#include "fclipbrd.h"
#include "fcopyman.h"

// define the columns indices for the wxListCtrl
const int LC_FILENAME	= 0;
const int LC_FILEDESC	= 1;
const int LC_RANDOM	= 2;
const int LC_FILESIZE	= 3;
const int LC_FILEDATE	= 4;
const int LC_FILEATTR	= 5;

#ifdef __WXMSW__
wxString FlexDiskListCtrl::fileViewer = "Notepad.exe";
#endif
#ifdef __WXGTK__
wxString FlexDiskListCtrl::fileViewer = "xedit";
#endif

int CALLBACK compareFlexListItems(long item1, long item2, long sortData)
{
	if (!item1 || !item2)
		return 0;
	FlexDirEntry *pObject1 = (FlexDirEntry *)item1;
	FlexDirEntry *pObject2 = (FlexDirEntry *)item2;
	switch (sortData) {
	 case LC_RANDOM:  return (pObject1->IsRandom() && !pObject2->IsRandom())? -1 : 1;
	 case LC_FILESIZE: return (pObject1->GetSize() < pObject2->GetSize())? -1 : 1;
	 case LC_FILEDATE: return (pObject1->GetDate() < pObject2->GetDate())? -1 : 1;
	 case LC_FILEATTR: return (pObject1->GetAttributesString() < pObject2->GetAttributesString())? -1 : 1;
	 case LC_FILEDESC: return (pObject1->GetFileDescription() < pObject2->GetFileDescription())? -1 : 1;
	 case LC_FILENAME:
	 default: return (pObject1->GetTotalFileName() < pObject2->GetTotalFileName()) ? -1 : 1;
	}
}
	// Event table
BEGIN_EVENT_TABLE(FlexDiskListCtrl, wxListCtrl)
#ifdef __WXMSW__
	EVT_LIST_BEGIN_DRAG(LIST_CTRL, FlexDiskListCtrl::OnBeginDrag)
#endif
	EVT_LIST_COL_CLICK(LIST_CTRL, FlexDiskListCtrl::OnColClick)
	EVT_LIST_KEY_DOWN(LIST_CTRL, FlexDiskListCtrl::OnListKeyDown)
	EVT_LIST_ITEM_SELECTED(LIST_CTRL, FlexDiskListCtrl::OnSelected)
	EVT_LIST_ITEM_DESELECTED(LIST_CTRL, FlexDiskListCtrl::OnDeselected)
	EVT_LIST_ITEM_ACTIVATED(LIST_CTRL, FlexDiskListCtrl::OnActivated)
	EVT_LIST_INSERT_ITEM(LIST_CTRL, FlexDiskListCtrl::OnInsertItem)
	EVT_LIST_DELETE_ITEM(LIST_CTRL, FlexDiskListCtrl::OnDeleteItem)
	EVT_LIST_DELETE_ALL_ITEMS(LIST_CTRL, FlexDiskListCtrl::OnDeleteAllItems)
	EVT_RIGHT_DOWN(FlexDiskListCtrl::OnRightMouseDown)
	EVT_LEFT_DCLICK(FlexDiskListCtrl::OnLeftMouseDClick)
	EVT_MENU(MDI_COPY, FlexDiskListCtrl::OnCopy)
	EVT_MENU(MDI_PASTE, FlexDiskListCtrl::OnPaste)
	EVT_MENU(MDI_DELETE, FlexDiskListCtrl::OnDelete)
	EVT_MENU(MDI_RENAME, FlexDiskListCtrl::OnRename)
	EVT_MENU(MDI_VIEW, FlexDiskListCtrl::OnView)
	EVT_MENU(MDI_SET_WRITEPROTECT, FlexDiskListCtrl::OnSetWriteProtect)
	EVT_MENU(MDI_CLEAR_WRITEPROTECT, FlexDiskListCtrl::OnClearWriteProtect)
	EVT_MENU(MDI_SET_READPROTECT, FlexDiskListCtrl::OnSetReadProtect)
	EVT_MENU(MDI_CLEAR_READPROTECT, FlexDiskListCtrl::OnClearReadProtect)
	EVT_MENU(MDI_SET_DELETEPROTECT, FlexDiskListCtrl::OnSetDeleteProtect)
	EVT_MENU(MDI_CLEAR_DELETEPROTECT, FlexDiskListCtrl::OnClearDeleteProtect)
	EVT_MENU(MDI_SET_CATALOGPROTECT, FlexDiskListCtrl::OnSetCatalogProtect)
	EVT_MENU(MDI_CLEAR_CATALOGPROTECT, FlexDiskListCtrl::OnClearCatalogProtect)
END_EVENT_TABLE()

FlexDiskListCtrl::FlexDiskListCtrl(wxWindow* parent, wxWindowID id,
		const wxPoint& pos, const wxSize& size, long style,
		FileContainerIf *container,
		const wxValidator& validator, const wxString& name) :
	wxListCtrl(parent, id, pos, size, style, validator, name), m_container(container),
		m_popupMenu(NULL), m_totalSize(0)
{
	// create all columns
	InsertColumn(LC_FILENAME, "Filename",   wxLIST_FORMAT_LEFT, 120);
	InsertColumn(LC_FILEDESC, "Filetype",   wxLIST_FORMAT_LEFT, 100);
	InsertColumn(LC_RANDOM,   "Random",     wxLIST_FORMAT_RIGHT, 55);
	InsertColumn(LC_FILESIZE, "Size",       wxLIST_FORMAT_RIGHT, 60);
	InsertColumn(LC_FILEDATE, "Date",       wxLIST_FORMAT_RIGHT, 90);
	InsertColumn(LC_FILEATTR, "Attributes", wxLIST_FORMAT_RIGHT, 55);

	m_popupMenu = new wxMenu;
	//m_popupMenu->SetTitle("Edit");
	m_popupMenu->Append(MDI_COPY, "&Copy");
	m_popupMenu->Append(MDI_PASTE, "&Paste");
	m_popupMenu->AppendSeparator();
	m_popupMenu->Append(MDI_DELETE, "&Delete");
	m_popupMenu->Append(MDI_RENAME, "&Rename");
	m_popupMenu->Append(MDI_VIEW, "&View");
	m_popupMenu->AppendSeparator();
	m_popupMenu->Append(MDI_SET_WRITEPROTECT,   "&Set Write Protect");
	m_popupMenu->Append(MDI_SET_READPROTECT,   "Set Read Protect");
	m_popupMenu->Append(MDI_SET_DELETEPROTECT,   "Set Delete Protect");
	m_popupMenu->Append(MDI_SET_CATALOGPROTECT,   "Set Catalog Protect");
	m_popupMenu->AppendSeparator();
	m_popupMenu->Append(MDI_CLEAR_WRITEPROTECT, "Clear &Write Protect");
	m_popupMenu->Append(MDI_CLEAR_READPROTECT, "Clear Read Protect");
	m_popupMenu->Append(MDI_CLEAR_DELETEPROTECT, "Clear Delete Protect");
	m_popupMenu->Append(MDI_CLEAR_CATALOGPROTECT, "Clear Catalog Protect");

	UpdateItems();

	if (m_container->IsWriteProtected()) {
		static const wxColour light_grey("LIGHT GREY");
		//SetBackgroundColour(wxLIGHT_GREY);
		SetBackgroundColour(light_grey);
	} else {
#ifdef __WXMSW__
		FlexFileDropTarget *dropTarget = new FlexFileDropTarget(this);
		SetDropTarget(dropTarget);
#endif
	}
}

int FlexDiskListCtrl::UpdateItems(void)
{
	int		index;
	FlexDirEntry	*pDe;
	wxListItem	anItem;

	index = 0;
	DeleteAllItems();
	if (!m_container)
		return index;
	m_container->OpenDirectory("*.*");
	// the new FlexDirEntry can be referenced by anItem.m_data !
	while (m_container->NextDirEntry(*(pDe = new FlexDirEntry))) {
		anItem.m_text   = pDe->GetTotalFileName().chars();
		anItem.m_itemId = index;
		anItem.m_col    = 0;
		anItem.m_data	= (long)pDe;
		anItem.m_mask   = wxLIST_MASK_TEXT | wxLIST_MASK_DATA;
		InsertItem(anItem);
		UpdateItem(index++, *pDe);
	}
	delete pDe; // last one was too much so kill it
	m_container->CloseDirectory();
	return index;
}

FlexDiskListCtrl::~FlexDiskListCtrl()
{
	DeleteAllItems();
	delete m_popupMenu;
	if (m_container) {
		m_container->Close();
		delete m_container;
	}
#ifdef __WXMSW__
//		delete dropTarget;
#endif
}

void FlexDiskListCtrl::UpdateItem(int item, FlexDirEntry &de)
{
	char buf[20];

	sprintf(buf, "%d", de.GetSize());
	SetItem(item, LC_RANDOM,   de.IsRandom() ? "Yes" : "");
	SetItem(item, LC_FILESIZE, buf);
	SetItem(item, LC_FILEDATE, de.GetDate().GetDateString());
	SetItem(item, LC_FILEATTR, de.GetAttributesString().chars());
	SetItem(item, LC_FILEDESC, de.GetFileDescription().chars());
}

const wxMenu *FlexDiskListCtrl::GetMenu(void)
{
	return m_popupMenu;
}

/* get the index of each selected item	*/
/* the pointer returned MUST BE DELETED */
/* with delete[]						*/
int FlexDiskListCtrl::GetSelections(int **pItems) const
{
	int item, *items;
	int i, selectionCount;

	*pItems = NULL;
	selectionCount = GetSelectedItemCount();
	if (selectionCount != 0) {
		*pItems = items = new int[selectionCount];
		item = -1;
		for (i = 0; i < selectionCount; i++) {
			item = GetNextItem(item, wxLIST_NEXT_ALL,
			   wxLIST_STATE_SELECTED);
			items[i] = item;
#ifdef __WXGTK__
			item++;
#endif
		} // for
	} // if
	return selectionCount;
}

void FlexDiskListCtrl::DeleteSelectedItems(bool askUser /* = TRUE */)
{
	wxString fileName;
	int item, *pItems;
	int count;
	wxMessageDialog *dialog = NULL;

	count = GetSelections(&pItems);
	if (m_container && count > 0) {
		dialog = new wxMessageDialog(this, "Delete all selected files",
			"Delete Files", wxYES_NO | wxICON_QUESTION | wxCENTRE);
		if (!askUser || (dialog->ShowModal() == wxID_YES)) {
			for (; count > 0; count--) {
				fileName = GetItemText(item = pItems[count-1]);
				try {
					m_container->DeleteFile(fileName.c_str());
				} catch (FlexException& e) {
					if (wxMessageBox(e.what(), "FLEXplorer Error",
						wxOK | wxCANCEL | wxCENTRE, this) != wxOK)
						break;
					continue; 
				}
				DeleteItem(item);
			} // for
		} // if
		delete dialog;
	} // if
	delete [] pItems;
	m_totalSize = 0;
	Notify();
}

void FlexDiskListCtrl::RenameSelectedItems(void)
{
	int			item, *pItems;
	wxString		itemText;
	wxTextEntryDialog	*dialog = NULL;

	int count = GetSelections(&pItems);
	if (m_container && count > 0) {
		// edit the last selected item
		item = pItems[count-1];
		itemText = GetItemText(item);
		dialog = new wxTextEntryDialog(this, "Input the new file name",
			"Rename file", itemText);
		if (dialog->ShowModal() == wxID_OK) {
			try {
				wxString fName = dialog->GetValue();
				FlexDirEntry *pDe;
				fName.MakeUpper();
				m_container->RenameFile(itemText, fName);
				pDe = (FlexDirEntry *)GetItemData(item);
				pDe->SetTotalFileName(fName);
				SetItemText(item, fName);
				UpdateItem(item, *pDe);
			} catch (FlexException &e) {
				wxMessageBox(e.what(), "FLEXplorer Error",
				   wxOK | wxCENTRE | wxICON_EXCLAMATION);
			}
		}
		delete dialog;
	} // if
	delete [] pItems;
}

void FlexDiskListCtrl::ViewSelectedItems(void)
{
	FlexFileBuffer	buffer;
	int		item, *pItems;
	wxString	fileName;

	int count = GetSelections(&pItems);
	if (m_container && count > 0) {
		// edit the last selected item
		item = pItems[count-1];
		fileName = GetItemText(item);
#ifdef __WXGTK__
		fileName.MakeLower();
#endif
		try {
			m_container->ReadToBuffer(fileName, buffer);
			if ((m_container->GetContainerType() & TYPE_CONTAINER) &&
				buffer.IsFlexTextFile()) {
				buffer.ConvertFromFlex();
			}
#ifdef __WXMSW__
			if (!getenv("TEMP")) {
				wxMessageBox("Environment variable 'TEMP' \
not set\nCan't create temporary file",
					"FLEXplorer Error", wxOK | wxCENTRE | wxICON_EXCLAMATION);
			} else {
				BString line(getenv("TEMP"));
				line += PATHSEPARATORSTRING + fileName;
				if (buffer.WriteToFile(line)) {
					if (!createProcess(fileViewer,	line, NULL)) {
						ex.setWindowsError(GetLastError(), fileViewer);
						throw ex;
					}
				}
			}
#else
			BString line("/tmp/");
			line += fileName;
			if (buffer.WriteToFile(line))
				if (createProcess(fileViewer, line, NULL) == -1) {
					ex.setString(FERR_CREATE_PROCESS,
						fileViewer, line);
					throw ex;
				}
#endif
		} catch (FlexException &e) {
			wxMessageBox(e.what(), "FLEXplorer Error", wxOK | wxCENTRE | wxICON_EXCLAMATION);
		}
	} // if
	delete [] pItems;
}

void FlexDiskListCtrl::OnBeginDrag(wxListEvent& event)
{

	if (m_container) {
		wxDragResult		effect;
		int					count, *pItems;
		FlexDnDFiles		*pData;
		FlexFileDataObject *dragData;

		count = GetSelections(&pItems);
		pData = new FlexDnDFiles(m_container, count);
		for (int i = 0; i < count; i++)
			pData->SetFileName(i, (FlexFileName *)GetItemText(pItems[i]).c_str());
		dragData = new FlexFileDataObject(pData);
		wxDropSource dragSource(*dragData, this);
		dragSource.SetData(*dragData);
		effect = dragSource.DoDragDrop(!m_container->IsWriteProtected());
		if (effect == wxDragMove) {
			// Unfinished file moved, source should be deleted;
		}
		delete pData;
		delete dragData; // savely delete after DoDragDrop, also deletes pData
		delete [] pItems;
	}
}

void FlexDiskListCtrl::OnSelected(wxListEvent& event)
{
	FlexDirEntry	de;
	wxString		t;

	if (m_container && m_container->FindFile(GetItemText(event.m_itemIndex).c_str(), de))
		m_totalSize += de.GetSize();
	Notify();
}

void FlexDiskListCtrl::OnDeselected(wxListEvent& event)
{
	FlexDirEntry	de;
	wxString		t;

	if (m_container && m_container->FindFile(GetItemText(event.m_itemIndex).c_str(), de))
		m_totalSize -= de.GetSize();
	Notify();
}

void FlexDiskListCtrl::Notify(void)
{
	if (m_statusbarObserver) {
		int id = OBSERVE_STATUS_BAR;
		m_statusbarObserver->Update(&id);
	}
}

void FlexDiskListCtrl::OnActivated(wxListEvent& WXUNUSED(event))
{
	Notify();
}

void FlexDiskListCtrl::OnColClick(wxListEvent& event)
{
	SortItems(compareFlexListItems, event.m_col);
}


void FlexDiskListCtrl::OnListKeyDown(wxListEvent& event)
{
	int keyCode = event.m_code;
	//char s[33];
	//sprintf((char *)s, "keyCode: %08x\n", keyCode);
	//wxMessageBox(s, "Key code", wxOK | wxCENTRE, this);
	switch (keyCode) {
	  // Delete-key: delete all selected files
	  case WXK_DELETE: DeleteSelectedItems(); break;
	  case WXK_F5: UpdateItems(); break;
	  // unfinished: not supported on Windows yet:
	  case 'c':
	  case 'C': CopyToClipboard(); break;
	  case 'v':
	  case 'V': PasteFromClipboard(); break;
	  default: keyCode = keyCode;
	} // switch
}

void FlexDiskListCtrl::SetPropertyOnSelectedItems(int protection, bool isToBeSet)
{
	wxString fileName;
	int item, *pItems;
	int count;
	int setMask;
	int clearMask;

	count = GetSelections(&pItems);
	if (m_container && count > 0) {
		for (; count > 0; count--) {
			fileName = GetItemText(item = pItems[count-1]);
			try {
				FlexDirEntry de, *pDe;
				setMask = isToBeSet ? protection : 0;
				clearMask = isToBeSet ? 0 : protection;
#ifdef __WXGTK__
				fileName.MakeLower();
#endif
				pDe = (FlexDirEntry *)GetItemData(item);
				m_container->SetAttributes(fileName.c_str(), setMask, clearMask);
				// read back properties from container
				// (because maybe not all properties are supported)
				m_container->FindFile(fileName.c_str(), de);
				if (!de.IsEmpty()) {
					setMask = de.GetAttributes();
					clearMask = ~setMask;
				}
				pDe->SetAttributes(setMask, clearMask);
				UpdateItem(item, *pDe);
			} catch (FlexException& e) {
				if (wxMessageBox(e.what(), "FlexDisk Error",
					wxOK | wxCANCEL | wxCENTRE, this) != wxOK)
					break;
			}
		} // for
	} // if
	delete [] pItems;
}

void FlexDiskListCtrl::OnViewProperties(wxCommandEvent& event)
{
	FlexContainerInfo	info;
	BString		str("", 200), caption;
	int			t, s;

	if (m_container) {
		try {
			m_container->GetInfo(info);
		} catch (FlexException& e) {
			wxMessageBox(e.what(), "FlexDisk Error", wxOK | wxCENTRE, this);
			return;
		}
		caption.printf("Container %s", info.GetName());
		info.GetTrackSector(&t, &s);
		str += "Path: ";
		str += info.GetPath();
		str += "\nType: ";
		str += info.GetTypeString();
		str += "\nDate: ";
		str += info.GetDate().GetDateString();
		if (t && s) {
			str += "\nTracks: ";
			str += t;
			str += "\nSectors: ";
			str += s;
		}
		str += "\nSize: ";
		str += info.GetTotalSize();
		str += " KByte\n";
		str += "Free: ";
		str += info.GetFree();
		str += " KByte\n";
		if (info.GetAttributes() & FLX_READONLY) {
			str += "Attributes: read-only\n";
		}
		wxMessageBox(str.chars(), caption.chars(), wxOK | wxCENTRE, this);

	}
}

void FlexDiskListCtrl::OnLeftMouseDClick(wxMouseEvent& WXUNUSED(event))
{
	ViewSelectedItems();
}

void FlexDiskListCtrl::OnRightMouseDown(wxMouseEvent& event)
{
	if (m_popupMenu)
		PopupMenu(m_popupMenu, event.GetX(), event.GetY());
}

void FlexDiskListCtrl::OnInsertItem(wxListEvent& event)
{
}

void FlexDiskListCtrl::OnDeleteItem(wxListEvent& event)
{
	// delete the FlexDirEntry * object
	// stored in m_data
	FlexDirEntry *pDe = (FlexDirEntry *)GetItemData(event.m_itemIndex);
	delete pDe;
}

void FlexDiskListCtrl::OnDeleteAllItems(wxListEvent& event)
{
}

IMPLEMENT_MODIFY_PROPERTY(OnSetWriteProtect,     WRITE_PROTECT,   TRUE)
IMPLEMENT_MODIFY_PROPERTY(OnClearWriteProtect,   WRITE_PROTECT,   FALSE)
IMPLEMENT_MODIFY_PROPERTY(OnSetReadProtect,      READ_PROTECT,    TRUE)
IMPLEMENT_MODIFY_PROPERTY(OnClearReadProtect,    READ_PROTECT,    FALSE)
IMPLEMENT_MODIFY_PROPERTY(OnSetDeleteProtect,    DELETE_PROTECT,  TRUE)
IMPLEMENT_MODIFY_PROPERTY(OnClearDeleteProtect,  DELETE_PROTECT,  FALSE)
IMPLEMENT_MODIFY_PROPERTY(OnSetCatalogProtect,   CATALOG_PROTECT, TRUE)
IMPLEMENT_MODIFY_PROPERTY(OnClearCatalogProtect, CATALOG_PROTECT, FALSE)

IMPLEMENT_SIMPLE_MENUCOMMAND(OnCopy, CopyToClipboard)
IMPLEMENT_SIMPLE_MENUCOMMAND(OnPaste, PasteFromClipboard)
IMPLEMENT_SIMPLE_MENUCOMMAND(OnDelete, DeleteSelectedItems)
IMPLEMENT_SIMPLE_MENUCOMMAND(OnRename, RenameSelectedItems)
IMPLEMENT_SIMPLE_MENUCOMMAND(OnView, ViewSelectedItems)

#ifdef __WXGTK__
IMPLEMENT_SORT_MENUCOMMAND(OnSortFilename, LC_FILENAME)
IMPLEMENT_SORT_MENUCOMMAND(OnSortRandom, LC_RANDOM)
IMPLEMENT_SORT_MENUCOMMAND(OnSortFilesize, LC_FILESIZE)
IMPLEMENT_SORT_MENUCOMMAND(OnSortFiledate, LC_FILEDATE)
IMPLEMENT_SORT_MENUCOMMAND(OnSortFileattr, LC_FILEATTR)
IMPLEMENT_SORT_MENUCOMMAND(OnSortFiledesc, LC_FILEDESC)
#endif

void FlexDiskListCtrl::GetFileList(FlexFileList &fileList)
{
	int	count, *pItems;
	BString	fileName;

	count = GetSelections(&pItems);
	for (int i = 0; i < count; i++) {
		fileName = GetItemText(pItems[i]);
#ifndef WIN32
		fileName.downcase();
#endif
		fileList.Append(new wxString(fileName));
	}
	delete [] pItems;
}


void FlexDiskListCtrl::CopyToClipboard(void)
{
	FlexFileList fileList;
	FlexDnDFiles  *flexFiles;
	FlexFileList::Node *node;
	int count = 0;

    if ( !wxTheClipboard->Open() ) {
        wxLogError(_T("Can't open clipboard.")); return;
    }

	GetFileList(fileList);
	for ( node = fileList.GetFirst(); node; node = node->GetNext())
		count++;
	flexFiles = new FlexDnDFiles(m_container, count);
	count = 0;
	for ( node = fileList.GetFirst(); node; node = node->GetNext())
		flexFiles->SetFileName(count++, (FlexFileName *)node->GetData()->c_str());
	if ( !wxTheClipboard->SetData(new FlexFileDataObject(flexFiles)) ) {
		wxLogError(_T("Can't copy data to the clipboard"));
	}
	delete flexFiles;
    wxTheClipboard->Close();
	fileList.DeleteContents(TRUE);
}

bool FlexDiskListCtrl::PasteFromClipboard(void)
{
    FlexFileDataObject flexFileData;

    if ( !wxTheClipboard->Open() ) {
        wxLogError(_T("Can't open clipboard."));
		wxBell();
		return false;
    }
    if ( !wxTheClipboard->IsSupported(FlexFileFormatId) ) {
        wxLogWarning(_T("No Flex file data on clipboard"));
        wxTheClipboard->Close();
		wxBell();
        return false;
    }

	if (!wxTheClipboard->GetData(flexFileData)) {
        wxLogWarning(_T("Unable to paste Flex file data from clipboard"));
        wxTheClipboard->Close();
		wxBell();
        return false;
	}
	int					index = 0;
	FlexDirEntry		*pDe;
	FileContainerIf		*srcContainer;
	FlexDnDFiles const	*flexFiles;

	flexFiles = flexFileData.GetFlexDnDFiles();
	srcContainer = flexFiles->GetSrcContainer();
	for (int i = 0; i < flexFiles->GetFileNameCount(); i++) {
		try {
			srcContainer->FileCopy((char *)flexFiles->GetFileName(i),
				(char *)flexFiles->GetFileName(i), *m_container);
			pDe = new FlexDirEntry();
			if (srcContainer->FindFile((char *)flexFiles->GetFileName(i), *pDe)) {
				wxListItem anItem;

				anItem.m_text   = pDe->GetTotalFileName().chars();
				anItem.m_itemId = index;
				anItem.m_col    = 0;
				anItem.m_data	= (long)pDe;
				anItem.m_mask   = wxLIST_MASK_TEXT | wxLIST_MASK_DATA;
				index = InsertItem(anItem);
				UpdateItem(index, *pDe);
			}
			delete pDe;
		} catch (FlexException &e) {
			if (e.GetErrorCode() == FERR_DISK_FULL_WRITING) {
				wxMessageBox(e.what(), "FLEXplorer Error", wxOK | wxCENTRE, this);
				return false;
			} else {
				if (wxMessageBox(e.what(), "FLEXplorer Error",
					wxOK | wxCANCEL | wxCENTRE, this) != wxOK)
				return true;
			}
		}
	} // for
    wxTheClipboard->Close();
	return true;
}
