/*
	e2floppy.cpp

	flexemu, an MC6809 emulator running FLEX
	Copyright (C) 1997-2000  W. Schwotzer

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <misc1.h>
#include <stdlib.h>
#include <string.h>
#ifdef _MSC_VER
#include <io.h>		// needed for access
#endif

#include "e2floppy.h"
#include "bstring.h"
#include "ffilecnt.h"
#include "rfilecnt.h"
#include "ndircont.h"
#include "fcinfo.h"
#include "flexerr.h"


E2floppy::E2floppy(Inout *x_io, Mc6809 *x_cpu)
{
	Word i;

	cpu = x_cpu;
	io  = x_io;

	disk_dir = "";
	for (i = 0; i <= 4; i++) {
		track[i]  = 1;	// position all drives to track != 0  !!!
		floppy[i] = NULL;
	}
	selected = 4;				// deselect all drives
	pfs = NULL;					// set pfs to selected drive
} // E2floppy


E2floppy::~E2floppy(void)
{
	for (int i = 0; i < 4; i++) {
		if (floppy[i] != NULL) {
			try {
				floppy[i]->Close();
			} catch (...) {
				// ignore errors
			}
		}
	}
} // ~E2floppy


// return != 0 on success
Byte E2floppy::umount_drive(Word drive_nr)
{
	FileContainerIf *pfloppy;

	if (drive_nr > 3 || (pfloppy = floppy[drive_nr]) == NULL)
		return 0;
	try {
		pfloppy->Close();
		delete pfloppy;
		floppy[drive_nr] = NULL;
	} catch (FlexException UNUSED(&e)) {
		// ignore errors
	}
	return 1;
} // umount_drive

// return != 0 on success
Byte E2floppy::mount_drive(const char *path, Word drive_nr, tMountOption option)
{
	FileContainerIf		*pfloppy = NULL;
	BString		containerPath;
	Byte 			result = 1;

	if (drive_nr > 3 || path == NULL || strlen(path) == 0)
		return 0;
	// check if already mounted
	if (floppy[drive_nr] != NULL)
		return 0;
	track[drive_nr] = 1;	// position to a track != 0  !!!

	// first try with given path
	containerPath = path;
#ifdef WIN32
	containerPath.replaceall('|', ':');
	containerPath.replaceall('/', '\\');
#endif
	try {
		if (option == MOUNT_RAM)
			pfloppy = new FlexRamFileContainer();
		else
			pfloppy = new FlexFileContainer();
		pfloppy->Open(containerPath);
	} catch (FlexException UNUSED(&e)) {
		result = 0;
		delete pfloppy;
		pfloppy = NULL;
#ifdef NAFS
		result = 1;
		try {
			pfloppy = new NafsDirectoryContainer();
			pfloppy->Open(containerPath);
		} catch (FlexException UNUSED(&e)) {
			result = 0;
			delete pfloppy;
			pfloppy = NULL;
		}
#endif
	}
	floppy[drive_nr] = pfloppy;
	if (result)
		return result;

	// second try with path within disk_dir directory
	containerPath = disk_dir;
	if (containerPath.lastchar() != PATHSEPARATOR)
		containerPath += PATHSEPARATORSTRING;
	containerPath += path;
	result = 1;
	try {
		pfloppy = new FlexFileContainer();
		pfloppy->Open(containerPath);
	} catch (FlexException UNUSED(&e)) {
		result = 0;
		delete pfloppy;
		pfloppy = NULL;
#ifdef NAFS
		result = 1;
		try {
			pfloppy = new NafsDirectoryContainer();
			pfloppy->Open(containerPath);
		} catch (FlexException UNUSED(&e)) {
			result = 0;
			delete pfloppy;
			pfloppy = NULL;
		}
#endif
	}
	floppy[drive_nr] = pfloppy;
	return result;
} // mount_drive



void E2floppy::disk_directory(const char *x_disk_dir)
{
	disk_dir = x_disk_dir;
}

void E2floppy::mount_all_drives(BString drive[])
{
	int i;

	for (i = 0; i < 4; i++)
		mount_drive(drive[i], i);
	selected = 4;			// deselect all drives
	pfs = NULL;
}  // mount_all_drives

Byte E2floppy::umount_all_drives(void)
{
	Word i;
	Byte result;

	result = 1;
	for (i = 0; i < 4; i++)
		if (!umount_drive(i))
			result = 0;
	return result;
}  // umount_all_drives

// get info for corresponding drive or NULL
// the info string should not exceed 512 Bytes
// it is dynamically allocated and should be freed
// by the calling program

BString E2floppy::drive_info(Word drive_nr)
{
	BString		str;
	FileContainerIf	*pfl;

	if (drive_nr <= 3) {
		if ((pfl = floppy[drive_nr]) == NULL) {
			str.printf("drive #%d not ready\n", drive_nr);
		} else {
			FlexContainerInfo info;
			int trk, sec;

			try {
				pfl->GetInfo(info);
			} catch (FlexException &e) {
				str.printf("%s\n", e.what());
				return str;
			}
			info.GetTrackSector(&trk, &sec);
			str.printf(
"drive       #%d\n"
"type:       %s\n"
"name:       %s\n"
"path:       %s\n"
"tracks:     %d\n"
"sectors:    %d\n"
"write-prot: %s\n", drive_nr, info.GetTypeString().chars(), info.GetName(),
				info.GetPath().chars(), trk, sec,
				pfl->IsWriteProtected() ? "yes" : "no");

		}
	}
	return str;
} // drive_info

char *E2floppy::open_mode(char *path)
{
	    int  wp;	// write protect flag
	char *mode;
	
	wp = access(path, W_OK);
#ifdef sun
	mode = wp ? "r" : "r+";
#else
	mode = wp ? "rb" : "rb+";
#endif
	return mode;
} // open_mode


Byte E2floppy::update_all_drives(void)
{
	FileContainerIf	*pfloppy;
	Word			i;
	Byte			result;

	result = 1;
	for (i = 0; i < 4; i++) {
		pfloppy = floppy[i];
		if (pfloppy == NULL)
			// no error if drive not ready
			continue;
		if (!update_drive(i))
			result = 0;
	}
	return result;
} // update_all_drives

Byte E2floppy::update_drive(Word drive_nr)
{
	FileContainerIf *pfloppy;

	if (drive_nr > 3)
		return 0;
	pfloppy = floppy[drive_nr];
	if (pfloppy == NULL)
		// error if drive not ready
		return 0;
	return 1;
} // update_drive

void E2floppy::resetIo()
{
	drisel = 0;
	Wd1793::resetIo();
}


Byte E2floppy::readIo(Word offset)
{
	if (offset <= 3)
		return Wd1793::readIo(offset);
	status = 0xff;	// unused is logical high
	if (!side)
		status &= 0xfd;
	if (!irq)
		status &= 0xbf;
	if (!drq)
		status &= 0x7f;
	return status;
} // readIo


void E2floppy::writeIo(Word offset, Byte val)
{
	if (offset <= 3)
		Wd1793::writeIo(offset, val);
	else {
		drisel = val;
		side = (drisel & 0x10) ? 1 : 0;
		track[selected] = tr;
		switch (drisel & 0x0f) {
			case 0x01 : selected = 0;
				    break;
			case 0x02 : selected = 1;
				    break;
			case 0x04 : selected = 2;
				    break;
			case 0x08 : selected = 3;
				    break;
			default   : selected = 4;
		};
		pfs = floppy[selected];
		tr = track[selected];
	}
} // writeIo


Byte E2floppy::readByte(Word index)
{
	if (pfs == NULL)
		return 0;

	if (index == 256 && tr == 0 && sr == 5)
		drq = drq;
	if (index == pfs->GetBytesPerSector()) {
		if (!pfs->ReadSector((Byte *)&sector_buffer, tr, sr)) {
			drq = 0;
			str = 0x10;
			byteCount = 0;
		}
	} // if
	return sector_buffer[SECTOR_SIZE - index];
} // readByte


void E2floppy::writeByte(Word index)
{
	//unsigned int error;

	sector_buffer[SECTOR_SIZE - index] = dr;
	if (index == 1) {
		if (!pfs->WriteSector((Byte *)&sector_buffer, tr, sr)) {
			// how to react on write error???
		}
	} // if
} // writeByte


Byte E2floppy::recordNotFound(void)
{
	if (pfs == NULL)
		return 1;
	return !pfs->IsSectorValid(tr, sr);
} // recordNotFound

Byte E2floppy::seekError(Byte new_track)
{
	if (pfs == NULL)
		return 1;
	return !pfs->IsTrackValid(new_track);
} // seekError

Byte E2floppy::driveReady(void)
{
	return pfs != NULL;
}  // driveReady

Byte E2floppy::writeProtect(void)
{
	if (pfs == NULL)
		return 1;
	return pfs->IsWriteProtected();
}  // writeProtect

Byte E2floppy::format_disk(SWord trk, SWord sec, char *name,
						   int type /* = TYPE_DSK_CONTAINER */)
{
	FileContainerIf *pfloppy = NULL;

	switch (type) {
#ifdef NAFS
		case TYPE_NAFS_DIRECTORY: pfloppy = new NafsDirectoryContainer(); break;
#endif
		case TYPE_DSK_CONTAINER: 
		case TYPE_FLX_CONTAINER:  pfloppy = new FlexFileContainer(); break;
		default:	      pfloppy = NULL;
	}
	try {
		if (pfloppy != NULL)
			pfloppy->Create(disk_dir, name, trk, sec, type);
	} catch (FlexException UNUSED(&e)) {
		delete pfloppy;
		return 0;
	}
	pfloppy->Close();
	delete pfloppy;
	return 1;
} // format_disk
