/*
    dircont.cpp


    FLEXplorer, An explorer for any FLEX file or disk container
    Copyright (C) 1998-2000  W. Schwotzer

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <misc1.h>
#ifdef _MSC_VER
#include <io.h>		// needed for access
#include <direct.h>
#endif

#ifndef WIN32
#include <sys/vfs.h>
#endif

#include <sys/stat.h>
#include <stdio.h>
#include <errno.h>
#include <time.h>

#include "bdir.h"
#include "bdate.h"
#include "fcinfo.h"
#include "flexerr.h"
#include "dircont.h"
#include "fdirent.h"
#include "fcopyman.h"
#include "ffilebuf.h"

/****************************************/
/* Constructor                          */
/****************************************/

DirectoryContainer::DirectoryContainer(void) :
	path(NULL),
	attributes(0),
	isOpened(0)	
{
	filePattern = "";
}

/****************************************/
/* Destructor                           */
/****************************************/

DirectoryContainer::~DirectoryContainer(void)
{
	// final cleanup: close if not already done
	try {
		Close();
	} catch (...) {
		// ignore exceptions
		// exceptions in destructors cause much problems
		// usually the file should be closed already
	}
}

/****************************************/
/* Public interface                     */
/****************************************/

int DirectoryContainer::IsWriteProtected(void)
{
	if (isOpened)
		return attributes & FLX_READONLY;
	else
		return 0;
}

int DirectoryContainer::IsTrackValid(int track)
{
	return 1;
}

int DirectoryContainer::IsSectorValid(int track, int sector)
{
	return 1;
}

int DirectoryContainer::GetBytesPerSector(void)
{
	return SECTOR_SIZE;
}

int DirectoryContainer::Open(const char *directoryPath)
{
	return OpenForDirectory(directoryPath);
}

// type, track and sectors parameter will be ignored
int DirectoryContainer::Create(const char *dir, const char *name,
							   int track, int sectors, int type)
{
	struct stat sbuf;
	BString *aPath;

	aPath = new BString(dir);
	if (aPath->lastchar() != PATHSEPARATOR)
		*aPath += PATHSEPARATORSTRING;
	*aPath += name;
	if (!stat(*aPath, &sbuf) && S_ISREG(sbuf.st_mode)) {
		// if a file exists with this name delete it
		remove(*aPath);
	}
	if (stat(*aPath, &sbuf) || !S_ISDIR(sbuf.st_mode)) {
		// directory does not exist
		if (!BDirectory::Create(*aPath, 0755)) {
			ex.setString(FERR_UNABLE_TO_CREATE, *aPath);
			throw ex;
		}
	}
	return OpenForDirectory(*aPath);
}

// return != 0 on success
// try to close all files still open
// this may cause exeptions!
// usually files should be closed explicitly
// with closeFile()
int DirectoryContainer::Close(void)
{
	isOpened = 0;
	delete path;
	path = NULL;
	return 1;
}

int	DirectoryContainer::OpenDirectory(const char *pattern)
{
	CHECK_NO_DCONTAINER_OPEN;	
	CHECK_DDIRECTORY_ALREADY_OPENED;	
	filePattern = pattern;
	dirHdl = NULL;
	return 1;
}

Word DirectoryContainer::IsInFileRandom(const char *ppath, const char *pfilename)
{
	FILE	*fp;
	char	str[PATH_MAX+1];
	char	lowFilename[14];

	strcpy(str, ppath);
	strcat(str, PATHSEPARATORSTRING "random");
	strncpy(lowFilename, pfilename, 13);
	lowFilename[13] = '\0';
	strlower((char *)lowFilename);
	if ((fp = fopen(str, "r")) != NULL) {
		while (!feof(fp)) {
			fgets(str, PATH_MAX, fp);
			if (strchr(str, '\n'))
				*strchr(str, '\n') = '\0';
			if (strcmp(lowFilename, str) == 0) {
				fclose(fp);
				return 1;
			}
		}
		fclose(fp);
	} // if
	return 0;

} // is_in_file_random

// return 0 if no more entries found otherwise != 0
int	DirectoryContainer::NextDirEntry(FlexDirEntry& entry)
{
	BString		str;
	int			attribs, sectorMap;
	bool			isValid;
#ifdef WIN32
	WIN32_FIND_DATA		dirEntry;
	SYSTEMTIME		systemTime;
#else
	struct dirent		*dirEntry = NULL;
	struct stat		sbuf;
	struct tm		*lt;
#endif

	CHECK_NO_DCONTAINER_OPEN;
	CHECK_DDIRECTORY_NOT_OPENED;
	entry.SetEmpty();
	// repeat until a valid directory entry found
#ifdef WIN32
	str = *path;
	str += PATHSEPARATOR;
	str += filePattern;
	do {
		isValid = false;
		if (dirHdl == NULL) {
			dirHdl = FindFirstFile(str, &dirEntry);
			if (dirHdl != INVALID_HANDLE_VALUE)
				isValid = true;
			else
				dirHdl = NULL;
		} else {
			if (FindNextFile(dirHdl, &dirEntry))
				isValid = true;
		}
	} while (isValid &&
		   ((dirEntry.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) ||
			(dirEntry.dwFileAttributes & FILE_ATTRIBUTE_OFFLINE) ||
			(dirEntry.dwFileAttributes & FILE_ATTRIBUTE_TEMPORARY) ||
			dirEntry.nFileSizeHigh != 0));
	if (isValid) {
	// ok, found a valid directory entry
		attribs = 0;
		sectorMap = 0;
		if (strlen(dirEntry.cFileName) > 12)
			entry.SetTotalFileName(dirEntry.cAlternateFileName);
		else
			entry.SetTotalFileName(dirEntry.cFileName);
		if (dirEntry.dwFileAttributes & FILE_ATTRIBUTE_HIDDEN)
			sectorMap = 2;
		// CDFS support:
		if (IsInFileRandom(*path, entry.GetTotalFileName()))
			sectorMap = 2;
		if (dirEntry.dwFileAttributes & FILE_ATTRIBUTE_READONLY)
			attribs |= WRITE_PROTECT;
		entry.SetSize(dirEntry.nFileSizeLow);
		FileTimeToSystemTime(&dirEntry.ftLastWriteTime, &systemTime);
		entry.SetDate(systemTime.wDay, systemTime.wMonth, systemTime.wYear);
		entry.SetAttributes(attribs);
		entry.SetSectorMap(sectorMap);
		entry.SetStartTrkSec(0, 0);
		entry.SetEndTrkSec(0, 0);
		entry.ClearEmpty();
	}
#else
	str = *path;
	do {
		isValid = false;
		if (dirHdl == NULL) {
			if ((dirHdl = opendir(str)) != NULL)
				isValid = true;
		}
		if (dirHdl != NULL && (dirEntry = readdir(dirHdl)) != NULL)
			isValid = true;
	} while (isValid &&
		(stat(str + PATHSEPARATORSTRING + dirEntry->d_name, &sbuf) ||
		 !IsFlexFilename(dirEntry->d_name) ||
		 !S_ISREG(sbuf.st_mode) ||
		 sbuf.st_size <= 0 || (strcmp(filePattern, "*.*") != 0 &&
		 strcmp(filePattern.chars(), dirEntry->d_name) != 0)));
	if (isValid) {
	// ok, found a valid directory entry
		attribs = 0;
		sectorMap = 0;
		if (sbuf.st_mode & S_IXUSR)
			sectorMap = 2;
		if (!(sbuf.st_mode & S_IWUSR))
			attribs |= WRITE_PROTECT;
		entry.SetTotalFileName(dirEntry->d_name);
		entry.SetSize(sbuf.st_size);
		lt = localtime(&(sbuf.st_mtime));
		entry.SetDate(lt->tm_mday, lt->tm_mon+1, lt->tm_year+1900);
		entry.SetAttributes(attribs);
		entry.SetSectorMap(sectorMap);
		entry.SetStartTrkSec(0, 0);
		entry.SetEndTrkSec(0, 0);
		entry.ClearEmpty();
	}
#endif
	return !entry.IsEmpty();
}

int	DirectoryContainer::CloseDirectory(void)
{
	CHECK_NO_DCONTAINER_OPEN;
	CHECK_DDIRECTORY_NOT_OPENED;

	if (dirHdl != NULL) {
#ifdef WIN32
		FindClose(dirHdl);
#else
		closedir(dirHdl);
#endif
		dirHdl = NULL;
	}
	filePattern = "";
	return 1;
}

// return != 0 if file found
// if file found can also be checked by
// !entry.isEmpty
int DirectoryContainer::FindFile(const char *fileName, FlexDirEntry& entry)
{
	int ret;

	CHECK_NO_DCONTAINER_OPEN;
	CHECK_DDIRECTORY_ALREADY_OPENED;
	ret = 0;
	if (OpenDirectory(fileName)) {
		try {
			ret = NextDirEntry(entry);
		} catch (FlexException UNUSED(&e)) {
				CloseDirectory();
				throw;
		}
		CloseDirectory();
	}
	return ret;
}

int	DirectoryContainer::DeleteFile(const char *fileName)
{
	BString str;

	CHECK_NO_DCONTAINER_OPEN;
	str = fileName;
#ifndef WIN32
	str.downcase();
#endif
	str = *path + PATHSEPARATOR + str;
	if (remove(str)) {
		// Unfinished
		if (errno == EACCES)
			ex.setString(FERR_REMOVE_FILE, fileName, *path);
		if (errno == ENOENT)
			ex.setString(FERR_NO_FILE_IN_CONTAINER, fileName, *path);
		throw ex;
	}
	return 1;
}

int	DirectoryContainer::RenameFile(const char *oldName, const char *newName)
{
	BString s, d;

	CHECK_NO_DCONTAINER_OPEN;
	s = oldName;
	d = newName;
#ifndef WIN32
	s.downcase();
	d.downcase();
#endif
	s = *path + PATHSEPARATORSTRING + s;
	d = *path + PATHSEPARATORSTRING + d;
	if (rename(s, d)) {
		// Unfinished
		if (errno == EEXIST)
			ex.setString(FERR_FILE_ALREADY_EXISTS, newName);
		if (errno == EACCES)
			ex.setString(FERR_RENAME_FILE, oldName, *path);
		if (errno == ENOENT)
			ex.setString(FERR_NO_FILE_IN_CONTAINER, oldName, *path);
		throw ex;
	}
	return 1;
}

int	DirectoryContainer::FileCopy(
		const char *sourceName, const char *destName,
		FileContainerIf& destination)
{
	FlexCopyManager copyMan;

	CHECK_NO_DCONTAINER_OPEN;
	CHECK_DDIRECTORY_ALREADY_OPENED;
	return copyMan.FileCopy(sourceName, destName,
		(FileContainerIf &)*this, destination);
}

int	DirectoryContainer::GetInfo(FlexContainerInfo& info)
{
#ifdef WIN32
	DWORD sectorsPerCluster = 0;
	DWORD bytesPerSector = 1;
	DWORD numberOfFreeClusters = 0;
	DWORD totalNumberOfClusters = 0;
#else
#endif
	char 		*p;
	BString	rootPath;
	struct stat	sbuf;
	struct tm	*timeStruct;


	CHECK_NO_DCONTAINER_OPEN;
	if (path->length() > 3) {
		path->at(0, 3, rootPath);
	}
#ifdef WIN32
	if (!GetDiskFreeSpace(rootPath, &sectorsPerCluster,
		&bytesPerSector, &numberOfFreeClusters, &totalNumberOfClusters)) {
		ex.setString(FERR_READING_DISKSPACE, *path);
		throw ex;
	}
	// free size in KByte
	info.SetFree(numberOfFreeClusters * sectorsPerCluster /
		1024 * bytesPerSector);
	// total size in KByte
	info.SetTotalSize(totalNumberOfClusters * sectorsPerCluster /
		1024 * bytesPerSector);
#else
	struct statfs fsbuf;

	if (statfs(*path, &fsbuf)) {
		ex.setString(FERR_READING_DISKSPACE, *path);
		throw ex;
	}
	info.SetFree(fsbuf.f_bsize * fsbuf.f_bfree / 1024);
	info.SetTotalSize(fsbuf.f_bsize * fsbuf.f_blocks / 1024);
#endif
	if (stat(*path, &sbuf) >= 0) {
		timeStruct = localtime(&sbuf.st_atime);
		info.SetDate(timeStruct->tm_mday, timeStruct->tm_mon+1, timeStruct->tm_year + 1900);
	} else
		info.SetDate(0, 0, 0);
	info.SetTrackSector(0, 0);
	if ((p = strrchr(*path, PATHSEPARATOR)) != NULL)
		info.SetName(p+1);
	else
		info.SetName(*path);
	info.SetPath(*path);
	info.SetType(param.type);
	info.SetAttributes(attributes);
	return 1;
}

// check if an container is opened
// If so return != 0
int DirectoryContainer::IsContainerOpened(void)
{
	return isOpened;
}


int DirectoryContainer::GetContainerType(void)
{
	return param.type;
}

void DirectoryContainer::ReadToBuffer(const char *fileName, FlexFileBuffer &buffer)
{
	BString filePath;

	filePath = *path + PATHSEPARATORSTRING + fileName;
	if (!buffer.ReadFromFile(filePath)) {
		ex.setString(FERR_READING_FROM, fileName);
		throw ex;
	}
}

int DirectoryContainer::WriteFromBuffer(const char *fileName, const FlexFileBuffer &buffer)
{
	BString filePath;

	filePath = *path + PATHSEPARATORSTRING + fileName;
	if (!buffer.WriteToFile(filePath)) {
		ex.setString(FERR_WRITING_TO, fileName);
		throw ex;
	}
	SetDate(fileName, buffer.GetDate());
	SetAttributes(fileName, buffer.GetAttributes());
	if (buffer.IsRandom())
		SetRandom(fileName);
	return 1;
}

/******************************/
/* private interface		  */
/******************************/

void DirectoryContainer::Initialize_header(Byte wp) {
	param.offset        = 0;
	param.write_protect = wp;
	param.max_sector    = 0;
	param.max_sector0   = 0;
	param.max_track     = 0;
	param.byte_p_sector = SECTOR_SIZE;
	param.byte_p_track0 = 0; 
	param.byte_p_track  = 0; 
	param.type			= TYPE_DIRECTORY;
}

int DirectoryContainer::OpenForDirectory(const char *aPath)
{
	struct stat sbuf;

	path = NULL;
	if (!stat(aPath, &sbuf) && !S_ISDIR(sbuf.st_mode)) {
		ex.setString(FERR_UNABLE_TO_OPEN, aPath);
		throw ex;
	}
	attributes = 0;
	if (access(aPath, W_OK))
		attributes |= FLX_READONLY;
	Initialize_header(attributes & FLX_READONLY);
	path = new BString(aPath);
	isOpened = 1;
	return 1;
}

// set the date of a file

int	DirectoryContainer::SetDate(const char *fileName, const BDate& date)
{
	struct stat    sbuf;
	struct utimbuf timebuf;
	struct tm      file_time;
	BString filePath;

	filePath = *path + PATHSEPARATORSTRING + fileName;
	if (stat(filePath, &sbuf) >= 0) {
		timebuf.actime = sbuf.st_atime;
		file_time.tm_sec   = 0;
		file_time.tm_min   = 0;
		file_time.tm_hour  = 0;
		file_time.tm_mon   = date.GetMonth()-1;
		file_time.tm_mday  = date.GetDay();
		file_time.tm_year  = date.GetYear() - 1900;
		file_time.tm_isdst = 0;
		timebuf.modtime    = mktime(&file_time);
		if (timebuf.modtime >= 0 && utime(filePath, &timebuf) >= 0)
			return 1;
		else
			return 0;
	} // if
	return 0;
}

// set the file attributes of a file
int	DirectoryContainer::SetAttributes(const char *fileName, int setMask, int clearMask /* = ~0 */)
{
	// only WRITE_PROTECT flag is supported
	if ((setMask & WRITE_PROTECT) || (clearMask & WRITE_PROTECT)) {
		BString filePath;
		filePath = *path + PATHSEPARATORSTRING + fileName;
#ifdef WIN32
		DWORD attrs = GetFileAttributes(filePath);
		if (clearMask & WRITE_PROTECT)
			attrs &= ~FILE_ATTRIBUTE_READONLY;
		if (setMask & WRITE_PROTECT)
			attrs |= FILE_ATTRIBUTE_READONLY;
		SetFileAttributes(filePath, attrs);
#else
		struct stat sbuf;
		if (!stat(filePath, &sbuf)) {
			if (clearMask & WRITE_PROTECT)
				chmod(filePath, sbuf.st_mode | S_IWUSR);
			if (setMask & WRITE_PROTECT)
				chmod(filePath, sbuf.st_mode & ~S_IWUSR);
		}
#endif
	}
	return 1;
}

// on WIN32 a random file will be represented by a hidden flag
int	DirectoryContainer::SetRandom(const char *fileName)
{
	BString filePath;
	filePath = *path + PATHSEPARATORSTRING + fileName;
#ifdef WIN32
	DWORD attrs = GetFileAttributes(filePath);
	SetFileAttributes(filePath, attrs | FILE_ATTRIBUTE_HIDDEN);
#else
	struct stat sbuf;
	if (!stat(filePath, &sbuf))
		chmod(filePath, sbuf.st_mode | S_IXUSR);
#endif
	return 1;
}

bool DirectoryContainer::IsFlexFilename(const char *pfilename,
					char *pname /* = NULL */,
					char *pext  /* = NULL */)
{
   int     result; // result from sscanf should be int
   char    dot;
   char    name[9];
   char    ext[4];

   dot    = '\0';
   result = sscanf(pfilename, "%1[a-z]%7[a-z0-9_-]",
	    (char *)&name, (char *)&name + 1);
   if (!result || result == EOF)
   	return false;
   if (result == 1)
   	result = sscanf(pfilename, "%*1[a-z]%c%1[a-z]%2[a-z0-9_-]",
        &dot, (char *)&ext, (char *)&ext + 1);
   else
        result = sscanf(pfilename, "%*1[a-z]%*7[a-z0-9_-]%c%1[a-z]%2[a-z0-9_-]",
           &dot, (char *)&ext, (char *)&ext + 1);
   if (!result || result == 1 || result == EOF)
           return false;
   if (strlen(name) + strlen(ext) + (dot == '.' ? 1 : 0) != strlen(pfilename))
           return false;
   strupper(name);
   strupper(ext);
   if (pname)
       strcpy(pname, name);
   if (pext)
       strcpy(pext, ext);
   return true;
} // IsFlexFilename

/****************************************/
/* low level routines                   */
/****************************************/

// low level routine to read a single sector
// should be used with care
// Does not throw any exception !
// returns 0 on failure
// Not supported in this class
int DirectoryContainer::ReadSector(Byte *pbuffer, int trk, int sec)
{
	return 0;
}

// low level routine to write a single sector
// should be used with care
// Does not throw any exception !
// returns 0 on failure
// Not supported in this class
int DirectoryContainer::WriteSector(const Byte *pbuffer, int trk, int sec)
{
	return 0;
}
