/*************************************************************************** 
  Originally posted to comp.sys.m6809 by Didier Derny (didier@aida.remcomp.fr)

  Minor hacks by Alan DeKok

 Fixed: D_Indexed addressing used prog[2] and prog[3] when it meant
        prog[pc+2] and prog[pc+3]:  Would produce flawed disassemblies!

        changed addresses in D_Indexed to be all hex.
	added 2 instances of 'extrabyte' in D_Indexed: would not skip them..
        Added PC offsets to D_Indexed ,PCR formats
	added SWI2 print out as OS9

	converted to a C++ Class
	to use within flexemu by W. Schwotzer

****************************************************************************/


#include <misc1.h>
#include <stdio.h>
#include <string.h>

#include "da6809.h"
#include "mc6809.h"
#include "extmem.h"


Da6809::Da6809(Mc6809 *x_cpu)
{
	// the cpu will be used as memory interface:
	cpu = x_cpu;
}

Da6809::~Da6809()
{
}

char *Da6809::FlexLabel(Word addr)
{
#ifdef FLEX_LABEL
	switch (addr) {
		// FLEX DOS entries:
		case 0xCD00: return "COLDS";
		case 0xCD03: return "WARMS";
		case 0xCD06: return "RENTER";
		case 0xCD09: return "INCH";
		case 0xCD0C: return "INCH2";
		case 0xCD0F: return "OUTCH";
		case 0xCD12: return "OUTCH2";
		case 0xCD15: return "GETCHR";
		case 0xCD18: return "PUTCHR";
		case 0xCD1B: return "INBUFF";
		case 0xCD1E: return "PSTRNG";
		case 0xCD21: return "CLASS";
		case 0xCD24: return "PCRLF";
		case 0xCD27: return "NXTCH";
		case 0xCD2A: return "RSTRIO";
		case 0xCD2D: return "GETFIL";
		case 0xCD30: return "LOAD";
		case 0xCD33: return "SETEXT";
		case 0xCD36: return "ADDBX";
		case 0xCD39: return "OUTDEC";
		case 0xCD3C: return "OUTHEX";
		case 0xCD3F: return "RPTERR";
		case 0xCD42: return "GETHEX";
		case 0xCD45: return "OUTADR";
		case 0xCD48: return "INDEC";
		case 0xCD4B: return "DOCMND";
		case 0xCD4E: return "STAT";
	
		// FLEX FMS entries:
		case 0xD400: return "FMSINI";	// FMS init
		case 0xD403: return "FMSCLS";	// FMS close
		case 0xD406: return "FMS";
		case 0xC840: return "FCB";	// standard system FCB

		// miscellenious:
		case 0xD435: return "VFYFLG";	// FMS verify flag
		case 0xC080: return "LINBUF";	// line buffer 
		case 0xCC00: return "TTYBS"; 
		case 0xCC01: return "TTYDEL"; 
		case 0xCC02: return "TTYEOL"; 
		case 0xCC03: return "TTYDPT"; 
		case 0xCC04: return "TTYWDT"; 
		case 0xCC11: return "TTYTRM"; 
		case 0xCC12: return "COMTBL"; // user command table
		case 0xCC14: return "LINBFP"; // line buffer pointer
		case 0xCC16: return "ESCRET"; // escape return register
		case 0xCC18: return "LINCHR"; // current char in linebuffer
		case 0xCC19: return "LINPCH"; // previous char in linebuffer
		case 0xCC1A: return "LINENR"; //line nr of current page 
		case 0xCC1B: return "LODOFS"; // loader address offset 
		case 0xCC1D: return "TFRFLG"; // loader  transfer flag
		case 0xCC1E: return "TFRADR"; // transfer address
		case 0xCC20: return "FMSERR"; // FMS error type
		case 0xCC21: return "IOFLG"; // special I/O flag
		case 0xCC22: return "OUTSWT"; // output switch
		case 0xCC23: return "INSWT"; // input switch
		case 0xCC24: return "OUTADR"; // file output address
		case 0xCC26: return "INADR"; // file input address
		case 0xCC28: return "COMFLG"; // command flag
		case 0xCC29: return "OUTCOL"; // current output column
		case 0xCC2A: return "SCRATC"; // system scratch
		case 0xCC2B: return "MEMEND"; // memory end
		case 0xCC2D: return "ERRVEC"; // error name vector
		case 0xCC2F: return "INECHO"; // file input echo flag

		// printer support
		case 0xCCC0: return "PRTINI"; // printer initialize
		case 0xCCD8: return "PRTCHK"; // printer check
		case 0xCCE4: return "PRTOUT"; // printer output

		default:     return NULL;
	}
#endif // #ifdef FLEX_LABEL
	return NULL;
}

char *Da6809::IndexedRegister(Byte which)
{
	switch (which & 0x03) {
		case 0: return "X";
		case 1: return "Y";
		case 2: return "U";
		case 3: return "S";
	} // switch
	return "?"; // should never happen
}

char *Da6809::InterRegister(Byte which)
{
	switch (which & 0x0f) {
		case 0x0: return "D";
		case 0x1: return "X";
		case 0x2: return "Y";
		case 0x3: return "U";
		case 0x4: return "S";
		case 0x5: return "PC";
		case 0x8: return "A";
		case 0x9: return "B";
		case 0xa: return "CC";
		case 0xb: return "DP";
		default: return "??";
	} // switch
}


char *Da6809::StackRegister(Byte which, char *not_stack)
{
	switch (which & 0x07) {
		case 0x0: return "CC";
		case 0x1: return "A";
		case 0x2: return "B";
		case 0x3: return "DP";
		case 0x4: return "X";
		case 0x5: return "Y";
		case 0x6: return not_stack;
		case 0x7: return "PC";
		default: return "??"; // this case should never happen
	} // switch
}


Byte Da6809::D_Illegal (char *name, Byte bytes, Word pc)
{
  Byte code;

  code = READ(pc);
  sprintf(code_buf, "%0.4X: %0.2X", pc, code);
  sprintf(mnem_buf, "%s ?????", name);
  return bytes;
}


Byte Da6809::D_Direct (char *name, Byte bytes, Word pc)
{
  Byte code, offset;

  code = READ(pc);
  offset = READ(pc+1);
  sprintf(code_buf, "%0.4X: %0.2X %0.2X", pc, code, offset);
  sprintf(mnem_buf, "%s $%0.2X", name, offset);
  return bytes;
}


Byte Da6809::D_Immediat (char *name, Byte bytes, Word pc)
{
  Byte code, offset;

  code = READ(pc);
  offset = READ(pc+1);
  sprintf(code_buf, "%0.4X: %0.2X %0.2X", pc, code, offset);
  sprintf(mnem_buf, "%s #$%0.2X", name, offset);
  return bytes;
}


Byte Da6809::D_ImmediatL (char *name, Byte bytes, Word pc)
{
  Byte code;
  Word offset;
  char *label;

    code = READ(pc);
    offset = (READ(pc+1) << 8) | READ(pc+2);
    sprintf(code_buf, "%0.4X: %0.2X %0.2X %0.2X", pc, code, READ(pc+1),
    	READ(pc+2));
  label = FlexLabel(offset);
  if (label == NULL)	
    	sprintf(mnem_buf, "%s #$%0.4X", name, offset);
  else
  	sprintf(mnem_buf, "%s #%s ; $%0.4X", name, label, offset);
    return bytes;
}


Byte Da6809::D_Inherent (char *name, Byte bytes, Word pc)
{
  int code;

    code = READ(pc);
    sprintf(code_buf, "%0.4X: %0.2X", pc, code);
    sprintf(mnem_buf, "%s", name);
    return bytes;
}  // D_Inherent


Byte Da6809::D_Indexed (char *name, Byte bytes, Word pc)
{
    Byte code, postbyte;
    char *s, *br1, *br2;
    Byte extrabytes, disp;
    Word offset;

    extrabytes = 0;
    code = READ(pc);
    postbyte = READ(pc+1);
    br1        = "";		// bracket on for indirect addressing
    br2        = "";		// bracket off for indirect addressing
    s 	       = "";		// minus sign for offset

       if ((postbyte & 0x80) == 0x00) {
          // ,R + 5 Bit Offset
          disp = postbyte & 0x1f;
          if ((postbyte & 0x10) == 0x10) {
             s = "-";
              disp=0x20-disp;
          }
          sprintf(code_buf, "%0.4X: %0.2X %0.2X", pc, code, postbyte);
          sprintf(mnem_buf, "%s %s$%0.2X,%s", name, s, disp,
          	 IndexedRegister(postbyte >> 5));
  } else {
      switch(postbyte & 0x1f) {
	 case 0x00 : // ,R+
            sprintf(code_buf, "%0.4X: %0.2X %0.2X", pc, code, postbyte);
            sprintf(mnem_buf, "%s ,%s+", name, IndexedRegister(postbyte>>5));
            break;
	 case 0x11 : // [,R++]
	    br1 = "[";
	    br2 = "]";
         case 0x01 : // ,R++
            sprintf(code_buf, "%0.4X: %0.2X %0.2X", pc, code, postbyte);
            sprintf(mnem_buf, "%s %s,%s++%s", name,
            	br1, IndexedRegister(postbyte>>5), br2);
            break;
         case 0x02 : // ,-R
            sprintf(code_buf, "%0.4X: %0.2X %0.2X", pc, code, postbyte);
            sprintf(mnem_buf, "%s ,-%s", name, IndexedRegister(postbyte>>5));
            break;
	 case 0x13 : // [,R--]
	    br1 = "[";
	    br2 = "]";
         case 0x03 : // ,--R
            sprintf(code_buf, "%0.4X: %0.2X %0.2X", pc, code, postbyte);
            sprintf(mnem_buf, "%s %s,--%s%s", name,
            	br1, IndexedRegister(postbyte>>5), br2);
            break;
	 case 0x14 : // [,R--]
	    br1 = "[";
	    br2 = "]";
         case 0x04 : // ,R
            sprintf(code_buf, "%0.4X: %0.2X %0.2X", pc, code, postbyte);
            sprintf(mnem_buf, "%s %s,%s%s", name,
            	br1, IndexedRegister(postbyte>>5), br2);
            break;
	 case 0x15 : // [B,R]
	    br1 = "[";
	    br2 = "]";
         case 0x05 : // B,R
            sprintf(code_buf, "%0.4X: %0.2X %0.2X", pc, code, postbyte);
            sprintf(mnem_buf, "%s %sB,%s%s", name,
            	br1, IndexedRegister(postbyte>>5), br2);
            break;
	 case 0x16 : // [A,R]
	    br1 = "[";
	    br2 = "]";
         case 0x06 : // A,R
            sprintf(code_buf, "%0.4X: %0.2X %0.2X", pc, code, postbyte);
            sprintf(mnem_buf, "%s %sA,%s%s", name,
            	br1, IndexedRegister(postbyte>>5), br2);
            break;
         case 0x18 : // [,R + 8 Bit Offset]
	    br1 = "[";
	    br2 = "]";
         case 0x08 : // ,R + 8 Bit Offset
            offset = READ(pc+2);
            if (offset < 128)
               s = "";
            else {
               s = "-";
               offset=0x0100-offset;
            }
            sprintf(code_buf, "%0.4X: %0.2X %0.2X %0.2X", pc, code, postbyte,
            	READ(pc+2));
            sprintf(mnem_buf, "%s %s%s$%0.2X,%s%s", name, br1, s, offset,
            	IndexedRegister(postbyte>>5), br2);
            extrabytes=1;
            break;
         case 0x19 : // [,R + 16 Bit Offset]
	    br1 = "[";
	    br2 = "]";
         case 0x09 : // ,R + 16 Bit Offset
            offset = (READ(pc+2) << 8) | READ(pc+3);
            if (offset < 32768)
               s = "";
            else {
               s = "-";
               offset=0xffff-offset+1;
            }
            sprintf(code_buf, "%0.4X: %0.2X %0.2X %0.2X %0.2X", pc, code,
            	postbyte, READ(pc+2), READ(pc+3));
            sprintf(mnem_buf, "%s %s%s$%0.4X,%s%s", name,
            	br1, s, offset, IndexedRegister(postbyte>>5), br2);
            extrabytes=2;
            break;
         case 0x1b : // [D,R]
	    br1 = "[";
	    br2 = "]";
         case 0x0b : // D,R
            sprintf(code_buf, "%0.4X: %0.2X %0.2X", pc, code, postbyte);
            sprintf(mnem_buf, "%s %sD,%s%s", name,
            	br1, IndexedRegister(postbyte>>5), br2);
            break;
         case 0x1c : // [,PC + 8 Bit Offset]
	    br1 = "[";
	    br2 = "]";
         case 0x0c : // ,PC + 8 Bit Offset
            offset = ((char)(READ(pc+2))+pc+3) & 0xFFFF;
            s = "<";
            sprintf(code_buf, "%0.4X: %0.2X %0.2X %0.2X", pc, code, postbyte,
            	READ(pc+2));
            sprintf(mnem_buf, "%s %s%s$%0.2X,PCR%s", name, br1, s, offset, br2);
            extrabytes=1;
            break;
         case 0x1d : // [,PC + 16 Bit Offset]
	    br1 = "[";
	    br2 = "]";
         case 0x0d :  // ,PC + 16 Bit Offset
            offset = ((READ(pc+2) << 8) |
            	READ(pc+3)+pc+4) & 0xFFFF;
            s = ">";
            sprintf(code_buf, "%0.4X: %0.2X %0.2X %0.2X %0.2X", pc, code,
            	postbyte, READ(pc+2), READ(pc+3));
            sprintf(mnem_buf, "%s %s%s$%0.4X,PCR%s", name, br1, s, offset, br2);
            extrabytes = 2;
            break;
         case 0x1f : // [n]
	    br1 = "[";
	    br2 = "]";
            offset = (READ(pc+2) << 8) | READ(pc+3);
            sprintf(code_buf, "%0.4X: %0.2X %0.2X %0.2X %0.2X", pc, code,
            	postbyte, READ(pc+2), READ(pc+3));
            sprintf(mnem_buf, "%s %s$%0.4X%s", name, br1, offset, br2);
            extrabytes = 2;
            break;
	 default: 
            sprintf(code_buf, "%0.4X: %0.2X %0.2X", pc, code, postbyte);
            sprintf(mnem_buf, "%s ????", name);
	  }
   }
   return bytes + extrabytes;
} // D_Indexed


Byte Da6809::D_Extended (char *name, Byte bytes, Word pc)
{
  Byte code;
  Word offset;
  char *label;

  code = READ(pc);
  offset = (READ(pc+1) << 8) | READ(pc+2);
  sprintf(code_buf, "%0.4X: %0.2X %0.2X %0.2X", pc, code,
  	READ(pc+1), READ(pc+2));
  label = FlexLabel(offset);
  if (label == NULL)	
  	sprintf(mnem_buf, "%s $%0.4X", name, offset);
  else
  	sprintf(mnem_buf, "%s %s ; $%0.4X", name, label, offset);
  return bytes;
}

Byte Da6809::D_Relative (char *name, Byte bytes, Word pc)
{
  Byte code;
  Word offset, disp;
  char *label;

  code = READ(pc);
  offset = READ(pc+1);
  if (offset < 127 )
	 disp   = pc + 2 + offset;
  else
	 disp   = pc + 2 - (256 - offset);
  sprintf(code_buf, "%0.4X: %0.2X %0.2X", pc, code, offset);
  label = FlexLabel(disp);
  if (label == NULL)	
  	sprintf(mnem_buf, "%s $%0.4X", name, disp);
  else
  	sprintf(mnem_buf, "%s %s ; $%0.4X", name, label, disp);
  return bytes;
}

Byte Da6809::D_RelativeL (char *name, Byte bytes, Word pc)
{
  Byte code;
  Word offset, disp;
  char *label;

  code = READ(pc);
  offset = (READ(pc+1) << 8) | READ(pc+2);
  if (offset < 32767 )
	 disp   = pc + 3 + offset;
  else
	 disp   = pc + 3 - (65536 - offset);
  sprintf(code_buf, "%0.4X: %0.2X %0.2X %0.2X", pc, code,
  	READ(pc+1), READ(pc+2));
  label = FlexLabel(disp);
  if (label == NULL)	
	sprintf(mnem_buf, "%s $%0.4X", name, disp);
  else
  	sprintf(mnem_buf, "%s %s ; $%0.4X", name, label, disp);
  return bytes;
}

Byte Da6809::D_Register0 (char *name, Byte bytes, Word pc)
{
  Byte code;
  Word postbyte;

  code = READ(pc);
  postbyte = READ(pc+1);

  sprintf(code_buf, "%0.4X: %0.2X %0.2X", pc, code, postbyte);
  sprintf(mnem_buf, "%s %s,%s", name, InterRegister(postbyte>>4),
  	InterRegister(postbyte & 0x0f));

  return bytes;
}

Byte Da6809::D_Register1 (char *name, Byte bytes, Word pc)
{
  Byte code, postbyte;
  Byte i, index, comma;

  code      = READ(pc);	
  postbyte  = READ(pc+1);	

  comma = 0;
  sprintf(code_buf, "%0.4X: %0.2X %0.2X", pc, code, postbyte);   	
  sprintf(mnem_buf, "%s ", name);
  index = strlen(name);
  for (i= 0; i < 8; i++) {
    if (postbyte & (1 << i)) {
      sprintf((char *)&mnem_buf[index], "%s%s",
      		comma ? "," : "", StackRegister(i, "U")); 
      index += strlen(StackRegister(i, "U")) + (comma ? 1 : 0); 
      comma = 1;
    } // if
  } // for
  return bytes;
} // D_Register1


Byte Da6809::D_Register2 (char *name, Byte bytes, Word pc)
{
  Byte code, postbyte;
  Byte i, index, comma;

  code      = READ(pc);	
  postbyte  = READ(pc+1);	

  comma = 0;
  sprintf(code_buf, "%0.4X: %0.2X %0.2X", pc, code, postbyte);   	
  sprintf(mnem_buf, "%s ", name);
  index = strlen(name);
  for (i= 0; i < 8; i++) {
    if (postbyte & (1 << i)) {
      sprintf((char *)&mnem_buf[index], "%s%s",
      		comma ? "," : "", StackRegister(i, "S")); 
      index += strlen(StackRegister(i, "S")) + (comma ? 1 : 0); 
      comma = 1;
    } // if
  } // for
  return bytes;
} // D_Register2


Byte Da6809::D_Page10 (Byte *pstep, Word address)
{
	Byte code;

	code = READ(address+1);
	switch(code) {
		case 0x21: return D_RelativeL("LBRN ", 4, address+1);
		case 0x22: return D_RelativeL("LBHI ", 4, address+1);
		case 0x23: return D_RelativeL("LBLS ", 4, address+1);
		case 0x24: return D_RelativeL("LBCC ", 4, address+1);
		case 0x25: return D_RelativeL("LBCS ", 4, address+1);
		case 0x26: return D_RelativeL("LBNE ", 4, address+1);
		case 0x27: return D_RelativeL("LBEQ ", 4, address+1);
		case 0x28: return D_RelativeL("LBVC ", 4, address+1);
		case 0x29: return D_RelativeL("LBVS ", 4, address+1);
		case 0x2a: return D_RelativeL("LBPL ", 4, address+1);
		case 0x2b: return D_RelativeL("LBMI ", 4, address+1);
		case 0x2c: return D_RelativeL("LBGE ", 4, address+1);
		case 0x2d: return D_RelativeL("LBLT ", 4, address+1);
		case 0x2e: return D_RelativeL("LBGT ", 4, address+1);
		case 0x2f: return D_RelativeL("LBLE ", 4, address+1);

		case 0x3f: *pstep = 0; return D_Inherent("SWI2 ", 2, address+1);

		case 0x83: return D_ImmediatL("CMPD ", 4, address+1);
		case 0x8c: return D_ImmediatL("CMPY ", 4, address+1);
		case 0x8e: return D_ImmediatL("LDY  ", 4, address+1);

		case 0x93: return D_Direct("CMPD ", 3, address+1);
		case 0x9c: return D_Direct("CMPY ", 3, address+1);
		case 0x9e: return D_Direct("LDY  ", 3, address+1);
		case 0x9f: return D_Direct("STY  ", 3, address+1);

		case 0xa3: return D_Indexed("CMPD ", 3, address+1);
		case 0xac: return D_Indexed("CMPY ", 3, address+1);
		case 0xae: return D_Indexed("LDY  ", 3, address+1);
		case 0xaf: return D_Indexed("STY  ", 3, address+1);

		case 0xb3: return D_Extended("CMPD ", 4, address+1);
		case 0xbc: return D_Extended("CMPY ", 4, address+1);
		case 0xbe: return D_Extended("LDY  ", 4, address+1);
		case 0xbf: return D_Extended("STY  ", 4, address+1);

		case 0xce: return D_ImmediatL("LDS  ", 4, address+1);

		case 0xde: return D_Direct("LDS  ", 3, address+1);
		case 0xdf: return D_Direct("STS  ", 3, address+1);

		case 0xee: return D_Indexed("LDS  ", 3, address+1);
		case 0xef: return D_Indexed("STS  ", 3, address+1);

		case 0xfe: return D_Extended("LDS  ", 4, address+1);
		case 0xff: return D_Extended("STS  ", 4, address+1);

		default:   return D_Illegal("", 1, address+1);
	} // switch
} // D_Page10


Byte Da6809::D_Page11 (Byte *pstep, Word address)
{
	Byte code;

	code = READ(address+1);
	switch(code) {
		case 0x3f: *pstep = 0; return D_Inherent("SWI3 ", 2, address+1);

		case 0x83: return D_ImmediatL("CMPU ", 4, address+1);
		case 0x8c: return D_ImmediatL("CMPS ", 4, address+1);

		case 0x93: return D_Direct("CMPU ", 3, address+1);
		case 0x9c: return D_Direct("CMPS ", 3, address+1);

		case 0xa3: return D_Indexed("CMPU ", 3, address+1);
		case 0xac: return D_Indexed("CMPS ", 3, address+1);

		case 0xb3: return D_Indexed("CMPU ", 4, address+1);
		case 0xbc: return D_Indexed("CMPS ", 4, address+1);

		default:   return D_Illegal("", 1, address+1);
	}  // switch
} // D_Page11


int Da6809::disassemble(Word address, Byte *pstep, char **pb1, char **pb2)
{
	Byte code;

	*pb1 = code_buf;
	*pb2 = mnem_buf;
	*pstep = 1;
	code = READ(address);
	switch(code) {
		case 0x00: return D_Direct("NEG  ", 2, address);
	    	// 0x01, 0x02 is illegal
		case 0x03: return D_Direct("COM  ", 2, address);
		case 0x04: return D_Direct("LSR  ", 2, address);
	    	// 0x05 is illegal
		case 0x06: return D_Direct("ROR  ", 2, address);
		case 0x07: return D_Direct("ASR  ", 2, address);
		case 0x08: return D_Direct("LSR  ", 2, address);
		case 0x09: return D_Direct("ROR  ", 2, address);
		case 0x0a: return D_Direct("DEC  ", 2, address);
	    	// 0x0b is illegal
		case 0x0c: return D_Direct("INC  ", 2, address);
		case 0x0d: return D_Direct("TST  ", 2, address);
		case 0x0e: return D_Direct("JMP  ", 2, address);
		case 0x0f: return D_Direct("CLR  ", 2, address);

		case 0x10: return D_Page10(pstep, address);
		case 0x11: return D_Page11(pstep, address);
		case 0x12: return D_Inherent("NOP  ", 1, address);
		case 0x13: return D_Inherent("SYNC ", 1, address);
	    	// 0x14, 0x15 is illegal
		case 0x16: return D_RelativeL("LBRA ", 3, address);
		case 0x17: *pstep = 0; return D_RelativeL("LBSR ", 3, address);
	    	// 0x18 is illegal
		case 0x19: return D_Inherent("DAA  ", 1, address);
		case 0x1a: return D_Immediat("ORCC ", 2, address);
	    	// 0x1b is illegal
		case 0x1c: return D_Immediat("ANDCC", 2, address);
		case 0x1d: return D_Inherent("SEX  ", 1, address);
		case 0x1e: return D_Register0("EXG  ", 2, address);
		case 0x1f: return D_Register0("TFR  ", 2, address);

		case 0x20: return D_Relative("BRA  ", 2, address);
		case 0x21: return D_Relative("BRN  ", 2, address);
		case 0x22: return D_Relative("BHI  ", 2, address);
		case 0x23: return D_Relative("BLS  ", 2, address);
		case 0x24: return D_Relative("BCC  ", 2, address);
		case 0x25: return D_Relative("BCS  ", 2, address);
		case 0x26: return D_Relative("BNE  ", 2, address);
		case 0x27: return D_Relative("BEQ  ", 2, address);
		case 0x28: return D_Relative("BVC  ", 2, address);
		case 0x29: return D_Relative("BVS  ", 2, address);
		case 0x2a: return D_Relative("BPL  ", 2, address);
		case 0x2b: return D_Relative("BMI  ", 2, address);
		case 0x2c: return D_Relative("BGE  ", 2, address);
		case 0x2d: return D_Relative("BLT  ", 2, address);
		case 0x2e: return D_Relative("BGT  ", 2, address);
		case 0x2f: return D_Relative("BLE  ", 2, address);

		case 0x30: return D_Indexed("LEAX ", 2, address);
		case 0x31: return D_Indexed("LEAY ", 2, address);
		case 0x32: return D_Indexed("LEAS ", 2, address);
		case 0x33: return D_Indexed("LEAU ", 2, address);
		case 0x34: return D_Register1("PSHS ", 2, address);
		case 0x35: return D_Register1("PULS ", 2, address);
		case 0x36: return D_Register2("PSHU ", 2, address);
		case 0x37: return D_Register2("PULU ", 2, address);
	    	// 0x38 is illegal
		case 0x39: return D_Inherent("RTS  ", 1, address);
		case 0x3a: return D_Inherent("ABX  ", 1, address);
		case 0x3b: return D_Inherent("RTI  ", 1, address);
		case 0x3c: return D_Immediat("CWAI ", 2, address);
		case 0x3d: return D_Inherent("MUL  ", 1, address);
	    	// 0x3e is illegal
		case 0x3f: *pstep = 0; return D_Illegal("SWI  ", 1, address);

		case 0x40: return D_Inherent("NEGA ", 1, address);
	    	// 0x41, 0x42 is illegal
		case 0x43: return D_Inherent("COMA ", 1, address);
		case 0x44: return D_Inherent("LSRA ", 1, address);
	    	// 0x45 is illegal
		case 0x46: return D_Inherent("RORA ", 1, address);
		case 0x47: return D_Inherent("ASRA ", 1, address);
		case 0x48: return D_Inherent("LSLA ", 1, address);
		case 0x49: return D_Inherent("ROLA ", 1, address);
		case 0x4a: return D_Inherent("DECA ", 1, address);
	    	// 0x4b is illegal
		case 0x4c: return D_Inherent("INCA ", 1, address);
		case 0x4d: return D_Inherent("TSTA ", 1, address);
	    	// 0x4e is illegal
		case 0x4f: return D_Inherent("CLRA ", 1, address);
                              
		case 0x50: return D_Inherent("NEGB ", 1, address);
	    	// 0x51, 0x52 is illegal
		case 0x53: return D_Inherent("COMB ", 1, address);
		case 0x54: return D_Inherent("LSRB ", 1, address);
	    	// 0x55 is illegal
		case 0x56: return D_Inherent("RORB ", 1, address);
		case 0x57: return D_Inherent("ASRB ", 1, address);
		case 0x58: return D_Inherent("LSLB ", 1, address);
		case 0x59: return D_Inherent("ROLB ", 1, address);
		case 0x5a: return D_Inherent("DECB ", 1, address);
	    	// 0x5b is illegal
		case 0x5c: return D_Inherent("INCB ", 1, address);
		case 0x5d: return D_Inherent("TSTB ", 1, address);
	    	// 0x5e is illegal
		case 0x5f: return D_Inherent("CLRB ", 1, address);
                                
		case 0x60: return D_Indexed("NEG  ", 2, address);
		// 0x61, 0x62 is illegal
		case 0x63: return D_Indexed("COM  ", 2, address);
		case 0x64: return D_Indexed("LSR  ", 2, address);
		// 0x65 is illegal
		case 0x66: return D_Indexed("ROR  ", 2, address);
		case 0x67: return D_Indexed("ASR  ", 2, address);
		case 0x68: return D_Indexed("LSL  ", 2, address);
		case 0x69: return D_Indexed("ROL  ", 2, address);
		case 0x6a: return D_Indexed("DEC  ", 2, address);
		// 0xc2 is illegal
		case 0x6c: return D_Indexed("INC  ", 2, address);
		case 0x6d: return D_Indexed("TST  ", 2, address);
		case 0x6e: return D_Indexed("JMP  ", 2, address);
		case 0x6f: return D_Indexed("CLR  ", 2, address);
                                
		case 0x70: return D_Extended("NEG  ", 3, address);
		// 0x71, 0x72 is illegal
		case 0x73: return D_Extended("COM  ", 3, address);
		case 0x74: return D_Extended("LSR  ", 3, address);
		// 0x75 is illegal
		case 0x76: return D_Extended("ROR  ", 3, address);
		case 0x77: return D_Extended("ASR  ", 3, address);
		case 0x78: return D_Extended("LSL  ", 3, address);
		case 0x79: return D_Extended("ROL  ", 3, address);
		case 0x7a: return D_Extended("DEC  ", 3, address);
		// 0x7b is illegal
		case 0x7c: return D_Extended("INC  ", 3, address);
		case 0x7d: return D_Extended("TST  ", 3, address);
		case 0x7e: return D_Extended("JMP  ", 3, address);
		case 0x7f: return D_Extended("CLR  ", 3, address);
                                
		case 0x80: return D_Immediat("SUBA ", 2, address);
		case 0x81: return D_Immediat("CMPA ", 2, address);
		case 0x82: return D_Immediat("SBCA ", 2, address);
		case 0x83: return D_ImmediatL("SUBD ", 3, address);
		case 0x84: return D_Immediat("ANDA ", 2, address);
		case 0x85: return D_Immediat("BITA ", 2, address);
		case 0x86: return D_Immediat("LDA  ", 2, address);
		// 0x87 is illegal
		case 0x88: return D_Immediat("EORA ", 2, address);
		case 0x89: return D_Immediat("ADCA ", 2, address);
		case 0x8a: return D_Immediat("ORA  ", 2, address);
		case 0x8b: return D_Immediat("ADDA ", 2, address);
		case 0x8c: return D_ImmediatL("CMPX ", 3, address);
		case 0x8d: *pstep = 0; return D_Relative("BSR  ", 2, address);
		case 0x8e: return D_ImmediatL("LDX  ", 2, address);
		// 0x8f is illegal

		case 0x90: return D_Direct("SUBA ", 2, address);
		case 0x91: return D_Direct("CMPA ", 2, address);
		case 0x92: return D_Direct("SBCA ", 2, address);
		case 0x93: return D_Direct("SUBD ", 2, address);
		case 0x94: return D_Direct("ANDA ", 2, address);
		case 0x95: return D_Direct("BITA ", 2, address);
		case 0x96: return D_Direct("LDA  ", 2, address);
		case 0x97: return D_Direct("STA  ", 2, address);
		case 0x98: return D_Direct("EORA ", 2, address);
		case 0x99: return D_Direct("ADCA ", 2, address);
		case 0x9a: return D_Direct("ORA  ", 2, address);
		case 0x9b: return D_Direct("ADDA ", 2, address);
		case 0x9c: return D_Direct("CMPX ", 2, address);
		case 0x9d: *pstep = 0; return D_Direct("JSR  ", 2, address);
		case 0x9e: return D_Direct("LDX  ", 2, address);
		case 0x9f: return D_Direct("STX  ", 2, address);

		case 0xa0: return D_Indexed("SUBA ", 2, address);
		case 0xa1: return D_Indexed("CMPA ", 2, address);
		case 0xa2: return D_Indexed("SBCA ", 2, address);
		case 0xa3: return D_Indexed("SUBD ", 2, address);
		case 0xa4: return D_Indexed("ANDA ", 2, address);
		case 0xa5: return D_Indexed("BITA ", 2, address);
		case 0xa6: return D_Indexed("LDA  ", 2, address);
		case 0xa7: return D_Indexed("STA  ", 2, address);
		case 0xa8: return D_Indexed("EORA ", 2, address);
		case 0xa9: return D_Indexed("ADCA ", 2, address);
		case 0xaa: return D_Indexed("ORA  ", 2, address);
		case 0xab: return D_Indexed("ADDA ", 2, address);
		case 0xac: return D_Indexed("CMPX ", 2, address);
		case 0xad: *pstep = 0; return D_Indexed("JSR  ", 2, address);
		case 0xae: return D_Indexed("LDX  ", 2, address);
		case 0xaf: return D_Indexed("STX  ", 2, address);

		case 0xb0: return D_Extended("SUBA ", 3, address);
		case 0xb1: return D_Extended("CMPA ", 3, address);
		case 0xb2: return D_Extended("SBCA ", 3, address);
		case 0xb3: return D_Extended("SUBD ", 3, address);
		case 0xb4: return D_Extended("ANDA ", 3, address);
		case 0xb5: return D_Extended("BITA ", 3, address);
		case 0xb6: return D_Extended("LDA  ", 3, address);
		case 0xb7: return D_Extended("STA  ", 3, address);
		case 0xb8: return D_Extended("EORA ", 3, address);
		case 0xb9: return D_Extended("ADCA ", 3, address);
		case 0xba: return D_Extended("ORA  ", 3, address);
		case 0xbb: return D_Extended("ADDA ", 3, address);
		case 0xbc: return D_Extended("CMPX ", 3, address);
		case 0xbd: *pstep = 0; return D_Extended("JSR  ", 3, address);
		case 0xbe: return D_Extended("LDX  ", 3, address);
		case 0xbf: return D_Extended("STX  ", 3, address);

		case 0xc0: return D_Immediat("SUBB ", 2, address);
		case 0xc1: return D_Immediat("CMPB ", 2, address);
		case 0xc2: return D_Immediat("SBCB ", 2, address);
		case 0xc3: return D_ImmediatL("ADDD ", 3, address);
		case 0xc4: return D_Immediat("ANDB ", 2, address);
		case 0xc5: return D_Immediat("BITB ", 2, address);
		case 0xc6: return D_Immediat("LDB  ", 2, address);
		// 0xc7 is illegal
		case 0xc8: return D_Immediat("EORB ", 2, address);
		case 0xc9: return D_Immediat("ADCB ", 2, address);
		case 0xca: return D_Immediat("ORB  ", 2, address);
		case 0xcb: return D_Immediat("ADDB ", 2, address);
		case 0xcc: return D_ImmediatL("LDD  ", 3, address);
		// 0xcd is illegal
		case 0xce: return D_ImmediatL("LDU  ", 3, address);
		// 0xcf is illegal

		case 0xd0: return D_Direct("SUBB ", 2, address);
		case 0xd1: return D_Direct("CMPB ", 2, address);
		case 0xd2: return D_Direct("SBCB ", 2, address);
		case 0xd3: return D_Direct("ADDD ", 2, address);
		case 0xd4: return D_Direct("ANDB ", 2, address);
		case 0xd5: return D_Direct("BITB ", 2, address);
		case 0xd6: return D_Direct("LDB  ", 2, address);
		case 0xd7: return D_Direct("STB  ", 2, address);
		case 0xd8: return D_Direct("EORB ", 2, address);
		case 0xd9: return D_Direct("ADCB ", 2, address);
		case 0xda: return D_Direct("ORB  ", 2, address);
		case 0xdb: return D_Direct("ADDB ", 2, address);
		case 0xdc: return D_Direct("LDD  ", 2, address);
		case 0xdd: return D_Direct("STD  ", 2, address);
		case 0xde: return D_Direct("LDU  ", 2, address);
		case 0xdf: return D_Direct("STU  ", 2, address);

		case 0xe0: return D_Indexed("SUBB ", 2, address);
		case 0xe1: return D_Indexed("CMPB ", 2, address);
		case 0xe2: return D_Indexed("SBCB ", 2, address);
		case 0xe3: return D_Indexed("ADDD ", 2, address);
		case 0xe4: return D_Indexed("ANDB ", 2, address);
		case 0xe5: return D_Indexed("BITB ", 2, address);
		case 0xe6: return D_Indexed("LDB  ", 2, address);
		case 0xe7: return D_Indexed("STB  ", 2, address);
		case 0xe8: return D_Indexed("EORB ", 2, address);
		case 0xe9: return D_Indexed("ADCB ", 2, address);
		case 0xea: return D_Indexed("ORB  ", 2, address);
		case 0xeb: return D_Indexed("ADDB ", 2, address);
		case 0xec: return D_Indexed("LDD  ", 2, address);
		case 0xed: return D_Indexed("STD  ", 2, address);
		case 0xee: return D_Indexed("LDU  ", 2, address);
		case 0xef: return D_Indexed("STU  ", 2, address);

		case 0xf0: return D_Extended("SUBB ", 3, address);
		case 0xf1: return D_Extended("CMPB ", 3, address);
		case 0xf2: return D_Extended("SBCA ", 3, address);
		case 0xf3: return D_Extended("ADDD ", 3, address);
		case 0xf4: return D_Extended("ANDB ", 3, address);
		case 0xf5: return D_Extended("BITB ", 3, address);
		case 0xf6: return D_Extended("LDB  ", 3, address);
		case 0xf7: return D_Extended("STB  ", 3, address);
		case 0xf8: return D_Extended("EORB ", 3, address);
		case 0xf9: return D_Extended("ADCB ", 3, address);
		case 0xfa: return D_Extended("ORB  ", 3, address);
		case 0xfb: return D_Extended("ADDB ", 3, address);
		case 0xfc: return D_Extended("LDD  ", 3, address);
		case 0xfd: return D_Extended("STD  ", 3, address);
		case 0xfe: return D_Extended("LDU  ", 3, address);
		case 0xff: return D_Extended("STU  ", 3, address);
		default:   return D_Illegal("", 1, address);
	}  // switch
} // disassemble
