/*
    cpuview.cpp


    flexemu, an MC6809 emulator running FLEX
    Copyright (C) 1997-2000  W. Schwotzer

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <misc1.h>

#ifdef WIN32
#include <string.h>
#include "typedefs.h"
#include "e2video.h"
#include "mc6809.h"
#include "extmem.h"
#include "win32gui.h"
#include "resource.h"
#endif // WIN32

#ifdef X11
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "typedefs.h"
#include "e2video.h"
#include "mc6809.h"
#include "extmem.h"
#include "xgui.h"

#ifdef XTK
#include "xtgui.h"
#include "flexcpu.xbm"
#endif
#endif

#ifdef WIN32
extern Win32Gui *wgui;
#endif

/***************************************/
/* CPU-view with XToolkit interface    */
/***************************************/

#ifdef X11
#ifdef XTK

extern int ok_width;

int radio_data[] = {NO_CHANGE, RUN, STOP, STEP, EXIT, RESET, NEXT};

void XtGui::popup_cpu(void)
{
        XtPopup(cpuframe, XtGrabNone);
        if (okpixmap != None)
                XtVaSetValues(entry24, XtNleftBitmap, okpixmap, NULL);
	cpu_popped_up = 1;
}

void XtGui::popdown_cpu(void)
{
        XtVaSetValues(entry24, XtNleftBitmap, None, NULL);
	XtPopdown(cpuframe);
	cpu_popped_up = 0;
}

void XtGui::toggle_cpu(void)
{
	if (cpu_popped_up)
		popdown_cpu();
	else
		popup_cpu();
}

void XtGui::popup_bp(unsigned int which)
{
	String bpstring;

	which_bp = which;
	bpstring = (String)XtMalloc(6);
	if (!cpu->is_bp_set(which))
		strcpy(bpstring, "");
	else
		sprintf(bpstring, "%04x", cpu->get_bp(which));
	XtVaSetValues(bptext, XtNlength, 6,
		XtNstring, (XtArgVal)bpstring, NULL);
	XtPopup(bpframe, XtGrabExclusive);
	XtFree(bpstring);
}

void XtGui::popdown_bp(void)
{
	String bpstring;
	unsigned int addr;

	XtVaGetValues(bptext, XtNstring, &bpstring, NULL);
	if (sscanf(bpstring, "%x", (int *)&addr) == 1 && addr <= 0xffff)
		cpu->set_bp(which_bp, (Word)addr);
	else
		if (strlen(bpstring) == 0)
		 	cpu->reset_bp(which_bp);
	XtPopdown(bpframe);
	update_cpuview(NO_CHANGE);
}

void XtGui::clear_bp(void)
{
	XtVaSetValues(bptext, XtNstring, (XtArgVal)"", NULL);
}


void XtGui::create_cpuview(Widget parent)
{
	cpu_popped_up = 0;
	cpuframe = XtVaCreatePopupShell("MC6809",
		topLevelShellWidgetClass, parent, NULL);
	cpuform = XtVaCreateManagedWidget("cpuForm",
		formWidgetClass, cpuframe, NULL);
	cputext = XtVaCreateManagedWidget("cpuText",
		asciiTextWidgetClass, cpuform, NULL);
	runbutton = XtVaCreateManagedWidget("runButton",
		toggleWidgetClass, cpuform,
		XtNradioData, (XtArgVal)&radio_data[RUN],
		XtNstate, (XtArgVal)1, NULL);
	stopbutton = XtVaCreateManagedWidget("stopButton",
		toggleWidgetClass, cpuform,
		XtNradioGroup, (XtArgVal)runbutton,
		XtNradioData, (XtArgVal)&radio_data[STOP], NULL);
	stepbutton = XtVaCreateManagedWidget("stepButton",
		toggleWidgetClass, cpuform,
		XtNradioGroup, (XtArgVal)runbutton,
		XtNradioData, (XtArgVal)&radio_data[STEP], NULL);
	nextbutton = XtVaCreateManagedWidget("nextButton",
		toggleWidgetClass, cpuform,
		XtNradioGroup, (XtArgVal)runbutton,
		XtNradioData, (XtArgVal)&radio_data[NEXT], NULL);
	resetbutton = XtVaCreateManagedWidget("resetButton",
		toggleWidgetClass, cpuform,
		XtNradioGroup, (XtArgVal)runbutton,
		XtNradioData, (XtArgVal)&radio_data[RESET], NULL);
	bp1button = XtVaCreateManagedWidget("bp1Button",
		commandWidgetClass, cpuform, NULL);
	bp2button = XtVaCreateManagedWidget("bp2Button",
		commandWidgetClass, cpuform, NULL);
	cpubutton = XtVaCreateManagedWidget("cpuButton",
		commandWidgetClass, cpuform, NULL);

	cpupixmap = XCreateBitmapFromData(XtDisplay(cpuframe),
		RootWindowOfScreen(XtScreen(cpuframe)),
		flexcpu_bits, flexcpu_width, flexcpu_height);
	if (cpupixmap != None)
		XtVaSetValues(cpuframe, XtNiconPixmap, cpupixmap, NULL);

	XtAddCallback(runbutton,   XtNcallback, setCpuRunCallback,
		(XtPointer)this);
	XtAddCallback(stopbutton,  XtNcallback, setCpuStopCallback,
		(XtPointer)this);
	XtAddCallback(stepbutton,  XtNcallback, setCpuStepCallback,
		(XtPointer)this);
	XtAddCallback(nextbutton,  XtNcallback, setCpuNextCallback,
		(XtPointer)this);
	XtAddCallback(resetbutton, XtNcallback, setCpuResetCallback,
		(XtPointer)this);
	XtAddCallback(bp1button,   XtNcallback, popupBp0Callback,
		(XtPointer)this);
	XtAddCallback(bp2button,   XtNcallback, popupBp1Callback,
		(XtPointer)this);
	XtAddCallback(cpubutton,   XtNcallback, popdownCpuCallback,
		(XtPointer)this);
} // create_cpuview

void XtGui::create_bp_dialog(Widget parent)
{
	bpframe = XtVaCreatePopupShell("breakpoint",
		transientShellWidgetClass, parent, NULL);
	bpform = XtVaCreateManagedWidget("bpForm",
		formWidgetClass, bpframe, NULL);
	bptext = XtVaCreateManagedWidget("bpText",
		asciiTextWidgetClass, bpform,
		XtNeditType, (XtArgVal)XawtextEdit, NULL);
	bpbutton = XtVaCreateManagedWidget("bpButton",
		commandWidgetClass, bpform, NULL);
	clearbutton = XtVaCreateManagedWidget("clearButton",
		commandWidgetClass, bpform, NULL);

	XtAddCallback(bpbutton, XtNcallback, popdownBpCallback,
		(XtPointer)this);
	XtAddCallback(clearbutton, XtNcallback, clearBpCallback,
		(XtPointer)this);

	XtInstallAllAccelerators(bpform, bptext);
} // create_bp_dialog

void XtGui::redraw_cpuview_impl(struct s_cpu_status *pstat, int state)
{
	int i;

	i = pstat->s & 7;
	text(5 + 3 * i, 10, "[");
	text(8 + 3 * i, 10, "]");
	XtVaSetValues(cputext, XtNlength, strlen(cpustring)+1,
			XtNstring, cpustring, NULL);
	if (state == RESET_RUN)
		XawToggleSetCurrent(runbutton, (XtPointer)&radio_data[RESET]);
	else
		XawToggleSetCurrent(runbutton, (XtPointer)&radio_data[state]);
	if (state == RUN) {
		XtVaSetValues(stepbutton, XtNsensitive, 0, NULL);
		XtVaSetValues(nextbutton, XtNsensitive, 0, NULL);
		XtVaSetValues(resetbutton, XtNsensitive, 0, NULL);
	} else {
		XtVaSetValues(stepbutton, XtNsensitive, 1, NULL);
		XtVaSetValues(nextbutton, XtNsensitive, 1, NULL);
		XtVaSetValues(resetbutton, XtNsensitive, 1, NULL);
	}
}
#endif // #ifdef XTK

/***********************************/
/* CPU view with Xlib interface    */
/***********************************/

void XGui::process_button_input(XEvent *pevent)
{
	Byte	user_input = NO_CHANGE;
	Word	x, y, i, j;

	if (pevent->xbutton.button == Button1) {
	   // calculate character coordinates
	   x = (pevent->xbutton.x - 4) / cw;
	   y = (pevent->xbutton.y + fs->descent - 4) / ch;
	   if (y == BUTTON_RUN_Y && x >= BUTTON_X && x <= (BUTTON_X + 2))
		  user_input = RUN;
	   if (y == BUTTON_STOP_Y && x >= BUTTON_X && x <= (BUTTON_X + 3))
		user_input = STOP;
	   if (y == BUTTON_STEP_Y && x >= BUTTON_X && x <= (BUTTON_X + 3))
		user_input = STEP;
	   if (y == BUTTON_NEXT_Y && x >= BUTTON_X && x <= (BUTTON_X + 3))
		user_input = NEXT;
	   if (y == BUTTON_RESET_Y && x >= BUTTON_X && x <= (BUTTON_X + 4))
		user_input = RESET;
	   if (y == BUTTON_EXIT_Y && x >= BUTTON_X && x <= (BUTTON_X + 3))
		user_input = EXIT;
	   set_new_state(user_input);
	   for (i=0; i < 2; i++) {
		if (y == (5+i) && x >= 25 && x <= 28) {
			if (!bp_input[i]) {
				for (j=0; j < 2; j++)
					bp_input[j] = 0;
				bp_input[i] = 1;
				cpu->reset_bp(i);
			} else
				bp_input[i] = 0;
			break;
		}
	   }  // for
	   update_cpuview(user_input);
	} // if
} // process_button_input

void XGui::create_cpuview(void)
{
	char		      **ret;
	int			count;
	XSetWindowAttributes	attrs;
	XGCValues		gcv;
	XSizeHints		sh;
	XClassHint		xch;
	XEvent			event;

	// get a fixed font
	ret = XListFonts(display, "*-fixed-*-*-*-*-10-*", 1, &count);
	if (ret) {
		fs = XLoadQueryFont(display, ret[0]);
		if (!fs) {
			fs = XLoadQueryFont(display, "fixed");
		}
		XFreeFontNames(ret);
	} else {
		fs = XLoadQueryFont(display, "fixed");
	}
	cw = XTextWidth(fs, " ", 1);
	ch = (fs->ascent + fs->descent);


	// Setup default window attributes
	attrs.background_pixel		= BlackPixelOfScreen(screen);
	attrs.do_not_propagate_mask	= 0;
//	attrs.event_mask =	StructureNotifyMask |
	attrs.event_mask		= ButtonPressMask | ExposureMask |
					  KeyPressMask;

	// Create the window
	ww = cw * 38 + 8;
	wh = ch * 14 + 8;
	cpu_window = XCreateWindow(display, RootWindowOfScreen(screen),
		0, 0, ww, wh, 2, CopyFromParent, CopyFromParent, CopyFromParent,
		CWEventMask | CWBackPixel | CWDontPropagate, &attrs);

	// Create a pixmap
	pix = XCreatePixmap(display, RootWindowOfScreen(screen),
		ww, wh, DefaultDepthOfScreen(screen));

	// Create a GC to go with it
	gcv.foreground = WhitePixelOfScreen(screen);
	gcv.background = BlackPixelOfScreen(screen);
	gcv.font = fs->fid;
	cpu_gc = XCreateGC(display, pix, GCForeground | GCBackground | GCFont,
		&gcv);

	// And an inverted GC
	gcv.foreground = BlackPixelOfScreen(screen);
	gcv.background = WhitePixelOfScreen(screen);
	inv_gc = XCreateGC(display, pix, GCForeground | GCBackground | GCFont,
		&gcv);

	// Set window size
	sh.width = sh.max_width = ww;
	sh.height = sh.max_height = wh;
	sh.flags = PSize | PMaxSize;
	XSetStandardProperties(display, cpu_window, "MC6809", "mc6809",
		None, NULL, 0, &sh);

	// Set window class
	xch.res_name = "mc6809";
	xch.res_class = "MC6809";
	XSetClassHint(display, cpu_window, &xch);

	// Put the window on the screen
	XMapWindow(display, cpu_window);
	// wait until window is visible
	XWindowEvent(display, cpu_window, ExposureMask, &event);
	
	update_cpuview(NO_CHANGE); // now display all
} // create_cpuview

void XGui::clear_cpuview(void)
{
	// clear the pixmap
	XFillRectangle(display, pix, inv_gc, 0, 0, ww, wh);
}

/*ARGSUSED*/
void XGui::text(int x, int y, const char *str, int rev /* = 0 */)
{
	x = x * cw + 4;
	y = (y + 1) * ch + 4 - fs->descent;
	XDrawImageString(display, pix, rev ? inv_gc : cpu_gc, x, y, str,
		strlen(str));
}

void XGui::redraw_cpuview_impl(struct s_cpu_status *pstat, int state)
{
	int i;

	i = pstat->s & 7;
	text(BUTTON_X, BUTTON_RUN_Y,   "RUN",   state == RUN);
	text(BUTTON_X, BUTTON_STOP_Y,  "STOP",  state == STOP);
	text(BUTTON_X, BUTTON_STEP_Y,  "STEP",  state == STEP);
	text(BUTTON_X, BUTTON_NEXT_Y,  "NEXT",  state == NEXT);
	text(BUTTON_X, BUTTON_RESET_Y, "RESET", state == RESET);
	text(BUTTON_X, BUTTON_EXIT_Y,  "EXIT",  state == EXIT);
	text(6 + 3 * i, 10, hexstr(READ(pstat->s)), 1);
	text(30 + i, 10, ascchr(READ(pstat->s)), 1);
	// optional print a message instead of stack
	if (message != NULL)
		cpu_message(&message);
	XCopyArea(display, pix, cpu_window, cpu_gc, 0, 0, ww, wh, 0, 0);
}

void XGui::process_bp_input(KeySym keysym, int which)
{
	unsigned int digit;
	char ch;

	switch (keysym) {
		case XK_0 :
		case XK_1 :
		case XK_2 :
		case XK_3 :
		case XK_4 :
		case XK_5 :
		case XK_6 :
		case XK_7 :
		case XK_8 :
		case XK_9 :
		case XK_a :
		case XK_b :
		case XK_c :
		case XK_d :
		case XK_e :
		case XK_f :
			ch = (char)keysym;
			digit = isdigit(ch) ? ch - '0' : ch - 'a' + 10;
			if (!cpu->is_bp_set(which))
				cpu->set_bp(which, 0);
			if (cpu->get_bp(which) < 0x1000)
				cpu->set_bp(which, 16 * cpu->get_bp(which) + digit);
			break;
		case XK_Delete: cpu->reset_bp(which);
		case XK_Return: bp_input[which] = 0;
			break;
	} // switch
}

void XGui::popup_message(char *pmessage)
{
	// will be printed within next update_cpuview()

	message = pmessage;
}

void XGui::cpu_message(char **ppmessage)
{
	int i, linenr;
	char *p, line[CPU_LINE_SIZE];

	sprintf(line, "%*.*s", CPU_LINE_SIZE-1, CPU_LINE_SIZE-1, "");
	for (linenr = 8; linenr <= 13; linenr++)
		text(0, linenr, line);
	p = *ppmessage;
	linenr = 8;
	do {
		i = 0;
		// copy first up to CPU_LINE_SIZE characters
		while (*p != '\0' && *p != '\n' && i < CPU_LINE_SIZE-1)
			line[i++] = *(p++);
		line[i] = '\0';
		text(0, linenr++, line);
		// skip to begin of next line
		while (*p != '\0' && *p != '\n')
			p++;
		if (*p == '\n')
			p++;
	} while (*p != '\0' && linenr <= 13);
	*ppmessage = NULL;
}
#endif // X11


/***************************************/
/* CPU-view with Win32 interface       */
/***************************************/

#ifdef WIN32

extern int ok_width;

int radio_data[] = {NO_CHANGE, RUN, STOP, STEP, EXIT, RESET, NEXT};

void Win32Gui::popup_cpu(void)
{
	ShowWindow(cpuform, SW_SHOW);
	cpu_popped_up = 1;
}

void Win32Gui::popdown_cpu(void)
{
	ShowWindow(cpuform, SW_HIDE);
	cpu_popped_up = 0;
}

void Win32Gui::toggle_cpu(void)
{
	if (cpu_popped_up)
		popdown_cpu();
	else
		popup_cpu();
}

LOGFONT *Win32Gui::getLogFontStruct(HDC hdc, int pointSize)
{
	static LOGFONT lf;

	lf.lfHeight		= -MulDiv(pointSize,
				  GetDeviceCaps(hdc, LOGPIXELSY), 72);
	lf.lfWidth		= 0;
	lf.lfEscapement		= 0;
	lf.lfOrientation	= 0;
	lf.lfWeight		= FW_DONTCARE;
	lf.lfItalic		= 0;
	lf.lfUnderline		= 0;
	lf.lfStrikeOut		= 0;
	lf.lfCharSet		= ANSI_CHARSET;
	lf.lfOutPrecision	= OUT_TT_PRECIS;
	lf.lfClipPrecision	= CLIP_DEFAULT_PRECIS;
	lf.lfQuality		= DEFAULT_QUALITY;
	lf.lfPitchAndFamily	= FIXED_PITCH | FF_DONTCARE;
	lf.lfFaceName[0]	= '\0';
	return &lf;
} // getLogFontStruct

void Win32Gui::create_cpuview(HWND parent, struct sGuiOptions *pOptions)
{
	DWORD	style;
	HFONT	hFontFixed;
	HDC	hdc;
	RECT	rect;

	cpu_popped_up = 0;
	rect.left	= (long)0;
	rect.right	= (long)CPUW;
	rect.top	= (long)0;
	rect.bottom	= (long)CPUH;
	AdjustWindowRectEx(&rect, WS_OVERLAPPEDWINDOW, TRUE, 0);
	cpuform = CreateWindowEx (0,
		"Mc6809",
		"MC6809",
		WS_OVERLAPPEDWINDOW,
		CW_USEDEFAULT,
		0,
		rect.right - rect.left,
		rect.bottom - rect.top,
		parent,
		NULL, 
		pOptions->hInstance,
		NULL);
	style = WS_CHILD | WS_VISIBLE;
	nextbutton = CreateWindow("button", "&Next", style, BL, BT, BW, BH,
		cpuform, (HMENU)IDP_NEXT, pOptions->hInstance, NULL);
	stepbutton = CreateWindow("button", "&Step", style, BL, BT+26, BW, BH,
		cpuform, (HMENU)IDP_STEP, pOptions->hInstance, NULL);
	runbutton = CreateWindow("button", "&Run", style, BL, BT+52, BW, BH,
		cpuform, (HMENU)IDP_RUN, pOptions->hInstance, NULL);
	stopbutton = CreateWindow("button", "S&top", style, BL, BT+78, BW, BH,
		cpuform, (HMENU)IDP_STOP, pOptions->hInstance, NULL);
	resetbutton = CreateWindow("button", "R&eset", style, BL, BT+104, BW,
		BH, cpuform, (HMENU)IDP_RESET, pOptions->hInstance, NULL);
	bp1button = CreateWindow("button", "Bp&1", style, BL, BT+130, BW, BH,
		cpuform, (HMENU)IDP_BP1, pOptions->hInstance, NULL);
	bp2button = CreateWindow("button", "Bp&2", style, BL, BT+156, BW, BH,
		cpuform, (HMENU)IDP_BP2, pOptions->hInstance, NULL);
	style |= BS_DEFPUSHBUTTON;
	cpubutton = CreateWindow("button", "&OK", style, BL, BT+182, BW, BH,
		cpuform, (HMENU)IDP_CPU, pOptions->hInstance, NULL);
	// change the font to a fixed font
	hdc = GetDC(cpuform);
	hFontFixed = CreateFontIndirect(getLogFontStruct(hdc, 8));
	SetWindowFont(cpuform,     hFontFixed, TRUE);
	SetWindowFont(nextbutton,  hFontFixed, TRUE);
	SetWindowFont(stepbutton,  hFontFixed, TRUE);
	SetWindowFont(runbutton,   hFontFixed, TRUE);
	SetWindowFont(stopbutton,  hFontFixed, TRUE);
	SetWindowFont(resetbutton, hFontFixed, TRUE);
	SetWindowFont(bp1button,   hFontFixed, TRUE);
	SetWindowFont(bp2button,   hFontFixed, TRUE);
	SetWindowFont(cpubutton,   hFontFixed, TRUE);
	// Accelerators
	hAccel = LoadAccelerators(hInstance, MAKEINTRESOURCE(IDR_CPU_ACCELERATOR));
	hAccelHwnd = cpuform;
}  // create_cpuview

BOOL Win32Gui::onCpuCommand(HWND hwnd, int cmd)
{
	switch (cmd) {
		// cpuview button controls
		case IDP_NEXT:	set_new_state(NEXT);		break;
		case IDP_STEP:	set_new_state(STEP);		break;
		case IDP_STOP:	set_new_state(STOP);		break;
		case IDP_RUN:	set_new_state(RUN);		break;
		case IDP_RESET:	set_new_state(RESET);		break;
		case IDP_BP1:	popup_bp(0, hwnd);		break;
		case IDP_BP2:	popup_bp(1, hwnd);		break;
		case IDP_CPU:	popdown_cpu();			break;
		default:	return FALSE;
	}
	return TRUE;
} // onCpuCommand


BOOL Win32Gui::onCpuPaint(HWND hwnd)
{
	redraw_cpuview(state_to_show);
	return TRUE;
}

BOOL Win32Gui::onCpuClose(HWND hwnd)
{
	popdown_cpu();
	return TRUE;
}

LRESULT CALLBACK cpuWindowWndProc(
	HWND hwnd,
	UINT message,
	WPARAM wParam,
	LPARAM lParam)
{
	switch(message) {
		case WM_PAINT:	return wgui->onCpuPaint(hwnd);
		case WM_COMMAND:return wgui->onCpuCommand(hwnd,LOWORD(wParam));
		case WM_CLOSE:	return wgui->onCpuClose(hwnd);
		default:	return DefWindowProc(hwnd, message, wParam, lParam);
	} // switch
	return FALSE;
} // cpuWindowWndProc

void Win32Gui::redraw_cpuview_impl(struct s_cpu_status *pstat,
	int state)
{
	int		i;
	HDC		hdc;
	PAINTSTRUCT	ps;
	HWND		hwnd;
	HFONT		hFontFixed, hPrevFont;
	RECT		rect;

	hwnd = cpuform;
	i = pstat->s & 7;
	text(5 + 3 * i, 10, "[");
	text(8 + 3 * i, 10, "]");
	rect.left	= CPUFRAME;
	rect.top	= CPUFRAME;
	rect.right	= BL-CPUFRAME;
	rect.bottom	= CPUH-CPUFRAME;
	hdc = BeginPaint(hwnd, &ps);
	hFontFixed = CreateFontIndirect(getLogFontStruct(hdc, 8));
	hPrevFont = SelectFont(hdc, hFontFixed);
	DrawText(hdc, (char *)&cpustring, strlen(cpustring),
		&rect, DT_LEFT | DT_NOPREFIX);
	SelectFont(hdc, hPrevFont);
	MoveToEx(hdc, rect.left-2, rect.top-2, NULL);
	LineTo(hdc, rect.right+2, rect.top-2);
	LineTo(hdc, rect.right+2, rect.bottom+2);
	LineTo(hdc, rect.left-2, rect.bottom+2);
	LineTo(hdc, rect.left-2, rect.top-2);
	EndPaint(hwnd, &ps);
} // redraw_cpuview_impl

void Win32Gui::update_cpuview(Byte x_state_to_show)
{
	state_to_show = x_state_to_show;
	RedrawWindow(cpuform, NULL, NULL, RDW_INVALIDATE);
}

BOOL Win32Gui::onBpPaint(HWND hwnd)
{
//	switch (cmd) {
//	}
	return FALSE;
} // onBpPaint

BOOL Win32Gui::onBpCommand(HWND hwnd, int cmd)
{
	switch (cmd) {
		// BP-Dialog button controls
		case ID_BP_OK:		popdown_bp(hwnd); break;
		case ID_BP_CLEAR:	clear_bp(hwnd); break;
		default:		return FALSE;
	}
	return TRUE;
} // onBpCommand

BOOL Win32Gui::onBpInit(HWND hwnd)
{
	char bpstring[16];
	HWND hEdit;

	sprintf((char *)bpstring, "Breakpoint %d", which_bp + 1);
	SetWindowText(hwnd, bpstring);
	if (!cpu->is_bp_set(which_bp))
		strcpy(bpstring, "");
	else
		sprintf(bpstring, "%04x", cpu->get_bp(which_bp));
	SetDlgItemText(hwnd, IDC_BP_ADDR, bpstring);
	if ((hEdit = GetDlgItem(hwnd, IDC_BP_ADDR))) {
		// Select contents of edit control
		SendMessage(hEdit, EM_SETSEL, 0, -1);
	}
	return TRUE;
}

BOOL Win32Gui::onBpClose(HWND hwnd)
{
	EndDialog(hwnd, 0);
	return TRUE;
}


BOOL CALLBACK bpDialogProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch(message) {
		case WM_INITDIALOG:
			return wgui->onBpInit(hwnd);
		case WM_PAINT:
			return wgui->onBpPaint(hwnd);
		case WM_COMMAND:
			return wgui->onBpCommand(hwnd,LOWORD(wParam));
		case WM_CLOSE:
			return wgui->onBpClose(hwnd);
	} // switch
	return FALSE;
} // bpDialogProc

void Win32Gui::popup_bp(unsigned int which, HWND hwnd)
{
	int  res;

	which_bp = which;
	res = DialogBox(hInstance, MAKEINTRESOURCE(IDD_BP_DIALOG), hwnd, CALLBACKCAST bpDialogProc);
}

void Win32Gui::popdown_bp(HWND hwnd)
{
	char		bpstring[16];
	unsigned int	addr;

	GetDlgItemText(hwnd, IDC_BP_ADDR, bpstring, 16);
	if (sscanf(bpstring, "%x", (int *)&addr) == 1 && addr <= 0xffff)
		cpu->set_bp(which_bp, (Word)addr);
	else
		if (strlen(bpstring) == 0)
		 	cpu->reset_bp(which_bp);
	EndDialog(hwnd, 0);
	update_cpuview(NO_CHANGE);
}

void Win32Gui::clear_bp(HWND hwnd)
{
	SetDlgItemText(hwnd, IDC_BP_ADDR, "");
}

#endif // WIN32
