/*
    bstring.cpp


    FLEXplorer, An explorer for any FLEX file or disk container
    Copyright (C) 1998-2000  W. Schwotzer

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <misc1.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <ctype.h>

#include "bstring.h"

#ifndef HAVE_VPRINTF
#error function vsprintf not available on this system
#endif

#if defined (WIN32) && !defined(_MSC_VER)
extern int vsnprintf(char *buf, size_t size, const char *format, va_list args);
#endif

void BString::initForChar(const char c)
{
	sz = 2;
	str = new char[sz];
	str[0] = c;
	str[1] = '\0';
}

void BString::init(const char *s, int maxlen)
{
	if (maxlen == 0) {
		initForChar('\0');
		return;
	}
	if (maxlen < 0)
		sz = strlen(s) + 1;
	else
		sz = maxlen + 1;
	if (sz < 2)
		sz = 2;
	str = new char[sz];
	strncpy(str, s, sz-1);
	str[sz-1] = '\0';
}

BString::BString(void)
{
	initForChar('\0');
}

BString::BString(const char c)
{
	initForChar(c);
}

BString::BString(const char *s, int maxlen)
{
	init(s, maxlen);
}

BString::BString(const BString& s, int maxlen)
{
	init(s.chars(), maxlen);
}

BString::~BString(void)
{
	delete [] str;
	sz = 0;
}

void FScat(const char *s1, const char *s2, BString& s3)
{
	int size;

	size = strlen(s1) + strlen(s2) + 1;
	char *newS = new char[size];
	strcpy(newS, s1);
	strcat(newS, s2);
	s3 = BString(newS);
	delete [] newS;
}

void BString::FSadd(const char *s1)
{
	int size = length() + strlen(s1) + 1;
	if (size <= allocation())
		strcat(str, s1);
	else {
		char *newS = new char[size];
		strcpy(newS, str);
		strcat(newS, s1);
		delete [] str;
		str = newS;
		sz = size;
	}
}

int BString::index(const char *s, int startpos) const
{
	int i1, i2, l1, l2;
	const char *p1, *p2;

	p1 = chars();
	p2 = s;
	l1 = strlen(p1);
	l2 = strlen(p2);
	for (i1 = startpos; i1 < l1; i1++) {
		for (i2 = 0; i2 < l2; i2++) {
			if (*(p1+i1+i2) != *(p2+i2))
				break;
		};
		if (i2 >= l2)
			return i1;
	}
	return -1;
}

void BString::alloc(int s)
{
	if (s > sz) {
		sz = s;
		char *newS = new char[s];
		strcpy(newS, str);
		delete [] str;
		str = newS;
	}
}
void BString::alloc(const char *s)
{
	int size;

	size = strlen(s) + 1;
	if (size > sz) {
		sz = size;
		char *newS = new char[size];
		strcpy(newS, s);
		delete [] str;
		str = newS;
	} else
		strcpy(str, s);
}

void BString::reverse(void)
{
	int i;
	char *p1, c;

	i = 1;
	if (this->length() % 2 == 1) {
		// odd number of characters
		p1 = &str[0] + this->length() / 2;
		while(*(p1+i) != '\0') {
			c = *(p1+i); *(p1+i) = *(p1-i); *(p1-i) = c; i++;
		};
	} else {
		// even number of characters
		p1 = &str[0] + this->length() / 2 - 1;
		while(*(p1+i) != '\0') {
			c = *(p1+i); *(p1+i) = *(p1-i+1); *(p1-i+1) = c; i++;
		};
	}

} 

void BString::upcase(void)
{
	char *p;

	p = &str[0];
	while (*p != '\0')
		*(p++) = toupper(*p);
} 

void BString::downcase(void)
{
	char *p;
	
	p = &str[0];
	while (*p != '\0')
		*(p++) = tolower(*p);
} 

char BString::firstchar(void) const
{
	return str[0];
}

char BString::lastchar(void) const
{
	return str[strlen(str)-1];
}

void BString::at(unsigned int pos, int len, BString& s)
{
	if (pos >= strlen(str))
		s = "";
	else
		s = BString(&str[pos], len);
}

int BString::printf(const char *format, ...)
{
	int                     size, res;
	char            *newS;
	va_list         arg_ptr;

	size = 80;
	do {
		size *= 2;
		if (size > sz) {
			newS = new char[size];
			delete [] str;
			str = newS;
			sz = size;
		}
		va_start(arg_ptr, format);
#ifdef WIN32
#ifdef _MSC_VER
		res = vsnprintf(str, sz, format, arg_ptr);
#else
		vsprintf(str, format, arg_ptr);
		res = 1;
#endif
#else
		res = vsnprintf(str, sz, format, arg_ptr);
#endif
		va_end(arg_ptr);
		if (res >= 0 && res > sz)
			res = -1;
	} while (res < 0);
	return length();
}

BString& BString::operator += (const int i)
{
	char s[16];

	sprintf((char *)s, "%d", i);
	FSadd((char *)s);
	return *this;
}

BString& BString::operator += (const unsigned int ui)
{
	char s[16];

	sprintf((char *)s, "%ud", ui);
	FSadd((char *)s);
	return *this;
}

void BString::replaceall(const char oldC, const char newC)
{
	if (newC == 0)
		return;
	for (int i = 0; i < sz && str[i]; i++)
		if (str[i] == oldC)
			str[i] = newC;
}

int BString::comparenocase(const BString &s)
{
	return stricmp(str, s.str);
}

bool BString::operator <  (const BString &s) const
{
	return strcmp(str, s.str)  < 0;
}

bool BString::operator <= (const BString &s) const
{
	return strcmp(str, s.str)  <= 0;
}

bool BString::operator == (const BString &s) const
{
	return strcmp(str, s.str)  == 0;
}

bool BString::operator >= (const BString &s) const
{
	return strcmp(str, s.str)  >= 0;
}

bool BString::operator >  (const BString &s) const
{
	return strcmp(str, s.str)  > 0;
}

