/*
    absgui.cc: abstract graphical user interface


    flexemu, an MC6809 emulator running FLEX
    Copyright (C) 1997-2000  W. Schwotzer

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <misc1.h>
#include "pia1.h"
#include "absgui.h"
#include "extmem.h"

Word internal_wct[2][8] = {
        {0x0300, 0x0C00, 0x3000, 0xC000,
	 0x0003, 0x000C, 0x0030, 0x00C0},
	{0x0003, 0x000C, 0x0030, 0x00C0,
        0x0300, 0x0C00, 0x3000, 0xC000}
};

/* ARGSUSED */
void AbstractGui::update_block(int block_number)
{
}

void AbstractGui::initialize(struct sGuiOptions *pOptions)
{
	switch_sp = 0;
	program_name = "";
	if (pOptions != NULL) {
		program_name	= pOptions->argv[0];
		switch_sp	= pOptions->switch_sp;
		guiXSize	= pOptions->guiXSize;
		guiYSize	= pOptions->guiYSize;
	}
	
}

void AbstractGui::initialize_word_conv_table(int lsb_first)
{
        int i, j;
	int type;

	type = lsb_first ? 1 : 0;
	for (i= 0; i < 256; i++) {
		word_conv_tab[i] = 0;
		for (j = 0; j < 8; j++) {
			if (i & (1 << j))
				word_conv_tab[i] |= internal_wct[type][j] ;
		} // for
	} // for
} // initialize_word_conv_table

void AbstractGui::event_queue(void)
{
	if (io->poll() && io->pia1 != NULL)
		io->pia1->activeTransition(CA1);
}

/* ARGSUSED */
void AbstractGui::update_cpuview(Byte state)
{
	redraw_cpuview(state);
}

/* ARGSUSED */
void AbstractGui::redraw_cpuview(Byte state)
{
        int i;

        clear_cpuview();
        text( 0, 0, "Cycl:");
        text( 0, 1, "Inst:");
        text( 0, 2, "  PC:");
        text( 0, 3, "   S:");
        text( 0, 4, "   U:");
        text( 0, 5, "   X:");
        text( 0, 6, "   Y:");

        text(15, 2, "EFHINZVC");
        text(10, 3, "  CC:");
        text(10, 4, "  DP:");
        text(10, 5, "   A:");
        text(10, 6, "   B:");
        text(19, 5, "  bp1:");
        text(19, 6, "  bp2:");
        text(20, 0, "Freq:");
        text(31, 0, "MHz");

        for (i = 0; i < 6; ++i) {
                text(4, i + 8, ":");
	}
	redraw_cpuview_contents(state);
}

/* ARGSUSED */
void AbstractGui::redraw_cpuview_contents(Byte state_to_show)
{
        static char             tmp[40];
        Byte                    i, j, step, state;
        char                    *pmnem_buf, *pbuffer;
        struct s_cpu_status     stat;

        cpu->get_status(&stat);
        text( 6, 2, hexstr(stat.pc));
        text( 6, 3, hexstr(stat.s));
        text( 6, 4, hexstr(stat.u));
        text( 6, 5, hexstr(stat.x));
	text( 6, 6, hexstr(stat.y));
	text(15, 3, binstr(stat.cc));
	text(15, 4, hexstr(stat.dp));
	text(15, 5, hexstr(stat.a));
	text(15, 6, hexstr(stat.b));
	sprintf(tmp, " %10lu", stat.cycles);
	text(6,  0, tmp);
	sprintf(tmp, "%5.2f", stat.freq > 99.999 ? 99.999 : stat.freq);
	text(25,  0, tmp);
	text(6, 1, "                        "); // first clear area
        if (cpu->disassemble(stat.pc, &step, &pbuffer, &pmnem_buf))
        	text(6, 1, pmnem_buf);
	else
	        text(6, 1, "sorry, no disassembler installed");
	if (state_to_show == NO_CHANGE)
	        state = stat.state;
	else
	        state = state_to_show;
	for (i=0; i < 2; i++) {
                if (!cpu->is_bp_set(i))
	                text(25, 5+i, "    ", bp_input[i]);
	        else
		        text(25, 5+i,
				hexstr((Word)cpu->get_bp(i)),
				        bp_input[i]);
	}  // for

	Word            stk = ((stat.s >> 3) << 3) - 16;

        for (i = 0; i < 6; ++i) {
		text(0, i + 8, hexstr(stk));

	        tmp[0] = ' ';
	        tmp[1] = '\0';
                for (j = 0; j < 8; ++j) {
	                Byte            ch = READ(stk + j);
                        strcat(tmp, hexstr(ch));
                        strcat(tmp, " ");
                } // for

		for (j = 0; j < 8; ++j) {
        		Byte            ch = READ(stk + j);
	        	strcat(tmp, ascchr(ch));
		} // for
		text(5, i + 8, tmp);
		stk+= 8;
	} // for
	redraw_cpuview_impl(&stat, state);
}

/* ARGSUSED */
void AbstractGui::redraw_cpuview_impl(struct s_cpu_status *pstat, int state)
{
}

/*ARGSUSED*/
void AbstractGui::text(int x, int y, const char *str, int rev)
{
        strncpy(&cpustring[CPU_LINE_SIZE*y + x], str, strlen(str));
}

void AbstractGui::clear_cpuview(void)
{
        int i;

        for (i = 1; i <= CPU_LINES*CPU_LINE_SIZE; i++)
                cpustring[i-1] = (i % CPU_LINE_SIZE == 0) ? '\n' : ' ';
}

/* ARGSUSED */
void AbstractGui::update_1_second(Byte state_to_show)
{
	if (state_to_show != NO_CPUVIEW_UPDATE)
		update_cpuview(state_to_show);
}

void AbstractGui::set_exit(void)
{
	exit_flag = 1;
}

void AbstractGui::update(void)
{
	static int event_loop_counter = 0;
	static int display_block = 0;

	update_block(display_block);
	display_block = ++display_block % YBLOCKS;
	if (event_loop_counter++ == 25) {
		event_loop_counter = 0;
		event_queue();
	} // if
}

void AbstractGui::set_new_state(Byte user_input)
{
	if (user_input != NO_CHANGE)
		cpu->set_new_state(user_input);
}

void AbstractGui::set_bell(int percent)
{
}

void AbstractGui::output_to_terminal(void)
{
	if (io->is_terminal_supported() && switch_sp)
		cpu->write(SERPAR, 0xff);
}

void AbstractGui::output_to_graphic(void)
{
	if (switch_sp)
		cpu->write(SERPAR, 0x00);
}

/* ARGSUSED */
void AbstractGui::popup_message(char *message)
{
}

int AbstractGui::gui_type(void)
{
	return -1;
}

AbstractGui::AbstractGui(
	Mc6809* x_cpu = NULL,
	Inout*  x_io = NULL,
	E2video* x_video = NULL,
	struct sGuiOptions *x_pOptions = NULL)
{
	cpu		= x_cpu;
	io		= x_io;
	e2video		= x_video;
	pOptions	= x_pOptions;
	exit_flag	= 0;

	if (e2video != NULL)
		e2video->resetIo();
} 

AbstractGui::~AbstractGui(void)
{
}

