/* Compile with -ml the Large Memory model && -P Cplusplus*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include "conio.h"
#include "dos.h"

#define flag_CF (1 << 0)
#define flag_ZF (1 << 6)

/* global variables */
static FILE *file = NULL;
static void (*read_key)(int *scanptr, int *asciiptr) = NULL;

#define KBD_STATUS_PORT 0x64
#define KBD_COMMAND_PORT 0x64
#define KBD_DATA_PORT 0x60
#define KBD_ACK_PORT 0x61

#define KBD_COMMAND_ENABLE  0xAE
#define KBD_COMMAND_DISABLE 0xAD
#define KBD_COMMAND_READ_OUTPUT_PORT  0xD0
#define KBD_COMMAND_READ_INPUT_PORT   0xC0
#define KBD_COMMAND_WRITE_OUTPUT_PORT 0xD1
#define KBD_COMMAND_READ_TEST_INPUTS  0xE0

#define HDW_PAUSE_PORT 0xE1

#define PIC_PORT 0x20
#define PIC_ACK  0x20


int test_key_extended(void);
void read_key_extended(int *scanptr, int *asciiptr);

typedef unsigned long Bit32u;
typedef Bit32u key_t;

typedef void interrupt (* intfunc_t)(...);

static intfunc_t *const inttable = MK_FP(0,0);

static intfunc_t old_int9;
static intfunc_t old_int1b;

#define QUEUE_SIZE 16000 /* guard against overflow :) */
static key_t queue[QUEUE_SIZE];
static unsigned volatile queue_head = 0;
static unsigned volatile queue_tail = 0;

#define BUF_SIZE 65000
#define BUF_PADDING 256
static unsigned char * buffer;
static unsigned buf_pos =0;


/* Note: We carefully avoid making calls to dos function
 * as they might peek with function int16 01 at the keyboard buffer.
 * This function will remove any extended keys from the keyboard buffer
 * causing grief in high speed regression tests.
 */

static void init_buffer(void)
{
	buffer = (unsigned char *)malloc(BUF_SIZE);
	if (buffer == 0) {
		fprintf(stderr, "Cannot allocate buffer\n");
		exit(3);
	}
	buf_pos = 0;
}

static void flush_buffer(FILE *f)
{
	size_t written = 0;
	size_t rc;
	do {
		rc = fwrite(buffer + written, 1, buf_pos -written, f);
		if (rc > 0) {
			written += rc;
		}
	} while (written < buf_pos);
	buf_pos = 0;
}

static void bprintf(FILE *file, char *format, ...)
{
	char *start;
	va_list args;

	/* handle the worst case */
	if ((buf_pos + BUF_PADDING) > BUF_SIZE) {
		flush_buffer(file);
	}
	start = buffer + buf_pos;
	va_start(args, format);
	vsprintf(start, format, args);
	va_end(args);
	
	buf_pos += strlen(start);
}


static void cli(void)
{
	asm cli;
}

static void sti(void)
{
	asm sti;
}

static inline intfunc_t get_int_func(unsigned char num)
{
	return inttable[num];
}

static inline void set_int_func(unsigned char num, intfunc_t func)
{
	cli();
	inttable[num] = func;
	sti();
}

static void interrupt int1b(void)
{
	return; /* ignore Ctrl-Break */
}
static void hook_int1b(void)
{
	old_int1b = get_int_func(0x1b);
	set_int_func(0x1b, (intfunc_t)int1b);
}
static void unhook_int1b(void)
{
	set_int_func(0x1b, old_int1b);
}

static void interrupt int9(void)
{
	static key_t last_scan = 0;
	static key_t partial_scan = 0;
	key_t scan;
	scan = inp(KBD_DATA_PORT);
	if ((scan == 0xe0) || (scan == 0xe1)) {
		partial_scan = scan;
	} else if (partial_scan == 0xe1) {
		partial_scan = 0xe100 | scan;
	} else {
		scan = (partial_scan  << 8) | scan;
		partial_scan = 0;
		if (last_scan != scan) { /* supress auto repeats */
			queue[queue_head] = scan;
			queue_head = (queue_head +1)%QUEUE_SIZE;
		}
		last_scan = scan;
	}
	_chain_intr(old_int9);
}

static void hook_int9(void)
{
	old_int9 = get_int_func(9);
	set_int_func(9, (intfunc_t)int9);
}
static void unhook_int9(void)
{
	set_int_func(9, old_int9);
}

void read_key_direct(int *scanptr, int *asciiptr)
{
	key_t scan = 0;

	do {
		cli();
		if (queue_head != queue_tail) {
			scan = queue[queue_tail];
			queue_tail = (queue_tail +1)%QUEUE_SIZE;
		}
		sti();
	if (test_key_extended()) {
		read_key_extended(NULL, NULL);
	}
	} while (scan == 0);

	cprintf("ps: scan = %06lx\r\n", scan);
	bprintf(file, "ps: scan = %06lx\n", scan);

	if (scanptr) {
		*scanptr = scan;
	}
	if (asciiptr) {
		*asciiptr = 0x1b; /* so pressing escape will exit */
	}
}

int test_key_extended(void)
{
	int not_ready;
	union REGS regs;
	regs.h.ah = 0x11;
	regs.x.flags = 0;
	int86(0x16, &regs, &regs);

	not_ready = (regs.x.flags & flag_ZF) != 0;
	return !not_ready;
}


void read_key_extended(int *scanptr, int *asciiptr)
{
	int scan;
	int ascii;
	union REGS regs;

	regs.h.ah = 0x10;
	regs.x.flags = 0;
	int86(0x16, &regs, &regs);
	scan = regs.h.ah;
	ascii = regs.h.al;

	if (scanptr) {
		cprintf("px: scan = %02x ascii = %02x\r\n",
		       scan, ascii);
		bprintf(file, "px: scan = %02x ascii = %02x\n",
			scan, ascii);
		*scanptr = scan;
	}
	if (asciiptr) {
		*asciiptr = ascii;
	}
}


void read_key_normal(int *scanptr, int *asciiptr)
{
	int scan;
	int ascii;
	union REGS regs;

	regs.h.ah = 0x00;
	regs.x.flags = 0;
	int86(0x16, &regs, &regs);
	scan = regs.h.ah;
	ascii = regs.h.al;

	cprintf("pn: scan = %02x ascii = %02x\r\n",
				scan, ascii);
	bprintf(file, "pn: scan = %02x ascii = %02x\n",
				scan, ascii);

	if (scanptr) {
		*scanptr = scan;
	}
	if (asciiptr) {
		*asciiptr = ascii;
	}
}


int c_break(void)
{
	return 1; /* don't abort on ctrl-break */
}

void parse_args(int argc, char *argv[])
{
	char *filename =0;
	int i;
	int error = 0;

	for(i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch(argv[i][1]) {
			case 'n':
				read_key = read_key_normal;
				break;
			case 'e':
				read_key = read_key_extended;
				break;
			case 's':
				read_key = read_key_direct;
				break;
			default:
				error = 1;
				break;
			}
		} else if (!filename) {
			filename = argv[i];

		} else {
			error = 1;
		}
		if (error) {
			break;
		}
	}
	if (!read_key) {
		error = 1;
	}
	if (error) {
		fprintf(stdout, "usage: %s -[sne] filename\n",
			argv[0]);
		exit(1);
	}
	file = fopen(filename, "w+b");
	if (file == NULL) {
		fprintf(stdout, "error couldn't open file %s\n",
			filename);
		exit(2);
	}
	if (read_key == read_key_direct) {
		hook_int9();
	}
}

int main(int argc, char *argv[])
{
	int escs=0;

	directvideo=1;
	init_buffer();
	parse_args(argc, argv);
	hook_int1b();
	ctrlbrk(c_break);
	fprintf(stdout, "Press Esc twice to exit\n");
	for(;;) {
		int scan, ascii;
		read_key(&scan, &ascii);

		/* It takes two escapes in a row to quit. */
		if (scan == 01 && ascii == 0x1b) {
			escs++;
		} else if (scan < 0x80) { /* ignore break events */
			escs = 0;
		}
		if (escs == 2) {
			break;
		}
	}
	if (read_key == read_key_direct) {
		unhook_int9();
	}
	flush_buffer(file);
	unhook_int1b();
	fclose(file);
	return 0;
}
