 /*
  * UAE - The Un*x Amiga Emulator
  * 
  * Main program 
  * 
  * (c) 1995 Bernd Schmidt, Ed Hanway
  */

#include "sysconfig.h"
#include "sysdeps.h"
#include <assert.h>

#include "config.h"
#include "options.h"
#include "memory.h"
#include "custom.h"
#include "newcpu.h"
#include "disk.h"
#include "debug.h"
#include "xwin.h"
#include "os.h"
#include "filesys.h"
#include "keybuf.h"
#include "gui.h"

int framerate = 1;
int dont_want_aspect = 1;
int use_debugger = 0;
int use_slow_mem = 0;
int use_gfxlib = 0;
int use_xhair = 0;
int use_lores = 0;
int automount_uaedev = 1;
int produce_sound = 1;
int fake_joystick = 0;
KbdLang keyboard_lang = KBD_LANG_US;
#ifndef __DOS__
int screen_res = 3;
#else
int screen_res = 4;
#endif
int color_mode = 0;
ULONG fastmem_size = 0;
char df0[256]="df0.adf", df1[256]="df1.adf", df2[256]="df2.adf", df3[256]="df3.adf";
char romfile[256] = "kick.rom";
#ifndef __DOS__
char prtname[256] = "lpr ";
#else
char prtname[256] = "LPT1:";
#endif

/* If you want to pipe Printer output to a file, put something like
 * "cat >>printerfile.tmp" above.
 * The printer support was only tested with the driver "PostScript" on
 * Amiga side, using apsfilter for linux to print ps-data.
 *
 * Note for the DOS-Port: Maybe it's only neccesary to use a
 * -p LPT1: or -p PRN: to print in a DOSBOX. I don't know,
 * I do not use DOS... (please try) -=SR=-
 */

#ifndef __bebox__  /* BeOS needs its own startup code */

static void usage(void)
{
    printf("UAE - The Un*x Amiga emulator\n");
    printf("Summary of command-line options:\n");
    printf("  -h                       : Print help\n");
    printf("  -m VOLNAME:mount_point   : mount file system at <mount point> as AmigaDOS\n"
	   "                             volume VOLNAME:\n");
    printf("  -M VOLNAME:mount_point   : like -m, but mount read-only\n");
    printf("  -a                       : Don't mount the harddisk file automatically.\n"
	   "                             Useful only for testing.\n");
    printf("  -s                       : Emulate 1 MB slow memory at 0xC00000\n");
    printf("  -F n                     : Emulate n MB fast memory at 0x200000\n");
    printf("  -f n                     : Set the frame rate to 1/n\n");
    printf("  -D                       : Start up the built-in debugger\n");
    printf("  -[0123] file             : Use file instead of df[0123].adf as disk image file\n");
    printf("  -r file                  : Use file as ROM image instead of kick.rom\n");
    printf("  -J                       : Fake joystick emulation with the numeric pad.\n");
/*   printf("  -g                       : Turn on gfx-lib replacement.\n");*/
    /* We'll have to sort out the following a little better in the future... */
    printf("  -S                       : Turn off sound support (if it is configured)\n");
    printf("  -x                       : Use visible cross-hair cursor (X Windows version)\n");
    printf("  -l lang                  : Set keyboard language to lang, where lang is\n"
	   "                             DE, SE, US, FR or IT\n");
#if defined(__unix) && !defined(__DOS__)
    printf("  -p command               : Use command to pipe printer output to.\n");
#else
    printf("  -p filename              : Use filename to save printer output (may be LPT1: ?)\n");
#endif
    printf("  -d mode                  : Select resolution with the mode parameter.\n");
    printf("  -H mode                  : Set the number of colors with the mode parameter.\n");
    printf("\n");
#ifndef __DOS__
    printf("Valid resolutions: 0 (320x200); 1 (320x240); 2 (320x400); 3 (800x600);\n"
	   "                   4 (800x600, correct aspect)\n"
	   "Valid color modes: 0 (256 colors); 1 (32768 colors); 2 (65536 colors)\n"
	   "UAE may choose to ignore the color mode/resolution setting.\n");
#else
    printf("Valid resolutions: 0 (320x200); 1 (320x240); 2 (320x400); 3 (640x480);\n"
	   "                   4 (800x600); 5 (800x600, correct aspect)\n"
	   "Valid color modes: 0 (256 colors); 1 (32768 colors); 2 (65536 colors)\n"
	   "UAE may choose to ignore the color mode/resolution setting.\n");
#endif
}

#ifdef __unix

static void parse_cmdline(int argc, char **argv)
{
    int c;
    extern char *optarg;

    while(((c = getopt(argc, argv, "l:Df:gd:hxF:asSJm:M:0:1:2:3:r:H:p:")) != EOF)) switch(c) {
     case 'h': usage();	exit(0);

     case '0': strncpy(df0, optarg, 255); df0[255] = 0;	break;
     case '1': strncpy(df1, optarg, 255); df1[255] = 0; break;
     case '2': strncpy(df2, optarg, 255); df2[255] = 0; break;
     case '3': strncpy(df3, optarg, 255); df3[255] = 0; break;
     case 'r': strncpy(romfile, optarg, 255); romfile[255] = 0; break;
     case 'p': strncpy(prtname, optarg, 255); prtname[255] = 0; break;		    
     case 'm':
     case 'M':
	{
	    /* mount file system (repeatable)
	     * syntax: [-m | -M] VOLNAME:/mount_point
	     * example: -M CDROM:/cdrom -m UNIXFS:./disk
	     */
	    static int mount_seen = 0;
	    char buf[256];
	    char *s2;
	    int readonly = (c == 'M');

	    if (mount_seen)
		fprintf(stderr, "warning: multiple mounts confuse Kickstart 1.3\n");
	    mount_seen = 1;
	    strncpy(buf, optarg, 255); buf[255] = 0;
	    s2 = strchr(buf, ':');
	    if(s2) {
		*s2++ = '\0';
#ifdef __DOS__
		{
		    char *tmp;

		    while ((tmp = strchr(s2, '\\')))
			*tmp = '/';
		}
#endif
		add_filesys_unit(buf, s2, readonly);
	    } else {
		fprintf(stderr, "Usage: [-m | -M] VOLNAME:/mount_point\n");
	    }
	}
	break;
	
     case 'S': produce_sound = 0; break;
     case 'f': framerate = atoi(optarg); break;
     case 'x': use_xhair = 1; break;
     case 'D': use_debugger = 1; break;
     case 'J': fake_joystick = 1; break;
     case 'a': automount_uaedev = 0; break;
     case 's': use_slow_mem = 1; break;
     case 'g': use_gfxlib = 1; break;

     case 'F':
	fastmem_size = atoi(optarg) * 0x100000;
	if (fastmem_size != 0x100000 && fastmem_size != 0x200000 
	    && fastmem_size != 0x400000 && fastmem_size != 0x800000) 
	{
	    fastmem_size = 0;
	    fprintf(stderr, "Unsupported fastmem size!\n");
	}
	
	break;
	
     case 'l':
	if (0 == strcasecmp(optarg, "de"))
	    keyboard_lang = KBD_LANG_DE;
	else if (0 == strcasecmp(optarg, "us"))
	    keyboard_lang = KBD_LANG_US;
	else if (0 == strcasecmp(optarg, "se"))
	    keyboard_lang = KBD_LANG_SE;
	else if (0 == strcasecmp(optarg, "fr"))
	    keyboard_lang = KBD_LANG_FR;
	else if (0 == strcasecmp(optarg, "it"))
	    keyboard_lang = KBD_LANG_IT;
	break;
	
     case 'd':
	screen_res = atoi(optarg);
#ifndef __DOS__
	if (screen_res >= 0 && screen_res <= 4) {
	} else {
	    fprintf(stderr, "Bad video mode selected. Using default.\n");
	    screen_res = 3;
	}
	dont_want_aspect = screen_res != 4;
#else
	if (screen_res >= 0 && screen_res <= 5) {
	} else {
	    fprintf(stderr, "Bad video mode selected. Using default.\n");
	    screen_res = 4;
	}
	dont_want_aspect = screen_res != 5;
#endif
	use_lores = screen_res < 3;
	break;

     case 'H':
	color_mode = atoi(optarg);
	if (color_mode < 0 || color_mode > 2) {
	    fprintf(stderr, "Bad color mode selected. Using default.\n");
	    color_mode = 0;
	}
	break;
    }
}
#endif

static void parse_cmdline_and_init_file(int argc, char **argv)
{
    FILE *f;
    char file[256], *home;
    char *buffer,*tmpbuf, *token;
    char smallbuf[256];
    int bufsiz, result;
    int n_args;
    char **new_argv;
    int new_argc;

    strcpy(file,"");

#if !defined(__DOS__) && !defined(__mac__)
    home = getenv("HOME");
    if (home != NULL && strlen(home) < 240)
    {
	strcpy(file, home);
	strcat(file, "/");
    }
#endif

#if defined(__unix) && !defined(__DOS__)
    strcat(file, ".uaerc");
#else
    strcat(file, "uae.rc");
#endif

    f = fopen(file,"rb");
    if (f == NULL) {
	parse_cmdline(argc, argv);
	return;
    }
    fseek(f, 0, SEEK_END);
    bufsiz = ftell(f);
    fseek(f, 0, SEEK_SET);

    buffer = (char *)malloc(bufsiz+1);
    buffer[bufsiz] = 0;
    if (fread(buffer, 1, bufsiz, f) < bufsiz) {
	fprintf(stderr, "Error reading configuration file\n");
	fclose(f);
	parse_cmdline(argc, argv);
	return;
    }
    fclose(f);

#ifdef __DOS__
    {
	char *tmp;

	while ((tmp = strchr(buffer, 0x0d)))
	    *tmp = ' ';
	while ((tmp = strchr(buffer, 0x0a)))
	    *tmp = ' ';
	while (buffer[0] == ' ')
	    strcpy(buffer, buffer+1);
	while ((strlen(buffer) > 0) && (buffer[strlen(buffer) - 1] == ' '))
	    buffer[strlen(buffer) - 1] = '\0';
	while ((tmp = strstr(buffer, "  ")))
	    strcpy(tmp, tmp+1);
    }
#endif

    tmpbuf = my_strdup (buffer);

    n_args = 0;
    if (strtok(tmpbuf, "\n ") != NULL) {
	do {
	    n_args++;
	} while (strtok(NULL, "\n ") != NULL);
    }
    free (tmpbuf);

    new_argv = (char **)malloc ((1 + n_args + argc) * sizeof (char **));
    new_argv[0] = argv[0];
    new_argc = 1;

    token = strtok(buffer, "\n ");
    while (token != NULL) {
	new_argv[new_argc] = my_strdup (token);
	new_argc++;
	token = strtok(NULL, "\n ");
    }
    for (n_args = 1; n_args < argc; n_args++)
	new_argv[new_argc++] = argv[n_args];
    new_argv[new_argc] = NULL;
    parse_cmdline(new_argc, new_argv);
}

int main(int argc, char **argv)
{
    parse_cmdline_and_init_file(argc, argv);
    if (produce_sound && !init_sound()) {
	fprintf(stderr, "Sound driver unavailable: Sound output disabled\n");
	produce_sound = 0;
    }

    if (gui_init() < 0) {
	fprintf(stderr, "Failed to initialize the GUI\n");
	/* abort()? Maybe. */
    }
    init_joystick();
    keybuf_init();
    memory_init();
    custom_init();
    DISK_init();
    init_m68k();
    
    if (graphics_init()) {
	MC68000_reset();

	debug();
    
	graphics_leave();
	close_joystick();
    }
#ifdef COUNT_INSTRS
    {
	extern void dump_counts(void);
	dump_counts();
    }
#endif
    gui_exit();
    return 0;
}
#endif /* not __bebox__ */
