/*
 *	The C64 emulator
 *
 *	Copyright 1995-96 by ALE.
 *	Written by Edgar Trnig
 *
 *	Video driver for X11 using the shared memory pixmap extension.
 *-----------------------------------------------------------------------------
 * $Id: video_x11.h,v 1.3 1996/06/16 00:50:53 johns Exp root $
 * $Log: video_x11.h,v $
 * Revision 1.3  1996/06/16 00:50:53  johns
 * Added support for 16bpp.
 *
 * Revision 1.2  1996/06/13 02:29:08  johns
 * Added overscan cache and VideoSync in n%.
 *
 * Revision 1.1  1996/04/18 01:17:15  root
 * Initial revision
 *
 *-----------------------------------------------------------------------------
 */

#define C64HSIZE	320
#define C64VSIZE	200

#define OVERSCAN_CACHE

#ifdef OVERSCAN

#define HPIXEL		(C64HSIZE+8)	/* pixel per line */
#ifdef COLORMAP16bpp
#define HSIZE		(C64HSIZE*2+16)	/* byte_per_line of the pixmap */
#else
#define HSIZE		(C64HSIZE+8)	/* byte_per_line of the pixmap */
#endif
#define VSIZE		(C64VSIZE+64)

#else

#define HPIXEL		(C64HSIZE)	/* pixel per line */
#ifdef COLORMAP16bpp
#define HSIZE		(C64HSIZE*2)	/* byte_per_line of the pixmap */
#else
#define HSIZE		(C64HSIZE)	/* byte_per_line of the pixmap */
#endif
#define VSIZE		(C64VSIZE)

#endif


/*---------------------------------------------------------------------------*/

static unsigned char* VideoPointer;	/* current pointer in video memory */
extern unsigned char *VideoMemory;	/* start pointer of video memory */
extern unsigned long QuadByte[256];	/* quadruple a byte */
extern void VideoArrange(int,int);	/* arrange video image */

/*---------------------------------------------------------------------------*/

/*
**	Suspend video.
*/
#define SuspendVideo()

/*
**	Resume video.
*/
#define ResumeVideo()

/*
**	Set overscan color.
*/
#define VideoSetOverscanColor(color)

/*
**	Update internal structures to monitor.
*/
extern void VideoRefresh(void);

/*-----------------------------------------------------------------------------
 *	OVERSCAN dependend
 *---------------------------------------------------------------------------*/

#ifdef OVERSCAN	/* { */

/*
**	Macro to include code only if compiling for overscan support.
*/
#define IfOverscan(x)	x

/*
**	Called only for blank overscanlines.
*/
#ifdef OVERSCAN_CACHE
#define VideoFillOverscanLine(quad_color) \
    if( VicOverscanCache[VicRasterLine]!=(quad_color&0xFF) ) {	\
	VicOverscanCache[VicRasterLine]=quad_color;		\
	VideoFillLine(quad_color);				\
	VicDrawOverscan=1;					\
    } else {							\
	VideoPointer+=HSIZE;					\
    }
#else
#define VideoFillOverscanLine(quad_color) \
    VideoFillLine(quad_color);
#endif

/*
**	Called only for open border overscanlines.
*/
#ifdef OVERSCAN_CACHE
#define VideoCopyOverscanLine(pointer) \
    VicOverscanCache[VicRasterLine]=0xFF;			\
    VicDrawOverscan=1;						\
    VideoCopyLine(pointer)
#else
#define VideoCopyOverscanLine(pointer) \
    VicOverscanCache[VicRasterLine]=0xFF;			\
    VideoCopyLine(pointer)
#endif

#else	/* }{ OVERSCAN */

#define IfOverscan(x)	/* x */
#define VideoFillOverscanLine(quad_color)	/* quad_color */
#define VideoCopyOverscanLine(pointer) 		/* pointer */

#endif	/* } !OVERSCAN */

/*---------------------------------------------------------------------------*/

/*
**	VideoPut: Put a byte into video memory.
*/
#ifdef OVERSCAN_CACHE
#  ifdef COLORMAP16bpp
#    define VideoPut(a,v) VicDrawOverscan=1,(*(short*)(a)=(v))
#  else
#    define VideoPut(a,v) VicDrawOverscan=1,(*(a)=(v))
#  endif
#else
#  ifdef COLORMAP16bpp
#    define VideoPut(a,v) (*(short*)(a)=(v))
#  else
#    define VideoPut(a,v) (*(a)=(v))
#  endif
#endif

#ifdef COLORMAP0	/* { */

extern unsigned char Pen2Pixel[];

/*
**	1 emulator pixel to video format.
*/
#define VIC2VIDEO(p) \
    (Pen2Pixel[p])

/*
**	Fill a scanline with 'quad_color'.
*/
#define VideoFillLine(quad_color) \
  do {									\
    IfOverscan(								\
	((unsigned long*)VideoPointer)[0]=				\
	    QuadByte[Pen2Pixel[VicExterior&0xff]]; )			\
    LongFill(VideoPointer IfOverscan(+4),				\
	QuadByte[Pen2Pixel[quad_color&0xFF]],C64HSIZE/4);		\
    IfOverscan(								\
	((unsigned long*)VideoPointer)[1+C64HSIZE/4]=			\
	    QuadByte[Pen2Pixel[VicExterior&0xff]]; )			\
    VideoPointer+=HSIZE;						\
  } while( 0 )

/*
**	Copy a scanline from 'src' to video.
**		(This is the hard way to get speed.)
*/
static inline void VideoCopyLine(unsigned char const* org)
{
    unsigned char* dst;
    unsigned char const* src;
    
    src=org;
    dst=VideoPointer;
    IfOverscan(
	*((unsigned long*)dst)++=QuadByte[Pen2Pixel[VicExterior&0xff]]; )
    do {
	dst[0]=Pen2Pixel[src[0]];
	dst[1]=Pen2Pixel[src[1]];
	dst[2]=Pen2Pixel[src[2]];
	dst[3]=Pen2Pixel[src[3]];
	dst[4]=Pen2Pixel[src[4]];
	dst[5]=Pen2Pixel[src[5]];
	dst[6]=Pen2Pixel[src[6]];
	dst[7]=Pen2Pixel[src[7]];
	dst[8]=Pen2Pixel[src[8]];
	dst[9]=Pen2Pixel[src[9]];
	dst+=10; src+=10;
    } while( src<org+C64HSIZE );
    IfOverscan(
	*((unsigned long*)dst)++=QuadByte[Pen2Pixel[VicExterior&0xff]]; )
    VideoPointer=dst;
}

#endif	/* } COLORMAP0 */

#ifdef COLORMAP16	/* { */

extern unsigned long VideoPixelAdd;	/* quad pixel index add */

/*
**	1 emulator pixel to video format.
*/
#define VIC2VIDEO(p) \
    (((p)&0x0F)+(VideoPixelAdd&0xFF))

/*
**	Fill a scanline with 'quad_color'.
*/
#define VideoFillLine(quad_color) \
  do {									\
    IfOverscan(								\
	((unsigned long*)VideoPointer)[0]=				\
	    (VicExterior&0x0F0F0F0F)+VideoPixelAdd; )			\
    LongFill(VideoPointer IfOverscan(+4),				\
	(quad_color&0x0F0F0F0F)+VideoPixelAdd,C64HSIZE/4);		\
    IfOverscan(								\
	((unsigned long*)VideoPointer)[1+C64HSIZE/4]=			\
	    (VicExterior&0x0F0F0F0F)+VideoPixelAdd; )			\
    VideoPointer+=HSIZE;						\
  } while( 0 )

/*
**	Copy a scanline from 'src' to video.
**		(This is the hard way to get speed.)
*/
static inline void VideoCopyLine(unsigned char const* org)
{
    unsigned long* dst;
    unsigned long const* src;
    
    src=(unsigned long*)org;
    dst=(unsigned long*)VideoPointer;
    IfOverscan(
	*dst++=(VicExterior&0x0F0F0F0F)+VideoPixelAdd; )
    do {
	dst[0]=(src[0]&0x0F0F0F0F)+VideoPixelAdd;
	dst[1]=(src[1]&0x0F0F0F0F)+VideoPixelAdd;
	dst[2]=(src[2]&0x0F0F0F0F)+VideoPixelAdd;
	dst[3]=(src[3]&0x0F0F0F0F)+VideoPixelAdd;
	dst[4]=(src[4]&0x0F0F0F0F)+VideoPixelAdd;
	dst[5]=(src[5]&0x0F0F0F0F)+VideoPixelAdd;
	dst[6]=(src[6]&0x0F0F0F0F)+VideoPixelAdd;
	dst[7]=(src[7]&0x0F0F0F0F)+VideoPixelAdd;
	dst[8]=(src[8]&0x0F0F0F0F)+VideoPixelAdd;
	dst[9]=(src[9]&0x0F0F0F0F)+VideoPixelAdd;
	dst+=10; src+=10;
    } while( src<(unsigned long*)(org+C64HSIZE) );
    IfOverscan(
	*dst++=(VicExterior&0x0F0F0F0F)+VideoPixelAdd; )
    VideoPointer=(unsigned char*)dst;
}

#endif	/* } COLORMAP16 */

#ifdef COLORMAP256	/* { */

/*
**	1 emulator pixel to video format.
*/
#define VIC2VIDEO(p) \
    (p)

/*
**	Fill a scanline with 'quad_color'.
*/
#define VideoFillLine(quad_color) \
  do {									\
    IfOverscan(								\
	((unsigned long*)VideoPointer)[0]=VicExterior; )		\
    LongFill(VideoPointer IfOverscan(+4),quad_color,C64HSIZE/4);	\
    IfOverscan(								\
	((unsigned long*)VideoPointer)[1+C64HSIZE/4]=VicExterior; )	\
    VideoPointer+=HSIZE;						\
  } while( 0 )

/*
**	Copy a scanline from 'src' to video.
*/
#define VideoCopyLine(src) \
  do {									\
    IfOverscan(								\
	((unsigned long*)VideoPointer)[0]=VicExterior; )		\
    LongCopy(VideoPointer IfOverscan(+4),src,C64HSIZE/4);		\
    IfOverscan(								\
	((unsigned long*)VideoPointer)[1+C64HSIZE/4]=VicExterior; )	\
    VideoPointer+=HSIZE;						\
  } while( 0 )

#endif	/* } COLORMAP256 */

#ifdef COLORMAP16bpp	/* { */

extern unsigned short Pen2Pixel[];

/*
**	1 emulator pixel to video format.
*/
#define VIC2VIDEO(p) \
    (Pen2Pixel[p])

/*
**	Fill a scanline with 'quad_color'.
*/
#define VideoFillLine(quad_color) \
  do {									\
    IfOverscan(								\
	((unsigned short*)VideoPointer)[0]=				\
	((unsigned short*)VideoPointer)[1]=				\
	((unsigned short*)VideoPointer)[2]=				\
	((unsigned short*)VideoPointer)[3]=				\
	    Pen2Pixel[VicExterior&0xff]; )				\
    LongFill(VideoPointer IfOverscan(+8),				\
	(Pen2Pixel[quad_color&0xFF]<<16)|Pen2Pixel[quad_color&0xFF],	\
	C64HSIZE/2);							\
    IfOverscan(								\
	((unsigned short*)VideoPointer)[4+C64HSIZE]=			\
	((unsigned short*)VideoPointer)[5+C64HSIZE]=			\
	((unsigned short*)VideoPointer)[6+C64HSIZE]=			\
	((unsigned short*)VideoPointer)[7+C64HSIZE]=			\
	    Pen2Pixel[VicExterior&0xff]; )				\
    VideoPointer+=HSIZE;						\
  } while( 0 )

/*
**	Copy a scanline from 'src' to video.
**		(This is the hard way to get speed.)
*/
static inline void VideoCopyLine(unsigned char const* org)
{
    unsigned short* dst;
    unsigned char const* src;
    
    src=org;
    dst=(unsigned short*)VideoPointer;
    IfOverscan(
	*((unsigned short*)dst)++=Pen2Pixel[VicExterior&0xff];
	*((unsigned short*)dst)++=Pen2Pixel[VicExterior&0xff];
	*((unsigned short*)dst)++=Pen2Pixel[VicExterior&0xff];
	*((unsigned short*)dst)++=Pen2Pixel[VicExterior&0xff]; )
    do {
	dst[0]=Pen2Pixel[src[0]];
	dst[1]=Pen2Pixel[src[1]];
	dst[2]=Pen2Pixel[src[2]];
	dst[3]=Pen2Pixel[src[3]];
	dst[4]=Pen2Pixel[src[4]];
	dst[5]=Pen2Pixel[src[5]];
	dst[6]=Pen2Pixel[src[6]];
	dst[7]=Pen2Pixel[src[7]];
	dst[8]=Pen2Pixel[src[8]];
	dst[9]=Pen2Pixel[src[9]];
	dst+=10; src+=10;
    } while( src<org+C64HSIZE );
    IfOverscan(
	*((unsigned short*)dst)++=Pen2Pixel[VicExterior&0xff];
	*((unsigned short*)dst)++=Pen2Pixel[VicExterior&0xff];
	*((unsigned short*)dst)++=Pen2Pixel[VicExterior&0xff];
	*((unsigned short*)dst)++=Pen2Pixel[VicExterior&0xff]; )
    VideoPointer=(unsigned char*)dst;
}

#endif	/* } COLORMAP16bpp */

/*-----------------------------------------------------------------------------
 *	VIDEOSYNC support
 *---------------------------------------------------------------------------*/

#ifdef VIDEOSYNC	/* { */

extern volatile int VideoInterrupts;	/* be happy, were are quicker */

/*
**	Sync video.
*/
#define VideoSync()	\
    do {						\
	while( VideoSyncSpeed				\
		&& VideoInterrupts<VicRefreshRate ) {	\
	    sigpause(0);				\
	}						\
	VideoInterrupts=0;				\
    } while( 0 )

#else	/* }{ VIDEOSYNC */

#define VideoSync()	\
    do {						\
	/* noop video-sync */				\
    } while( 0 )

#endif	/* } !VIDEOSYNC */
