/*
 *	The C64 emulator
 *
 *	Copyright 1995-96 by ALE.
 *	Written by Edgar Trnig
 *
 *	Video driver for X11 using the shared memory pixmap extension.
 *-----------------------------------------------------------------------------
 * $Id: video_x11.c,v 1.4 1996/06/16 00:50:15 johns Exp root $
 * $Log: video_x11.c,v $
 * Revision 1.4  1996/06/16 00:50:15  johns
 * Added support for 16bpp.
 *
 * Revision 1.3  1996/06/13 02:28:06  johns
 * VideoSync in n%.
 *
 * Revision 1.2  1996/05/20 23:01:45  root
 * XSync configurable, Center image in small window, Setup window manager hints
 * for olvwm (didn't like none), Bugfix of Frank Wuebbeling XStoreColor didn't
 * return something.
 *
 * Revision 1.1  1996/04/18 01:21:07  root
 * Initial revision
 *
 *-----------------------------------------------------------------------------
 */

#include "config.h"
#ifdef X11	/* { */

#include "c64.h"
#include "vic.h"
#include "video_x11.h"

#include <stdio.h>
#include <unistd.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/XShm.h>
#include <X11/cursorfont.h>

/*---------------------------------------------------------------------------*/

static struct {	
    unsigned short Red,Green,Blue;
} CONST Palette[16] = {
#ifdef __ET__
    { 0x0000, 0x0000, 0x0000 },       /* 00 Black */
    { 0xffff, 0xffff, 0xffff },       /* 01 White */
    { 0x9000, 0x0000, 0x0000 },       /* 02 Red */
    { 0x0000, 0xc000, 0xc000 },       /* 03 Turquoise	08 48 48 */
    { 0xc000, 0x0000, 0xc000 },       /* 04 Violet	48 00 48 */
    { 0x0000, 0x9000, 0x0000 },       /* 05 Green */
    { 0x0000, 0x0000, 0x9000 },       /* 06 Blue */
    { 0xffff, 0xffff, 0x0000 },       /* 07 Yellow */
    { 0xffff, 0x6000, 0x0000 },       /* 08 Orange	63 22 00 */
    { 0x8000, 0x4000, 0x0000 },       /* 09 Brown	31 15 00 */
    { 0xffff, 0x0000, 0x0000 },       /* 0A Lite Red */
    { 0x4000, 0x4000, 0x4000 },       /* 0B Gray 1 */
    { 0x8000, 0x8000, 0x8000 },       /* 0C Gray 2 */
    { 0x0000, 0xffff, 0x0000 },       /* 0D Lite Green */
    { 0x0000, 0x0000, 0xffff },       /* 0E Lite Blue */
    { 0xc000, 0xc000, 0xc000 }        /* 0F Gray 3 */
#else
    { 0x0000, 0x0000, 0x0000 },       /* 00 Black */
    { 0xffff, 0xffff, 0xffff },       /* 01 White */
    { 0x9000, 0x0000, 0x0000 },       /* 02 Red */
    { 0x0000, 0xc000, 0xc000 },       /* 03 Turquoise	08 48 48 */
    { 0xc000, 0x0000, 0xc000 },       /* 04 Violet	48 00 48 */
    { 0x0000, 0x9000, 0x0000 },       /* 05 Green */
    { 0x0000, 0x0000, 0x9000 },       /* 06 Blue */
    { 0xffff, 0xffff, 0x0000 },       /* 07 Yellow */
    { 0xffff, 0x6000, 0x0000 },       /* 08 Orange	63 22 00 */
    { 0x8000, 0x4000, 0x0000 },       /* 09 Brown	31 15 00 */
    { 0xffff, 0x0000, 0x0000 },       /* 0A Lite Red */
    { 0x4000, 0x4000, 0x4000 },       /* 0B Gray 1 */
    { 0x8000, 0x8000, 0x8000 },       /* 0C Gray 2 */
    { 0x0000, 0xffff, 0x0000 },       /* 0D Lite Green */
    { 0x0000, 0x8000, 0xffff },       /* 0E Lite Blue */
    { 0xc000, 0xc000, 0xc000 }        /* 0F Gray 3 */
#endif
};

Display *display = 0;
unsigned char *VideoMemory;
static Window window;
static Pixmap pixmap;
static GC gc_copy;
static XShmSegmentInfo shminfo;
static unsigned long Pixels[16];	/* X11-pixels for the Palette */
static int XOff,YOff;
unsigned char Pixel2Pen[256];		/* convert pixmap back to c64 */
Atom WM_Delete_Window;


static void
fatal(char *str, ...)
{
    char buf[256];

    write(2, "c64emu: fatal error: ", 21);
    write(2, buf, vsprintf(buf, str, &str + 1));
    exit(1);
}

#ifdef VIDEOSYNC	/* { */

#include <signal.h>
#include <sys/time.h>

volatile int VideoInterrupts;		/* be happy, were are quicker */

/*
**	Called from SIGALRM.
*/
static void VideoSyncHandler(int unused)
{
    ++VideoInterrupts;
}

/*
**	Initialise video sync.
*/
static void InitVideoSync(void)
{
    struct sigaction sa;
    struct itimerval itv;

    if( !VideoSyncSpeed ) {
	return;
    }

    itv.it_interval.tv_sec=itv.it_value.tv_sec=
	(100/FRAMES_PER_SECOND)/VideoSyncSpeed;
    itv.it_interval.tv_usec=itv.it_value.tv_usec=
	(100000000/FRAMES_PER_SECOND)/VideoSyncSpeed-
	itv.it_value.tv_sec*100000;
    if( setitimer(ITIMER_REAL,&itv,NULL) ) {
	printf("Can't set itimer\n");
    }
    sa.sa_handler=VideoSyncHandler;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags=SA_RESTART;
    if( sigaction(SIGALRM,&sa,NULL) ) {
	printf("Can't set signal\n");
    }
}

#else	/* }{ VIDEOSYNC */

#define InitVideoSync()

#endif	/* } !VIDEOSYNC */


#ifdef COLORMAP0	/* { */

unsigned char Pen2Pixel[256];	/* pen to pixel mapping */

static void InitPalette(void)
{
    Colormap cm = DefaultColormap(display, DefaultScreen(display));
    XColor color;
    int i;

    for (i = 0; i < 16; ++i)
    {
	color.red = Palette[i].Red;
	color.green = Palette[i].Green;
	color.blue = Palette[i].Blue;
	if (XAllocColor(display, cm, &color) == 0)
	    fatal("cannot allocate color %d\n", i);
	Pixels[i] = color.pixel;
	Pixel2Pen[color.pixel]=i;
    }
    for (i = 0; i < 256; ++i) {
	Pen2Pixel[i] = Pixels[i & 15];
    }
}

#endif /* } COLORMAP0 */

#ifdef COLORMAP16	/* { */

unsigned long VideoPixelAdd;		/* quad pixel index add */

static void InitPalette(void)
{
    Colormap cm;
    XColor color;
    int i;

    cm=DefaultColormap(display,DefaultScreen(display));

    /*
    **	Allocate 16 continuous colors.
    */
    if( XAllocColorCells(display,cm,1,0L,0,Pixels,16)==0 ) {
	fatal("XAllocColorCells fails, can't allocate 16 continuous colors\n");
    }

    /*
    **	Setup the 16 continuous colors with original c64 colors.
    */
    for( i=0; i<16; ++i ) {
	color.pixel=Pixels[i];
	color.red=Palette[i].Red;
	color.green=Palette[i].Green;
	color.blue=Palette[i].Blue;
	color.flags=DoRed|DoGreen|DoBlue;
	XStoreColor(display,cm,&color);
	if( 0 ) {			/* don't know how to check error? */
	    fatal("can't store color %d\n", i);
	}
	Pixel2Pen[color.pixel]=i;
    }

    VideoPixelAdd=QuadByte[Pixels[0]];
}

#endif	/* } COLORMAP16 */

#ifdef COLORMAP256	/* { */

static void InitPalette(void)
{
    Colormap cm;
    unsigned long pixels[256];
    XColor color;
    int i;

    cm=XCreateColormap(display,
	DefaultRootWindow(display),
	DefaultVisual(display,DefaultScreen(display)),
	AllocNone);
    if( !cm ) {
	fatal("Can't allocate color map\n");
    }

    if( XAllocColorCells(display,cm,1,0L,0,pixels,256)==0 ) {
	fatal("Can't XAllocColorCells\n");
    }

    for( i = 0; i < 256; ++i) {
	color.pixel=pixels[i];
	color.red = Palette[i&15].Red;
	color.green = Palette[i&15].Green;
	color.blue = Palette[i&15].Blue;
	color.flags=DoRed|DoGreen|DoBlue;
	if (XStoreColor(display, cm, &color) == 0)
	    fatal("cannot store color %d\n", i);
	Pixel2Pen[color.pixel]=i&16;
    }
    memcpy(Pixels,pixels,16*sizeof(*Pixels));

    XSetWindowColormap(display,window,cm);
}

#endif	/* } COLORMAP256 */

#ifdef COLORMAP16bpp	/* { */

unsigned short Pen2Pixel[256];	/* pen to pixel mapping */

static void InitPalette(void)
{
    Colormap cm = DefaultColormap(display, DefaultScreen(display));
    XColor color;
    int i;

    for (i = 0; i < 16; ++i)
    {
	color.red = Palette[i].Red;
	color.green = Palette[i].Green;
	color.blue = Palette[i].Blue;
	if (XAllocColor(display, cm, &color) == 0)
	    fatal("cannot allocate color %d\n", i);
	Pixels[i] = color.pixel;
	// Pixel2Pen[color.pixel]=i;
    }
    for (i = 0; i < 256; ++i) {
	Pen2Pixel[i] = Pixels[i & 15];
    }
}

#endif	/* } COLORMAP16bpp */

/*
** Modifies the XSizeHints to reflect the -geometry specs.
*/
static void
SetupGeometry(Display *dpy, char *geo, XSizeHints *hints)
{
    int f = XParseGeometry(geo, &hints->x, &hints->y, &hints->width, &hints->height);

    if (f & XValue)
    {
	if (f & XNegative)
	    hints->x += DisplayWidth(dpy, DefaultScreen(dpy)) - hints->width;
	hints->flags |= USPosition;
	/* FIXME: win gravity */
    }
    if (f & YValue)
    {
	if (f & YNegative)
	    hints->y += DisplayHeight(dpy, DefaultScreen(dpy)) - hints->height;
	hints->flags |= USPosition;
	/* FIXME: win gravity */
    }
    if (f & WidthValue)
	hints->flags |= USSize;
    if (f & HeightValue)
	hints->flags |= USSize;
}



void EnterVideo(int argc, char **argv, char *geometry)
{
    XSetWindowAttributes attributes;
    XWMHints wm_hints;
    XSizeHints hints;
    XClassHint class_hints;
    int shm_major, shm_minor;
    Bool pixmap_support;

    if (display)
	return;

    if (!(display = XOpenDisplay(NULL)))
	fatal("Cannot connect to X-Server.\n");

    if (!XShmQueryVersion(display, &shm_major, &shm_minor, &pixmap_support))
	fatal("SHM-Extensions required.\n");
	
    if (!pixmap_support)
	fatal("SHM-Extensions with pixmap supported required.\n");

#ifdef	COLORMAP16bpp			/* 16bpp X11 server */
    if( DefaultDepth(display,DefaultScreen(display))!=16 )
	fatal("Color depth of 16 needed.\n");
#else
    if( DefaultDepth(display,DefaultScreen(display))!=8 )
	fatal("Color depth of 8 needed.\n");
#endif

    //attributes.cursor=XCreateFontCursor(display,XC_coffee_mug);
    attributes.cursor=XCreateFontCursor(display,XC_trek);
    /* FIXME: Build color map after window open !!! */
    /* FIXME: attributes.background_pixel = Pixels[0]; */
    attributes.bit_gravity=CenterGravity;
    attributes.event_mask = KeyPressMask | KeyReleaseMask | FocusChangeMask
			| KeymapStateMask | StructureNotifyMask | ExposureMask;

    hints.x = 0;
    hints.y = 0;
    hints.base_width = hints.max_width = hints.width = HPIXEL;
    hints.base_height = hints.max_height = hints.height = VSIZE;
    hints.flags = PSize | PMaxSize;

    if (geometry)
	SetupGeometry(display, geometry, &hints);

    window = XCreateWindow(display, DefaultRootWindow(display),
			    hints.x, hints.y, hints.width, hints.height, 3,
			    CopyFromParent, InputOutput, CopyFromParent,
			    CWBitGravity |
			    /* CWBackPixel |*/ CWEventMask | CWCursor,
			    &attributes);

    InitPalette();

    shminfo.shmid = shmget(IPC_PRIVATE, HSIZE * VSIZE, IPC_CREAT|0777);
    if (shminfo.shmid == -1)
	fatal("shmget failed.\n");
    shminfo.shmaddr = shmat(shminfo.shmid, 0, 0);
    shmctl(shminfo.shmid, IPC_RMID, 0);
    if (shminfo.shmaddr == (void *)-1)
	fatal("shmget failed.\n");
    shminfo.readOnly = False;

    if (!XShmAttach(display, &shminfo))
	fatal("XShmAttach failed.\n");

#ifdef	COLORMAP16bpp			/* 16bpp X11 server */
    pixmap = XShmCreatePixmap(display, window, shminfo.shmaddr, &shminfo,
	HPIXEL, VSIZE, 16);
#else
    pixmap = XShmCreatePixmap(display, window, shminfo.shmaddr, &shminfo,
	HPIXEL, VSIZE, 8);
#endif

    gc_copy = XCreateGC(display, window, 0, NULL);
    XSetGraphicsExposures(display, gc_copy, False);

    class_hints.res_class = "C64";
    class_hints.res_name = "alec64";
	
    /*wm_hints.initial_state = NormalState;*/
    wm_hints.input = True;
    wm_hints.flags = /*StateHint |*/ InputHint;
    XSetStandardProperties(display, window,
	"ALE C64 Emulator", "ALE C64 Emulator",
	None, argv, argc, &hints);
    XSetClassHint(display, window, &class_hints);
    XSetWMHints(display, window, &wm_hints);

    WM_Delete_Window = XInternAtom(display, "WM_DELETE_WINDOW", False);
    XSetWMProtocols(display, window, &WM_Delete_Window, 1);

    XMapWindow(display, window);

    VideoMemory = shminfo.shmaddr;

    VideoArrange(hints.width,hints.height);
    
    InitVideoSync();
}

void LeaveVideo(void)
{
    if (!display)
	return;

    XUnmapWindow(display, window);
    /* FreeColors(...) */
    XFreeGC(display, gc_copy);

    XFreePixmap(display, pixmap);
    XShmDetach(display, &shminfo);
    shmdt(shminfo.shmaddr);

    XDestroyWindow(display, window);
    XCloseDisplay(display);

    display = 0;
}

void
VideoArrange(int width,int height)
{
    XOff=(HPIXEL-width)/2;
    YOff=(VSIZE-height)/2;
    VicDrawOverscan=1;
    // printf("Win: %dx%d -> %d,%d\n",width,height,XOff,YOff);
}

void
VideoRefresh(void)
{
    int i;
    char buf[80];
    struct timeval end;
    static struct timeval start;
    static int count;

    if( --count<0 ) {
	count=FRAMES_PER_SECOND*5;

	gettimeofday(&end,0);
	if( start.tv_sec || start.tv_usec ) {
	    i=(end.tv_sec-start.tv_sec)*1000
		+(end.tv_usec-start.tv_usec)/1000;

	    i=(((long long)500000
		*CYCLES_PER_LINE*LINES_PER_FRAME*FRAMES_PER_SECOND/CPU_CLK)
		*VicRefreshRate*VicEmulateRate)/i;
	    sprintf(buf,"ALE C64 Emulator at %d%% speed",i+1);
	    XSetStandardProperties(display,window,buf,NULL,None,NULL,0,NULL);
	}
	start=end;
    }

#ifdef OVERSCAN_CACHE
    if( VicDrawOverscan ) {
	VicDrawOverscan=0;
	XCopyArea(display, pixmap, window, gc_copy,
	    XOff, YOff, HPIXEL-(XOff*2), VSIZE-(YOff*2), 0, 0);
    } else {
	XCopyArea(display, pixmap, window, gc_copy,
	    XOff, (VSIZE-C64VSIZE)/2,
	    HPIXEL-(XOff*2), VSIZE-(VSIZE-C64VSIZE),
	    0, (VSIZE-C64VSIZE)/2-YOff);
    }
#else
    XCopyArea(display, pixmap, window, gc_copy,
	    XOff, YOff, HPIXEL-(XOff*2), VSIZE-(YOff*2), 0, 0);
#endif
    XFlush(display);
#ifdef XSYNC
    XSync(display,0);
#endif
}

#endif	/* } X11 */
