/*
 *	The C64 emulator
 *
 *	Copyright 1996 by ALE.
 *	written by Lutz Sammer.
 *
 *	Video driver for svgalib and linux.
 *------------------------------------------------------------------------------
 * $Id: video_svga.c,v 1.3 1996/06/13 00:00:16 ari Exp root $
 * $Log: video_svga.c,v $
 * Revision 1.3  1996/06/13 00:00:16  ari
 * Made svgalib quiet. Added volatile to VideoIsDisplayed!
 *
 * Revision 1.2  1996/06/12 23:33:12  johns
 * Added video sync support and overscan support.
 *
 * Revision 1.1  1996/04/26 23:08:44  johns
 * Initial revision
 *
 *------------------------------------------------------------------------------
 */

#include "config.h"

#ifdef SVGALIB	/* { */

#include "c64.h"
#include "vic.h"
#include "sid.h"
#include "video_svga.h"

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <signal.h>
#include <unistd.h>
#include <sys/vt.h>
#include <sys/kd.h>
#include <sys/ioctl.h>

volatile int VideoIsDisplayed;		/* GLOBAL Flag to indicate
					** VT-Screen active or switched away */

/*---------------------------------------------------------------------------*/

static struct {
    char Red,Green,Blue;
} CONST Palette[16] = {
    { 0,  0,  0},			/* 00 Black */
    {63, 63, 63},			/* 01 White */
    {39,  0,  0},			/* 02 Red */
    { 0, 47, 47},			/* 03 Turquoise 08 48 48 */
    {47,  0, 47},			/* 04 Violet 48 0 48 */
    { 0, 39,  0},			/* 05 Green */
    { 0,  0, 39},			/* 06 Blue */
    {63, 63,  0},			/* 07 Yellow */
    {63, 22,  0},			/* 08 Orange 63 22 0 */
    {30, 14,  0},			/* 09 Brown  31 15 0 */
    {63,  0,  0},			/* 0A Lite Red */
    {15, 15, 15},			/* 0B Gray 1 */
    {31, 31, 31},			/* 0C Gray 2 */
    { 0, 63,  0},			/* 0D Lite Green */
    { 0,  0, 63},			/* 0E Lite Blue */
    {47, 47, 47} 			/* 0F Gray 3 */
};

/*
**	Load my own palette into DAC.
*/
static void InitPalette()
{
    int i;

    for( i=0; i<256; ++i ) {
	vga_setpalette(i,Palette[i&0xF].Red,
		Palette[i&0xF].Green,
		Palette[i&0xF].Blue);
    }
}

/*
**	Set asyncron overscan color.
*/
void SetOverscanColor(unsigned char c)
{
    if( VideoIsDisplayed ) {
	/* HELPME: portable WAY for svgalib ?? */
	inb(0x3DA);
	outb(0x3C0,0x31);
	outb(0x3C0,c);
    }
}

/*---------------------------------------------------------------------------*/

#ifdef VIDEOSYNC	/* { */

#include <signal.h>
#include <sys/time.h>

volatile int VideoInterrupts;		/* be happy, were are quicker */

/*
**	Called from SIGALRM.
*/
static void VideoSyncHandler(int unused)
{
    ++VideoInterrupts;
}

/*
**	Initialise video sync.
*/
static void InitVideoSync(void)
{
    struct sigaction sa;
    struct itimerval itv;

    if( !VideoSyncSpeed ) {
	return;
    }

    itv.it_interval.tv_sec=itv.it_value.tv_sec=
	(100/FRAMES_PER_SECOND)/VideoSyncSpeed;
    itv.it_interval.tv_usec=itv.it_value.tv_usec=
	(100000000/FRAMES_PER_SECOND)/VideoSyncSpeed-
	itv.it_value.tv_sec*100000;
    if( setitimer(ITIMER_REAL,&itv,NULL) ) {
	printf("Can't set itimer\n");
    }
    sa.sa_handler=VideoSyncHandler;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags=SA_RESTART;
    if( sigaction(SIGALRM,&sa,NULL) ) {
	printf("Can't set signal\n");
    }
}

#else	/* }{ VIDEOSYNC */

#define InitVideoSync()

#endif	/* } !VIDEOSYNC */

/*---------------------------------------------------------------------------*/

/*
**	Setup my vga-mode.
*/
static void SVGA_SetupVideoMode(void)
{
    vga_setmode(VGAMODE);
#ifdef NEW_SVGA
    vga_setlogicalwidth(1024);
    {
	int i;
	for( i=0; i<8; ++i ) {
	    vga_setpage(i);
	    LongFill(VideoMemory,0x00000000,(64*1024)/4);
	}
    }
#endif
}

/*---------------------------------------------------------------------------*/

#define SIG_REL	SIGUSR1
#define SIG_ACQ	SIGUSR2

/*
**	this is the signal handler for when the user screen flips
**	away from us.
*/
static void rel_screen(int sig)
{
    signal(SIG_REL,rel_screen);
    vga_setlogicalwidth(640);
    vga_setpage(0);
    vga_setmode(TEXT);
    VideoIsDisplayed=0;
    SuspendSound();
    /*
    ** Tell the video driver that you have saved your state
    ** and it can now have the card to switch to the new screen.
    ** The video driver waits (forever) for this ioctl before
    ** it will complete the screen switch requested by the user.
    ** If you don't make this ioctl the screen switcher will
    ** be wedged until it gets one.  It is best to have a
    ** small one line reldisp.c program to unwedge your screen
    ** switcher when development programs screw up from time
    ** to time.
    */
    ioctl(0,VT_RELDISP,1);

    SuspendKeyboard();
#ifdef NEW_SVGA
    pause();
#endif
}

/*
**	this is the signal handler for when the user screen flips
**	back to us.
*/
static void acq_screen(int sig)
{
    signal(SIG_ACQ,acq_screen);

    SVGA_SetupVideoMode();

    InitPalette();
    VicOverscanColor=-1;
    VideoIsDisplayed=1;
    ResumeSound();
    /*
    **	Tell the video driver that you have restored your state
    **	and screen switching can now continue.
    */
    ioctl(0,VT_RELDISP,VT_ACKACQ);

    ResumeKeyboard();
}

/*
**	Turn console switching on.
*/
void SwitchingOn(void)
{
    struct vt_mode smode;

    /*
    **	Set up to catch the screen switch signals.
    */
    signal(SIG_REL,rel_screen);
    signal(SIG_ACQ,acq_screen);

    /*
    ** Set up the data structure that asks the driver
    ** to send you signals when the screens are switched.
    ** mode == VT_PROCESS means send screen switch signals.
    ** mode == VT_AUTO means turn off screen switch signals (regular mode).
    ** relsig == the signal you want when the user switches away.
    ** acqsig == the signal you want when the user switches back to you.
    */
    smode.mode  =VT_PROCESS;
    smode.waitv =0;			/* not implemented, reserved */
    smode.relsig=SIG_REL;
    smode.acqsig=SIG_ACQ;
    smode.frsig =SIGINT;		/* not implemented, reserved */

    if( ioctl(0,VT_SETMODE,&smode)==-1 ) {
	perror("Screen switch signal ioctl VT_SETMODE");
	abort();
    }
}

/*
**	Turn console switching off.
*/
void SwitchingOff(void)
{
    struct vt_mode smode;

    smode.mode  =VT_AUTO;
    smode.waitv =0;			/* not implemented, reserved */
    smode.relsig=0;
    smode.acqsig=0;
    smode.frsig =SIGINT;		/* not implemented, reserved */

    if( ioctl(0,VT_SETMODE,&smode)==-1 ) {
	perror("Screen switch signal ioctl VT_SETMODE");
	abort();
    }

    signal(SIG_REL,SIG_DFL);
    signal(SIG_ACQ,SIG_DFL);
}

/*---------------------------------------------------------------------------*/

/*
**	Setup video card for C64 emulator.
*/
void EnterVideo(int argc,char** argv,char* geometry)
{
    extern int __svgalib_driver_report;

    iopl(3);
#ifndef AOUT
    // FIXME: MY OLD SGVALIB VERSION: didn't have it
    __svgalib_driver_report = 0;
#endif
    vga_init();
    iopl(3);		/* SVGALIB BUGFIX:
			** Stupid ET4000W32 detection does give up IOPL
			** although it shouldn't touch it!
			*/
    setuid(getuid());	/* finally give up SUSER priv */

    SVGA_SetupVideoMode();

    SwitchingOff();	/* turn off svgalib's switch handlers.. */

    SwitchingOn();	/* install our own instead */
    InitPalette();
    VideoIsDisplayed=1;

    InitVideoSync();
}

/*
**	Restore video card.
*/
void LeaveVideo(void)
{
    if( VideoIsDisplayed ) {
	vga_setlogicalwidth(640);
	vga_setpage(0);
	vga_setmode(TEXT);
	SwitchingOff();
	VideoIsDisplayed=0;
    }
}

/*
**	Suspend video.
*/
void SuspendVideo(void)
{
    LeaveVideo();
}

/*
**	Resume video.
*/
void ResumeVideo(void)
{
    if( !VideoIsDisplayed ) {
	SwitchingOn();
	SVGA_SetupVideoMode();

	InitPalette();
	VideoIsDisplayed=1;
    }
    VicOverscanColor=-1;
    /*
    **	Contents automatic restored after 1/50s or 1/60s.
    */
}

#endif	/* } SVGALIB */
