/*
 *	The C64 emulator
 *
 *	Copyright 1992-96 by ALE.
 *	written by Lutz Sammer.
 *
 *-----------------------------------------------------------------------------
 * $Id: vic.c,v 1.15 1996/07/13 00:43:39 johns Exp root $
 * $Log: vic.c,v $
 * Revision 1.15  1996/07/13 00:43:39  johns
 * Correct idle state display written.
 *
 * Revision 1.14  1996/07/04 21:41:33  johns
 * New display enable detection, frame end changed for NTSC.
 *
 * Revision 1.13  1996/06/21 01:48:41  johns
 * Added Overscan cache, new interrupt handline, correct sid cycles.
 * Moved some range checks 63 us.
 *
 * Revision 1.12  1996/05/03 17:47:19  johns
 * Cleanup, Added open-borders support, Hardware depend moved to own files,
 * Changed mob handling, Added text display for messages.
 *
 * Revision 1.11  1994/12/18  00:32:11  root
 * non-gcc feature macro support, icc support
 *
 * Revision 1.10  1994/06/02  14:54:24  johns
 * version 1.07 ci
 *
 * Revision 1.9  1993/12/30  11:48:12  johns
 * go32 changes
 *
 * Revision 1.8  1993/08/31  20:30:03  johns
 * go32 initial support
 *
 * Revision 1.7  1993/06/13  12:26:14  johns
 * final vt-switching mods
 *
 * Revision 1.6  1993/05/09  17:07:52  johns
 * Checkin after Wuebbel Patches
 *
 * Revision 1.5  1993/01/05  12:41:24  johns
 * Checkin before applying wuebbel patches
 *
 * Revision 1.4  1992/07/28  19:47:42  johns
 * Sparc changes added.
 *
 * Revision 1.3  1992/07/20  04:26:49  johns
 * Sprite emulation in speed improved.
 * Privat mode for emulator written.
 *
 * Revision 1.2  1992/07/13  04:36:26  johns
 * Rasterline irq now an action, EmulVic now a variable -> State.
 *
 * Revision 1.1  1992/07/11  21:52:24  johns
 * Initial revision
 *
 *-----------------------------------------------------------------------------
 */

#include "c64.h"
#include "vic.h"
#include "sid.h"
#include VIDEO_H

#include <stdio.h>
#include <string.h>
#include <signal.h>

#if !defined(PCSOLARIS)

/* LAST TEST: PentiumPro assembler <-> C, 273% <-> 288% */
#define	MULTI_COLOR_TEXT_DL		/* assembler version in mctdl*.S */

#endif

#define VIC_FIRST_DISPLAYED	0x013	/* first displayed line */

#if LINES_PER_FRAME==312
#define VIC_LAST_DISPLAYED	0x11B	/* last displayed line */
#else
#define VIC_LAST_DISPLAYED	(LINES_PER_FRAME-2)
#endif

/*-----------------------------------------------------------------------------
 *	VIC variables
 *---------------------------------------------------------------------------*/

/*
**	I mixed here locals and globals, to get same cache lines!
*/

char VicOverscanCache[LINES_PER_FRAME];	/* to reduce overscan draws */
char VicDrawOverscan=1;			/* redraw overscan area */

/*
**	8=Scrolling 24=Sprite 320=Displayed 48=Sprite
*/
unsigned char DispBuf[8+24+320+48 +8];

unsigned int VicLineBuffer[40];		/* video line buffer */

static char VicMessageBuffer[41];	/* Message buffer */
static int VicMessageCount;		/* Frames to display message */
static int VicLedCount;			/* VIC LED counter HZ */
static int VicLedStart;			/* VIC LED start of counter HZ */
static int VicLedOnOff;			/* VIC LED state */

void (*VerticalRetraceHook)(void);	/* Insert vertical retrace functions*/

int VicOverscanColor;			/* Overscan color to be set */

int VideoSyncSpeed=0;			/* Video sync with % rate, 0=off */

int VicEmulateRate=VIC_EMULATE_RATE;	/* Calculate frame if */
static int VicEmulateCount;		/* EmulateCount counts to zero */
int VicRefreshRate=VIC_REFRESH_RATE;	/* Refresh frame if */
static int VicRefreshCount;		/* RefreshCount counts to zero */

int VicIrqRasterLine;			/* Irq raster line */
unsigned VicIrqAction;			/* Irq raster line action */
unsigned VicAction;			/* Emulation register */

void (*EmulVic)(void);			/* Emulalion state */
static void (*EmulVicNext)(void);	/* Emulalion state for open borders */
void (*VicDisplayLine)(void);		/* Display line function */

int VicRasterLine;			/* Current raster line */
int VicScanLine;			/* Current scan line */

int VicOffset;				/* Current video offset */
int VicYOffset;				/* Y offset raster lines */
int VicXOffset;				/* X offset raster points */

static unsigned char* VideoPointer;	/* current pointer in video memory */

unsigned char* VicBank;			/* Vic bank 0 16 32 48 */
unsigned char* VicVRam;			/* Video ram */
unsigned char* VicBRam;			/* Bitmap ram */
unsigned char* VicCRam;			/* Character rom */
unsigned char* VicColorRam;		/* Current color ram pointer */

int VicFetch=VIC_FETCH_DEFAULT;		/* Fetch position cycles irq->fetch*/
int VicFetchAdd=VIC_FETCH_ADD_DEFAULT;	/* Fetch position add cmp->fetch */

unsigned long VicExterior;		/* Exterior Color */
unsigned long VicBackground0;		/* Background #0 Color */
unsigned long VicBackground1;		/* Background #1 Color */
unsigned long VicBackground2;		/* Background #2 Color */
unsigned long VicBackground3;		/* Background #3 Color */

struct sprite VicSprite[8];		/* 8 Sprites data */
int VicLastSprite;			/* Last sprite number BA calculation */

#ifdef NEW_SPRITES
unsigned char	VicSpriteDMA[256];	/* sprite dma flags */
#endif

static void VicState00(void);

static void VicState01C(void);
static void VicState01D(void);

static void VicState02A(void);
static void VicState02B(void);
static void VicState02C(void);
static void VicState02D(void);

static void VicState03A(void);
static void VicState03B(void);
static void VicState03C(void);
static void VicState03D(void);

static void VicState04A(void);

static void VicState05C(void);
static void VicState05D(void);

static void VicState06(void);

/*-----------------------------------------------------------------------------
 *	Vic calculation tables
 *---------------------------------------------------------------------------*/

/*
**	Table for multi-color modes	Pixel1+2
*/
static unsigned char CONST MCT0[512] = {
    0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,	/* 00 */
    0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,	/* 10 */
    0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,	/* 20 */
    0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,	/* 30 */
    1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,	/* 40 */
    1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,	/* 50 */
    1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,	/* 60 */
    1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,	/* 70 */
    3, 3, 3, 3,  3, 3, 3, 3,  3, 3, 3, 3,  3, 3, 3, 3,	/* 80 */
    3, 3, 3, 3,  3, 3, 3, 3,  3, 3, 3, 3,  3, 3, 3, 3,	/* 90 */
    3, 3, 3, 3,  3, 3, 3, 3,  3, 3, 3, 3,  3, 3, 3, 3,	/* A0 */
    3, 3, 3, 3,  3, 3, 3, 3,  3, 3, 3, 3,  3, 3, 3, 3,	/* B0 */
    2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,	/* C0 */
    2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,	/* D0 */
    2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,	/* E0 */
    2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,	/* F0 */

    0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,	/* 00 */
    0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,	/* 10 */
    0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,	/* 20 */
    0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,	/* 30 */
    5, 5, 5, 5,  5, 5, 5, 5,  5, 5, 5, 5,  5, 5, 5, 5,	/* 40 */
    5, 5, 5, 5,  5, 5, 5, 5,  5, 5, 5, 5,  5, 5, 5, 5,	/* 50 */
    5, 5, 5, 5,  5, 5, 5, 5,  5, 5, 5, 5,  5, 5, 5, 5,	/* 60 */
    5, 5, 5, 5,  5, 5, 5, 5,  5, 5, 5, 5,  5, 5, 5, 5,	/* 70 */
    7, 7, 7, 7,  7, 7, 7, 7,  7, 7, 7, 7,  7, 7, 7, 7,	/* 80 */
    7, 7, 7, 7,  7, 7, 7, 7,  7, 7, 7, 7,  7, 7, 7, 7,	/* 90 */
    7, 7, 7, 7,  7, 7, 7, 7,  7, 7, 7, 7,  7, 7, 7, 7,	/* A0 */
    7, 7, 7, 7,  7, 7, 7, 7,  7, 7, 7, 7,  7, 7, 7, 7,	/* B0 */
    2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,	/* C0 */
    2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,	/* D0 */
    2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,	/* E0 */
    2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,	/* F0 */
};

/*
**	Table for multi-color modes	Pixel2+3
*/
static unsigned char CONST MCT1[512] = {
    0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  /* 00 */
    1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  /* 10 */
    3, 3, 3, 3,  3, 3, 3, 3,  3, 3, 3, 3,  3, 3, 3, 3,  /* 20 */
    2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  /* 30 */
    0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  /* 40 */
    1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  /* 50 */
    3, 3, 3, 3,  3, 3, 3, 3,  3, 3, 3, 3,  3, 3, 3, 3,  /* 60 */
    2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  /* 70 */
    0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  /* 80 */
    1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  /* 90 */
    3, 3, 3, 3,  3, 3, 3, 3,  3, 3, 3, 3,  3, 3, 3, 3,  /* A0 */
    2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  /* B0 */
    0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  /* C0 */
    1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  /* D0 */
    3, 3, 3, 3,  3, 3, 3, 3,  3, 3, 3, 3,  3, 3, 3, 3,  /* E0 */
    2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  /* F0 */

    0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  /* 00 */
    5, 5, 5, 5,  5, 5, 5, 5,  5, 5, 5, 5,  5, 5, 5, 5,  /* 10 */
    7, 7, 7, 7,  7, 7, 7, 7,  7, 7, 7, 7,  7, 7, 7, 7,  /* 20 */
    2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  /* 30 */
    0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  /* 40 */
    5, 5, 5, 5,  5, 5, 5, 5,  5, 5, 5, 5,  5, 5, 5, 5,  /* 50 */
    7, 7, 7, 7,  7, 7, 7, 7,  7, 7, 7, 7,  7, 7, 7, 7,  /* 60 */
    2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  /* 70 */
    0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  /* 80 */
    5, 5, 5, 5,  5, 5, 5, 5,  5, 5, 5, 5,  5, 5, 5, 5,  /* 90 */
    7, 7, 7, 7,  7, 7, 7, 7,  7, 7, 7, 7,  7, 7, 7, 7,  /* A0 */
    2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  /* B0 */
    0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0,  /* C0 */
    5, 5, 5, 5,  5, 5, 5, 5,  5, 5, 5, 5,  5, 5, 5, 5,  /* D0 */
    7, 7, 7, 7,  7, 7, 7, 7,  7, 7, 7, 7,  7, 7, 7, 7,  /* E0 */
    2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  2, 2, 2, 2,  /* F0 */
};

/*
**	Table for multi-color modes	Pixel4+5
*/
static unsigned char CONST MCT2[512] = {
    0, 0, 0, 0,  1, 1, 1, 1,  3, 3, 3, 3,  2, 2, 2, 2,  /* 00 */
    0, 0, 0, 0,  1, 1, 1, 1,  3, 3, 3, 3,  2, 2, 2, 2,  /* 10 */
    0, 0, 0, 0,  1, 1, 1, 1,  3, 3, 3, 3,  2, 2, 2, 2,  /* 20 */
    0, 0, 0, 0,  1, 1, 1, 1,  3, 3, 3, 3,  2, 2, 2, 2,  /* 30 */
    0, 0, 0, 0,  1, 1, 1, 1,  3, 3, 3, 3,  2, 2, 2, 2,  /* 40 */
    0, 0, 0, 0,  1, 1, 1, 1,  3, 3, 3, 3,  2, 2, 2, 2,  /* 50 */
    0, 0, 0, 0,  1, 1, 1, 1,  3, 3, 3, 3,  2, 2, 2, 2,  /* 60 */
    0, 0, 0, 0,  1, 1, 1, 1,  3, 3, 3, 3,  2, 2, 2, 2,  /* 70 */
    0, 0, 0, 0,  1, 1, 1, 1,  3, 3, 3, 3,  2, 2, 2, 2,  /* 80 */
    0, 0, 0, 0,  1, 1, 1, 1,  3, 3, 3, 3,  2, 2, 2, 2,  /* 90 */
    0, 0, 0, 0,  1, 1, 1, 1,  3, 3, 3, 3,  2, 2, 2, 2,  /* A0 */
    0, 0, 0, 0,  1, 1, 1, 1,  3, 3, 3, 3,  2, 2, 2, 2,  /* B0 */
    0, 0, 0, 0,  1, 1, 1, 1,  3, 3, 3, 3,  2, 2, 2, 2,  /* C0 */
    0, 0, 0, 0,  1, 1, 1, 1,  3, 3, 3, 3,  2, 2, 2, 2,  /* D0 */
    0, 0, 0, 0,  1, 1, 1, 1,  3, 3, 3, 3,  2, 2, 2, 2,  /* E0 */
    0, 0, 0, 0,  1, 1, 1, 1,  3, 3, 3, 3,  2, 2, 2, 2,  /* F0 */

    0, 0, 0, 0,  5, 5, 5, 5,  7, 7, 7, 7,  2, 2, 2, 2,  /* 00 */
    0, 0, 0, 0,  5, 5, 5, 5,  7, 7, 7, 7,  2, 2, 2, 2,  /* 10 */
    0, 0, 0, 0,  5, 5, 5, 5,  7, 7, 7, 7,  2, 2, 2, 2,  /* 20 */
    0, 0, 0, 0,  5, 5, 5, 5,  7, 7, 7, 7,  2, 2, 2, 2,  /* 30 */
    0, 0, 0, 0,  5, 5, 5, 5,  7, 7, 7, 7,  2, 2, 2, 2,  /* 40 */
    0, 0, 0, 0,  5, 5, 5, 5,  7, 7, 7, 7,  2, 2, 2, 2,  /* 50 */
    0, 0, 0, 0,  5, 5, 5, 5,  7, 7, 7, 7,  2, 2, 2, 2,  /* 60 */
    0, 0, 0, 0,  5, 5, 5, 5,  7, 7, 7, 7,  2, 2, 2, 2,  /* 70 */
    0, 0, 0, 0,  5, 5, 5, 5,  7, 7, 7, 7,  2, 2, 2, 2,  /* 80 */
    0, 0, 0, 0,  5, 5, 5, 5,  7, 7, 7, 7,  2, 2, 2, 2,  /* 90 */
    0, 0, 0, 0,  5, 5, 5, 5,  7, 7, 7, 7,  2, 2, 2, 2,  /* A0 */
    0, 0, 0, 0,  5, 5, 5, 5,  7, 7, 7, 7,  2, 2, 2, 2,  /* B0 */
    0, 0, 0, 0,  5, 5, 5, 5,  7, 7, 7, 7,  2, 2, 2, 2,  /* C0 */
    0, 0, 0, 0,  5, 5, 5, 5,  7, 7, 7, 7,  2, 2, 2, 2,  /* D0 */
    0, 0, 0, 0,  5, 5, 5, 5,  7, 7, 7, 7,  2, 2, 2, 2,  /* E0 */
    0, 0, 0, 0,  5, 5, 5, 5,  7, 7, 7, 7,  2, 2, 2, 2,  /* F0 */
};

/*
**	Table for multi-color modes	Pixel6+7
*/
static unsigned char CONST MCT3[512] = {
    /* 4 */
    0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  /* 00 */
    0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  /* 10 */
    0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  /* 20 */
    0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  /* 30 */
    0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  /* 40 */
    0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  /* 50 */
    0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  /* 60 */
    0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  /* 70 */
    0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  /* 80 */
    0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  /* 90 */
    0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  /* A0 */
    0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  /* B0 */
    0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  /* C0 */
    0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  /* D0 */
    0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  /* E0 */
    0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  0, 1, 3, 2,  /* F0 */

    0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  /* 00 */
    0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  /* 10 */
    0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  /* 20 */
    0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  /* 30 */
    0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  /* 40 */
    0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  /* 50 */
    0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  /* 60 */
    0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  /* 70 */
    0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  /* 80 */
    0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  /* 90 */
    0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  /* A0 */
    0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  /* B0 */
    0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  /* C0 */
    0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  /* D0 */
    0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  /* E0 */
    0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  0, 5, 7, 2,  /* F0 */
};

/*
**	C64 byte Bitmap -> 2 Longs
**		(512K Table quicker)
**		A nibble is converted into a long (4 pixel).
*/
static unsigned long ColorT[256*16];

/*---------------------------------------------------------------------------*/

#ifdef BIG_ENDIAN
#   define __BIT_1	8
#   define __BIT_2	4
#   define __BIT_3	2
#   define __BIT_4	1
#endif
#ifdef LITTLE_ENDIAN
#   define __BIT_1	1
#   define __BIT_2	2
#   define __BIT_3	4
#   define __BIT_4	8
#endif

/*
**	Calculate table for standard text and bitmap mode.
*/
static void InitTranslation(void)
{
    int i,j;

    for( i=0; i<256; ++i ) {
	for( j=0; j<16; ++j ) {
	    long b;
	    long f;

	    b=(i>>4)|0x10;
	    f=(i&0xF);
	    ColorT[i*16+j]=(b<<24)|(b<<16)|(b<<8)|b;
	    if( j&__BIT_4 ) {
		ColorT[i*16+j]&=0xFFFFFF00;
		ColorT[i*16+j]|=f;
	    }
	    if( j&__BIT_3 ) {
		ColorT[i*16+j]&=0xFFFF00FF;
		ColorT[i*16+j]|=(f<<8);
	    }
	    if( j&__BIT_2 ) {
		ColorT[i*16+j]&=0xFF00FFFF;
		ColorT[i*16+j]|=(f<<16);
	    }
	    if( j&__BIT_1 ) {
		ColorT[i*16+j]&=0x00FFFFFF;
		ColorT[i*16+j]|=(f<<24);
	    }
	}
    }
}

/*-----------------------------------------------------------------------------
 *	VIC global functions
 *---------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
 *	Vic 1541 LED support
 *---------------------------------------------------------------------------*/

/*
**	Turn 1541 LED on.
*/
void LedOn(void)
{
    VicLedCount=FRAMES_PER_SECOND;
    VicLedStart=0;
    VicLedOnOff=1;
}

/*
**	Turn 1541 LED off.
*/
void LedOff(void)
{
    VicLedCount=1;
    VicLedStart=0;
    VicLedOnOff=0;
}

/*
**	Turn 1541 LED flashing on.
*/
void LedFlash(void)
{
    VicLedCount=VicLedStart=FRAMES_PER_SECOND;
    VicLedOnOff=1;
}

/*-----------------------------------------------------------------------------
 *	Vic raster line interrupt
 *---------------------------------------------------------------------------*/

/*
**	Emulate raster line IRQ action.
*/
void EmulRasterLineIrq(void)
{
    /*
    **	Calculate the next interrupt point.
    */
    VicIrqAction+=LINES_PER_FRAME*CYCLES_PER_LINE;
    ChangeAction(VicIrqAction,EmulRasterLineIrq);

    VIC[0x19]|=0x01;
    if( VIC[0x1A]&0x01 ) {		/* Rasterline IRQ enabled */
	VIC[0x19]|=0x80;
	Irq();
    }
}

/*-----------------------------------------------------------------------------
 *	Vic monitor support
 *---------------------------------------------------------------------------*/

#ifdef MONITOR	/* { */

/*
**	Display vic emulator internals.
*/
void VicInfo(void)
{
    printf("Vic:  Action=%d IrqAction=%d\n",VicAction,VicIrqAction);
    printf("  Rasterline=%d ($%03X) IrqRasterline=%d ($%03X)",
	VicRasterLine,VicRasterLine,
	VicIrqRasterLine,VicIrqRasterLine);
    printf("  Scanline=%d Matrixline=%d\n",VicScanLine,VicOffset);
    printf("  VicFetch %d VicFetchAdd %d\n",VicFetch,VicFetchAdd);
    printf("  Bank $%04X V-Ram $%04X B-Ram $%04X C-Ram $%04X\n",
	(unsigned int)(VicBank-Ram),
	(unsigned int)(VicVRam-Ram),
	(unsigned int)(VicBRam-Ram),
	VicCRam==CharacterRom ? 0xC000 : 
	VicCRam==CharacterRom+0x0800 ? 0xC800 : (unsigned int)(VicCRam-Ram));
    printf("  Displaymode: ");
    if( VicDisplayLine==TextDL )
	printf("Text\n");
    else if( VicDisplayLine==MultiColorTextDL )
	printf("Multi-color-text\n");
    else if( VicDisplayLine==ExtendedColorTextDL )
	printf("Extended-color-text\n");
    else if( VicDisplayLine==BitmapDL )
	printf("Bitmap\n");
    else if( VicDisplayLine==MultiColorBitmapDL )
	printf("Multi-color-bitmap\n");
    else
	printf("Illegal\n");
}

#endif	/* } MONITOR */

/*-----------------------------------------------------------------------------
 *	VIC Emulation
 *---------------------------------------------------------------------------*/

void VerticalRetraceNoop(void) {};	/* no operation for vertical hook */
#define VicNoop		VerticalRetraceNoop

/*
**	Initialise VIC.
*/
void EnterVic(void)
{
    InitTranslation();			/* Longword color translation */
#if 1
    VerticalRetraceHook=VerticalRetraceNoop;
#else
    { extern void TimePer20s(void);
     extern void TimePerFrame(void);
    VerticalRetraceHook=TimePerFrame;
    }
#endif
    VicVRam=Ram+0x400;
    VicCRam=CharacterRom;
}

/*
**	Reset of VIC.
*/
void VicReset(void)
{
    int i;

    EmulVic=VicState06;
    EmulVicNext=VicState01C;
    VicDisplayLine=TextDL;
    VicIrqRasterLine=VicRasterLine=0;
    VicEmulateCount=1;
    VicRefreshCount=1;
    VicAction=0;
    VicIrqAction=-1;

    // VicLastSprite=10;
    LongFill(DispBuf,0x10101010,24/4);	/* Clear sprite work area */
    LongFill(DispBuf+0x20+320,0x10101010,48/4);

#ifdef NEW_SPRITES
    LongFill(VicSpriteDMA,0,256/4);
    for( i=0; i<8; ++i ) {
	VicSprite[i].Counter=VicSprite[i].Advance=0;
    }
#endif

    memset(VicOverscanCache,0xFF,sizeof(VicOverscanCache));
}

/*-----------------------------------------------------------------------------
 *	VIC action handling
 *---------------------------------------------------------------------------*/

#ifdef NEW_ACTION

#undef ChangeAction

#define ChangeAction(a,f)	\
do{				\
    Action=a;			\
    ActionFunction=f;		\
    if( a>HwAction ) {		\
	UpdateAction();		\
    }				\
}while( 0 )

#endif

/*** FIXME: { START OF CLEANED UP CODE ****/

/*-----------------------------------------------------------------------------
 *	VIC message
 *---------------------------------------------------------------------------*/

/*
**	Display text at (x,y).
*/
void VicText(int x,int y,unsigned char* s)
{
    unsigned char* p;
    int c;
    int px,py,b;

    while( (c=*s++) ) {
	if( x<0 || x>HSIZE-8 || y<0 || y>VSIZE-8 ) {
	    printf("Give correct coordinates\n");
	    return;
	}
	if( '@'<c && c<='_' ) {
	    c-='@';
	}
	if( 0x60<c && c<=0x7F ) {
	    c=c-0x60+0x100;
	}
	p=VideoMemory+x+y*HSIZE;
	for( py=0; py<8; ++py ) {
	    b=CharacterRom[py+c*8];
	    for( px=0; px<8; ++px ) {
		if( b&(1<<(7-px)) ) {
		    VideoPut(p+px,VIC2VIDEO(0x01));
		} else {
		    VideoPut(p+px,VIC2VIDEO(0x00));
		}
	    }
	    p+=HSIZE;
	}
	x+=8;
    }
}

/*
**	Display message for count frames.
*/
void VicMessage(const char* message,int count)
{
    strncpy(VicMessageBuffer,message,sizeof(VicMessageBuffer)-1);
    VicMessageCount=count;
}

/*-----------------------------------------------------------------------------
 *	Vic scanline utilities
 *---------------------------------------------------------------------------*/

/*
**	Handle a scanline.
*/
#define ONE_LINE() \
    CHAR( 0); CHAR( 1); CHAR( 2); CHAR( 3); CHAR( 4);	\
    CHAR( 5); CHAR( 6); CHAR( 7); CHAR( 8); CHAR( 9);	\
    CHAR(10); CHAR(11); CHAR(12); CHAR(13); CHAR(14);	\
    CHAR(15); CHAR(16); CHAR(17); CHAR(18); CHAR(19);	\
    CHAR(20); CHAR(21); CHAR(22); CHAR(23); CHAR(24);	\
    CHAR(25); CHAR(26); CHAR(27); CHAR(28); CHAR(29);	\
    CHAR(30); CHAR(31); CHAR(32); CHAR(33); CHAR(34);	\
    CHAR(35); CHAR(36); CHAR(37); CHAR(38); CHAR(39);

/*-----------------------------------------------------------------------------
 *	Vic scanline functions
 *---------------------------------------------------------------------------*/

#ifndef TEXT_DL
/*
**	Standart text mode display line
**		1	=	ColorRAM color
**		0	=	Background0 color
*/
void TextDL(void)
{
    unsigned long* ctable;
    unsigned char* fram;
    unsigned char* cram;

    cram=VicCRam+VicScanLine;
    fram=VicColorRam;

    ctable=ColorT+VIC[0x21]*256;	/* start of color table, background0 */

#define CHAR(x) \
    {								\
	unsigned long* ct;					\
	unsigned c;						\
								\
	ct=ctable+fram[x];					\
	c=cram[VicLineBuffer[x]];				\
	((unsigned long*)(DispBuf+0x20))[x*2+0]=ct[c>>4];	\
	((unsigned long*)(DispBuf+0x20))[x*2+1]=ct[c&0xF];	\
    }

    ONE_LINE();

#undef CHAR
}
#endif

#ifndef MULTI_COLOR_TEXT_DL
/* assembler version in mctdl.S */
/*
**	Multi-color text mode display line
**
**	0 1 2 3 4 5 6 7 8
**	0 0 3 3 0 1 1 2 2
**
**	00	Color 0		Multi-color
**	01	Color 5
**	10	Color 7
**	11	Color 2
**
**	00	Color 0		Normal
**	01	Color 1
**	10	Color 3
**	11	Color 2
*/
void MultiColorTextDL(void)
{
    unsigned c;
    unsigned char* fram;
    unsigned char* cram;
    static unsigned char color[9+2];

    cram=VicCRam+VicScanLine;
    fram=VicColorRam;

    // COULD be moved into memory interface
    color[0]=color[1]=color[4]=VicBackground0;
    color[5]=color[6]=VIC[0x22]|0x10;	/* VicBackground1 */
    color[7]=color[8]=VIC[0x23];	/* VicBackground2 */

#if 1
#define CHAR(x) \
	color[2]=color[3]=(fram[x]>>4)&0x7;		\
	c=cram[VicLineBuffer[x]]|((fram[x]<<1)&0x100);	\
	((unsigned short*)(DispBuf+0x20))[x*4+0]=	\
		*(unsigned short*)(color+MCT0[c]);	\
	((unsigned short*)(DispBuf+0x20))[x*4+1]=	\
		*(unsigned short*)(color+MCT1[c]);	\
	((unsigned short*)(DispBuf+0x20))[x*4+2]=	\
		*(unsigned short*)(color+MCT2[c]);	\
	((unsigned short*)(DispBuf+0x20))[x*4+3]=	\
		*(unsigned short*)(color+MCT3[c])
#else	/* LONG VERSION slower on i486/dx4 */
#define CHAR(x) \
	color[2]=color[3]=(fram[x]>>4)&0x7;			\
	c=cram[VicLineBuffer[x]]|((fram[x]&0x80)<<1);		\
	((unsigned long*)(DispBuf+0x20))[x*2+0]=		\
		  *(unsigned short*)(color+MCT0[c])		\
		| *(unsigned short*)(color+MCT1[c])<<16;	\
	((unsigned long*)(DispBuf+0x20))[x*2+1]=		\
		  *(unsigned short*)(color+MCT2[c])		\
		| *(unsigned short*)(color+MCT3[c])<<16
#endif

    ONE_LINE();

#undef CHAR
}
#endif

#ifndef EXTENDED_COLOR_TEXT_DL
/*
**	Extended-color text mode display line
**		1	=	Colorram
**		0	=	Background from 2 high bits
**			00	VicBackground0
**			01	VicBackground1
**			10	VicBackground2
**			11	VicBackground3
*/
void ExtendedColorTextDL(void)
{
    unsigned c;
    unsigned long* ct;
    unsigned char* fram;
    unsigned char* cram;

    cram=VicCRam+VicScanLine;
    fram=VicColorRam;

#define CHAR(x) \
	c=VicLineBuffer[x];					\
	ct=ColorT+VIC[0x21+(c>>9)]*256+fram[x];			\
	c=cram[c&0x1F8];					\
	((unsigned long*)(DispBuf+0x20))[x*2+0]=ct[c>>4];	\
	((unsigned long*)(DispBuf+0x20))[x*2+1]=ct[c&0xF]

    ONE_LINE();
#undef CHAR
}
#endif


#ifndef BITMAP_DL
/*
**	Standart bitmap mode display line
**		1	=	ColorRAM color
**		0	=	Background0 color
*/
void BitmapDL(void)
{
    unsigned char c;
    unsigned char* bram;
    unsigned long* ct;

    bram=VicBRam+VicScanLine+VicOffset*8;

#define CHAR(x) \
	ct=ColorT+((VicLineBuffer[x]>>3)&0x0F)*256		\
		+((VicLineBuffer[x]>>3)&0xF0);			\
	c=bram[x*8];						\
	((unsigned long*)(DispBuf+0x20))[x*2+0]=ct[c>>4];	\
	((unsigned long*)(DispBuf+0x20))[x*2+1]=ct[c&0xF]

    ONE_LINE();

#undef CHAR
}
#endif

#ifndef MULTI_COLOR_BITMAP_DL
/*
**	Multi-color bitmap mode display line
**		FIXME: this function could be improved!
*/
void MultiColorBitmapDL(void)
{
    unsigned char c;
    unsigned char* bram;
    unsigned char* fram;
    unsigned char color[9+2];

    bram=VicBRam+VicScanLine+VicOffset*8;
    fram=VicColorRam;

    color[0]=color[1]=VicBackground0;				/* 00 */

#if 1
#define CHAR(x) \
	color[2]=color[3]=fram[x]>>4;				/* 11 */\
	color[5]=color[6]=((VicLineBuffer[x]>>7)&0xF)|0x10;	/* 01 */\
	color[7]=color[8]=((VicLineBuffer[x]>>3)&0xF);		/* 10 */\
	c=bram[x*8];							\
	((unsigned short*)(DispBuf+0x20))[x*4+0]=			\
		*(unsigned short*)(color+MCT0[0x100+c]);		\
	((unsigned short*)(DispBuf+0x20))[x*4+1]=			\
		*(unsigned short*)(color+MCT1[0x100+c]);		\
	((unsigned short*)(DispBuf+0x20))[x*4+2]=			\
		*(unsigned short*)(color+MCT2[0x100+c]);		\
	((unsigned short*)(DispBuf+0x20))[x*4+3]=			\
		*(unsigned short*)(color+MCT3[0x100+c])

#else	/* LONG VERSION slower on i486/dx4 */
#define CHAR(x) \
	color[2]=color[3]=fram[x]>>4;				/* 11 */\
	color[5]=color[6]=((VicLineBuffer[x]>>7)&0xF)|0x10;	/* 01 */\
	color[7]=color[8]=((VicLineBuffer[x]>>3)&0xF);		/* 10 */\
	c=bram[x*8];							\
	((unsigned long*)(DispBuf+0x20))[x*2+0]=			\
		  *(unsigned short*)(color+MCT0[0x100+c])		\
		| *(unsigned short*)(color+MCT1[0x100+c])<<16;		\
	((unsigned long*)(DispBuf+0x20))[x*2+1]=			\
		  *(unsigned short*)(color+MCT2[0x100+c])		\
		| *(unsigned short*)(color+MCT3[0x100+c])<<16;
#endif

    ONE_LINE();

#undef CHAR
}
#endif

/*
**	Illegal display mode.
*/
void IllegalDL(void)
{
    LongFill(DispBuf+0x20,0x00000000,C64HSIZE/4);
}

/*
**	Idle display mode.
**
**	Display byte on VicBank+0x39FF (ECM) or VicBank+0x3FFF (others)
**	0	VicBankground0
**	1	always black
*/
void IdleDL(void)
{
    unsigned c;
    unsigned long fill1;
    unsigned long fill2;

    if( VicDisplayLine==IllegalDL ) {	/* illegal modes have no idle */
	LongFill(DispBuf+0x20,0x00000000,C64HSIZE/4);
	return;
    }
    c=VicBank[VIC[0x11]&0x40 ? 0x39FF : 0x3FFF];
    fill1=ColorT[VIC[0x21]*256+(c>>4)];	/* background0 */
    fill2=ColorT[VIC[0x21]*256+(c&0xF)];

#define CHAR(x) \
	((unsigned long*)(DispBuf+0x20))[x*2+0]=fill1;	\
	((unsigned long*)(DispBuf+0x20))[x*2+1]=fill2

    ONE_LINE();

#undef CHAR
}

/*-----------------------------------------------------------------------------
 *	Vic state utilities
 *---------------------------------------------------------------------------*/

/*
**	Fetch a line 40 bytes into line buffer.
**		Multiplied by 8 for character rom index.
**		Unrolled for speed.
**	FIXME: Original C64 also fetches the color memory.
*/
#define FETCH(i) \
    (VicLineBuffer[i]=VicVRam[VicOffset+i]*8)

#define VicLineFetch() \
    VicScanLine=0;						\
    FETCH( 0); FETCH( 1); FETCH( 2); FETCH( 3); FETCH( 4);	\
    FETCH( 5); FETCH( 6); FETCH( 7); FETCH( 8); FETCH( 9);	\
    FETCH(10); FETCH(11); FETCH(12); FETCH(13); FETCH(14);	\
    FETCH(15); FETCH(16); FETCH(17); FETCH(18); FETCH(19);	\
    FETCH(20); FETCH(21); FETCH(22); FETCH(23); FETCH(24);	\
    FETCH(25); FETCH(26); FETCH(27); FETCH(28); FETCH(29);	\
    FETCH(30); FETCH(31); FETCH(32); FETCH(33); FETCH(34);	\
    FETCH(35); FETCH(36); FETCH(37); FETCH(38); FETCH(39);	\
    Cycle+=43;		/* 40 characters + 3 bus acknowledge */

/*---------------------------------------------------------------------------*/

#ifndef NEW_SPRITES

/*
**	Test if a sprite must displayed.
**
**	FIXME: This could be speedup if I use an array of functions.
**		void (*VicSprite0Func[LINES_PER_FRAME*2])(void);
**		VicSprite0Func[VicRasterLine]();
*/
#define SPRITE(x) \
  if( VicSprite[x].Counter						\
	|| ((enab&(1<<x)) && VicSprite[x].Y==(VicRasterLine&0xFF)) ) {	\
    Sprite(x);								\
  }

/*
**	Handle all sprites.
*/
#define VicAllSprites() \
do{\
    register unsigned char enab;					\
									\
    if( (enab=VIC[0x15]) ) {		/* Some sprites enabled */	\
	SPRITE(7) SPRITE(6) SPRITE(5) SPRITE(4)				\
	SPRITE(3) SPRITE(2) SPRITE(1) SPRITE(0)				\
									\
	if( VicLastSprite!=10 ) {	/* Some sprites displayed */	\
	    VicLastSprite=10;						\
									\
	    /* Clear sprite work area */				\
	    LongFill(DispBuf,0x10101010,24/4);				\
	    LongFill(DispBuf+0x20+C64HSIZE,0x10101010,48/4);		\
									\
	    /* Collision: 2 Sprite-Background, 4 Sprite-Sprite */	\
	    VIC[0x19]|=((VIC[0x1F]!=0)<<1)|((VIC[0x1E]!=0)<<2);		\
	    if( VIC[0x19]&VIC[0x1A]&0x06 ) { /* Look if collision irq */\
		VIC[0x19]|=0x80;					\
		Irq();							\
	    }								\
	}								\
    }									\
}while( 0 )

#else

#if 0

int SpriteDMA;				/* sprite DMA active */

/*
**	Handle a single sprite.
*/
#define SPRITE(x) \
if( SpriteDMA&(1<<x) ) {						\
    Sprite(x);								\
} else if( (VIC[0x15]&(1<<x)) && VicSprite[x].Y==(VicRasterLine&0xFF) ) {\
    SpriteDMA|=(1<<x);							\
    VicSprite[x].Counter=0;						\
    VicSprite[x].Advance=0;						\
    Sprite(x);								\
}

/*
**	Handle all sprites.
*/
#define VicAllSprites()		\
do{\
    SPRITE(7) SPRITE(6) SPRITE(5) SPRITE(4)				\
    SPRITE(3) SPRITE(2) SPRITE(1) SPRITE(0)				\
									\
    if( VicLastSprite!=10 ) {		/* Some sprites displayed */	\
	VicLastSprite=10;						\
									\
	/* Clear sprite work area */					\
	((unsigned long*)DispBuf)[0]=0x10101010;			\
	((unsigned long*)DispBuf)[1]=0x10101010;			\
	((unsigned long*)DispBuf)[2]=0x10101010;			\
	((unsigned long*)DispBuf)[3]=0x10101010;			\
	((unsigned long*)DispBuf)[4]=0x10101010;			\
	((unsigned long*)DispBuf)[5]=0x10101010; /* first 24 pixels */	\
									\
	LongFill(DispBuf+0x20+C64HSIZE,0x10101010,48/4);		\
									\
	/* Collision: 2 Sprite-Background, 4 Sprite-Sprite */		\
	VIC[0x19]|=((VIC[0x1F]!=0)<<1)|((VIC[0x1E]!=0)<<2);		\
	if( VIC[0x19]&VIC[0x1A]&0x06 ) { /* Look if collision irq */	\
	    VIC[0x19]|=0x80;						\
	    Irq();							\
	}								\
    }									\
}while( 0 )

#endif

#undef VicAllSprites

/*
**	Handle all sprites.
*/
static void VicAllSprites(void)
{
    int dma;
    static char sprite_cycles[256] = {
	 3,  5,  5,  7,   5,  9,  7,  9, 
	 8, 10,  9, 11,   7, 11,  9, 11, 	/* 00 */
	 8, 10, 10, 12,   9, 13, 11, 13, 
	10, 12, 11, 13,   9, 13, 11, 13, 	/* 10 */
	 8, 10, 10, 12,  10, 14, 12, 14, 
	12, 14, 13, 15,  11, 15, 13, 15, 	/* 20 */
	10, 12, 12, 14,  11, 15, 13, 15, 
	12, 14, 13, 15,  11, 15, 13, 15, 	/* 30 */
	 8, 10, 10, 12,  10, 14, 12, 14, 
	13, 15, 14, 16,  12, 16, 14, 16, 	/* 40 */
	12, 14, 14, 16,  13, 17, 15, 17, 
	14, 16, 15, 17,  13, 17, 15, 17, 	/* 50 */
	10, 12, 12, 14,  12, 16, 14, 16, 
	14, 16, 15, 17,  13, 17, 15, 17, 	/* 60 */
	12, 14, 14, 16,  13, 17, 15, 17, 
	14, 16, 15, 17,  13, 17, 15, 17, 	/* 70 */
	 8, 10, 10, 12,  10, 14, 12, 14, 
	13, 15, 14, 16,  12, 16, 14, 16, 	/* 80 */
	13, 15, 15, 17,  14, 18, 16, 18, 
	15, 17, 16, 18,  14, 18, 16, 18, 	/* 90 */
	12, 14, 14, 16,  14, 18, 16, 18, 
	16, 18, 17, 19,  15, 19, 17, 19, 	/* A0 */
	14, 16, 16, 18,  15, 19, 17, 19, 
	16, 18, 17, 19,  15, 19, 17, 19, 	/* B0 */
	10, 12, 12, 14,  12, 16, 14, 16, 
	15, 17, 16, 18,  14, 18, 16, 18, 	/* C0 */
	14, 16, 16, 18,  15, 19, 17, 19, 
	16, 18, 17, 19,  15, 19, 17, 19, 	/* D0 */
	12, 14, 14, 16,  14, 18, 16, 18, 
	16, 18, 17, 19,  15, 19, 17, 19, 	/* E0 */
	14, 16, 16, 18,  15, 19, 17, 19, 
	16, 18, 17, 19,  15, 19, 17, 19, 	/* F0 */
    };
	

    dma=VicSpriteDMA[VicRasterLine&0xFF];
    if( !dma ) {			/* nothing todo */
	return;
    }

    if( dma&0x80 )	Sprite(7);
    if( dma&0x40 )	Sprite(6);
    if( dma&0x20 )	Sprite(5);
    if( dma&0x10 )	Sprite(4);
    if( dma&0x08 )	Sprite(3);
    if( dma&0x04 )	Sprite(2);
    if( dma&0x02 )	Sprite(1);
    if( dma&0x01 )	Sprite(0);

    Cycle+=sprite_cycles[dma];

    /*
    **	Clear sprite work area
    */
    ((unsigned long*)DispBuf)[0]=0x10101010;
    ((unsigned long*)DispBuf)[1]=0x10101010;
    ((unsigned long*)DispBuf)[2]=0x10101010;
    ((unsigned long*)DispBuf)[3]=0x10101010;
    ((unsigned long*)DispBuf)[4]=0x10101010;
    ((unsigned long*)DispBuf)[5]=0x10101010; /* first 24 pixels */

    ((unsigned long*)(DispBuf+0x20+C64HSIZE))[0]=0x10101010;
    ((unsigned long*)(DispBuf+0x20+C64HSIZE))[1]=0x10101010;
    ((unsigned long*)(DispBuf+0x20+C64HSIZE))[2]=0x10101010;
    ((unsigned long*)(DispBuf+0x20+C64HSIZE))[3]=0x10101010;
    ((unsigned long*)(DispBuf+0x20+C64HSIZE))[4]=0x10101010;
    ((unsigned long*)(DispBuf+0x20+C64HSIZE))[5]=0x10101010;
    ((unsigned long*)(DispBuf+0x20+C64HSIZE))[6]=0x10101010;
    ((unsigned long*)(DispBuf+0x20+C64HSIZE))[7]=0x10101010;
    ((unsigned long*)(DispBuf+0x20+C64HSIZE))[8]=0x10101010;
    ((unsigned long*)(DispBuf+0x20+C64HSIZE))[9]=0x10101010;
    ((unsigned long*)(DispBuf+0x20+C64HSIZE))[10]=0x10101010;
    ((unsigned long*)(DispBuf+0x20+C64HSIZE))[11]=0x10101010; /* last 48 pixels*/

    /*
    **	Collision:	2 Sprite-Background, 4 Sprite-Sprite
    */
    VIC[0x19]|=((VIC[0x1F]!=0)<<1)|((VIC[0x1E]!=0)<<2);
    if( VIC[0x19]&VIC[0x1A]&0x06 ) {	/* Look if collision irq */
	VIC[0x19]|=0x80;
	Irq();
    }
}

#endif

/*
**	Function for new sprite routines.
**	Setup counters after non-displayed area.
*/
static void VicFirstSprites(void)
{
#ifndef NEW_SPRITES
    int x;

    for( x=0; x<8; ++x ) {
	VicSprite[x].Counter=0;
	if( ((VIC[0x15]&(1<<x)) && VicSprite[x].Y<=VicRasterLine
	    && (VicSprite[x].Y+((VIC[0x17]&(1<<x)) ? 41 : 20))
		>VicRasterLine) ) {
	    VicSprite[x].Counter=VicSprite[x].Y;
	    if( VIC[0x17]&(1<<x) ) {		/* Sprite expanded in Y */
		VicSprite[x].Stop=41;
	    } else {
		VicSprite[x].Stop=20;
	    }
	}
    }
#else

#if 0
    int x;

    /* FIXME: This isn't correct !! Must do more like VicAllSprites! */
    SpriteDMA=0;
    for( x=0; x<8; ++x ) {
	if( ((VIC[0x15]&(1<<x)) && VicSprite[x].Y<VicRasterLine
	    && (VicSprite[x].Y+((VIC[0x17]&(1<<x)) ? 41 : 20))
		>=VicRasterLine) ) {
	    SpriteDMA|=(1<<x);
	    if( VIC[0x17]&(1<<x) ) {		/* Sprite expanded in Y */
		VicSprite[x].Counter=((VicRasterLine-VicSprite[x].Y)/2)*3;
		VicSprite[x].Advance=(VicRasterLine-VicSprite[x].Y)&1;
	    } else {
		VicSprite[x].Counter=(VicRasterLine-VicSprite[x].Y)*3;
		VicSprite[x].Advance=0;
	    }
	}
    }
#endif

    int x;

    /* FIXME: This isn't correct !! Must do more like VicAllSprites! */
    for( x=0; x<8; ++x ) {
	if( (VIC[0x15]&(1<<x)) ) {
	    if( (VicSprite[x].Y<VicRasterLine)
		&& ((VicSprite[x].Y+((VIC[0x17]&(1<<x)) ? 41 : 20))
		    >=VicRasterLine) ) {
		VicSpriteDMA[VicRasterLine]|=(1<<x);
		if( VIC[0x17]&(1<<x) ) {	/* Sprite expanded in Y */
		    VicSprite[x].Counter=((VicRasterLine-VicSprite[x].Y)/2)*3;
		    VicSprite[x].Advance=(VicRasterLine-VicSprite[x].Y)&1;
		} else {
		    VicSprite[x].Counter=(VicRasterLine-VicSprite[x].Y)*3;
		    VicSprite[x].Advance=0;
		}
	    } else {
		VicSpriteDMA[VicSprite[x].Y]|=(1<<x);
	    }
	}
    }

#endif
}

/*
**	Function for new sprite routines.
**	Cleanup counters after non-displayed area.
*/
static void VicLastSprites(void)
{
    VicSpriteDMA[(VicRasterLine)&0xFF]=0;
    VicSpriteDMA[(VicRasterLine+1)&0xFF]=0;
    VicSprite[0].Counter=VicSprite[0].Advance=0;
    VicSprite[1].Counter=VicSprite[1].Advance=0;
    VicSprite[2].Counter=VicSprite[2].Advance=0;
    VicSprite[3].Counter=VicSprite[3].Advance=0;
    VicSprite[4].Counter=VicSprite[4].Advance=0;
    VicSprite[5].Counter=VicSprite[5].Advance=0;
    VicSprite[6].Counter=VicSprite[6].Advance=0;
    VicSprite[7].Counter=VicSprite[7].Advance=0;
}

/*---------------------------------------------------------------------------*/

/*
**	Called in vertical retrace with 50/60 Hz.
*/
static void VerticalRetrace(void)
{
    VerticalRetraceHook();
    if( Cycle>0x0FFFFFFF ) {		/* Avoid clocks overflows */
#ifdef JOHNS
	printf("Clock adjust\n");
#endif
	Cycle-=0x0F0FFFFF;
	VicAction-=0x0F0FFFFF;
	if( VicIrqAction!=-1 ) {
	    VicIrqAction-=0x0F0FFFFF;
	}
	if( Cia1TimerA_Action!=-1 ) {
	    Cia1TimerA_Action-=0x0F0FFFFF;
	}
	if( Cia1TimerB_Action!=-1 ) {
	    Cia1TimerB_Action-=0x0F0FFFFF;
	}
	if( Cia2TimerA_Action!=-1 ) {
	    Cia2TimerA_Action-=0x0F0FFFFF;
	}
	if( Cia2TimerB_Action!=-1 ) {
	    Cia2TimerB_Action-=0x0F0FFFFF;
	}
#ifdef MONITOR
	if( MonAction!=-1 ) {
	    MonAction-=0x0F0FFFFF;
	}
#endif
#ifdef REAL1541
	{ extern unsigned long R1541Cycle;
	  extern unsigned CheckAction;
	  extern int DoMystic;
	R1541Cycle-=((long unsigned long)0x0F0FFFFF*DoMystic)/1000;
	CheckAction-=((long unsigned long)0x0F0FFFFF*DoMystic)/1000;
	}
#endif
	SidCycle-=0x0F0FFFFF;
	SidVoice[0].Cycle-=0x0F0FFFFF;
	SidVoice[1].Cycle-=0x0F0FFFFF;
	SidVoice[2].Cycle-=0x0F0FFFFF;
    }
    EmulCia1Tod();
    EmulCia2Tod();
    EmulSid();
    EmulKey();
    EmulJoy();

#if 1
    /*
    **	NEW: 1541 Drive LED emulation.
    */
#define S(pointer,pixel) \
    VideoPut((pointer),VIC2VIDEO(pixel))

    if( VicLedCount || VicLedOnOff ) {
	unsigned char* x;
	int c;

	x=VideoMemory-8+HSIZE*(VSIZE-7);
	c=VicLedOnOff ? 2 : 0;
		  S(x+2,0);S(x+3,0);S(x+4,0);S(x+5,0); x+=HSIZE;
	 S(x+1,0);S(x+2,c);S(x+3,c);S(x+4,c);S(x+5,c);S(x+6,0); x+=HSIZE;
S(x+0,0);S(x+1,c);S(x+2,c);S(x+3,c);S(x+4,c);S(x+5,c);S(x+6,c);S(x+7,0); x+=HSIZE;
S(x+0,0);S(x+1,c);S(x+2,c);S(x+3,c);S(x+4,c);S(x+5,c);S(x+6,c);S(x+7,0); x+=HSIZE;
S(x+0,0);S(x+1,c);S(x+2,c);S(x+3,c);S(x+4,c);S(x+5,c);S(x+6,c);S(x+7,0); x+=HSIZE;
S(x+0,0);S(x+1,c);S(x+2,c);S(x+3,c);S(x+4,c);S(x+5,c);S(x+6,c);S(x+7,0); x+=HSIZE;
	 S(x+1,0);S(x+2,c);S(x+3,c);S(x+4,c);S(x+5,c);S(x+6,0); x+=HSIZE;
		  S(x+2,0);S(x+3,0);S(x+4,0);S(x+5,0); x+=HSIZE;

	if( VicLedCount && !--VicLedCount ) {
	    VicLedCount=VicLedStart;
	    if( VicLedCount || VicLedOnOff ) {
		VicLedOnOff^=1;
	    }
	}
    }

#undef S

#endif
}

/*-----------------------------------------------------------------------------
 *	Vic state executer
 *---------------------------------------------------------------------------*/

#if 0

Video Areas:
0:	Non displayed upper-blank
1:	Displayed upper-blank
2:	Upper-blank with bad lines.
3:	Displayed video area.
4:	
5:	Displayed lower-blank.
6:	Non displayed lower-blank.

Modes:
A	Border, bad-lines.
B	No Border, bad-lines.
C	Border, no bad-lines.
D	No Border, no bad-lines.


|-------------------| 0x000
| 00 | 00 | 00 | 00 |
|-------------------| 0x013 Displayed
| 1C   1C | 1D | 1C |
|-------------------| 0x030 Start bad-lines
| 2A | 2C | 2B | 2D |
|-------------------| 0x033 Start of 25 lines display
| 3A | 3C | 3B | 3D |
|-------------------| 0x037 Start of 24 lines display
| 4A                |
|-------------------| 0x0F7 End of 24 lines display and end bad-lines
|                   |
|-------------------| 0x0FB End of 25 lines display
| 5C | 5D | 5C | 5D |
|-------------------| 0x11B End of Displayed
| 06 | 06 | 06 | 06 |
|-------------------| 0x138

#endif

/*
**	Line 0x000-0x012	Upper blank (Fly back)
**		Nothing todo.
*/
static void VicState00(void)
{
    if( VicRasterLine==VIC_FIRST_DISPLAYED-1 ) {
	if( EmulVic!=VicState01C ) {
	    VicFirstSprites();
	}
	EmulVic=EmulVicNext;		/* open borders -or- closed borders */
    }

    ++VicRasterLine;
    VicAction+=CYCLES_PER_LINE;
    ChangeAction(VicAction,EmulVic);
}

/*
**	Line 0x013-0x02F	Upper blank
**		Filled with exterior color.
*/
static void VicState01C(void)
{
    VideoFillOverscanLine(VicExterior);

    if( VicRasterLine==0x30-1 ) {	/* start of bad lines */
	EmulVic=VicState02A;
    }

    ++VicRasterLine;
    VicAction+=CYCLES_PER_LINE;
    ChangeAction(VicAction,EmulVic);
}

/*
**	Line 0x013-0x02F	Upper blank
**		Open borders, display sprites and idle state.
*/
static void VicState01D(void)
{
    /* FIXME: if we don't support overscan this functions is not needed */
    IdleDL();

    VicAllSprites();

    if( VIC[0x16]&0x08 ) {			/* 40 characters */
	/* For X scrolling */
	((unsigned long*)(DispBuf+0x18))[0]=VicBackground0;
	((unsigned long*)(DispBuf+0x18))[1]=VicBackground0;
    } else {					/* 38 characters */
	unsigned long* p;

	p=(unsigned long*)(DispBuf+0x1F-VicXOffset);
	p[0]=VicExterior;
	p[1]=VicExterior;			/* 1 byte too much */
	p[78]=VicExterior;
	p[79]=VicExterior;
	p[80]=VicExterior;			/* 3 bytes too much */
    }
    VideoCopyOverscanLine(DispBuf+0x20-VicXOffset);

    if( VicRasterLine==0x30-1 ) {		/* start of bad-lines */
	EmulVic=VicState02B;
    }

    ++VicRasterLine;
    VicAction+=CYCLES_PER_LINE;
    ChangeAction(VicAction,EmulVic);
}

/*
**	Line 0x030-0x032	Blank and bad-lines.
**		Filled with exterior color.
*/
static void VicState02A(void)
{
    if( VicRasterLine==0x30
	    && !(VIC[0x11]&0x10) ) {		/* line to test DEN */
	EmulVic=VicState02C;
	VicState02C();				/* continue with no-bad */
	return;
    }

    if( VicRasterLine==0x33 ) {
	if( (VIC[0x11]&0x18)==0x18 ) {		/* 25 lines and DEN */
	    VicFirstSprites();
	    EmulVic=VicState03B;
	    VicState03B();
	} else {				/* 24 lines or !DEN */
	    EmulVic=VicState03A;
	    VicState03A();
	}
	return;
    }

    if( (VicRasterLine&7)==VicYOffset ) {	/* Fetch line */
	if( VicScanLine>7 ) {
	    VicOffset+=40;
	}
	VicColorRam=ColorRam+VicOffset;
	VicLineFetch();				/* into buffer */
    }

    VideoFillOverscanLine(VicExterior);
    ++VicScanLine;

    ++VicRasterLine;
    VicAction+=CYCLES_PER_LINE;
    ChangeAction(VicAction,EmulVic);
}

/*
**	Line 0x030-0x032	Blank+Bad-lines
**		Open borders, display sprites and idle state.
*/
static void VicState02B(void)
{
    if( VicRasterLine==0x30
	    && !(VIC[0x11]&0x10) ) {		/* line to test DEN */
	EmulVic=VicState02D;
	VicState02D();				/* continue with no-bad */
	return;
    }

    if( VicRasterLine==0x33 ) {
	EmulVic=VicState03B;
	VicState03B();
	return;
    }

    if( (VicRasterLine&7)==VicYOffset ) {
	if( VicScanLine>7 ) {
	    VicOffset+=40;
	}
	VicColorRam=ColorRam+VicOffset;
	VicLineFetch();				/* into buffer */
    }

    if( VicScanLine>7 ) {			/* illegal line */
	IdleDL();
    } else {
	(*VicDisplayLine)();
    }

    VicAllSprites();

    if( VIC[0x16]&0x08 ) {			/* 40 characters */
	/* For X scrolling */
	((unsigned long*)(DispBuf+0x18))[0]=VicBackground0;
	((unsigned long*)(DispBuf+0x18))[1]=VicBackground0;
    } else {					/* 38 characters */
	unsigned long* p;

	p=(unsigned long*)(DispBuf+0x1F-VicXOffset);
	p[0]=VicExterior;
	p[1]=VicExterior;			/* 1 byte too much */
	p[78]=VicExterior;
	p[79]=VicExterior;
	p[80]=VicExterior;			/* 3 bytes too much */
    }

    VideoCopyOverscanLine(DispBuf+0x20-VicXOffset);

    ++VicScanLine;

    ++VicRasterLine;
    VicAction+=CYCLES_PER_LINE;
    ChangeAction(VicAction,EmulVic);
}

/*
**	Line 0x031-0x032	Blank
**		Filled with exterior color.
*/
static void VicState02C(void)
{
    if( VicRasterLine==0x33 ) {
	if( (VIC[0x11]&0x18)==0x18 ) {		/* 25 lines and DEN */
	    VicFirstSprites();
	    EmulVic=VicState03D;
	    VicState03D();
	} else {				/* 24 lines or !DEN */
	    VicLastSprites();
	    EmulVic=VicState03C;
	    VicState03C();
	}
	return;
    }

    VideoFillOverscanLine(VicExterior);

    ++VicScanLine;

    ++VicRasterLine;
    VicAction+=CYCLES_PER_LINE;
    ChangeAction(VicAction,EmulVic);
}

/*
**	Line 0x031-0x032	Blank
**		Open borders, display sprites and idle state.
*/
static void VicState02D(void)
{
    if( VicRasterLine==0x33 ) {
	EmulVic=VicState03D;
	VicState03D();
	return;
    }

    /* FIXME: if we don't support overscan this functions is not needed */
    IdleDL();

    VicAllSprites();

    if( VIC[0x16]&0x08 ) {			/* 40 characters */
	/* For X scrolling */
	((unsigned long*)(DispBuf+0x18))[0]=VicBackground0;
	((unsigned long*)(DispBuf+0x18))[1]=VicBackground0;
    } else {					/* 38 characters */
	unsigned long* p;

	p=(unsigned long*)(DispBuf+0x1F-VicXOffset);
	p[0]=VicExterior;
	p[1]=VicExterior;			/* 1 byte too much */
	p[78]=VicExterior;
	p[79]=VicExterior;
	p[80]=VicExterior;			/* 3 bytes too much */
    }
    VideoCopyOverscanLine(DispBuf+0x20-VicXOffset);

    ++VicRasterLine;
    VicAction+=CYCLES_PER_LINE;
    ChangeAction(VicAction,EmulVic);
}

/*
**	Line 0x033-0x036	Upper blank+bad-lines 24 lines
**		Filled with exterior color.
*/
static void VicState03A(void)
{
    if( VicRasterLine==0x37 ) {
	if( (VIC[0x11]&0x18)==0x10 ) {		/* 24 lines and DEN */
	    VicFirstSprites();
	    EmulVic=VicState03B;
	    VicState03B();
	} else {				/* stay blank */
	    VicLastSprites();
	    EmulVic=VicState04A;
	    VicState04A();
	}
	return;
    }

    if( (VicRasterLine&7)==VicYOffset ) {	/* Fetch line */
	if( VicScanLine>7 ) {
	    VicOffset+=40;
	}
	VicColorRam=ColorRam+VicOffset;
	VicLineFetch();				/* into buffer */
    }

    VideoFillLine(VicExterior);

    ++VicScanLine;

    ++VicRasterLine;
    VicAction+=CYCLES_PER_LINE;
    ChangeAction(VicAction,EmulVic);
}

/*
**	Line 0x033-0FB	Displayed 25 lines
**	Line 0x037-0F7	Displayed 24 lines
*/
static void VicState03B(void)
{
    /*
    **	End of displayed area.
    */
    if( VicRasterLine>=0x0F7 ) {
	if( VicRasterLine==0x0F7
		&& !(VIC[0x11]&0x08) ) {	/* 24 lines */
	    VicLastSprites();
	    EmulVic=VicState04A;
	    VicState04A();
	    return;
	}
	if( VicRasterLine==0x0FB ) {
	    if( VIC[0x11]&0x08 ) {		/* 25 lines */
		VicLastSprites();
		EmulVic=VicState05C;
		VicState05C();
		return;
	    } else {				/* open border */
		EmulVic=VicState05D;
		VicState05D();
		return;
	    }
	}
    }

    if( (VicRasterLine&7)==VicYOffset && VicRasterLine<=0x0F7 ) {
#if 0	/* above seems ok */
    if( (VicRasterLine&7)==VicYOffset /* && VicOffset<40*24 */ ) {
#endif
	if( VicScanLine>7 ) {
	    VicOffset+=40;
	}
	VicColorRam=ColorRam+VicOffset;
	VicLineFetch();				/* into buffer */
    }

    if( VicScanLine>7 ) {			/* illegal line */
	IdleDL();
    } else {
	(*VicDisplayLine)();
    }

    VicAllSprites();

    if( VIC[0x16]&0x08 ) {			/* 40 characters */
	/* For X scrolling */
	((unsigned long*)(DispBuf+0x18))[0]=VicBackground0;
	((unsigned long*)(DispBuf+0x18))[1]=VicBackground0;
    } else {					/* 38 characters */
	unsigned long* p;

	p=(unsigned long*)(DispBuf+0x1F-VicXOffset);
	p[0]=VicExterior;
	p[1]=VicExterior;			/* 1 byte too much */
	p[78]=VicExterior;
	p[79]=VicExterior;
	p[80]=VicExterior;			/* 3 bytes too much */
    }
    VideoCopyLine(DispBuf+0x20-VicXOffset);
#ifdef _JOHNS
    VideoPointer[-HSIZE+0]=VIC[0x15];
    VideoPointer[-HSIZE+1]=VIC[0x15];
    VideoPointer[-HSIZE+2]=SpriteDMA;
    VideoPointer[-HSIZE+3]=SpriteDMA;
#endif

    ++VicScanLine;

    ++VicRasterLine;
    VicAction+=CYCLES_PER_LINE;
    ChangeAction(VicAction,EmulVic);
}

/*
**	Line 0x33-0FB	25 lines blank
**	Line 0x37-0F7	24 lines blank
*/
static void VicState03C(void)
{
    /*
    **	End of displayed area.
    */
    if( VicRasterLine==0x0FB ) {
	EmulVic=VicState05C;
    }

    VideoFillLine(VicExterior);

    ++VicRasterLine;
    VicAction+=CYCLES_PER_LINE;
    ChangeAction(VicAction,EmulVic);
}

/*
**	Line 0x033-0FB	Displayed 25 lines no bad-lines
**	Line 0x037-0F7	Displayed 24 lines no bad-lines
*/
static void VicState03D(void)
{
    /*
    **	End of displayed area.
    */
    if( VicRasterLine>=0x0F7 ) {
	if( VicRasterLine==0x0F7
		&& !(VIC[0x11]&0x08) ) {	/* 24 lines */
	    VicLastSprites();
	    EmulVic=VicState03C;
	    VicState03C();
	    return;
	}
	if( VicRasterLine==0x0FB ) {
	    if( VIC[0x11]&0x08 ) {		/* 25 lines */
		VicLastSprites();
		EmulVic=VicState05C;
		VicState05C();
		return;
	    } else {				/* open border */
		EmulVic=VicState05D;
		VicState05D();
		return;
	    }
	}
    }

    IdleDL();

    VicAllSprites();

    if( VIC[0x16]&0x08 ) {			/* 40 characters */
	/* For X scrolling */
	((unsigned long*)(DispBuf+0x18))[0]=VicBackground0;
	((unsigned long*)(DispBuf+0x18))[1]=VicBackground0;
    } else {					/* 38 characters */
	unsigned long* p;

	p=(unsigned long*)(DispBuf+0x1F-VicXOffset);
	p[0]=VicExterior;
	p[1]=VicExterior;			/* 1 byte too much */
	p[78]=VicExterior;
	p[79]=VicExterior;
	p[80]=VicExterior;			/* 3 bytes too much */
    }
    VideoCopyLine(DispBuf+0x20-VicXOffset);

    ++VicScanLine;

    ++VicRasterLine;
    VicAction+=CYCLES_PER_LINE;
    ChangeAction(VicAction,EmulVic);
}

/*
**	Line 0x037-0x0FB	25 lines blank bad-lines
*/
static void VicState04A(void)
{
    /*
    **	End of displayed area.
    */
    if( VicRasterLine==0x0FB ) {
	VicLastSprites();
	EmulVic=VicState05C;
	VicState05C();
	return;
    }

    if( (VicRasterLine&7)==VicYOffset ) {	/* bad line */
	Cycle+=43;			/* 40 characters + 3 bus acknowledge */
    }

    VideoFillLine(VicExterior);

    ++VicRasterLine;
    VicAction+=CYCLES_PER_LINE;
    ChangeAction(VicAction,EmulVic);
}

/*
**	Line 0x0FC-0x11A	Lower blank
**		Filled with exterior color.
*/
static void VicState05C(void)
{
    VideoFillOverscanLine(VicExterior);

    if( VicRasterLine==VIC_LAST_DISPLAYED-1 ) {
	EmulVic=VicState06;
	EmulVicNext=VicState01C;
    }

    ++VicRasterLine;
    VicAction+=CYCLES_PER_LINE;
    ChangeAction(VicAction,EmulVic);
}

/*
**	Line 0x0FC-0x11A	Lower blank
**		Open borders, display sprites and idle state.
*/
static void VicState05D(void)
{
    if( VicScanLine>7 ) {			/* illegal line */
	IdleDL();
    } else {
	(*VicDisplayLine)();
	++VicScanLine;
    }

    VicAllSprites();

    if( VIC[0x16]&0x08 ) {			/* 40 characters */
	/* For X scrolling */
	((unsigned long*)(DispBuf+0x18))[0]=VicBackground0;
	((unsigned long*)(DispBuf+0x18))[1]=VicBackground0;
    } else {					/* 38 characters */
	unsigned long* p;

	p=(unsigned long*)(DispBuf+0x1F-VicXOffset);
	p[0]=VicExterior;
	p[1]=VicExterior;			/* 1 byte too much */
	p[78]=VicExterior;
	p[79]=VicExterior;
	p[80]=VicExterior;			/* 3 bytes too much */
    }
    VideoCopyOverscanLine(DispBuf+0x20-VicXOffset);

    if( VicRasterLine==VIC_LAST_DISPLAYED-1 ) {
	VicLastSprites();
	EmulVic=VicState06;
	EmulVicNext=VicState01D;
    }

    ++VicRasterLine;
    VicAction+=CYCLES_PER_LINE;
    ChangeAction(VicAction,EmulVic);
}

/*
**      Line 0x11B-0x137	Lower blank	(Sync)
*/
static void VicState06(void)
{
    if( VicRasterLine==LINES_PER_FRAME-1 ) {	/* PAL: 311 */
	VideoPointer=VideoMemory;
	VicOffset=-40;
	VicRasterLine=-1;
	VicScanLine=8;

	if( VicMessageCount ) {
	    --VicMessageCount;
	    if( !VicMessageCount ) {		/* last time: clear it */
		memset(VicMessageBuffer,' ',
		    strlen(VicMessageBuffer));
	    }
	    VicText(0,0,VicMessageBuffer);
	}

	/*
	**	Hook for non syncronous overscan colors.
	*/
	if( VIC[0x20]!=VicOverscanColor ) {
	    VideoSetOverscanColor(VicOverscanColor=VIC[0x20]);
	}

	/*
 	**	Do all things with 50 Hz or 60 Hz.
	*/
	VerticalRetrace();

	if( --VicEmulateCount<=0 ) {	/* calculate each n'th frame */
	    VicEmulateCount=VicEmulateRate;
	    EmulVic=VicState00;
	    /*
	    **	Need only refresh, if emulated.
	    */
	    if( --VicRefreshCount<=0 ) {/* refresh each n'th frame */
		VicRefreshCount=VicRefreshRate;
		VideoRefresh();
		VideoSync();
	    }
	}
    }

    ++VicRasterLine;
    VicAction+=CYCLES_PER_LINE;
    ChangeAction(VicAction,EmulVic);
}

/*** FIXME: } END OF CLEANED UP CODE ****/
