/*
 *	The C64 emulator
 *
 *	Copyright 1992-96 by ALE.
 *	written by Lutz Sammer.
 *
 *	Sid emulator
 *-----------------------------------------------------------------------------
 * $Id: sid.c,v 1.12 1996/07/27 23:15:10 johns Exp root $
 * $Log: sid.c,v $
 * Revision 1.12  1996/07/27 23:15:10  johns
 * Speaker bug fixed, Sync+Ring modulation completed.
 *
 * Revision 1.11  1996/06/22 22:09:23  ari
 * Corrected DOS soundblaster support interrupt handling.
 *
 * Revision 1.10  1996/06/16 00:47:16  johns
 * Turn on opl3 for resume and toggle, Turn off all 18 opl3 voices.
 *
 * Revision 1.9  1996/06/13 21:23:04  johns
 * DSP support rewritten, OPL2/OPL3 support implemented.
 * Better Envelope read.
 *
 * Revision 1.8  1996/05/17 16:47:15  johns
 * Cleanup, DSP Sound driver written. Bugfix in SPEAKER driver.
 *
 * Revision 1.7  1995/02/13  11:27:22  ari
 * second linux port ( by ari ) integration
 *
 * Revision 1.6  1994/06/02  14:54:24  johns
 * Use demosum.h, always revert meaning of '-j'.
 *
 * Revision 1.5  1993/08/31  20:30:03  johns
 * go32 initial support
 *
 * Revision 1.4  1993/06/13  12:25:58  johns
 * wuebbel patches, cleanup
 *
 * Revision 1.2  1992/07/20  04:26:49  johns
 * More doc and better sound chip emulation written.
 * Sid emulation info printing.
 *
 * Revision 1.1  1992/07/11  21:52:24  johns
 * Initial revision
 *
 *-----------------------------------------------------------------------------
 */

#include "c64.h"
#include "sid.h"
#include <stdio.h>

#if 0

Sound chip doco:

	  /|\				- Reg 18: 0-3	Max. volume level
	/  |  \				^
      /    |    \______________		| - Reg 06,0D,14: 4-7	Sustain level
    /	   |	 |	      |\	| ^
  /	   |	 |	      |  \ 	v v
-|---------|-----|------------|---|---  --- 0
 |	   |	 |	      |   |
 |<------->|<--->|	      |<->|
      |	      | Decay		| Release
      | Attack

Voice 1
	Reg 00: LSB Frequency
	Reg 01: MSB Frequency
	Reg 02: LSB Pulse width
	Reg 03: MSB Pulse width
	Reg 04: Control
		0 GATE: 0->1 Start 1->0 Stop
		1 SYNC: 1= Sync with 2
		2 RING: 1= TRI mixed of 0 and 2
		3 TEST: 1= Reset
		4 TRI:  1= Triangle waveform
		5 SAW:  1= Sawtooth waveform
		6 PUL:  1= Pulse    waveform
		7 NSE:  1= Noise    waveform
	Reg 05: Attack/Decay
		0-3: Decay
		4-7: Attack
	Reg 06: Sustain/Release
		0-3: Release
		4-7: Sustain

Voice 2
	Reg 07: LSB Frequency
	Reg 08: MSB Frequency
	Reg 09: LSB Pulse width
	Reg 0A: MSB Pulse width
	Reg 0B: Control
		0 GATE: 0->1 Start 1->0 Stop
		1 SYNC: 1= Sync with 0
		2 RING: 1= TRI mixed of 1 and 0
		3 TEST: 1= Reset
		4 TRI:  1= Triangle waveform
		5 SAW:  1= Sawtooth waveform
		6 PUL:  1= Pulse    waveform
		7 NSE:  1= Noise    waveform
	Reg 0C: Attack/Decay
		0-3: Decay
		4-7: Attack
	Reg 0D: Sustain/Release
		0-3: Release
		4-7: Sustain

Voice 3
	Reg 0E: LSB Frequency
	Reg 0F: MSB Frequency
	Reg 10: LSB Pulse width
	Reg 11: MSB Pulse width
	Reg 12: Control
		0 GATE: 0->1 Start 1->0 Stop
		1 SYNC: 1= Sync with 2
		2 RING: 1= TRI mixed of 2 and 1
		3 TEST: 1= Reset
		4 TRI:  1= Triangle waveform
		5 SAW:  1= Sawtooth waveform
		6 PUL:  1= Pulse    waveform
		7 NSE:  1= Noise    waveform
	Reg 13: Attack/Decay
		0-3: Decay
		4-7: Attack
	Reg 14: Sustain/Release
		0-3: Release
		4-7: Sustain

All channels:

	Reg 15: 0-2 LSB Filter
	Reg 16: MSB Filter	F=(30+W*5.8) Hz

	Reg 17: Filter Control
		0 FILT1:	1= Filter of voice 1
		1 FILT2:	1= Filter of voice 2
		2 FILT3:	1= Filter of voice 3
		3 FILTEX:	1= Filter of external
		4-7 RES0-RES3:  resonance of the filter
	
	Reg 18: Control
		0-3	Volume
		4	1= Low-Pass  filter
		5	1= Band-Pass filter
		6	1= High-Pass filter
		7	1= voice 3 unhearable

	Reg 19: A/D Wandler 1
		Updated every 512 Q2 clock cycles
	Reg 1A: A/D Wandler 2
		Updated every 512 Q2 clock cycles

	Reg 1B: OSC 3 / RANDOM
		Output of the waveform generator of voice 3
		
	Reg 1C: Envelope 3
		Output of the envelope generator of voice 3

Envelope rates:

      Value  |  Attack rate |   Decay/Release rate
	 0   |     2 ms     |       6 ms
	 1   |     8 ms     |      24 ms
	 2   |    16 ms     |      48 ms
	 3   |    24 ms     |      72 ms
	 4   |    38 ms     |     114 ms
	 5   |    56 ms     |     168 ms
	 6   |    68 ms     |     204 ms
	 7   |    80 ms     |     240 ms
	 8   |   100 ms     |     300 ms
	 9   |   250 ms     |     750 ms
	10   |   500 ms     |       1.5 s
	11   |   800 ms     |       2.4 s
	12   |     1   s    |       3   s
	13   |     3   s    |       9   s
	14   |     5   s    |      15   s
	15   |     8   s    |      24   s

Formels:
	Frequency:	Fn=$XXXX
		Frq = Fn * 0.059604645 Hz
		Fn  = Frq / 0.06097 Hz
	Pulse width:    Pn=$0XXX
		PW  = Pn / 40.95 %

#endif

/*-----------------------------------------------------------------------------
 * SID Emulation
 *---------------------------------------------------------------------------*/

/*
**	Attack and Decay/Release in clock counts
*/
static CONST unsigned int SidTiming[16]={
      2*1000,				/* 2ms */
      8*1000,				/* 8ms */
     16*1000,				/* 16ms */
     24*1000,				/* 24ms */
     38*1000,				/* 38ms */
     56*1000,				/* 56ms */
     68*1000,				/* 68ms */
     80*1000,				/* 80ms */
    100*1000,				/* 100ms */
    250*1000,				/* 250ms */
    500*1000,				/* 500ms */
    800*1000,				/* 800ms */
      1*1000*1000,			/* 1s */
      3*1000*1000,			/* 3s */
      5*1000*1000,			/* 5s */
      8*1000*1000,			/* 8s */
}; 

struct voice SidVoice[3];		/* 3 Voices */
int	SidSoundOff;			/* True no sound */
int	SidCycle;			/* Cycle until sid is handled */

/*
**	Some shortcuts to access sid registers.
*/
#define FREQUENCY(v)	(SID[(v)+0x00]+(SID[(v)+0x01]<<8))
#define PULSEWIDTH(v)	((SID[(v)+0x02]+(SID[(v)+0x03]<<8))&0x0FFF)
#define ATTACK(v)	(SID[(v)+0x05]>>4)
#define DECAY(v)	(SID[(v)+0x05]&0xF)
#define SUSTAIN(v)	(SID[(v)+0x06]>>4)
#define RELEASE(v)	(SID[(v)+0x06]&0xF)

#define VOLUME()	(SID[0x18]&0xF)

/*--------------------------------------------------------------------------*/

/*
**	Return the current envelope level of voice.
*/
int SidEnvelope(int voice)
{
    int i;

    switch( SidVoice[voice].State ) {
	case SID_OFF:
	    return 0;

	case SID_ATTACK:
	    if( SidVoice[voice].Cycle+SidTiming[ATTACK(voice*7)]<Cycle ) {
		/* DECAY STATE */
		SidVoice[voice].Cycle+=SidTiming[ATTACK(voice*7)];
		SidVoice[voice].State=SID_DECAY;
		/*
		**	SidVoice.Cycle+ATTACK		= 255
		**	SidVoice.Cycle+ATTACK+RELEASE	= SUSTAIN
		*/
		i=Cycle-SidVoice[voice].Cycle;
		return 255-
		    ((255-17*SUSTAIN(voice*7))*i)/(SidTiming[DECAY(voice*7)]*3);
	    }
	    if( SidVoice[voice].Cycle+SidTiming[ATTACK(voice*7)]
		    +SidTiming[DECAY(voice*7)]*3<Cycle ) {
		/* SUSTAIN STATE */
		SidVoice[voice].State=SID_SUSTAIN;
		return 17*SUSTAIN(voice*7);
	    }
	    /* ATTACK STATE */
	    i=Cycle-SidVoice[voice].Cycle;
	    /*
            **	SidVoice.Cycle		= 0
	    **	SidVoice.Cycle+ATTACK	= 255
	    */
	    return (i*255)/SidTiming[ATTACK(voice*7)];

	case SID_DECAY:
	    if( SidVoice[voice].Cycle+SidTiming[DECAY(voice*7)]*3<Cycle ) {
		/* SUSTAIN STATE */
		SidVoice[voice].State=SID_SUSTAIN;
		return 17*SUSTAIN(voice*7);
	    }
	    /*
	    **	SidVoice.Cycle+ATTACK		= 255
	    **	SidVoice.Cycle+ATTACK+RELEASE	= SUSTAIN
	    */
	    i=Cycle-SidVoice[voice].Cycle;
	    return 255-
		((255-17*SUSTAIN(voice*7))*i)/SidTiming[DECAY(voice*7)];

	case SID_SUSTAIN:
	    return 17*SUSTAIN(voice*7);

	case SID_RELEASE:
	    if( SidVoice[voice].Cycle+SidTiming[RELEASE(voice*7)]<Cycle ) {
		/* OFF STATE */
		SidVoice[voice].State=SID_OFF;
		return 0;
	    }
	    /* RELEASE STATE */
	    i=Cycle-SidVoice[voice].Cycle;
	    /*
            **	SidVoice.Cycle		= SUSTAIN
	    **  SidVoice.Cycle+RELEASE	= 0
	    */
	    return (17*SUSTAIN(voice*7))-
		(17*SUSTAIN(voice*7)*i)/SidTiming[RELEASE(voice*7)];
    }
    /* unreached */
    return 255;
}

/*--------------------------------------------------------------------------*/

#ifdef MONITOR	/* { */

/*
**	Print information for sid.
*/
void SidInfo(void)
{
    int c,i;
    char* cp;
    static char* SidT1[16] = {
	"2ms",  "8ms",  "16ms", "24ms", "38ms", "56ms", "68ms", "80ms",
	"100ms","250ms","500ms","800ms","1s",   "3s",   "5s",   "8s" }; 
    static char* SidT2[16] = {
	"6ms",  "24ms", "48ms", "72ms", "114ms","168ms","204ms","240ms",
	"300ms","750ms","1s",   "2s",   "3s",   "9s",   "15s",  "24s" }; 

#ifdef SID_DSP
    printf("Sid emulation with DSP");
#endif
#ifdef SID_DSP_SPEAKER
    printf("Sid emulation with DIGITAL SPEAKER");
#endif
#ifdef SID_SPEAKER
    printf("Sid emulation with SPEAKER");
#endif
#ifdef SID_OPL3
    printf("Sid emulation with OPL2/OPL3");
#endif
    printf(" SidOff: %d\n",SidSoundOff);
    printf("Overall volume %d\n",VOLUME());
    for( c=0; c<3; ++c ) {
	printf("Voice %d:\n",c);
	i=FREQUENCY(c*7);
	printf("Frequency: $%04X=%d = %d Hz, ",i,i,
		(int)((double)i*CPU_CLK/16777216));
	i=PULSEWIDTH(c*7);
	printf("Pulsewidth: $%04X = %d = %d %%\n",i,i,(i&0xFFF)/41);
	printf("Control: %s%s%s%s%s%s%s%s\n",
	    (SID[c*7+0x4]&0x80) ? "NSE " : "",
	    (SID[c*7+0x4]&0x40) ? "PUL " : "",
	    (SID[c*7+0x4]&0x20) ? "SAW " : "",
	    (SID[c*7+0x4]&0x10) ? "TRI " : "",
	    (SID[c*7+0x4]&0x08) ? "TEST " : "",
	    (SID[c*7+0x4]&0x04) ? "RING " : "",
	    (SID[c*7+0x4]&0x02) ? "SYNC " : "",
	    (SID[c*7+0x4]&0x01) ? "GATE " : "");
	printf("Attack %d=%s",ATTACK(c*7),SidT1[ATTACK(c*7)]);
	printf(" Decay %d=%s",DECAY(c*7),SidT2[DECAY(c*7)]);
	printf(" Sustain %d",SUSTAIN(c*7));
	printf(" Release %d=%s",RELEASE(c*7),SidT2[RELEASE(c*7)]);
	printf("\nEmulation: Voice: %s Cycle: %d State: ",
	    SidVoice[c].On ? "YES" : "NO",SidVoice[c].Cycle);
	cp="Off";
	if( SidVoice[c].State==SID_ATTACK ) {
	    cp="Attack";
	    if( SidVoice[c].Cycle+SidTiming[ATTACK(c*7)]<Cycle ) {
		cp="Decay";
	    }
	    if( SidVoice[c].Cycle+SidTiming[ATTACK(c*7)]
		    +SidTiming[DECAY(c*7)]*3<Cycle ) {
		cp="Sustain";
	    }
	} else if( SidVoice[c].State==SID_DECAY ) {
	    cp="Decay";
	    if( SidVoice[c].Cycle+SidTiming[DECAY(c*7)]*3<Cycle ) {
		cp="Sustain";
	    }
	} else if( SidVoice[c].State==SID_SUSTAIN ) {
	    cp="Sustain";
	} else if( SidVoice[c].State==SID_RELEASE ) {
	    cp="Release";
	    if( SidVoice[c].Cycle+SidTiming[RELEASE(c*7)]<Cycle ) {
		cp="Off";
	    }
	}
	printf("%s\t",cp);
	printf("Envelope %d\n",SidEnvelope(c));
	printf("\n");
    }
}

#endif	/* } MONITOR */

/*
**	Reset of SID.
*/
void SidReset(void)
{
    SidSoundOff&=2;			/* Turn sound again on */
    SidCycle=0;
}

/*
**	PORTING:
**
**		EnterSound	Initialise and turn sound on.
**		LeaveSound	Turn sound off and cleanup.
**		SuspendSound	Turn sound temporary off.
**		ResumeSound	Turn sound on again.
**		ToggleSound	Toggle sound emulation on/off
**		EmulSid		Emulate sid chip: Make noise
**		DigitalSound	Try to emulate sid sample mode.
*/

#define NeedIOPriv()	0

/* FIXME: remove this chaos !!!!!! */
#define JumpTable
#define Switch(x)	switch( x )
#define Case(x)		case x:
#define Default		default:
#define Break		break;
#define BreakLabel


#ifdef SID_SPEAKER	/* { */

/******************************************************************************
 * Speaker SID Emulation
 *****************************************************************************/

/*
**	Speaker port and values definition
*/
#if M_I386 && M_UNIX || defined(__GO32__) || defined(__linux__)  /* { */

#define PORT_TIMER0	0x40
#define PORT_TIMER1	0x41
#define PORT_TIMER2	0x42
#define PORT_MODUS	0x43

/* Modus: sc1 sc0 rl1 rl0 m2 m1 m0 bcd */

/*
 *	Modus bits.
 */
#define PM_0		0x00		/* Access timer 0 */
#define PM_1		0x40		/* Access timer 1 */
#define PM_2		0x80		/* Access timer 2 */
#define PM_E		0xC0		/* Invalid */

#if 0
#define PM_LOCK		0x00		/* Lock timer */
#define PM_LSB		0x40		/* Set lowest significant byte */
#define PM_MSB		0x80		/* Set most significant byte */
#define PM_LSB_MSB	0xC0		/* Set lsb than msb byte */
#endif
#define PM_LOCK		0x00		/* Lock timer */
#define PM_LSB		0x10		/* Set lowest significant byte */
#define PM_MSB		0x20		/* Set most significant byte */
#define PM_LSB_MSB	0x30		/* Set lsb than msb byte */

#define PM_M0		0x00		/* Timer modes */
#define PM_M1		0x02
#define PM_M2		0x04
#define PM_M3		0x06
#define PM_M4		0x08
#define PM_M5		0x0A
#define PM_M6		0x0C
#define PM_M7		0x0E

#define PM_BCD		0x01		/* Timer is BCD counter */

#define PORT_PPI	0x61
#define TIMER_ON	0x1		/* Timer counts */
#define SPEAKER_ON	0x2		/* Turn speaker output on */

#endif  /* } !M_I386 && !M_UNIX */

static int Voice;			/* current emulated voice */

/*
**	Turn speaker and timer on.
*/
static void SpeakerOff(void)
{
    unsigned char b;

    b=inb(PORT_PPI);			/* Turn speaker off */
    outb(PORT_PPI,b&~(TIMER_ON|SPEAKER_ON));
}


/*
**	Initialition for sound.
*/
void EnterSound(void)
{
    if( (SidSoundOff&2)
	    || NeedIOPriv() ) {		/* For speaker we need I/O privelege */
	SidSoundOff=3;			/* Turn sound always off */
    }
}

/*
**	Cleanup for sound.
*/
void LeaveSound(void)
{
    if( !(SidSoundOff&2) ) {		/* initialise ok */
	SpeakerOff();
    }
}

/*
**	Suspend sound.
*/
void SuspendSound(void)
{
    LeaveSound();
}

/*
**	Resume sound.
*/
void ResumeSound(void)
{
    /* Done automatic with next EmulSid. */
}


/*
**	Turn sound on/off
*/
void ToggleSound(void)
{
    SidSoundOff^=1;
    if( !SidSoundOff ) {
	SpeakerOff();
    }
}

/*
**	DigitalSound.
*/
void DigitalSound(int volume)
{
    unsigned char b;

    if( volume>2 ) {
	b=inb(PORT_PPI);		/* Turn speaker on */
	outb(PORT_PPI,b|SPEAKER_ON);
    } else {
	b=inb(PORT_PPI);		/* Turn speaker off */
	outb(PORT_PPI,b&~SPEAKER_ON);
    }
}

/*
**	Emulate sid chip
*/
void EmulSid(void)
{
    unsigned i,j;
    unsigned frq;

    if( SidSoundOff&2 ) {		/* off by -q or no privilege */
	return;
    }
    if( !SidSoundOff && (SID[0x18]&0x0F)>1 ) {	/* Volume on */
	for( i=0; i<3; ++i ) {
	    ++Voice;
	    if( Voice>2 )			/* Next channel */
		Voice=0;
	    /*
	    **	 This channel on
	    */
	    if( SidVoice[Voice].On && SidVoice[Voice].State ) {
		j=Voice*7;
		/*
		 *  Sid  frq: f=frq*0.059604645 Hz
		 *  8253 frq: f=1193180/frq Hz
		 *  20018239/frq2=frq1
		 */
		frq=(SID[j+0x01]<<8)|SID[j+0x00];
		if( frq ) {
		    if( SID[j+0x4]&0x80 )	/* Noise */
			frq=(20018239+2000)/frq;
		    else
			frq=20018239/frq;
		} else {			/* 0 Hz = off */
		    continue;
		}

		outb(PORT_MODUS,PM_2|PM_LSB_MSB|PM_M3);
		outb(PORT_TIMER2,frq);
		outb(PORT_TIMER2,frq>>8);
		outb(PORT_PPI,inb(PORT_PPI)|TIMER_ON|SPEAKER_ON);

		/* FIXME: calculate volume of channel */
		if( SidVoice[Voice].State==SID_ATTACK ) {
		    if( SUSTAIN(j)<2 ) {
			if( SidVoice[Voice].Cycle+SidTiming[ATTACK(j)]
				+SidTiming[DECAY(j)]*3<Cycle ) {
			    SidVoice[Voice].State=SID_OFF;
			}
		    }
		} else if( SidVoice[Voice].State==SID_RELEASE ) {
		    if( (SUSTAIN(j)<2) || (SidVoice[Voice].Cycle
			    +SidTiming[RELEASE(j)]*3<Cycle) ) {
			SidVoice[Voice].State=SID_OFF;
		    }
		}

		return;
	    }
	}
    }
    SpeakerOff();			/* Turn speaker off */
}

#endif	/* } SID_SPEAKER */

#ifdef SID_DSP	/* { */

/******************************************************************************
 * DSP (Digital Sound Processor) SID Emulation
 *****************************************************************************/

#define AudioSampleRate	22050		/* sample rate of dsp */
//##define AudioSampleRate	8192		/* sample rate of dsp */

int AudioFrequency=AudioSampleRate;	/* selected dsp frequency */

static unsigned char TriangleWave[4096];/* triangle wave table */
static unsigned char SawtoothWave[4096];/* sawtooth wave table */
static unsigned char PulseWave[4096*2];	/* pulse wave table */

static unsigned char AudioBuffer[32768];/* Audio output buffer */
static int AudioBufferIndex;		/* Index into audio buffer */

typedef struct __dsp_info__ DspInfo;

struct __dsp_info__ {
    /*
    **	Waveform generator.
    */
    int		WaveSteps;		/* divider steps */
    int		WaveCount;		/* divider counter */
    int		WaveShift;		/* frequency shift */
    int		WaveIndex;		/* wave table index */
    int		PulseOffset;		/* offset into pulse table */
    int		NoiseRegister;		/* noise generator register */
    int		NoiseOutput;		/* noise generator output */
    int		NoiseIndex;		/* noise generator counter */
    /*
    **	Envelope generator.
    */
    enum SidState	State;		/* Envelope state */
    int		EnvVolume;		/* current envelope volume */
    int		EnvStart;		/* divider start */
    int		EnvSteps;		/* divider steps */
    int		EnvCount;		/* divider counter */
    int		EnvIndex;		/* envelope index */
    int		Sustain;		/* sustain for envelope */

    int		Off;			/* voice off for sync/ring */

    DspInfo*	Modulator;		/* connected voice */
    DspInfo*	Carrier;		/* connected voice */
};

#define NOISE_SEED	0x007FFFF8	/* start of noise random generator */

static DspInfo DspVoices[3];		/* dsp information */

static unsigned SidAttackIndex[16];	/* precalculated attack index */
static unsigned SidAttackSteps[16];	/* precalculated attack steps */
static unsigned SidAttackStart[16];	/* precalculated attack start */

static unsigned SidDecayIndex[16];	/* precalculated decay index */
static unsigned SidDecaySteps[16];	/* precalculated decay steps */
static unsigned SidDecayStart[16];	/* precalculated decay start */

#ifdef __linux__    /* { */

/*.............................................................................
 *	LINUX DEPENDEND DSP PART
 *...........................................................................*/

#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/soundcard.h>

#ifndef AudioDeviceName
#define AudioDeviceName	"/dev/dsp"	/* dsp device */
#endif

static int AudioFildes;			/* audio file descriptor */

/*
**	Machine depend DSP part: Initialise.
*/
static int EnterDSP(void)
{
    int dummy;

    /*
    **	Open dsp device, 8bit samples, mono. 
    */
    AudioFildes=open(AudioDeviceName,O_WRONLY);
    if( AudioFildes==-1 ) {
	printf("Can't open audio device `%s'\n",AudioDeviceName);
	return 1;
    }
    dummy=8;
    if( ioctl(AudioFildes,SNDCTL_DSP_SAMPLESIZE,&dummy)==-1 ) {
	perror("sid");
	close(AudioFildes);
	return 1;
    }
    dummy=0;
    if( ioctl(AudioFildes,SNDCTL_DSP_STEREO,&dummy)==-1 ) {
	perror("sid");
	close(AudioFildes);
	return 1;
    }
    dummy=AudioFrequency;
    if( ioctl(AudioFildes,SNDCTL_DSP_SPEED,&dummy)==-1 ) {
	perror("sid");
	close(AudioFildes);
	return 1;
    }
#if 1
    //dummy=((4<<16) | 14);
    //dummy=((2<<16) | 10);	/* 2 Buffers a 1024 Bytes 1/10s */
    //dummy=((3<<16) | 10);	/* 3 Buffers a 1024 Bytes 7/50s */
    // dummy=((2<<16) | 9);	/* 2 Buffers a 512 Bytes BAD */
    // dummy=((3<<16) | 9);	/* 3 Buffers a 512 Bytes BAD */
    dummy=((6<<16) | 9);	/* 6 Buffers a 512 Bytes */
    // dummy=((16<<16) | 8);	/* 16 Buffers a 256 Bytes BAD */
    //dummy=((5<<16) | 8);	/* 5 Buffers a 256 Bytes BAD */
    // dummy=((4<<16) | 7);	/* 4 Buffers a 64 Bytes */
    if( ioctl(AudioFildes,SNDCTL_DSP_SETFRAGMENT,&dummy)==-1 ) {
	perror("sid");
	close(AudioFildes);
	return 1;
    }
#endif
#if 0
    dummy=4;
    if( ioctl(AudioFildes,SNDCTL_DSP_SUBDIVIDE,&dummy)==-1 ) {
	perror("sid");
	close(AudioFildes);
	return;
    }
#endif
    if( ioctl(AudioFildes,SNDCTL_DSP_GETBLKSIZE,&dummy)==-1 ) {
	perror("sid");
	close(AudioFildes);
	return 1;
    }
#if defined(X11) && defined(DEBUG)
    printf("DSP block size %d\n",dummy);
    printf("DSP sample speed %d\n",AudioFrequency);
#endif
    return 0;
}

/*
**	Machine depend DSP part: Flush.
*/
static void FlushDSP(void)
{
    write(AudioFildes,AudioBuffer,0);	/* flush the buffer */
}

/*
**	Machine depend DSP part: Suspend.
*/
static void SuspendDSP(void)
{
    FlushDSP();
}

/*
**	Machine depend DSP part: Resume.
*/
static void ResumeDSP(void)
{
    /* Nothing to do */
}

/*
**	Machine depend DSP part: Cleanup.
*/
static void LeaveDSP(void)
{
    FlushDSP();
}

/*
**	Machine depend DSP part: Play it.
*/
static void SendToDSP(unsigned char* buffer,int length)
{
    write(AudioFildes,buffer,length);
}

#endif	/* } __linux__ */

#ifdef __GO32__    /* { */

/*.............................................................................
 *	MSDOS DEPENDEND DSP PART
 *...........................................................................*/

#include <stdlib.h>
#include <string.h>
#include <go32.h>
#include <dos.h>
#include <dpmi.h>

#include VIDEO_H

/*
**	GO32 DPMI structs for accessing DOS memory.
*/
static _go32_dpmi_seginfo DosMem;	/* DOS (conventional) memory buffer */

static _go32_dpmi_seginfo OldIrqRm;	/* original real mode IRQ */
static _go32_dpmi_seginfo RmSi;		/* real mode interrupt segment info */

/*
**	Offsets relative to base I/O address.
*/
#define SB_LEFT_FM_STATUS	0x00    /* Pro only */
#define SB_LEFT_FM_ADDRESS	0x00    /* Pro only */
#define SB_LEFT_FM_DATA		0x01    /* Pro only */
#define SB_RIGHT_FM_STATUS	0x02    /* Pro only */
#define SB_RIGHT_FM_ADDRESS	0x02    /* Pro only */
#define SB_RIGHT_FM_DATA	0x03    /* Pro only */
#define SB_MIXER_ADDRESS	0x04    /* Pro only */
#define SB_MIXER_DATA		0x05    /* Pro only */
#define SB_DSP_RESET		0x06
#define SB_FM_STATUS		0x08
#define SB_FM_ADDRESS		0x08
#define SB_FM_DATA		0x09
#define SB_DSP_READ_DATA	0x0A
#define SB_DSP_WRITE_DATA	0x0C
#define SB_DSP_WRITE_STATUS	0x0C
#define SB_DSP_DATA_AVAIL	0x0E
#define SB_CD_ROM_DATA		0x10    /* Pro only */
#define SB_CD_ROM_STATUS	0x11    /* Pro only */
#define SB_CD_ROM_RESET		0x12    /* Pro only */
#define SB_CD_ROM_ENABLE	0x13    /* Pro only */

#define SB_ADLIB_FM_STATUS	0x388
#define SB_ADLIB_FM_ADDRESS	0x388
#define SB_ADLIB_FM_DATA	0x389

/* Defines for 8237 DMA Controller IO addresses */
#define SB_DMA		0
#define SB_CH0_BASE	(SB_DMA+0)
#define SB_CH0_COUNT	(SB_DMA+1)
#define SB_CH1_BASE	(SB_DMA+2)
#define SB_CH1_COUNT	(SB_DMA+3)
#define SB_CH2_BASE	(SB_DMA+4)
#define SB_CH2_COUNT	(SB_DMA+5)
#define SB_CH3_BASE	(SB_DMA+6)
#define SB_CH3_COUNT	(SB_DMA+7)
#define SB_DMA_STATUS	(SB_DMA+8)
#define SB_DMA_CMD	(SB_DMA+8)
#define SB_DMA_REQUEST	(SB_DMA+9)
#define SB_DMA_MASK	(SB_DMA+10)
#define SB_DMA_MODE	(SB_DMA+11)
#define SB_DMA_FF	(SB_DMA+12)
#define SB_DMA_TMP	(SB_DMA+13)
#define SB_DMA_CLEAR	(SB_DMA+13)
#define SB_DMA_CLRMSK	(SB_DMA+14)
#define SB_DMA_WRMSK	(SB_DMA+15)
#define SB_DMAPAGE	0x80

/* Types of Soundblaster Cards */
#define SB_TYPE_15	1
#define SB_TYPE_PRO	2
#define SB_TYPE_20	3

/* DSP Commands */
#define SB_DIRECT_8_BIT_DAC	0x10
#define SB_DMA_8_BIT_DAC	0x14
#define SB_DMA_2_BIT_DAC	0x16
#define SB_DMA_2_BIT_REF_DAC	0x17
#define SB_DIRECT_ADC		0x20
#define SB_DMA_ADC		0x24
#define SB_MIDI_READ_POLL	0x30
#define SB_MIDI_READ_IRQ	0x31
#define SB_MIDI_WRITE_POLL	0x38
#define SB_TIME_CONSTANT	0x40
#define SB_DMA_4_BIT_DAC	0x74
#define SB_DMA_4_BIT_REF_DAC	0x75
#define SB_DMA_26_BIT_DAC	0x76
#define SB_DMA_26_BIT_REF_DAC	0x77
#define SB_HALT_DMA		0xD0
#define SB_CONTINUE_DMA		0xD4
#define SB_SPEAKER_ON		0xD1
#define SB_SPEAKER_OFF		0xD3
#define SB_DSP_ID		0xE0
#define SB_DSP_VER		0xE1
#define SB_MDAC1		0x61
#define SB_MDAC2		0x62
#define SB_MDAC3		0x63
#define SB_MDAC4		0x64
#define SB_MDAC5		0x65
#define SB_MDAC6		0x66
#define SB_MDAC7		0x67

#define SB_SET_BLOCKSIZE	0x48
#define SB_HIGH_DMA_8_BIT_DAC	0x91
#define SB_HIGH_DMA_8_BIT_ADC	0x99

/*
**	Card parameters
*/
static unsigned SbIoAddr;
static unsigned SbIrq;
static unsigned SbDmaChan;

static int SbHighSpeed;			/* Flag: high sample rate */

static volatile int SbDmaActive;	/* Flag: dma active */

/*
**	Conventional memory buffers for DMA.
*/
#define DMA_BUFFERS	2
static volatile int SbBufNum;
static char* SbBuf[DMA_BUFFERS];
static volatile unsigned int SbBufLen[DMA_BUFFERS];

/*
**	Write to DSP.
*/
static void inline SbWriteDac(int x)
{
    while( inb(SbIoAddr+SB_DSP_WRITE_STATUS)&0x80 ) {
    }
    outb(SbIoAddr+SB_DSP_WRITE_DATA,x);
}

/*
**	Read from DSP.
*/
static int SbReadDac(int base)
{
    int i;

    for( i=0; i<10000; i++ ) {
	if( inb(base+SB_DSP_DATA_AVAIL)&0x080 ) {
	    break;
	}
    }
    return inb(base+SB_DSP_READ_DATA);
}

/*
** Install our interrupt as the real mode interrupt handler for 
** the IRQ the soundblaster is on.
**
** We accomplish this by have GO32 allocate a real mode callback for us.
** The callback packages our protected mode code up in a real mode wrapper.
*/
static void SbInstallRealModeInterrupt(void (*sb_intr)(_go32_dpmi_registers *))
{
    int ret;
    static _go32_dpmi_registers rm_regs;

    memset(&rm_regs,0,sizeof(_go32_dpmi_registers));
    RmSi.pm_offset=(int)sb_intr;
    RmSi.pm_selector=_go32_my_cs();
    ret=_go32_dpmi_allocate_real_mode_callback_iret(&RmSi,&rm_regs);
    if( ret ) {
	Cleanup();
        printf("cannot allocate real mode callback, error=%04x\n",ret);
        exit(1);
    }

    disable();
    _go32_dpmi_get_real_mode_interrupt_vector(8+SbIrq,&OldIrqRm);
    _go32_dpmi_set_real_mode_interrupt_vector(8+SbIrq,&RmSi);
    enable();
}

/*
**	Install interrupts.
*/
static void SbInstallInterrupts(void (*sb_intr)(_go32_dpmi_registers *))
{
    SbInstallRealModeInterrupt(sb_intr);
}

/*
**	Remove our real mode interrupt handler.
*/
static void SbCleanupRealModeInterrupt(void)
{
    disable();
    if( RmSi.size!=-1 ) {
	_go32_dpmi_set_real_mode_interrupt_vector(8+SbIrq,&OldIrqRm);

        _go32_dpmi_free_real_mode_callback(&RmSi);
	RmSi.size=-1;
    }
    enable();
}

/*
**	Remove interrupt handlers
*/
static void SbCleanupInterrupts(void)
{
    SbCleanupRealModeInterrupt();	/* remove our interrupt */
}

/*
**	Set sampling/playback rate.
**	Parameter is rate in Hz (samples per second).
*/
static void SbSetSampleRate(unsigned int rate)
{
    unsigned char tc;

    SbHighSpeed=(rate>37000);
    if( SbHighSpeed ) {
	tc=(unsigned char)((65536-256000000/rate)>>8);
    } else {
	tc=(unsigned char)(256-1000000/rate);
    }

    SbWriteDac(SB_TIME_CONSTANT);	/* Command byte for sample rate */
    SbWriteDac(tc);			/* Sample rate time constant */
}

/*
**	Turn speaker on/off for sampling.
*/
static void SbVoice(int state)
{
    SbWriteDac(state ? SB_SPEAKER_ON : SB_SPEAKER_OFF);
}

/*
**	Play buffer with DMA.
*/
static void SbPlayBuffer(register unsigned i)
{
    int t;
    unsigned char im;
    unsigned char tm;
    static int dma_page_port[] = { 0x87,0x83,0x81,0x82, 0x00,0x8B,0x89,0x8A };

    if( SbBufLen[i]<=0 ) {		/* See if we're already done */
        SbDmaActive=0;
        return;
    }
    disable();

    im=inb(0x21);			/* Enable interrupts on PIC */
    tm=~(1<<SbIrq);
    outb(0x21,im&tm);

    /*
    **	Block DMA channel.
    **	Set DMA mode 'play'.
    */
    outb(SB_DMA_MASK,4+(SbDmaChan%4));
    outb(SB_DMA_FF,0);
    outb(SB_DMA_MODE,0x48+(SbDmaChan%4));/* channel 1 */

    SbBufNum=i;

    /*
    **	Set transfer address.
    */
    t=(int) ((unsigned long) SbBuf[i]>>16);
    outb(dma_page_port[SbDmaChan],t);	/* DMA PAGE Channel 1 */
    t=(int) ((unsigned long) SbBuf[i]&0xFFFF);
    outb(SB_DMA+2*SbDmaChan,t&0xFF);
    outb(SB_DMA+2*SbDmaChan,t>>8);

    /*
    **	Set transfer length byte count.
    */
    outb(SB_DMA+2*SbDmaChan+1,(SbBufLen[i]-1)&0xFF);
    outb(SB_DMA+2*SbDmaChan+1,(SbBufLen[i]-1)>>8);

    /*
    **	Unmask DMA channel.
    */
    outb(SB_DMA_MASK,SbDmaChan%4);

    if( SbHighSpeed ) {
	SbWriteDac(SB_SET_BLOCKSIZE);	/* prepare block programming */
    } else {
	SbWriteDac(SB_DMA_8_BIT_DAC);	/* command byte for DMA DAC transfer */
    }

    SbWriteDac((SbBufLen[i]-1)&0xFF);	/* sb_write length */
    SbWriteDac((SbBufLen[i]-1)>>8);

    if( SbHighSpeed ) {
	SbWriteDac(SB_HIGH_DMA_8_BIT_DAC);/* high speed DMA DAC transfer */
    }

    SbDmaActive=1;			/* A sound is playing now. */
}

/*
**	Interrupt player.
*/
void SbIntrPlay(_go32_dpmi_registers *reg)
{
    register int i;

    inb(SbIoAddr+SB_DSP_DATA_AVAIL);	/* Acknowledge soundblaster */
    i=SbBufNum;
    SbBufLen[i]=0;
    
    ++i;
    if( i==DMA_BUFFERS ) {
	i=0;
    }
    SbPlayBuffer(i);			/* Start next buffer player */
    
    outb(0x20,0x20);			/* Acknowledge the interrupt */
}

/*
**	Machine depend DSP part: Initialise.
*/
static int EnterDSP(void)
{
    int i;
    char* t;
    static int Bases[] = {0x210,0x220,0x230,0x240,0x250,0x260};

    /*
    **	Preset some variables.
    */
    DosMem.size=-1;
    RmSi.size=-1;

    /*
    **	Set arguments to Soundblaster defaults
    */
    SbIoAddr=0;
    SbIrq=7;
    SbDmaChan=1;
    
    if( (t=getenv("BLASTER")) ) {
	char* blaster;

	blaster=strdup(t);		/* Get a copy */

	/*
	**	Parse the BLASTER variable
	*/
	for( t=strtok(blaster," \t"); t; t=strtok(NULL," \t") ) {
	    switch( t[0] ) {
		case 'A':
		case 'a':		/* I/O address */
		    sscanf(t+1,"%x",&SbIoAddr);
		    break;
		case 'I':
		case 'i':		/* IRQ */
		    SbIrq=atoi(t+1);
		    break;
		case 'D':
		case 'd':		/* DMA channel */
		    SbDmaChan=atoi(t+1);
		    break;
		case 'T':
		case 't':		/* Card type */
		    break;

		case 'H':
		case 'h':
		case 'P':
		case 'p':
		case 'E':
		case 'e':
		    break;
		    
		default:
		    printf("Unknown BLASTER option %c\n",t[0]);
		    break;
	    }
	    
	}

	free(blaster);  
    }
    
    /*
    **	Prove current I/O addresse.
    */
    if( !SbIoAddr ) {
	int j;
	int base;
	int error;

	for( error=i=0; i<sizeof(Bases)/sizeof(Bases) && !error; ++i ) {
	    base=Bases[i];

	    outb(base+SB_DSP_RESET,1);

	    inb(base+SB_DSP_RESET);		/* Kill some time */
	    inb(base+SB_DSP_RESET);
	    inb(base+SB_DSP_RESET);
	    inb(base+SB_DSP_RESET);
	
	    outb(base+SB_DSP_RESET,0);

	    for( j=0; j<100; j++) {
		if( SbReadDac(base)==0xAA ) {
		    error=1;
		    break;
		}
	    }
	}
	if( error ) {
	    SbIoAddr=base;
	}
    }

    if( !SbIoAddr ) {				/* not found! */
	printf("No soundblaster found!\n");
	return 1;
    }

    /*
    **	Allocate conventional memory for our DMA buffers.
    **		Each DMA buffer must be aligned on a 64K boundary in
    **		physical memory.
    */
    DosMem.size=65536*(DMA_BUFFERS+1)/16;
    if( _go32_dpmi_allocate_dos_memory(&DosMem) ) {
        printf("Unable to allocate dos memory - max size is %lu\n",
	    DosMem.size);
	DosMem.size=-1;
        return 1;
    }

    /*
    **	Setup DMA Buffers, 64k aligned.
    */
    i=DosMem.rm_segment*16;
    SbBuf[0]= (char*)((i+0x0FFFFL) & 0xFFFF0000L);
    for( i=1; i<DMA_BUFFERS; ++i ) {
	SbBuf[i]=(char*)((unsigned long)SbBuf[i-1]+0x10000);
    }

    SbVoice(1);
    SbSetSampleRate(AudioFrequency);

    SbInstallInterrupts(SbIntrPlay);

    printf("Init: Port %d Irq %d Dma %d\n",SbIoAddr,SbIrq,SbDmaChan);
    fflush(stdout);

    return 0;
}

/*
**	Machine depend DSP part: Flush.
*/
static void FlushDSP(void)
{
    while( SbDmaActive )
 	;
}

/*
**	Machine depend DSP part: Suspend.
*/
static void SuspendDSP(void)
{
    FlushDSP();
}

/*
**	Machine depend DSP part: Resume.
*/
static void ResumeDSP(void)
{
}

/*
**	Machine depend DSP part: Cleanup.
*/
static void LeaveDSP(void)
{
    int j;

    /*
    **	Reset DSP.
    */
    outb(SbIoAddr+SB_DSP_RESET,1);

    inb(SbIoAddr+SB_DSP_RESET);		/* Kill some time */
    inb(SbIoAddr+SB_DSP_RESET);
    inb(SbIoAddr+SB_DSP_RESET);
    inb(SbIoAddr+SB_DSP_RESET);
    
    outb(SbIoAddr+SB_DSP_RESET,0);
    for( j=0; j<100; j++) {
        if( SbReadDac(SbIoAddr)==0xAA ) {
            break;
        }
    }

    if( DosMem.size!=-1 ) {
	if( !_go32_dpmi_free_dos_memory(&DosMem) ) {
	    printf("Unable to free dos memory");
	}
    }
    SbCleanupInterrupts();
}

/*
**	Machine depend DSP part: Play it.
*/
static void SendToDSP(unsigned char* buffer,int length)
{
    int i;
    // extern void VicText(int, int, char *);
    #define VicText(a,b,c)

    /*
    **	Find free DMA buffer.
    */
    if( length<=0 || length>1024 ) {
	return;
    }
    // printf("Current buffer %d\n",SbBufNum);
    // fflush(stdout);
    if( !(i=SbBufNum) ) {
	i=DMA_BUFFERS;
    }
    --i;

    // FIXME: currently support only 2 buffers.
    if( SbBufLen[i] ) {			/* buffer full, wait for next free */
	++i;
	if( i==DMA_BUFFERS ) {
	    i=0;
	}
	while( SbBufLen[i] ) {		/* wait for current buffer empty */
	    VicText(0,0,"Wait  DMA");
	    ;
	}
    }

    // FIXME: we can also append to unused buffer.
    if( i<0 || i==DMA_BUFFERS ) {
	VicText(0,0,"Illegal DMA buffer");
	;
    }
    dosmemput(buffer,length,(unsigned long)SbBuf[i]);
    SbBufLen[i]=length;

    if( !SbDmaActive ) {		/* start dma transfer */
	VicText(0,0,"Start DMA");
	SbPlayBuffer(i);		/* Start first buffer player */
	enable();
    }
}

#endif	/* } __GO32__ */

/*.............................................................................
 *	DSP GENERAL PART
 *...........................................................................*/

/*
**	Build wave tables.
*/
static void BuildWaveTables(void)
{
    int i;
    unsigned char* p;

    /*
    **	Build triangle waveform.
    */
    p=TriangleWave;
    for( i=0; i<256; ++i ) {
	memset(p,i,sizeof(TriangleWave)/(256*2));
	p+=sizeof(TriangleWave)/(256*2);
    }
    for( i=0; i<256; ++i ) {
	memset(p,255-i,sizeof(TriangleWave)/(256*2));
	p+=sizeof(TriangleWave)/(256*2);
    }

    /*
    **	Build saw waveform.
    */
    p=SawtoothWave;
    for( i=0; i<256; ++i ) {
	memset(p,i,sizeof(SawtoothWave)/(256));
	p+=sizeof(SawtoothWave)/(256);
    }

    /*
    **	Build pulse waveform.
    */
    p=PulseWave;
    memset(p,0,sizeof(PulseWave)/(2));
    p+=sizeof(PulseWave)/(2);
    memset(p,255,sizeof(PulseWave)/(2));
}

/*
**	Build envelope tables.
*/
static void BuildEnvelopeTables(void)
{
    int i;

    for( i=0; i<16; ++i ) {
	SidAttackIndex[i]=(255*CPU_CLK)/SidTiming[i]/AudioFrequency;
	SidAttackStart[i]=(double)SidTiming[i]*AudioFrequency*1000/CPU_CLK;
	SidAttackSteps[i]=255*1000-(SidAttackIndex[i]*SidAttackStart[i]);

	SidDecayIndex[i]=(255*CPU_CLK)/SidTiming[i]/3/AudioFrequency;
	SidDecayStart[i]=(double)SidTiming[i]*3*AudioFrequency*1000/CPU_CLK;
	SidDecaySteps[i]=255*1000-(SidDecayIndex[i]*SidDecayStart[i]);
    }
}

/*
**	Initialition for sound.
*/
void EnterSound(void)
{
    /*
    **	This is always need, I don't check everywhere for off.
    */
    DspVoices[0].Off=0;
    DspVoices[1].Off=0;
    DspVoices[2].Off=0;

    DspVoices[0].Modulator=&DspVoices[2];
    DspVoices[1].Modulator=&DspVoices[0];
    DspVoices[2].Modulator=&DspVoices[1];
    DspVoices[0].Carrier=&DspVoices[1];
    DspVoices[1].Carrier=&DspVoices[2];
    DspVoices[2].Carrier=&DspVoices[0];

    DspVoices[0].NoiseRegister=NOISE_SEED;
    DspVoices[1].NoiseRegister=NOISE_SEED;
    DspVoices[2].NoiseRegister=NOISE_SEED;

    if( SidSoundOff ) {			/* sound turned already off */
	return;
    }
    SidSoundOff=3;			/* Turn sound always off */

    if( EnterDSP() ) {			/* machine depend dsp part */
	return;
    }

    BuildWaveTables();
    BuildEnvelopeTables();

    SidSoundOff=0;			/* turn sound emulation on */
}

/*
**	Cleanup for sound.
*/
void LeaveSound(void)
{
    if( !(SidSoundOff&2) ) {		/* initialise ok */
	LeaveDSP();
    }
}

/*
**	Suspend sound.
*/
void SuspendSound(void)
{
    if( !(SidSoundOff&2) ) {		/* initialise ok */
	SuspendDSP();
    }
}

/*
**	Resume sound.
*/
void ResumeSound(void)
{
    if( !(SidSoundOff&2) ) {		/* initialise ok */
	ResumeDSP();
    }
}

/*
**	Turn sound on/off
*/
void ToggleSound(void)
{
    if( !(SidSoundOff&3) ) {		/* initialise ok and on */
	FlushDSP();
    }
    SidSoundOff^=1;
}

/*
**	Advance index into wave table.
*/
static void WaveAdvance(DspInfo* voice)
{
    int temp;

    temp=voice->WaveCount;
    temp-=voice->WaveSteps;
    if( temp<0 ) {			/* divide remainder */
	temp+=AudioFrequency;
	voice->WaveIndex++;
	voice->NoiseIndex++;
    }
    voice->WaveCount=temp;
    voice->WaveIndex+=voice->WaveShift;
    if( voice->WaveIndex>(4096-1) ) {
	voice->WaveIndex&=4096-1;
	if( voice->Off&2 ) {
	    voice->Carrier->NoiseIndex=0;
	    voice->Carrier->WaveIndex=0;
	    voice->Carrier->WaveCount=AudioFrequency;
	}
    }

    voice->NoiseIndex+=voice->WaveShift;
    if( voice->NoiseIndex>(256-1) ) {
	voice->NoiseIndex&=256-1;
	/*
	**	Advance voice output.
	*/
	temp=voice->NoiseRegister;
	voice->NoiseOutput=
	     ((temp>>(22-7))&0x80)
	    |((temp>>(20-6))&0x40)
	    |((temp>>(16-5))&0x20)
	    |((temp>>(13-4))&0x10)
	    |((temp>>(11-3))&0x08)
	    |((temp>>( 7-2))&0x04)
	    |((temp>>( 4-1))&0x02)
	    |((temp>>( 2-0))&0x01);
	temp<<=1;
	temp|=((temp>>23)^(temp>>18))&1;
	voice->NoiseRegister=temp;
    }
}

/*
**	Calculate wave advance.
*/
static void CalculateAdvance(DspInfo* voice,int value)
{
    int frequency;

    frequency=(int)((double)value*CPU_CLK/16777216);
    voice->WaveShift=(frequency*4095)/AudioFrequency;
    voice->WaveSteps=(frequency-(AudioFrequency*voice->WaveShift)/4095)*4095;

    /*
    printf("Frequency: %d (=%d)\n",
	(voice->WaveSteps*AudioFrequency)/(AudioFrequency*4096)+
	(AudioFrequency*voice->WaveShift)/4096,
	frequency);
    */
}

/*
**	Sample one byte of voice wave.
*/
static int SampleVoiceWave(DspInfo* voice,int control)
{
#define NSE	0x80
#define PUL	0x40
#define SAW	0x20
#define TRI	0x10
#define TEST	0x08
#define RING	0x04
#define SYNC	0x02
#define GATE	0x01
#define NONE	0x00

#define CASE(x)	\
    Case(x+0) Case(x+1) Case(x+2) Case(x+3) \
    Case(x+4) Case(x+5) Case(x+6) Case(x+7)

#define CASE_NR(x)	\
    Case(x+0) Case(x+1) Case(x+2) Case(x+3)

#define CASE_R(x)	\
    Case(x+RING+0) Case(x+RING+1) Case(x+RING+2) Case(x+RING+3)

#define RING_MODULATION	\
    (PulseWave[2048+voice->WaveIndex]			\
	^PulseWave[2048+voice->Modulator->WaveIndex])

    int out;

    /*
    **	Generate wave form.
    */
    Switch( control ) {
	CASE(NONE)
	    out=0;
	    Break;

	CASE(TEST)
	CASE(TRI+TEST)
	CASE(SAW+TEST)
	CASE(SAW+TRI+TEST)
	CASE(PUL+TEST)
	CASE(PUL+TRI+TEST)
	CASE(PUL+SAW+TEST)
	CASE(PUL+SAW+TRI+TEST)
	CASE(NSE+TEST)
	CASE(NSE+TRI+TEST)
	CASE(NSE+SAW+TEST)
	CASE(NSE+SAW+TRI+TEST)
	CASE(NSE+PUL+TEST)
	CASE(NSE+PUL+TRI+TEST)
	CASE(NSE+PUL+SAW+TEST)
	CASE(NSE+PUL+SAW+TRI+TEST)
	    /* Reset generator registers */
	    voice->NoiseRegister=NOISE_SEED;
	    voice->NoiseOutput=0xFE;
	    voice->WaveCount=0;
	    voice->WaveIndex=0;
	    voice->NoiseIndex=0;
	    out=0;
	    Break;

	CASE_NR(TRI)
	    out=TriangleWave[voice->WaveIndex];
	    WaveAdvance(voice);
	    Break;

	CASE_R(TRI)
	    out=RING_MODULATION;
	    WaveAdvance(voice);
	    Break;

	CASE(SAW)
	    out=SawtoothWave[voice->WaveIndex];
	    WaveAdvance(voice);
	    Break;

	CASE_NR(SAW+TRI)
	    out=SawtoothWave[voice->WaveIndex];
	    out&=TriangleWave[voice->WaveIndex];
	    WaveAdvance(voice);
	    Break;
	CASE_R(SAW+TRI)
	    out=SawtoothWave[voice->WaveIndex];
	    out&=RING_MODULATION;
	    WaveAdvance(voice);
	    Break;

	CASE(PUL)
	    out=PulseWave[voice->PulseOffset+voice->WaveIndex];
	    WaveAdvance(voice);
	    Break;
	CASE_NR(PUL+TRI)
	    out=PulseWave[voice->PulseOffset+voice->WaveIndex];
	    out&=TriangleWave[voice->WaveIndex];
	    WaveAdvance(voice);
	    Break;
	CASE_R(PUL+TRI)
	    out=PulseWave[voice->PulseOffset+voice->WaveIndex];
	    out&=RING_MODULATION;
	    WaveAdvance(voice);
	    Break;
	CASE(PUL+SAW)
	    out=PulseWave[voice->PulseOffset+voice->WaveIndex];
	    out&=SawtoothWave[voice->WaveIndex];
	    WaveAdvance(voice);
	    Break;
	CASE_NR(PUL+SAW+TRI)
	    out=PulseWave[voice->PulseOffset+voice->WaveIndex];
	    out&=SawtoothWave[voice->WaveIndex];
	    out&=TriangleWave[voice->WaveIndex];
	    WaveAdvance(voice);
	    Break;
	CASE_R(PUL+SAW+TRI)
	    out=PulseWave[voice->PulseOffset+voice->WaveIndex];
	    out&=SawtoothWave[voice->WaveIndex];
	    out&=RING_MODULATION;
	    WaveAdvance(voice);
	    Break;

	CASE(NSE)
	    out=voice->NoiseOutput;
	    WaveAdvance(voice);
	    Break;
	CASE_NR(NSE+TRI)
	    out=voice->NoiseOutput;
	    out&=TriangleWave[voice->WaveIndex];
	    WaveAdvance(voice);
	    Break;
	CASE_R(NSE+TRI)
	    out=voice->NoiseOutput;
	    out&=RING_MODULATION;
	    WaveAdvance(voice);
	    Break;
	CASE(NSE+SAW)
	    out=voice->NoiseOutput;
	    out&=SawtoothWave[voice->WaveIndex];
	    WaveAdvance(voice);
	    Break;
	CASE_NR(NSE+SAW+TRI)
	    out=voice->NoiseOutput;
	    out&=SawtoothWave[voice->WaveIndex];
	    out&=TriangleWave[voice->WaveIndex];
	    WaveAdvance(voice);
	    Break;
	CASE_R(NSE+SAW+TRI)
	    out=voice->NoiseOutput;
	    out&=SawtoothWave[voice->WaveIndex];
	    out&=RING_MODULATION;
	    WaveAdvance(voice);
	    Break;
	CASE(NSE+PUL)
	    out=voice->NoiseOutput;
	    out&=PulseWave[voice->PulseOffset+voice->WaveIndex];
	    WaveAdvance(voice);
	    Break;
	CASE_NR(NSE+PUL+TRI)
	    out=voice->NoiseOutput;
	    out&=PulseWave[voice->PulseOffset+voice->WaveIndex];
	    out&=TriangleWave[voice->WaveIndex];
	    WaveAdvance(voice);
	    Break;
	CASE_R(NSE+PUL+TRI)
	    out=voice->NoiseOutput;
	    out&=PulseWave[voice->PulseOffset+voice->WaveIndex];
	    out&=RING_MODULATION;
	    WaveAdvance(voice);
	    Break;
	CASE(NSE+PUL+SAW)
	    out=voice->NoiseOutput;
	    out&=PulseWave[voice->PulseOffset+voice->WaveIndex];
	    out&=SawtoothWave[voice->WaveIndex];
	    WaveAdvance(voice);
	    Break;
	CASE_NR(NSE+PUL+SAW+TRI)
	    out=voice->NoiseOutput;
	    out&=PulseWave[voice->PulseOffset+voice->WaveIndex];
	    out&=SawtoothWave[voice->WaveIndex];
	    out&=TriangleWave[voice->WaveIndex];
	    WaveAdvance(voice);
	    Break;
	CASE_R(NSE+PUL+SAW+TRI)
	    out=voice->NoiseOutput;
	    out&=PulseWave[voice->PulseOffset+voice->WaveIndex];
	    out&=SawtoothWave[voice->WaveIndex];
	    out&=RING_MODULATION;
	    WaveAdvance(voice);
	    Break;

	Default
#if defined(X11) || 1
	    printf("Wrong control %x\n",control);
#endif
	    out=0;
	    Break;
    }
    return out;
}

/*
**	Sample one byte of voice envelope.
*/
static int SampleVoiceEnvelope(DspInfo* voice,int index)
{
    switch( voice->State ) {
	case SID_OFF:
	    break;

	case SID_ATTACK:
	    voice->EnvVolume+=voice->EnvIndex;
	    voice->EnvCount-=voice->EnvSteps;
	    if( voice->EnvCount<0 ) {
		voice->EnvCount+=voice->EnvStart;
		++voice->EnvVolume;
	    }
	    // printf("%d ",voice->EnvVolume);
	    if( voice->EnvVolume>=255 ) {
		voice->EnvVolume=255;
		// printf("\n DECAY: ");
		voice->State=SID_DECAY;
		voice->EnvIndex=SidDecayIndex[DECAY(index*7)];
		voice->EnvCount=
		voice->EnvStart=SidDecayStart[DECAY(index*7)];
		voice->EnvSteps=SidDecaySteps[DECAY(index*7)];

		// printf("Index %d Start %d Steps %d\n",
		//     voice->EnvIndex,voice->EnvStart,voice->EnvSteps);
	    }
	    break;

	case SID_DECAY:
	    voice->EnvVolume-=voice->EnvIndex;
	    voice->EnvCount-=voice->EnvSteps;
	    if( voice->EnvCount<0 ) {
		voice->EnvCount+=voice->EnvStart;
		--voice->EnvVolume;
	    }
	    // printf("%d ",voice->EnvVolume);
	    if( voice->EnvVolume<=voice->Sustain ) {
		voice->EnvVolume=voice->Sustain;
		// printf("\n SUSTAIN\n");
		voice->State=SID_SUSTAIN;
	    }
	    break;

	case SID_SUSTAIN:
	    voice->EnvVolume=voice->Sustain;
	    break;

	case SID_RELEASE:
	    voice->EnvVolume-=voice->EnvIndex;
	    voice->EnvCount-=voice->EnvSteps;
	    if( voice->EnvCount<0 ) {
		voice->EnvCount+=voice->EnvStart;
		--voice->EnvVolume;
	    }
	    // printf("%d ",voice->EnvVolume);
	    if( voice->EnvVolume<=0 ) {
		voice->EnvVolume=0;
		// printf("\n OFF\n");
		voice->State=SID_OFF;
	    }
	    break;
    }
    return voice->EnvVolume;
}

/*
**	Make one sample.
*/
static void OneSample(void)
{
    static int VolumeTable[16] = {
	0*17*768, 1*17*768, 2*17*768, 3*17*768,
	4*17*768, 5*17*768, 6*17*768, 7*17*768,
	8*17*768, 9*17*768, 10*17*768, 11*17*768,
	12*17*768, 13*17*768, 14*17*768, 15*17*768
    };
    int i;

    i =SampleVoiceWave(DspVoices+0,SID[0*7+0x4])
	*SampleVoiceEnvelope(DspVoices+0,0);
    i+=SampleVoiceWave(DspVoices+1,SID[1*7+0x4])
	*SampleVoiceEnvelope(DspVoices+1,1);
    if( !(SID[0x18]&0x80) ) {       	/* voice 3 turned off */
	i+=SampleVoiceWave(DspVoices+2,SID[2*7+0x4])
	    *SampleVoiceEnvelope(DspVoices+2,2);
    }
    i/=256;
    AudioBuffer[AudioBufferIndex++]=
	(VolumeTable[VOLUME()]-i*VOLUME()*17)/(3*256);
}

/*
**	Sample to cycle.
*/
static void SampleToNow(void)
{
    static int divider;
    int cycle;

    cycle=Cycle;
    Cycle=SidCycle;
    while( Cycle<cycle ) {
	OneSample();
	Cycle+=CPU_CLK/AudioFrequency;	/* ca 44us */
	divider-=AudioFrequency;
	if( divider<0 ) {		/* handle remainder */
	    ++Cycle;
	    divider+=CPU_CLK;
	}
    }
    SidCycle=Cycle;
    Cycle=cycle;
}

/*
**	Emulate sid chip
*/
void EmulSid(void)
{
    if( !SidSoundOff ) {
	SampleToNow();
	SendToDSP(AudioBuffer,AudioBufferIndex);
	// write(1,AudioBuffer,AudioBufferIndex);
    }
    AudioBufferIndex=0;
}

void SidFrqLo(int voice,int value)
{
    SampleToNow();
    SID[voice*7+0]=value;
    CalculateAdvance(DspVoices+voice,FREQUENCY(voice*7));
}

void SidFrqHi(int voice,int value)
{
    SampleToNow();
    SID[voice*7+1]=value;
    CalculateAdvance(DspVoices+voice,FREQUENCY(voice*7));
}

void SidPulLo(int voice,int value)
{
    SampleToNow();
    SID[voice*7+2]=value;
    DspVoices[voice].PulseOffset=4096-PULSEWIDTH(voice*7);
}

void SidPulHi(int voice,int value)
{
    SampleToNow();
    SID[voice*7+3]=value;
    DspVoices[voice].PulseOffset=4096-PULSEWIDTH(voice*7);
}

void SidCtrl(int voice,int value)
{
    SampleToNow();
#if defined(X11) && defined(DEBUG)
    if( value&2 ) {
	printf("SYNC:\n");
    }
    if( value&4 )
	printf("RING:\n");
#endif
    
    DspVoices[voice].Modulator->Off=value&6;	/* SYNC/RING */

    if( (value^SID[voice*7+4])&1 ) {	/* Gate switched */
	if( value&1 ) {
	    // printf("Attack: ");
	    DspVoices[voice].State=SID_ATTACK;
	    DspVoices[voice].EnvIndex=SidAttackIndex[ATTACK(voice*7)];
	    DspVoices[voice].EnvCount=
	    DspVoices[voice].EnvStart=SidAttackStart[ATTACK(voice*7)];
	    DspVoices[voice].EnvSteps=SidAttackSteps[ATTACK(voice*7)];

	    // printf("Index %d Start %d Steps %d\n",
 	// 	DspVoices[voice].EnvIndex,DspVoices[voice].EnvStart,
	// 	DspVoices[voice].EnvSteps);
	} else {
	    // printf("Release: ");
	    DspVoices[voice].State=SID_RELEASE;
	    DspVoices[voice].EnvIndex=SidDecayIndex[RELEASE(voice*7)];
	    DspVoices[voice].EnvCount=
	    DspVoices[voice].EnvStart=SidDecayStart[RELEASE(voice*7)];
	    DspVoices[voice].EnvSteps=SidDecaySteps[RELEASE(voice*7)];

	    // printf("Index %d Start %d Steps %d\n",
	// 	DspVoices[voice].EnvIndex,DspVoices[voice].EnvStart,
	// 	DspVoices[voice].EnvSteps);
	}
    }
}

void SidAtkDcy(int voice,int value)
{
    SampleToNow();
    /*
    **	Change current attack!!
    */
    if( DspVoices[voice].State==SID_ATTACK ) {
	DspVoices[voice].EnvIndex=SidAttackIndex[value>>4];
	DspVoices[voice].EnvCount=
	DspVoices[voice].EnvStart=SidAttackStart[value>>4];
	DspVoices[voice].EnvSteps=SidAttackSteps[value>>4];
    }
    /*
    **	Change current decay!!
    */
    if( DspVoices[voice].State==SID_DECAY ) {
	DspVoices[voice].EnvIndex=SidDecayIndex[value&0xF];
	DspVoices[voice].EnvCount=
	DspVoices[voice].EnvStart=SidDecayStart[value&0xF];
	DspVoices[voice].EnvSteps=SidDecaySteps[value&0xF];
    }
}

void SidStnRls(int voice,int value)
{
    SampleToNow();
    /*
    **	Change current release!!
    */
    if( DspVoices[voice].State==SID_RELEASE ) {
	DspVoices[voice].EnvIndex=SidDecayIndex[value&0xF];
	DspVoices[voice].EnvCount=
	DspVoices[voice].EnvStart=SidDecayStart[value&0xF];
	DspVoices[voice].EnvSteps=SidDecaySteps[value&0xF];
    }
    DspVoices[voice].Sustain=(value>>4)*17;
}

void SidVolume(int value)
{
    SampleToNow();
}

void SidWavRd()
{
    SampleToNow();
}

void SidEnvRd()
{
    SampleToNow();
}

#endif	/* } SID_DSP */

#ifdef SID_OPL3	/* { */

/******************************************************************************
 * OPL3 and Yamaha YM3812 SID Emulation
 *****************************************************************************/

#include <string.h>

#ifdef __linux__    /* { */
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/soundcard.h>

/*
**	Linux version without kernal support.
*/
#endif	/* } __linux__ */

/*-----------------------------------------------------------------------------
 *	OPL3 low-level part.
 *---------------------------------------------------------------------------*/

static int IsOpl3;			/* flag is opl3 */

#undef FM_PORT
#include "opl3.h"

/*
**	Output to opl3-register port.
**	Handles delays.
*/
static void Opl3Cmd(unsigned port,unsigned index,unsigned data)
{
    if( SidSoundOff ) {			/* sound disabled or not initialised */
	return;
    }
    outb(port+0,index);
    DELAY1();
    outb(port+1,data);
    DELAY2();
}

/*
**	Shortcut left-port. (not quite correct, but so called in literature)
*/
static void LeftPort(unsigned index,unsigned data)
{
    /* THIS SHOULD BE IN THE KERNAL !!!! */
    Opl3Cmd(FM_PORT,index,data);
}

/*
**	Shortcut right-port. (not quite correct, but so called in literature)
*/
static void RightPort(unsigned index,unsigned data)
{
    /* THIS SHOULD BE IN THE KERNAL !!!! */
    Opl3Cmd(FM_PORT+2,index,data);
}

/*
**	Detect the opl3 card.
**	Returns:
**		0	failure
**		2	opl2
**		3	opl3
*/
static int Opl3Detect(void)
{
    unsigned temp1;
    unsigned temp2;
    int i;

    /*
    **	Reset timer 1 and timer 2.
    **	Reset the IRQ.
    */
    LeftPort(TIMER_CONTROL_REGISTER,TIMER1_MASK|TIMER2_MASK);
    LeftPort(TIMER_CONTROL_REGISTER,IRQ_RESET);

    /*
    **	Read status register:
    */
    temp1=inb(FM_PORT);

    if( temp1&0xE0 ) {			/* must be zero */
	return 0;
    }

    /*
    **	Set timer 1 to 255 and unmask and start timer 1.
    */
    LeftPort(TIMER1_REGISTER, 0xff);
    LeftPort(TIMER_CONTROL_REGISTER,TIMER2_MASK|TIMER1_START);

    /*
    **	Now we have to delay at least 80 usec.
    */
    for( i=0; i<36*4; i++)
	inb(FM_PORT);

    /*
    **	Read status after timers have expired.
    */
    temp2=inb(FM_PORT);

    /*
    **	Reset timer 1 and timer 2.
    **	Reset the IRQ.
    */
    if( (temp2&0xE0)!=0xC0 ) {		/* must be 0xC0 */
	return 0;
    }

    /*
    **	FIXME: continue with opl3,opl4 detection.
    */
    // printf("%02X %02X\n",temp1,temp2);

    if( temp1&0x06 ) {			/* OPL2 */
	IsOpl3=0;
	return 2;
    }
    IsOpl3=1;
    return 3;
}

/*
**	Turn all voices of OPL3 off.
*/
void Opl3Silent(void)
{
    int i;

    /*
    **	Turn all voices off. (Release quick and KEYON off).
    */
    for( i=0; i<9*3; ++i ) {
	LeftPort(SUSTAIN_RELEASE+i,0x0F);
	LeftPort(KEYON_BLOCK+i,0);

	if( IsOpl3 ) {
	    RightPort(SUSTAIN_RELEASE+i,0x0F);
	    RightPort(KEYON_BLOCK+i,0);
	}
    }
}

/*
**	Reset OPL3.
*/
void Opl3Reset(void)
{
    /*
    **	Reset common registers.
    */
    LeftPort(TEST_REGISTER,ENABLE_WAVE_SELECT);
    LeftPort(TIMER_CONTROL_REGISTER,0);
    LeftPort(KBD_SPLIT_REGISTER,0);
    LeftPort(PERCUSSION_REGISTER,0);

    /*
    **	Disable 4-op mode.
    **	Enable opl3.
    */
    if( IsOpl3 ) {
	RightPort(OPL3_MODE_REGISTER,OPL3_ENABLE);
	RightPort(CONNECTION_SELECT_REGISTER,0x00);
	// RightPort(OPL3_MODE_REGISTER,0x00);
    }

    /*
    **	Disable all voices.
    */
    Opl3Silent();
}

/*
**	Frequency to block and fnum for the FM chip.
*/
static int FrqToFNum(int freq,int* fnum)
{
    int f;
    int block;

    f=freq;
    block=5;
    if( f==0 ) {
	block=0;
    } else if( f<261 ) {
	while( f<261 ) {
	    block--;
	    f<<=1;
	}
    } else if(f>493) {
	while(f>493) {
	    block++;
	    f>>=1;
	}
    }

    if( block>7 )
	block=7;
    else if( block<0 )
	block=0;

    *fnum=freq*(1<<(20-block))/49716;
    if( *fnum<1 )
	*fnum=1;
    return block;
}

/*-----------------------------------------------------------------------------
 *	Device driver hooks.
 *---------------------------------------------------------------------------*/

#ifdef SID_LOG_VOLUME

/*
**	SID - OPL3 Volume	log!
*/
static unsigned char Opl3Volume[16]= {
    0x3F,	/*  0 db */
    0x3C,	/*  3 db */
    0x38,	/*  6 db */
    0x30,	/* 12 db */
    0x2C,	/* 15 db */
    0x28,	/* 18 db */
    0x24,	/* 21 db */
    0x20,	/* 24 db */
    0x1C,	/* 27 db */
    0x18,	/* 30 db */
    0x14,	/* 33 db */
    0x10,	/* 36 db */
    0x0C,	/* 39 db */
    0x08,	/* 42 db */
    0x04,	/* 45 db */
    0x00	/* 24+12+6+3+1.5+.75: 47.25 db */
};

#else

/*
**	SID - OPL3 Volume	lin!
*/
static unsigned char Opl3Volume[16]= {
    0x3F,0x0E,0x0D,0x0C, 0x0B,0x0A,0x09,0x08,
    0x07,0x06,0x05,0x04, 0x03,0x02,0x01,0x00
};

#endif

/*
**	Change the frequency or key state of a voice.
*/
static void FrqKey(int voice)
{
    int frq;
    int fnum;
    int block;
   
    frq=FREQUENCY(voice*7);
    block=FrqToFNum( ((frq*(CPU_CLK/100))/(16777216/100)), &fnum );

    /*
    printf("FrqKey: %d %d %d | %02x %02x\n",frq,
	fnum&0xFF,block,fnum,
	((SID[voice*7+4]&1) ? KEYON_BIT : 0)
	|(block<<2)|(fnum>>8));
    */
    LeftPort(FNUM_LOW+voice,fnum&0xFF);
    LeftPort(KEYON_BLOCK+voice,
	((SID[voice*7+4]&1) ? KEYON_BIT : 0)
	|(block<<2)|(fnum>>8));
}

/*
**	Turn noice for voice on.
*/
static void NoiseOn(int voice)
{
    /*
    **	Attack:0, Decay:0, Sustain level:15,
    **	Release:15, Sustain on, Frequency Multiplier:11
    */
    LeftPort(ATTACK_DECAY+voice,0xFF);
    LeftPort(SUSTAIN_RELEASE+voice,0x0F);
    LeftPort(AM_VIB+voice,SUSTAIN_ON|11);
    LeftPort(WAVE_SELECT+voice,0x00);		/* sin */
    LeftPort(KSL_LEVEL+voice,0);
    LeftPort(FEEDBACK_CONNECTION+voice,STEREO_BITS|0x0E);

    LeftPort(WAVE_SELECT+3+voice,0x01);		/* dps */
}

/*
**	Turn noise for voice off.
*/
static void NoiseOff(int voice)
{
    LeftPort(KSL_LEVEL+voice,63);
    LeftPort(FEEDBACK_CONNECTION+voice,STEREO_BITS);
}


void SidFrqLo(int voice,int value)
{
    SID[voice*7+0]=value;
    FrqKey(voice);
}

void SidFrqHi(int voice,int value)
{
    SID[voice*7+1]=value;
    FrqKey(voice);
}

void SidPulLo(int voice,int value)
{
}

void SidPulHi(int voice,int value)
{
}

void SidCtrl(int voice,int value)
{
#define NSE	0x80
#define PUL	0x40
#define SAW	0x20
#define TRI	0x10
#define TEST	0x08
#define NONE	0x00
    /*
    printf("voice %d register %d = %02X\n",voice,voice*7+4,value);
    */
    SID[voice*7+4]=value;
    if( (value&TEST) || !(value&0xF0) ) {		/* off */
	LeftPort(KSL_LEVEL+3+voice,63);
    } else if( voice!=2 || !(SID[0x18]&0x80) ) {
	LeftPort(KSL_LEVEL+3+voice,Opl3Volume[SID[0x18]&0x0F]);
	if( value&NSE ) {
	    NoiseOn(voice);
	} else if( value&PUL ) {
	    NoiseOff(voice);
	    if( IsOpl3 ) {
		LeftPort(WAVE_SELECT+3+voice,0x06);	/* sqr */
	    } else {
		LeftPort(WAVE_SELECT+3+voice,0x01);	/* half/sin */
	    }
	} else if( value&SAW ) {
	    NoiseOff(voice);
	    if( IsOpl3 ) {
		LeftPort(WAVE_SELECT+3+voice,0x07);	/* derived square */
	    } else {
		LeftPort(WAVE_SELECT+3+voice,0x03);	/* sin */
	    }
	} else {					/* TRI */
	    NoiseOff(voice);
	    LeftPort(WAVE_SELECT+3+voice,0x03);		/* pulse sin */
	}
    }
    FrqKey(voice);
}

void SidAtkDcy(int voice,int value)
{
    static unsigned char attack[16]= {
	0xF0,0xF0,0xE0,0xD0, 0xC0,0xB0,0xA0,0x90,
	0x80,0x70,0x60,0x50, 0x40,0x30,0x20,0x10
    };
    static unsigned char decay[16]= {
	0xF,0xF,0xE,0xD, 0xC,0xB,0xA,0x9,
	0x8,0x7,0x6,0x5, 0x4,0x3,0x2,0x1
    };

    // printf("Attack/Decay %02X\n",value);
    LeftPort(ATTACK_DECAY+3+voice,attack[value>>4]|decay[value&0xF]);
}

void SidStnRls(int voice,int value)
{
    static unsigned char sustain[16]= {
#if 0
	0xF0,0xE0,0xD0,0xC0, 0xB0,0xA0,0x90,0x80,
	0x70,0x60,0x50,0x40, 0x30,0x20,0x10,0x00
#endif
	0xF0,0xE0,0x60,0x60, 0x50,0x50,0x40,0x40,
	0x30,0x30,0x20,0x20, 0x10,0x10,0x00,0x00
    };
    static unsigned char release[16]= {
	0xF,0xF,0xE,0xD, 0xC,0xB,0xA,0x9,
	0x8,0x7,0x6,0x5, 0x4,0x3,0x2,0x1
    };

    // printf("Sustain/Release %02X\n",value);
    LeftPort(SUSTAIN_RELEASE+3+voice,sustain[value>>4]|release[value&0xF]);
}

void SidVolume(int value)
{
    int i;

    i=Opl3Volume[value&0x0F];
    LeftPort(KSL_LEVEL+C1OP2,i);
    LeftPort(KSL_LEVEL+C2OP2,i);
    if( value&0x80 ) 
	LeftPort(KSL_LEVEL+C3OP2,63);
    else
	LeftPort(KSL_LEVEL+C3OP2,i);

    /*
    **	This is the sample emulator :)
    */
    RightPort(KSL_LEVEL+C1OP1,i);
    RightPort(KSL_LEVEL+C1OP2,i);
}

#ifdef __linux__

#ifndef AudioDeviceName
#define AudioDeviceName	"/dev/sequencer"	/* opl3 device */
#endif

static int AudioFildes;			/* audio file descriptor */

#endif

/*
**	Initialition for sound.
*/
void EnterSound(void)
{
#ifdef __linux__    /* { */
    int dummy;
    struct synth_info info;
    int dev;
    int i;
#endif	/* } __linux__ */

    if( SidSoundOff ) {			/* sound turned already off */
	return;
    }
    SidSoundOff=3;			/* Turn sound always off */

#ifdef __linux__    /* { */
    /*
    **	Open sequencer device. 
    */
    AudioFildes=open(AudioDeviceName,O_WRONLY);
    if( AudioFildes==-1 ) {
	printf("Can't open audio device `%s'\n",AudioDeviceName);
	return;
    }

    /*
    **	Find correct device.
    */
    if( ioctl(AudioFildes,SNDCTL_SEQ_NRSYNTHS,&dummy)==-1 ) {
	perror("Can't get number of synths");
	return;
    }
    dev=-1;
    for( i=0; i<dummy /*&& dev==-1*/; i++ ) {
	info.device=i;
	if( ioctl(AudioFildes,SNDCTL_SYNTH_INFO,&info)==-1 ) {
	    perror("Can't get info of synth");
	    return;
	}
	if( info.synth_type==SYNTH_TYPE_FM ) {
	    dev=i;
#if defined(X11)
	    printf("Name %s ",info.name);
	    switch( info.synth_subtype ) {
		case FM_TYPE_ADLIB:
		    printf("synth adlib");
		    break;
		case FM_TYPE_OPL3:
		    printf("synth opl3");
		    break;
		default:
		    printf("synth %d",info.synth_subtype);
		    break;
	    }
	    printf("\n");
#endif
	    break;
	}
    }
    if( dev==-1 ) {
	fprintf(stderr,"FM synthesizer not detected\n");
	return;
    }
#endif	/* } __linux__ */

    SidSoundOff=0;			/* Turn sound on for testing */
    if( !Opl3Detect() ) {
	printf("No opl2 or opl3 present!\n");
	SidSoundOff=3;
	return;
    }

    Opl3Reset();

    /*
    **	Set my emulation defaults:
    */
    LeftPort(FEEDBACK_CONNECTION+C1OP1,STEREO_BITS|CONNECTION_BIT);
    LeftPort(FEEDBACK_CONNECTION+C2OP1,STEREO_BITS|CONNECTION_BIT);
    LeftPort(FEEDBACK_CONNECTION+C3OP1,STEREO_BITS|CONNECTION_BIT);

    LeftPort(AM_VIB+C1OP2,SUSTAIN_ON|1);
    LeftPort(AM_VIB+C2OP2,SUSTAIN_ON|1);
    LeftPort(AM_VIB+C3OP2,SUSTAIN_ON|1);

    LeftPort(KSL_LEVEL+C1OP2,0x00);	/* loudt */
    LeftPort(KSL_LEVEL+C2OP2,0x00);	/* loudt */
    LeftPort(KSL_LEVEL+C3OP2,0x00);	/* loudt */

    LeftPort(WAVE_SELECT+C1OP2,0x00);	/* sin */
    LeftPort(WAVE_SELECT+C2OP2,0x00);	/* sin */
    LeftPort(WAVE_SELECT+C3OP2,0x00);	/* sin */

    LeftPort(ATTACK_DECAY+C1OP2,0xFF);
    LeftPort(ATTACK_DECAY+C2OP2,0xFF);
    LeftPort(ATTACK_DECAY+C3OP2,0xFF);

    LeftPort(SUSTAIN_RELEASE+C1OP2,0x0F);
    LeftPort(SUSTAIN_RELEASE+C2OP2,0x0F);
    LeftPort(SUSTAIN_RELEASE+C3OP2,0x0F);
    
    RightPort(FEEDBACK_CONNECTION+C1OP1,STEREO_BITS|CONNECTION_BIT);

    RightPort(AM_VIB+C1OP1,SUSTAIN_ON|1);
    RightPort(AM_VIB+C1OP2,SUSTAIN_ON|1);
    RightPort(KSL_LEVEL+C1OP1,63);		/* off */
    RightPort(KSL_LEVEL+C1OP2,63);		/* off */
    RightPort(WAVE_SELECT+C1OP1,0x00);		/* sin */
    RightPort(WAVE_SELECT+C1OP2,0x00);		/* sin */
    RightPort(ATTACK_DECAY+C1OP1,0xFF);		/* fastest */
    RightPort(ATTACK_DECAY+C1OP2,0xFF);		/* fastest */
    RightPort(SUSTAIN_RELEASE+C1OP1,0x0F);	/* fastest/loudt */
    RightPort(SUSTAIN_RELEASE+C1OP2,0x0F);	/* fastest/loudt */
    RightPort(FNUM_LOW+C1OP1,0xFF);
    RightPort(FNUM_LOW+C1OP2,0xFF);
    RightPort(KEYON_BLOCK+C1OP1,KEYON_BIT);

    SidSoundOff=0;			/* turn sound emulation on */
}

/*
**	Cleanup for sound.
*/
void LeaveSound(void)
{
    if( !(SidSoundOff&2) ) {		/* initialise ok */
	Opl3Silent();
#ifdef __linux__    /* { */
	close(AudioFildes);
#endif	/* } __linux__ */
    }
}

/*
**	Suspend sound.
*/
void SuspendSound(void)
{
    if( !(SidSoundOff&2) ) {		/* initialise ok */
	Opl3Silent();
    }
}

/*
**	Resume sound.
*/
void ResumeSound(void)
{
    if( !(SidSoundOff&2) ) {		/* initialise ok */
	SidStnRls(0,SID[0x06]);
	FrqKey(0);
	SidStnRls(1,SID[0x0D]);
	FrqKey(1);
	SidStnRls(1,SID[0x14]);
	FrqKey(2);			/* restore values of silent */
	SidVolume(SID[0x18]);
    }
}

/*
**	Turn sound on/off
*/
void ToggleSound(void)
{
    if( SidSoundOff&2 ) {		/* initialise bad */
	return;
    }
    if( !SidSoundOff ) {		/* initialise ok and off */
	Opl3Silent();
	SidSoundOff^=1;
    } else {
	SidSoundOff^=1;
	SidStnRls(0,SID[0x06]);
	FrqKey(0);
	SidStnRls(1,SID[0x0D]);
	FrqKey(1);
	SidStnRls(1,SID[0x14]);
	FrqKey(2);			/* restore values of silent */
	SidVolume(SID[0x18]);
    }
}

/*
**	Emulate sid chip
*/
void EmulSid(void)
{
    return;
}

#endif	/* } SID_OPL3 */
