/*
**	The C64 emulator
**
**	Copyright 1996 by ALE.
**	written by Lutz Sammer.
**
**	A real 1541 hardware emulation.
**-----------------------------------------------------------------------------
** $Id: real1541.c,v 1.4 1996/07/11 20:26:57 johns Exp root $
** $Log: real1541.c,v $
** Revision 1.4  1996/07/11 20:26:57  johns
** Fixed bug in job execute.
**
** Revision 1.3  1996/07/08 03:34:03  johns
** Some fastloader are working, must get cycle syncron!!!
**
** Revision 1.1  1996/07/06 21:37:40  johns
** Initial revision
**-----------------------------------------------------------------------------
*/

/* QUICK HACK: A REAL CHAOS !!!! */

#include "c64.h"

#ifdef REAL1541	/* { */

#include "vic.h"
#include "6502.h"

#include <stdio.h>
#include <memory.h>
#include <fcntl.h>
#include <unistd.h>

/*---------------------------------------------------------------------------*/

#undef RBDP
#undef WBDP
#undef RBMem
#undef WBMem
#undef RWVec
#undef RLMem

/*
**	Macro to define memory read functions.
*/
#define R_FUN(n) \
	static int REGPARM1 n (int x) \

/*
**	Macro to define memory write functions.
*/
#define W_FUN(n) \
	static void REGPARM2 n (int x,int v) \

#define VAL     v
#define ADR     x

#define MEM_BANK_SIZE	1024		/* size of memory segments */
#define MEM_BANKS	65		/* number of segments */
#define MEM_SHIFT	10		/* shift for segments */

/*
**	Read a byte from memory with current configuration.
*/
#if 0
#define RB_MEM(adr) \
    (R1541RdIt[(adr)>>MEM_SHIFT]((adr)))
#else
    /* A good c-compiler should produce this */
#define RB_MEM(adr) \
    ( (*(regparm1**) ((unsigned)R1541RdIt \
	    + (((unsigned)(adr)>>(MEM_SHIFT-2))&~3)) ) ((adr)))
#endif

/*
**	Write a byte to memory with current configuration.
*/
#if 0
#define WB_MEM(adr,val) \
    (R1541WrIt[(adr)>>MEM_SHIFT]((adr),(val)))
#else
    /* A good c-compiler should produce this */
#define WB_MEM(adr,val) \
    ( (*(regparm2**) ((unsigned)R1541WrIt	\
	    + (((unsigned)(adr)>>(MEM_SHIFT-2))&~3)) ) ((adr),(val)))
#endif

#ifdef __GNUC__
#define RBMem(adr) \
    ({	int __rx=(adr); RB_MEM(__rx); })
#else
#define RBMem(adr) \
    ( __tmpvar__=(int)(adr), RB_MEM(__tmpvar__) )
#endif

#define WBMem(adr,val) \
    do{ int __wx=(adr); WB_MEM(__wx,(val)); }while(0)

/*-----------------------------
**	Zero/Direct page access
*/

/*
**	Read byte from zero page.
*/
#define RBDP(adr) \
    R1541Ram[adr]

/*
**	Write byte to zero page. Special Processor port data+direction
*/
#define WBDP(adr,val) \
    { R1541Ram[adr]=val; if( adr<5 ) CheckWorkQueue(); }

/*
**	Read word from zero page for indirect-x and indirect-y.
*/
#ifdef __GNUC__
#define RWDP(adr) \
    ({ int __x=(adr); Ram[__x]|(Ram[(__x+1)&0xFF]<<8); })
#else
static int __tmpvar__;
#define RWDP(adr) \
    (__tmpvar__=(int)(adr), Ram[__tmpvar__]|(Ram[(__tmpvar__+1)&0xFF]<<8))
#endif

/*
**	Processor vector fetch.
*/
#ifdef BIG_ENDIAN	/* { */

#define RWVec(adr) \
    (R1541Rom[adr-0xC000]|(R1541Rom[adr-0xC000+1]<<8))

#endif	/* } BIG_ENDIAN */

#ifdef LITTLE_ENDIAN	/* { */

#define RWVec(adr) \
    (*(unsigned short*)(R1541Rom+adr-0xC000))

#endif	/* } LITTLE_ENDIAN */

#ifdef LITTLE_ENDIAN	/* { */

#define RLMem(adr,val) \
    val=RBMem(adr+0)|(RBMem(adr+1)<<8)|(RBMem(adr+2)<<16);

#endif	/* } LITTLE_ENDIAN */

/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/

int FastLoaderOff=1;			/* Fastloader supported turned off */
int SerialTimeout;			/* Serial restart timeout */

char* R1541ImageName;			/* 1541 ROM Image file name */

static unsigned char R1541Rom[16438];	/* 1541 rom at $C000-$FFFF */
static unsigned char R1541Ram[2048];	/* 1541 ram at $0000-$07FF */

static unsigned char VIA1[16];		/* Via1 chip register */
static int Via1TimerA=1;		/* Loaded count */
static unsigned Via1TimerA_Action;	/* Next action */
// static int Via1TimerB;			/* Loaded count */
// static unsigned Via1TimerB_Action;	/* Next action */

static unsigned char VIA2[16];		/* Via2 chip register */
static int Via2TimerA=1;		/* Loaded count */
static unsigned Via2TimerA_Action;	/* Next action */
// static int Via2TimerB;			/* Loaded count */
// static unsigned Via2TimerB_Action;	/* Next action */

static int	R1541HeadTrack;		/* current head position */
static int	R1541HeadSector;	/* current head position */
static int	R1541HeadCount;		/* count until sync */
static int	R1541HeadIndex;		/* index into gcr buffer */
static int	R1541HeadFlag;		/* flag gcr buffer */
// static int	R1541ID0;		/* ID 0 */
// static int	R1541ID1;		/* ID 1 */

unsigned CheckAction;		/* counter for irq jobs */
static unsigned R1541SerialLine;	/* 1541 serial line */
static unsigned C64SerialLine;		/* C64 serial line */
static int C64OldOut;			/* old c64 output */

extern unsigned R1541Cycle;		/* 6502 Processor cycles */
extern union reg6502 R1541GlobalRegA;	/* 6502 Processor accu */
extern unsigned	R1541GlobalRegPC;	/* 6502 Processor counter */

extern void R1541Reg(void);		/* 6502 register */

static void CheckWorkQueue(void);

extern int ImageRO;
extern int AbsoluteSector(unsigned int,unsigned int);
extern unsigned char* SectorPointer(unsigned int,unsigned int);
extern int SectorError(unsigned int,unsigned int);
extern void SectorChanged(unsigned int,unsigned int);

/*---------------------------------------------------------------------------*/

/*
**	Sector header:
**		SYNC 08 PARITY SECTOR TRACK ID1 ID0 FIL1 FIL0
**
**	Data header:
**		SYNC 07 256*BYTES PARITY FIL1 FIL0
**
*/

static unsigned char HeadGCR[10];	/* buffer for header generated gcr */
static unsigned char DataGCR[325];	/* buffer for block generated gcr */

/*
**	Bytes to GCR code table.
*/
static CONST unsigned char Byte2GCR[16] = {
    0x0A,0x0B,0x12,0x13, 0x0E,0x0F,0x16,0x17,
    0x09,0x19,0x1A,0x1B, 0x0D,0x1D,0x1E,0x15
};

/*
**	Block of 4 bytes to 5 GCR bytes.
*/
static void Block2GCR(CONST unsigned char* src,unsigned char* dst)
{
    int i;

    memset(dst,0,5);

    i=(Byte2GCR[(*src>>4)]<<5)|Byte2GCR[(*src&0xF)];
    ++src;
    *dst++|=(i>>2)&0xFF;		/* 8 */
    *dst|=(i<<6)&0xC0;			/* 2 */

    i=(Byte2GCR[(*src>>4)]<<5)|Byte2GCR[(*src&0xF)];
    ++src;
    *dst++|=(i>>4)&0x3F;		/* 6 */
    *dst|=(i<<4)&0xF0;			/* 4 */

    i=(Byte2GCR[(*src>>4)]<<5)|Byte2GCR[(*src&0xF)];
    ++src;
    *dst++|=(i>>6)&0x0F;		/* 4 */
    *dst|=(i<<2)&0xFC;			/* 6 */

    i=(Byte2GCR[(*src>>4)]<<5)|Byte2GCR[(*src&0xF)];
    ++src;
    *dst++|=(i>>8)&0x03;		/* 2 */
    *dst|=(i<<0)&0xFF;			/* 8 */
}

/*
**	Build head gcr-code.
*/
static void Head2GCR(int track,int sector)
{
    unsigned char buf[4];

    /*	08 PARITY SECTOR TRACK ID1 ID0 0 0 */
#ifdef JOHNS
//     printf("TR: %02x SE: %02x - %02x %02x \n",
//	R1541Ram[0x18],R1541Ram[0x19],track,sector);
#endif

    buf[0]=0x08;
    buf[1]=track^sector^R1541Ram[0x16]^R1541Ram[0x17];
    buf[2]=sector;
    buf[3]=track;
    Block2GCR(buf,HeadGCR);

    buf[0]=R1541Ram[0x17];
    buf[1]=R1541Ram[0x16];
    buf[2]=0;
    buf[3]=0;
    Block2GCR(buf,HeadGCR+5);
}

/*
**	Build data gcr-code.
*/
static void Data2GCR(int track,int sector)
{
    unsigned char buf[4];
    unsigned char* cp;
    unsigned char* dp;
    int i;
    int j;

    cp=SectorPointer(track,sector);
    if( cp ) {
	buf[0]=0x07;
	buf[1]=cp[0];
	buf[2]=cp[1];
	buf[3]=cp[2];
	Block2GCR(buf,DataGCR);
	dp=DataGCR+5;

	for( i=3; i<255; i+=4 ) {
	    buf[0]=cp[i+0];
	    buf[1]=cp[i+1];
	    buf[2]=cp[i+2];
	    buf[3]=cp[i+3];
	    Block2GCR(buf,dp);
	    dp+=5;
	}
	buf[0]=cp[255];
	j=0;
	for( i=0; i<256; ++i ) {
	    j^=cp[i];
	}
	buf[1]=j;
	buf[2]=0;
	buf[3]=0;
	Block2GCR(buf,dp);
    }
}

/*---------------------------------------------------------------------------*/

/*
**       Emulator special instructions.
*/
static void R1541EmulatorTrap(int cmd)
{
    switch( cmd ) {

	case 1:				/* move head one track in or out */

#ifdef JOHNS
	    printf("Track move $%02X\n",Integer(R1541GlobalRegA));
	    if( Integer(R1541GlobalRegA)<0 ) {
	    } else if( Integer(R1541GlobalRegA)>0 ) {
	    }
#endif
	    R1541GlobalRegPC=0xD6A5;	/* RTS */
	    break;

	default:
	    printf("1541: Illegal emulator trap %d at $%04X\n",
		cmd,R1541GlobalRegPC++);
	    break;
    }
}

/*=============================================================================
 *	2K 1541 RAM
 *===========================================================================*/

R_FUN( R_Mirror )			/* Read RAM mirror */
{
    return R1541Ram[ADR&0x07FF];
}

W_FUN( W_Mirror )			/* Write RAM mirror */
{
    R1541Ram[ADR&0x07FF]=VAL;
}

R_FUN( R_Ram )				/* Read RAM */
{
    return R1541Ram[ADR];
}

W_FUN( W_Ram )				/* Write RAM */
{
    R1541Ram[ADR]=VAL;
}

/*=============================================================================
 *	VIA1
 *===========================================================================*/

/*
**	PortB:
**		PB7	ATN-IN
**		PB6	DEV-NR-IN
**		PB5	DEV-NR-IN
**		PB4	ATN-ACK-OUT
**		PB3	CLOCK OUT
**		PB2	CLOCK IN
**		PB1	DATA OUT
**		PB0	DATA IN
*/

R_FUN( R_VIA1 )				/* Read VIA1 */
{
    int i;

    // printf("VIA1: read $%04X\n",ADR);
    switch( ADR&0xF ) {
	case 0:				/* PB */
	    i=VIA1[0]|(VIA1[2]^0xFF);
	    i&=(8<<5)|0x9F;		/* id 8,9,10,11 */
	    i&=((~R1541SerialLine	/* ATN-IN,CLK-IN,DAT-IN from C64 */
		|((C64SerialLine>>7)
		|(C64SerialLine>>4)))|~0x85);
	    
	    return i;

	case 4:				/* timer A count low */
	    /*
	    printf("Timer LOW %d %d\n",Via1TimerA,
		(Via1TimerA-((R1541Cycle-Via1TimerA_Action)%Via1TimerA))&0xFF
	    );
	    */
	    VIA1[0xD]&=~0x40;		/* read clears irq */
	    return
		(Via1TimerA-((R1541Cycle-Via1TimerA_Action)%Via1TimerA))&0xFF;

	case 5:				/* timer A count high */
	    /*
	    printf("Timer HIGH %d %d %d %d\n",
		Via1TimerA_Action,R1541Cycle,Via1TimerA,
		(Via1TimerA-((R1541Cycle-Via1TimerA_Action)%Via1TimerA))>>8
	    );
	    */
	    return
		(Via1TimerA-((R1541Cycle-Via1TimerA_Action)%Via1TimerA))>>8;

	case 0xD:			/* interrupt flag register */
	    // printf("Interrupt flags\n");
	    if( R1541Cycle>Via1TimerA_Action+Via1TimerA ) {
		// printf("Underflow TimerA\n");
		Via1TimerA_Action=R1541Cycle+
		    (Via1TimerA-((R1541Cycle-Via1TimerA_Action)%Via1TimerA));
		VIA1[0xD]|=0x40;
	    }
	    return VIA1[0xD];

	default:
	    return VIA1[ADR&0xF];
    }
}

//static int flag;

W_FUN( W_VIA1 )				/* Write VIA1 */
{
    int o;

    // printf("VIA1: write $%04X,%02X\n",ADR,VAL);

    switch( ADR&0xF ) {
	case 0:				/* Port B */
	case 2:				/* Data direction port B */
	    VIA1[ADR&0xF]=VAL;
	    o=~VIA1[0]&VIA1[2];
#if 0
	    flag=1;
	    printf(">: %02X %d\n",o,R1541Cycle);
	    printf("1541: %08d: %s%s%s $%04X\n",R1541Cycle,
		    !(o&0x10) ? "ACK " : "___ ",
		    !(o&0x02) ? "DAT " : "___ ",
		    !(o&0x08) ? "CLK " : "___ ",
		    R1541GlobalRegPC
		);
#endif
	    C64SerialLine=(~o<<3)&0x40;	/* connect 1541 -> C64 */
	    C64SerialLine|=
		(((o<<6)^0x80)		/* 1541 DAT 0x02 -> 0x80 */
		| (((R1541SerialLine)^(~o<<3))^0x80))&0x80;
		//| (((R1541SerialLine)|(~o<<3))^0x80))&0x80;
	    // printf("ATN: %02X %02X %02X\n",C64SerialLine,R1541SerialLine,o);

	    break;

	case 1:				/* Port A */
	case 3:				/* Data direction port A */
	    VIA1[ADR&0xF]=VAL;
	    break;

	case 4:				/* timer A count low */
	    VIA1[4]=VAL;
	    break;

	case 5:				/* timer A count high */
	    VIA1[5]=VAL;
	    Via1TimerA=(VIA1[4]|(VIA1[5]<<8))+1;
	    Via1TimerA_Action=R1541Cycle;
	    VIA1[0xD]&=0xBF;		/* clear interrupt flag */
	    break;

	case 6:				/* timer A latch low */
	    VIA1[6]=VAL;
	    break;

	case 7:				/* timer A latch high */
	    VIA1[7]=VAL;
	    break;

	case 8:				/* timer B latch/count low */
	    VIA1[8]=VAL;
	    break;

	case 9:				/* timer B latch/count high */
	    VIA1[9]=VAL;
	    // ?? VIA1[0xD]&=0xBF;		/* clear interrupt flag */
	    break;

	case 0xA:
	case 0xB:
	case 0xC:
	    VIA1[ADR&0xF]=VAL;
	    break;

	case 0xD:			/* interrupt flag register */
	    VIA1[0xD]&=~VAL;
	    break;

	case 0xE:
	    VIA1[0xE]=VAL|0x80;
	    break;

	case 0xF:
	    VIA1[0xF]=VAL;
	    break;
    }
}

/*=============================================================================
 *	VIA2
 *===========================================================================*/

/*
**	PortB:
**		7:	SYNC IN
**		6,5:	Bit rate D1 and DO
**		4:	WPS - Write Protect Sense (1 = On)
**		3:	ACT - LED on drive
**		2:	MTR - drive motor
**		0,1:	step motor for head movement
**	PortA:
**		read/write gcr data
**	CA1:	Byte ready
**	CA2:	SOE - Set Overflow Enable for 6502.
**	CA3:	read/write
*/

R_FUN( R_VIA2 )				/* Read VIA2 */
{
    static CONST int ZBR_ns[43] = {
	0,
	21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21,
	19, 19, 19, 19, 19, 19, 19,
	18, 18, 18, 18, 18, 18,
	17, 17, 17, 17, 17,
	17, 17, 17, 17, 17, 17, 17,
    };

    // printf("VIA2: read $%04X\n",ADR);
    switch( ADR&0xF ) {
	case 0:				/* Port B */
	    if( --R1541HeadCount<0 ) {
		R1541HeadCount=4;
#ifdef JOHNS
		// printf("SYNC: %d %d\n",R1541HeadTrack,R1541HeadSector);
		// printf("SYNC:");
#endif
		R1541HeadIndex=0;
		R1541HeadFlag^=1;
		if( R1541HeadFlag ) {
		    ++R1541HeadSector;
		    if( R1541HeadSector>=ZBR_ns[R1541HeadTrack/2] )
			R1541HeadSector=0;
		    // printf("HEADER %d",R1541HeadSector);
		} else {
		    // printf("DATA %d",R1541HeadSector);
		}
		// printf("\n");
		return VIA2[ADR&0xF]&0x7F;
	    }
	    return VIA2[ADR&0xF]|0x80;

	case 1:				/* Port A */
	    if( R1541HeadFlag ) {	/* header */
		if( !R1541HeadIndex++ ) {
		    Head2GCR(R1541HeadTrack/2,R1541HeadSector);
		    return 0xFF;
		}
		// printf(" %02x\n",R1541HeadIndex<12 ? HeadGCR[R1541HeadIndex-2] : 0);
		if( R1541HeadIndex<12 ) {
		    return HeadGCR[R1541HeadIndex-2];
		}
		R1541HeadCount=4;
		return 0;
	    } else {			/* data */
		// printf("gcr read:");
		if( !R1541HeadIndex++ ) {
		    Data2GCR(R1541HeadTrack/2,R1541HeadSector);
		    return 0xFF;
		}
		// printf(" %02x\n",R1541HeadIndex<325+2 ? DataGCR[R1541HeadIndex-2] : 0);
		if( R1541HeadIndex<325+2 ) {
		    return DataGCR[R1541HeadIndex-2];
		}
		R1541HeadCount=4;
		return 0;
	    }
	    break;

	case 3:				/* Data direction port A */
	    break;

	case 4:				/* timer A count low */
	    /*
	    printf("Timer LOW %d %d\n",Via2TimerA,
		(Via2TimerA-((R1541Cycle-Via2TimerA_Action)%Via2TimerA))&0xFF
	    );
	    */
	    VIA2[0xD]&=~0x40;		/* read clears irq */
	    return
		(Via2TimerA-((R1541Cycle-Via2TimerA_Action)%Via2TimerA))&0xFF;

	case 5:				/* timer A count high */
	    /*
	    printf("Timer HIGH %d %d %d %d\n",
		Via2TimerA_Action,R1541Cycle,Via2TimerA,
		(Via2TimerA-((R1541Cycle-Via2TimerA_Action)%Via2TimerA))>>8
	    );
	    */
	    return
		(Via2TimerA-((R1541Cycle-Via2TimerA_Action)%Via2TimerA))>>8;

	case 0xD:			/* interrupt flag register */
	    // printf("Interrupt flags\n");
	    if( R1541Cycle>Via2TimerA_Action+Via2TimerA ) {
		// printf("Underflow TimerA\n");
		Via2TimerA_Action=R1541Cycle+
		    (Via2TimerA-((R1541Cycle-Via2TimerA_Action)%Via2TimerA));
		VIA2[0xD]|=0x40;
	    }
	    return VIA2[0xD];

    }
    return VIA2[ADR&0xF];
}

W_FUN( W_VIA2 )				/* Write VIA2 */
{
    int o;
    static int ob;

    // printf("VIA2: write $%04X,%02X\n",ADR,VAL);
    switch( ADR&0xF ) {

	case 0:				/* Port B */
	case 2:				/* Data direction port B */
	    VIA2[ADR&0xF]=VAL;
	    o=~VIA2[0]&VIA2[2];

#if 1
/* not nice */
	    if( o&0x08 ) {		/* LED State */
		LedOff();
	    } else {
		LedOn();
	    }
#endif

	    /*
	    **	Emulate head movement!
	    */
	    if( (o^ob)&3 ) {
		if( ((o+1)&3)==(ob&3) ) {
		    if( R1541HeadTrack<88 )
			++R1541HeadTrack;
		    // printf("OUT> %d\n",R1541HeadTrack/2);
		} else {
		    if( R1541HeadTrack>2 )
			--R1541HeadTrack;
		    // printf("IN> %d\n",R1541HeadTrack/2);
		}
	    }

#if 0
	    if( o!=ob ) {
		printf("Bit: %d %s %s Head: %d\n",
		    (o&0x60)>>5,
		    (o&0x08) ? "ACT" : "___",
		    (o&0x04) ? "MTR" : "___",
		    (o&0x03)
		);
	    }
#endif
	    ob=o;
	    break;

	case 1:				/* Port A */
	case 3:				/* Data direction port A */
	    VIA2[ADR&0xF]=VAL;
	    o=~VIA2[1]&VIA2[3];
	    break;

	case 4:				/* timer A count low */
	    VIA2[4]=VAL;
	    break;

	case 5:				/* timer A count high */
	    VIA2[5]=VAL;
	    Via2TimerA=(VIA2[4]|(VIA2[5]<<8))+1;
	    Via2TimerA_Action=R1541Cycle;
	    VIA2[0xD]&=0xBF;		/* clear interrupt flag */
	    break;

	case 6:				/* timer A latch low */
	    VIA2[6]=VAL;
	    break;

	case 7:				/* timer A latch high */
	    VIA2[7]=VAL;
	    break;

	case 8:				/* timer B latch/count low */
	    VIA2[8]=VAL;
	    break;

	case 9:				/* timer B latch/count high */
	    VIA2[9]=VAL;
	    // ?? VIA2[0xD]&=0xBF;		/* clear interrupt flag */
	    break;

	case 0xA:
	case 0xB:
	case 0xC:
	    VIA2[ADR&0xF]=VAL;
	    break;

	case 0xD:			/* interrupt flag register */
	    VIA2[0xD]&=~VAL;
	    break;

	case 0xE:
	    VIA2[0xE]=VAL|0x80;
	    break;

	case 0xF:
	    VIA2[0xF]=VAL;
	    break;
    }
}

/*=============================================================================
 *	16K 1541 ROM
 *===========================================================================*/

R_FUN( R_Rom )				/* Read ROM at C000-FFFF */
{
    return R1541Rom[ADR-0xC000];
}

/*-----------------------------------------------------------------------------
 *	Unconnected adress space
 *---------------------------------------------------------------------------*/

R_FUN( R_Free )
{
    return 0xFF;
}

W_FUN( W_Free )
{
}

/*---------------------------------------------------------------------------*/

/*
**	Read memory 1541 table.
*/
static regparm1* CONST R1541RdIt[MEM_BANKS] = {
    R_Ram,				/* 0000-03FF */
    R_Ram,				/* 0400-07FF */
    R_Free,				/* 0800-0BFF */
    R_Free,				/* 0C00-0FFF */
    R_Free,				/* 1000-13FF */
    R_Free,				/* 1400-17FF */
    R_VIA1,				/* 1800-1BFF */
    R_VIA2,				/* 1C00-1FFF */
    R_Free,				/* 2000-23FF */
    R_Free,				/* 2400-27FF */
    R_Free,				/* 2800-2BFF */
    R_Free,				/* 2C00-2FFF */
    R_Free,				/* 3000-33FF */
    R_Free,				/* 3400-37FF */
    R_Free,				/* 3800-3BFF */
    R_Free,				/* 3C00-3FFF */
    R_Free,				/* 4000-43FF */
    R_Free,				/* 4400-47FF */
    R_Free,				/* 4800-4BFF */
    R_Free,				/* 4C00-4FFF */
    R_Free,				/* 5000-53FF */
    R_Free,				/* 5400-57FF */
    R_Free,				/* 5800-5BFF */
    R_Free,				/* 5C00-5FFF */
    R_Free,				/* 6000-63FF */
    R_Free,				/* 6400-67FF */
    R_Free,				/* 6800-6BFF */
    R_Free,				/* 6C00-6FFF */
    R_Free,				/* 7000-73FF */
    R_Free,				/* 7400-77FF */
    R_Free,				/* 7800-7BFF */
    R_Free,				/* 7C00-7FFF */
    R_Free,				/* 8000-83FF */
    R_Free,				/* 8400-87FF */
    R_Free,				/* 8800-8BFF */
    R_Free,				/* 8C00-8FFF */
    R_Free,				/* 9000-93FF */
    R_Free,				/* 9400-97FF */
    R_Free,				/* 9800-9BFF */
    R_Free,				/* 9C00-9FFF */
    R_Free,				/* A000-03FF */
    R_Free,				/* A400-A7FF */
    R_Free,				/* A800-ABFF */
    R_Free,				/* AC00-AFFF */
    R_Free,				/* B000-B3FF */
    R_Free,				/* B400-B7FF */
    R_Free,				/* B800-BBFF */
    R_Free,				/* BC00-BFFF */
    R_Rom,				/* C000-C3FF */
    R_Rom,				/* C400-C7FF */
    R_Rom,				/* C800-CBFF */
    R_Rom,				/* CC00-CFFF */
    R_Rom,				/* D000-D3FF */
    R_Rom,				/* D400-D7FF */
    R_Rom,				/* D800-DBFF */
    R_Rom,				/* DC00-DFFF */
    R_Rom,				/* E000-E3FF */
    R_Rom,				/* E400-E7FF */
    R_Rom,				/* E800-EBFF */
    R_Rom,				/* EC00-EFFF */
    R_Rom,				/* F000-F3FF */
    R_Rom,				/* F400-F7FF */
    R_Rom,				/* F800-FBFF */
    R_Rom,				/* FC00-FFFF */
    R_Mirror				/* 0000-03FF */
};

/*
**	Write memory 1541 table.
*/
static regparm2* CONST R1541WrIt[MEM_BANKS] = {
    W_Ram,				/* 0000-03FF */
    W_Ram,				/* 0400-07FF */
    W_Free,				/* 0800-0BFF */
    W_Free,				/* 0C00-0FFF */
    W_Free,				/* 1000-13FF */
    W_Free,				/* 1400-17FF */
    W_VIA1,				/* 1800-1BFF */
    W_VIA2,				/* 1C00-1FFF */
    W_Free,				/* 2000-23FF */
    W_Free,				/* 2400-27FF */
    W_Free,				/* 2800-2BFF */
    W_Free,				/* 2C00-2FFF */
    W_Free,				/* 3000-33FF */
    W_Free,				/* 3400-37FF */
    W_Free,				/* 3800-3BFF */
    W_Free,				/* 3C00-3FFF */
    W_Free,				/* 4000-43FF */
    W_Free,				/* 4400-47FF */
    W_Free,				/* 4800-4BFF */
    W_Free,				/* 4C00-4FFF */
    W_Free,				/* 5000-53FF */
    W_Free,				/* 5400-57FF */
    W_Free,				/* 5800-5BFF */
    W_Free,				/* 5C00-5FFF */
    W_Free,				/* 6000-63FF */
    W_Free,				/* 6400-67FF */
    W_Free,				/* 6800-6BFF */
    W_Free,				/* 6C00-6FFF */
    W_Free,				/* 7000-73FF */
    W_Free,				/* 7400-77FF */
    W_Free,				/* 7800-7BFF */
    W_Free,				/* 7C00-7FFF */
    W_Free,				/* 8000-83FF */
    W_Free,				/* 8400-87FF */
    W_Free,				/* 8800-8BFF */
    W_Free,				/* 8C00-8FFF */
    W_Free,				/* 9000-93FF */
    W_Free,				/* 9400-97FF */
    W_Free,				/* 9800-9BFF */
    W_Free,				/* 9C00-9FFF */
    W_Free,				/* A000-03FF */
    W_Free,				/* A400-A7FF */
    W_Free,				/* A800-ABFF */
    W_Free,				/* AC00-AFFF */
    W_Free,				/* B000-B3FF */
    W_Free,				/* B400-B7FF */
    W_Free,				/* B800-BBFF */
    W_Free,				/* BC00-BFFF */
    W_Free,				/* C000-C3FF */
    W_Free,				/* C400-C7FF */
    W_Free,				/* C800-CBFF */
    W_Free,				/* CC00-CFFF */
    W_Free,				/* D000-D3FF */
    W_Free,				/* D400-D7FF */
    W_Free,				/* D800-DBFF */
    W_Free,				/* DC00-DFFF */
    W_Free,				/* E000-E3FF */
    W_Free,				/* E400-E7FF */
    W_Free,				/* E800-EBFF */
    W_Free,				/* EC00-EFFF */
    W_Free,				/* F000-F3FF */
    W_Free,				/* F400-F7FF */
    W_Free,				/* F800-FBFF */
    W_Free,				/* FC00-FFFF */
    W_Mirror				/* 0000-03FF */
};

/*
**	Read byte from 1541 memory.
*/
int R1541RBMem(int adr)
{
    if( adr==0x1C01 ) {			/* NON DESTRUCTIVE !! */
	if( R1541HeadFlag ) {		/* header */
	    if( !R1541HeadIndex ) {
		Head2GCR(R1541HeadTrack/2,R1541HeadSector);
		return 0xFF;
	    }
	    if( R1541HeadIndex<12 ) {
		return HeadGCR[R1541HeadIndex-2];
	    }
	    return 0;
	} else {			/* data */
	    if( !R1541HeadIndex ) {
		Data2GCR(R1541HeadTrack/2,R1541HeadSector);
		return 0xFF;
	    }
	    if( R1541HeadIndex<325+2 ) {
		return DataGCR[R1541HeadIndex-2];
	    }
	    return 0;
	}
    }
    return RBMem(adr);
}

/*
**	Write byte to 1541 memory.
*/
void R1541WBMem(int adr,int val)
{
    WBMem(adr,val);
}


#define _1541_

/*
**	Map C64 6502 Emulation names to 1541 names.
*/
#define GlobalRegA		R1541GlobalRegA
#define GlobalRegX		R1541GlobalRegX
#define GlobalRegY		R1541GlobalRegY
#define GlobalRegP		R1541GlobalRegP
#define GlobalRegS		R1541GlobalRegS
#define GlobalRegPC		R1541GlobalRegPC
#define IrqLine			R1541IrqLine
#define Cycle			R1541Cycle
#define Action			R1541Action
#define ActionFunction		R1541ActionFunction

#define UpdateFlags		R1541UpdateFlags
#define Emul6502Loop		R1541Emul6502Loop
#define Reset			R1541Reset
#define Nmi			R1541Nmi
#define Irq			R1541Irq
#define EmulatorTrap		R1541EmulatorTrap

#define Ram			R1541Ram
#define Dis			R1541Dis

#define HardwareReset()

#include "6502.c"

/*---------------------------------------------------------------------------*/

/*
**	C64 disassembler
*/
int R1541Dis(int ip)
{
    return Disassemble(
	ip,
	Integer(R1541GlobalRegX),
	Integer(R1541GlobalRegY),
	Integer(R1541GlobalRegS),
	R1541RBMem);
}

/*---------------------------------------------------------------------------*/

/*
**	Print register
*/
void R1541Reg(void)
{
    UpdateFlags();			/* Recalculate flags */
    printf("15=$%04X ",RegPC);
    printf("A=$%02X ",Integer(RegA));
    printf("X=$%02X ",Integer(RegX));
    printf("Y=$%02X ",Integer(RegY));
    printf("S=$%02X ",Integer(RegS));
    printf("P=$%02X ",Integer(RegP));
    printf("%c",(Integer(RegP)&FLAG_NEGATIVE)  ? 'N' : '-');
    printf("%c",(Integer(RegP)&FLAG_OVERFLOW)  ? 'O' : '-');
    printf("%c",(Integer(RegP)&FLAG_UNDEFINED) ? '*' : '-');
    printf("%c",(Integer(RegP)&FLAG_BREAK)     ? 'B' : '-');
    printf("%c",(Integer(RegP)&FLAG_INTERRUPT) ? 'I' : '-');
    printf("%c",(Integer(RegP)&FLAG_DECIMAL)   ? 'D' : '-');
    printf("%c",(Integer(RegP)&FLAG_ZERO)      ? 'Z' : '-');
    printf("%c",(Integer(RegP)&FLAG_CARRY)     ? 'C' : '-');
    if( Integer(RegS)<0xFE ) {
	printf(" [$%04X",RWStack(1+Integer(RegS)));
	if( Integer(RegS)<0xFC )
	    printf(",$%04X",RWStack(3+Integer(RegS)));
	printf("] ");
    }
    printf(" Cycle(%d)",R1541Cycle);
    printf("\n");
    printf("$%04X: ",RegPC);
    R1541Dis(RegPC);
    printf("\n");
}

/*---------------------------------------------------------------------------*/

#define JOB_OK			0x01	/* Everything OK */
#define JOB_NO_HEADER		0x02	/* Header block not found */
#define JOB_NO_SYNC		0x03	/* SYNC not found */
#define JOB_NO_BLOCK		0x04	/* Data block not found */
#define JOB_BAD_CRC_BLOCK	0x05	/* Checksum error in data block */
#define JOB_VERIFY_ERROR	0x07	/* Verify error */
#define JOB_WRITE_PROTECT	0x08	/* Disk write protected */
#define JOB_BAD_CRC_DATA	0x09	/* Checksum error in header block */
#define JOB_ID_MISMATCH		0x0B	/* Id mismatch */
#define JOB_NOT_READY		0x0F	/* Disk not inserted */

#define JOB_READ	0x80		/* Read sector */
#define JOB_WRITE	0x90		/* Write sector */
#define JOB_VERIFY	0xA0		/* Verify sector */
#define JOB_SEEK	0xB0		/* Find sector */
#define JOB_BUMP	0xC0		/* Bump, Find track 1 */
#define JOB_JUMP	0xD0		/* Execute program in buffer */
#define JOB_EXECUTE	0xE0		/* Execute program */

static unsigned OldAction;
static void (*OldActionFunction)(void);

/*
**	Can't be done in the job.
*/
static void ExecuteAction(void)
{
    /* THIS IS ABSOLUTLEY NECASSARY !!!!! */
#if 0
    PushW(R1541GlobalRegPC);

    R1541GlobalRegPC=0xF36E;	/* execute program in buffer */
    Integer(R1541GlobalRegP)|=FLAG_INTERRUPT;
#endif
    VIA2[0xD]|=0x40;
    Irq();
    IrqLine=0;

    R1541Action=OldAction;
    R1541ActionFunction=OldActionFunction;
}

/*
**	Execute a floppy job.
**	(Done by the real 1541 in the IRQ handler).
*/
static int DoJob(int command,int track,int sector,unsigned char* buffer)
{
    int e;
    unsigned char* tmp;

    // R1541Cycle=R1541Action;		/* skip remaining cycles */

    if( !FloppyImage || (command&1) ) {
	return JOB_NOT_READY;
    }

    tmp=SectorPointer(18,0);
    R1541Ram[0x16]=tmp[162];		/* ID1 */
    R1541Ram[0x17]=tmp[163];		/* ID2 */
    R1541Ram[0x18]=track;
    R1541Ram[0x19]=sector;

    switch( command ) {
	case JOB_READ:
#ifdef JOHNS
	    printf("Read %02d,%02d ",track,sector);
#endif
	    R1541HeadTrack=track*2;
	    R1541Ram[0x22]=track;
	    tmp=SectorPointer(track,sector);
	    if( tmp ) {
		memcpy(buffer,tmp,256);
		e=SectorError(track,sector);
#ifdef JOHNS
		printf("%d\n",e);
#endif
		return 1;
	    }
#ifdef JOHNS
	    printf("NO-Block\n");
#endif
	    return JOB_NO_BLOCK;

	case JOB_WRITE:
#ifdef JOHNS
	    printf("Write %02d,%02d\n",track,sector);
#endif
	    R1541HeadTrack=track*2;
	    R1541Ram[0x22]=track;
	    if( ImageRO ) {
		return JOB_WRITE_PROTECT;
	    }
	    tmp=SectorPointer(track,sector);
	    if( tmp ) {
		memcpy(tmp,buffer,256);
		SectorChanged(track,sector);
		return JOB_OK;
	    }
	    return JOB_NO_BLOCK;

	case JOB_VERIFY:
#ifdef JOHNS
	    printf("Verify %02d,%02d\n",track,sector);
#endif
	    R1541HeadTrack=track*2;
	    R1541Ram[0x22]=track;
	    tmp=SectorPointer(track,sector);
	    if( tmp ) {
		return memcmp(tmp,buffer,256) ? JOB_VERIFY_ERROR : JOB_OK;
		// return SectorError(track,sector);
	    }
	    return JOB_NO_BLOCK;

	case JOB_SEEK:
#ifdef JOHNS
	    printf("Seek %02d,%02d\n",track,sector);
#endif
	    R1541HeadTrack=track*2;
	    R1541Ram[0x22]=track;
	    e=AbsoluteSector(track,sector);
	    if( e==-1 ) {
		return JOB_NO_BLOCK;
	    }
	    return JOB_OK;

	case JOB_BUMP:
#ifdef JOHNS
	    printf("Bump %02d,%02d\n",track,sector);
#endif
	    R1541Ram[0x22]=R1541HeadTrack=2;
	    return JOB_OK;

	case JOB_EXECUTE:
#if 1
	    printf("Execute command $%02X %02d %02d %x\n",
		command,track,sector,buffer-R1541Ram);
	    /* Catch format: */
	    if( buffer==R1541Ram+0x600
		    && R1541Ram[0x600]==0x4C
		    && R1541Ram[0x601]==0xC7
		    && R1541Ram[0x602]==0xFA ) {
		printf("Format detected %s\n",R1541Ram+0x200);
		cmd_new(R1541Ram+0x200);
		return JOB_OK;
	    }
#endif
	    R1541HeadTrack=track*2;
	    R1541Ram[0x22]=track;
	    R1541Ram[0x32]=((buffer-R1541Ram-0x0300)>>7)+0x06;

	    /* FALL THROUGH */
	case JOB_JUMP:
#ifdef __JOHNS
	    printf("Execute command $%02X %02d %02d %x\n",
		command,track,sector,buffer-R1541Ram);
#endif

	    R1541Ram[0x3F]=(buffer-R1541Ram-0x0300)>>8;

	    if( R1541ActionFunction==ExecuteAction ) {
#ifdef JOHNS
		printf("Already done\n");
#endif
		return JOB_JUMP;
	    }
	    OldAction=R1541Action;	/* Can't change PC here !!! */
	    OldActionFunction=R1541ActionFunction;
	    R1541Action=R1541Cycle;
	    R1541ActionFunction=ExecuteAction;

	    return JOB_JUMP;

	default:
#ifdef X11
	    printf("Unknown command $%02X %02d %02d %x\n",
		command,track,sector,buffer-R1541Ram);
#endif
	    return JOB_OK;
    }
}

/*
**	Look if any commands are available in the job queue.
*/
static void CheckWorkQueue(void)
{
    if( (Integer(GlobalRegP)&FLAG_INTERRUPT) ) {
	return;
    }

#if 1
    /*
    **	Head move ?
    */
    if( R1541Ram[0x02FE] ) {
#ifdef JOHNS
	printf("Head irq $%02X\n",R1541Ram[0x02FE]);
#endif

	R1541HeadTrack+=R1541Ram[0x02FE];

	if( R1541HeadTrack<2 )
	    R1541HeadTrack=2;
	else if( R1541HeadTrack>87 )
	    R1541HeadTrack=87;

	R1541Ram[0x02FE]=0;
    }
#endif

    if( R1541Ram[0]&0x80 ) {
	// printf("JOB $%02X,$%02X\n",0,R1541Ram[0]);
	R1541Ram[0]=DoJob(R1541Ram[0],R1541Ram[6],R1541Ram[7],
		R1541Ram+0x300);
    }
    if( R1541Ram[1]&0x80 ) {
	// printf("JOB $%02X,$%02X\n",1,R1541Ram[1]);
	R1541Ram[1]=DoJob(R1541Ram[1],R1541Ram[8],R1541Ram[9],
		R1541Ram+0x400);
    }
    if( R1541Ram[2]&0x80 ) {
	// printf("JOB $%02X,$%02X\n",2,R1541Ram[2]);
	R1541Ram[2]=DoJob(R1541Ram[2],R1541Ram[10],R1541Ram[11],
		R1541Ram+0x500);
    }
    if( R1541Ram[3]&0x80 ) {
	// printf("JOB $%02X,$%02X\n",3,R1541Ram[3]);
	R1541Ram[3]=DoJob(R1541Ram[3],R1541Ram[12],R1541Ram[13],
		R1541Ram+0x600);
    }
    if( R1541Ram[4]&0x80 ) {
	// printf("JOB $%02X,$%02X\n",4,R1541Ram[4]);
	R1541Ram[4]=DoJob(R1541Ram[4],R1541Ram[14],R1541Ram[15],
		R1541Ram+0x700);
    }
}

/*---------------------------------------------------------------------------*/

/*
**	Reset 1541. Called from C64 reset.
*/
void ResetReal1541(void)
{
    if( FastLoaderOff&2 ) {		/* NO ROMS! */
	return;
    }

    C64SerialLine=0;			/* C64 serial line */
    R1541SerialLine=0;			/* 1541 serial line */
    C64OldOut=0;			/* old c64 serial output */
    CheckAction=0;

    R1541Reset();			/* processor reset */

    R1541ActionFunction=0;
    R1541Action=1016419;
    R1541Emul6502Loop();		/* emulate until wait loop */
}

/*---------------------------------------------------------------------------*/

/*
**	Initialise real 1541.
*/
void EnterReal1541(void)
{
    char* cp;
    int f;

    FastLoaderOff|=2;			/* initialise fails */

    /*
    **	Read 1541 ROM.
    */
    cp=GetDefault("R1541IMAGE",R1541ImageName,R1541IMAGE);
    f=open(cp,O_RDONLY|BINARY_FILE);
    if( f==-1 ) {
	f=open(R1541IMAGE,O_RDONLY|BINARY_FILE);
	if( f==-1 ) {
	    printf("Can't find 1541 ROM images\n");
	    return;
	}
    }

    if( read(f,R1541Rom,16384)!=16384 ) {
	printf("Read error: 1541 ROM\n");
	Exit(1);
    }
    close(f);

    FastLoaderOff&=~2;			/* initialise ok */

#ifdef X11
    printf("Reseting 1541\n");
#endif
    ResetReal1541();

    /*
    **	Patch the kernel.
    R1541Rom[0xD693-0xC000]=0x02;
    R1541Rom[0xD694-0xC000]=0x01;
    **	Must fix the checksum !!!
    */
}

/*
**	Cleanup for real 1541.
*/
void LeaveReal1541(void)
{
}

/*---------------------------------------------------------------------------*/

#if 0

Cycle>=Action stops 1541 emulation

	LDA	(C64)
	     S
	     T S
	     A T S
	     * A T S
Action--------------------------------
	L      * A T S
	D        * A T
	A          * A
	*            *


#endif

/*---------------------------------------------------------------------------*/

#undef Cycle

/*
**	Emulate 1541.
*/
void EmulReal1541(void)
{
    unsigned action;
    extern int DoMystic;

    /* Correct 1541 CPU CLOCK FACTOR: 1.0149728799246484133 */
#ifdef JOHNS
    //action=(long long)(Cycle+DoMystic)*1000000/CPU_CLK;
    // action=(unsigned long long)Cycle*1000000/CPU_CLK;
    //action=((double)(Cycle)*DoMystic)/1000;
    // action=Cycle;
#else
#endif
    action=((unsigned long long)(Cycle)*DoMystic)/1000;

    /*
    **	Round about.
    */
    if( action<=R1541Cycle ) {
	if( action<R1541Cycle-12 ) {
#ifdef JOHNS
	    printf("Restart %d\n",action-R1541Cycle);
#endif
	    R1541Cycle=action-10;
	    // CheckWorkQueue();	BREAKS MANY PROGRAMS !!!
	} else {
#ifdef JOHNS
	    static int flag=999;

	    if( (action-R1541Cycle)<flag ) {
		flag=(action-R1541Cycle);
		printf("Ignore %d\n",action-R1541Cycle);
	    }
#endif
	    return;
	}
    }

#if 1
    /*
    **	X frames no access, restart syncron.
    **	Defender of the Crown need *30
    **	Heart of Africa needs *100
    **  Turican II needs restart time.
    **  Red strom needs restart time.
    **	The Pawn needs *100 (FIXME: Check shorter time!)
    */
    else if( action>R1541Cycle+CYCLES_PER_LINE*LINES_PER_FRAME*SerialTimeout ) {
#ifdef JOHNS
	printf("Restart %d\n",action-R1541Cycle);
#endif
	//R1541Cycle=action-10;
	/* This needed by Red Strom and Turrican II */
	R1541Cycle=action-CYCLES_PER_LINE*LINES_PER_FRAME*SerialTimeout;
	// CheckWorkQueue();		BREAKS MANY PROGRAMS !!!
	// CheckAction=action+10000;
    }
#endif

    if( action>CheckAction ) {		/* test atlast each 10ms */
// FIXME: CheckAction must be updated by clock adjust
//	  Via1TimerA_Action must be updated by clock adjust
//	  Via2TimerA_Action must be updated by clock adjust
	CheckWorkQueue();
#ifdef JOHNS
	CheckAction=action+Via2TimerA;
	CheckAction=action+10;
#else
	CheckAction=action+10000;
#endif
    }

    R1541Action=action;
    R1541Emul6502Loop();
}

/*
**	IEC read serial port trap.
*/
int IEC_ReadPortTrap(void)
{
    if( FastLoaderOff ) {
	return 0;
    }

    // Cycle-=2;
    // Cycle--;
    EmulReal1541();
    // Cycle++;
    // Cycle+=2;
    /*
    printf("<: %02X %d\t",C64SerialLine,Cycle);
    if( !flag ) printf("!!!!!!!!!!!!!!!!!!!!!!!!!!");
    printf("\n");
    flag=0;
    */

    return C64SerialLine;
}

/*
**	IEC write serial port trap.
*/
void IEC_WritePortTrap(int val)
{
    int atn;

    if( FastLoaderOff ) {
	return;
    }

    val&=0x38;				/* C64_DAT_O|C64_CLK_O|C64_ATN_O */
    if( val!=C64OldOut ) {
	// Cycle+=2;
	EmulReal1541();
	// Cycle-=2;

	C64OldOut=val;
#if 0
	printf("C64:  %08d: %s%s%s $%04X\n",Cycle,
	    (C64OldOut&0x08) ? "___ " : "ATN ",
	    (C64OldOut&0x10) ? "___ " : "CLK ",
	    (C64OldOut&0x20) ? "___ " : "DAT ",
	    GlobalRegPC
	);
#endif
	atn=(val<<4)&0x80;

	/*
	**	ATN Changed?
	*/
	if( (R1541SerialLine^atn)&0x80 ) {
	    int o;

	    /*
	    **	Set ATN -> interrupt!
	    */
	    if( !atn ) {
		// printf("New ATN **************************************\n");
		WB_MEM(0x7C,1);		/* simulate ATN irq */
	    }

	    o=~VIA1[0]&VIA1[2];
#if 0
	    /*
	    **	line=0 -> C64SerialLine=1
	    */
	    C64SerialLine=		/* connect 1541 -> C64 */
		( (((o&0x02)<<6)	/* DAT */
		& (((atn^(o<<3))&0x80)^0x80) )
		//& (((atn|(~o<<3))&0x80)^0x00) )
		|((o&0x08)<<3) )^0xC0;	/* CLK */
#else
	    C64SerialLine&=0x40;	/* connect 1541 -> C64 */
	    C64SerialLine|=
		(((o<<6)^0x80)	/* 1541 DAT 0x02 -> 0x80 */
		| (((atn)^(~o<<3))^0x80))&0x80;
		//| (((atn)|(~o<<3))^0x80))&0x80;
#endif
	    // printf("ATN: %02X %02X ->%02X\n",val,o,C64SerialLine);
	}

	R1541SerialLine=		/* connect c64 -> 1541 */
	    atn				/* ATN 0x08 -> 0x80 */
	    |((val&0x10)>>2)		/* CLK 0x10 -> 0x04 */
	    |((val&0x20)>>5);		/* DAT 0x20 -> 0x01 */
    }
}

#endif	/* } REAL1541 */
