/*
 *	The C64 emulator
 *
 *	Copyright 1992,93,96 by ALE.
 *	written by Lutz Sammer.
 *
 *	Monitor
 *-----------------------------------------------------------------------------
 *	$Id: mon.c,v 1.12 1996/07/03 03:14:18 johns Exp root $
 *	$Log: mon.c,v $
 *	Revision 1.12  1996/07/03 03:14:18  johns
 *	Many new commands and cleanup.
 *
 * Revision 1.11  1994/06/02  14:54:24  johns
 * Use demosum.h, always revert meaning of '-j'.
 *
 * Revision 1.10  1993/10/02  13:27:46  root
 * cleanup
 *
 * Revision 1.9  1993/10/01  20:43:44  root
 * added missing '\n'
 *
 * Revision 1.8  1993/09/17  11:27:39  johns
 * support for wide ( > 80 char ) disassembly
 *
 * Revision 1.7  1993/08/31  20:30:03  johns
 * go32 initial support
 *
 * Revision 1.6  1993/06/16  23:03:58  johns
 * support VT-switching correctly.
 *
 * Revision 1.5  1993/01/05  12:41:24  johns
 * Checkin before applying wuebbel patches
 *
 * Revision 1.4  1992/07/28  19:47:42  johns
 * Change of 6502 register names.
 *
 * Revision 1.3  1992/07/20  04:23:33  johns
 * New functions (f)ill (i)nfo and (I)nfo sid written and some bugs removed.
 *
 * Revision 1.2  1992/07/13  04:36:26  johns
 * Rasterline irq now an action.
 *
 * Revision 1.1  1992/07/11  18:34:53  johns
 * Initial revision
 *
 *-----------------------------------------------------------------------------
 */

#include "c64.h"

#ifdef MONITOR	/* { */

#include "vic.h"
#include VIDEO_H
#include "sid.h"

#include <stdio.h>
#include <stdlib.h>
#include <time.h>


#if defined(__linux__) || defined(__GO32__)

#include <sys/time.h>
#include <unistd.h>

#define TIMER(x)	gettimeofday(&x,0)
typedef struct timeval	TimerVal;
#define TIMER_USEC(x)	((double)(x).tv_sec*1000+(double)((x).tv_usec/1000))

#else

#define TIMER(x)	time(&x)
typedef unsigned long	TimerVal;
#define TIMER_USEC(x)	((double)x*1000)

#endif

#define RegP	GlobalRegP
#define RegA	GlobalRegA
#define RegX	GlobalRegX
#define RegY	GlobalRegY
#define RegS	GlobalRegS
#define RegPC	GlobalRegPC

#ifdef REAL1541
extern void R1541Reg(void);
extern int R1541RBMem(int);
extern int R1541Dis(int);
#endif

unsigned MonAction;
void	(*MonFunction)(void);

static unsigned BreakPoint;
static unsigned TempBreakPoint;
static void	Monitor (CONST char*);

#define TimeCycles	((long)LINES_PER_FRAME*CYCLES_PER_LINE*2000)

static TimerVal StartTime;		/* speed checker */
static TimerVal EndTime;

static int MonDefaultBase=16;		/* default base */

static int MonLastAddr;			/* default start */

/*-----------------------------------------------------------------------------
 *	Monitor entry points
 *---------------------------------------------------------------------------*/

/*
**	Action: Monitor single step
*/
static void MonSingleStep(void)
{
    Monitor((char *)0);
}

/*
**	Action: Breakpoint
*/
static void MonBreakPoint(void)
{
    if( BreakPoint==RegPC ) {
	Monitor("Breakpoint");
    } else {
	MonAction=Cycle+1;
	UpdateAction();
    }
}

/*
**	Action: Tempory Breakpoint
*/
static void MonTempBreakPoint(void)
{
    if( TempBreakPoint==RegPC ) {
	TempBreakPoint=0;
	Monitor("Breakpoint (tempory)");
    } else if( BreakPoint==RegPC ) {
	Monitor("Breakpoint");
    } else {
	MonAction=Cycle+1;
	UpdateAction();
    }
}

/*
**	Action: Timing
*/
static void MonTiming(void)
{
    char buf[1024];

    TIMER(EndTime);
    sprintf(buf,"Timing: Time=%.2fs for %ld Cycles (C64=%.2fs) %.2f%%",
	(TIMER_USEC(EndTime)-TIMER_USEC(StartTime))/1000,
	TimeCycles,(double)TimeCycles/CPU_CLK,
	((double)TimeCycles/CPU_CLK)/
	(((double)TIMER_USEC(EndTime)-TIMER_USEC(StartTime))/100000) );
    Monitor(buf);
}

/*
**	Turn monitor on
*/
void MonitorOn(void)
{
    MonAction=Cycle+1;
    MonFunction=MonSingleStep;
    UpdateAction();
}

/*-----------------------------------------------------------------------------
 *	Monitor sub functions
 *---------------------------------------------------------------------------*/

/*
**	Cbm character to ascii.
*/
static int Cbm2Ascii(int c)
{
    if( 'A'<=c && c<='Z' )
	c|=0x20;
    else if( ('A'|0x80)<=c && c<=('Z'|0x80) )
	c&=0x7F;
    else if( c<' ' || ('a'<=c && c<='z') || (0x80<=c && c<=0x9F) )
	c='.';
    return c;
}

/*-----------------------------------------------------------------------------
 *	Monitor functions
 *---------------------------------------------------------------------------*/

/*
**	Show pc each frame.
*/
static void ShowPC(void)
{
    char buf[40];

    sprintf(buf,"$%04X",RegPC);
    VicText(34*8,0,buf);
}

/*
**	Show time per frame.
*/
/*static*/ void TimePerFrame(void)
{
    static struct timeval start;
    struct timeval end;
    static char buf[40]="Please Wait";

    gettimeofday(&end,0);
    if( start.tv_sec || start.tv_usec ) {
	long l;
	l=(end.tv_sec-start.tv_sec)*100000+(end.tv_usec-start.tv_usec)/10;
	sprintf(buf,"%4ldms/100 frames",l);
    }
    VicText(13*8,0,buf);
    start=end;
}

static long Min20s;
static long Avg20s;

/*
**	Show time for longer running.
*/
/*static*/ void TimePer20s(void)
{
    static struct timeval start;
    struct timeval end;
    static char buf[40]="Please Wait";

    long i;
    static long count;
    
    if( --count<0 ) {
	count=1000;
	gettimeofday(&end,0);
	if( start.tv_sec || start.tv_usec ) {
	    i=(end.tv_sec-start.tv_sec)*1000+(end.tv_usec-start.tv_usec)/1000;
	    if( Min20s>i ) {
		Min20s=i;
	    }
	    if( Avg20s ) {
		Avg20s=(Avg20s*2+i)/3;
	    } else {
		Avg20s=i;
	    }
	    sprintf(buf,"%5ld>%5ld~%5ld",i,Min20s,Avg20s);
	}
	start=end;
    }
    VicText(13*8,0,buf);
}

/*
**	Print register
*/
static void Reg(void)
{
    int s1,s2,s3,s4;

#if 0 && !defined(__GO32__)
    extern int DisStringLen;

    UpdateFlags();			/* Recalculate flags */
    printf("$%04X: ",RegPC);
    s1=CIA1[CIA_ICR];			/* Don't destroy ICR */
    s2=CIA2[CIA_ICR];
    s3=VIC[0x1E];			/* Don't destroy collision */
    s4=VIC[0x1F];
    Dis(RegPC);
    CIA1[CIA_ICR]=s1;
    CIA2[CIA_ICR]=s2;
    VIC[0x1E]=s3;
    VIC[0x1F]=s4;
    if( DisStringLen%8 ) {		/* calc tab */
	printf("\t");
	DisStringLen = ((DisStringLen + 7)/8)*8;
    }
    while (DisStringLen < 32) {
	printf("\t");
	DisStringLen += 8;
    }
    printf("A=$%02X ",RegA.Int);
    printf("X=$%02X ",RegX.Int);
    printf("Y=$%02X ",RegY.Int);
    printf("S=$%02X ",RegS.Int);
    printf("P=$%02X ",RegP.Int);
    printf("%c",(RegP.Int&FLAG_NEGATIVE)  ? 'N' : '-');
    printf("%c",(RegP.Int&FLAG_OVERFLOW)  ? 'O' : '-');
    printf("%c",(RegP.Int&FLAG_UNDEFINED) ? '*' : '-');
    printf("%c",(RegP.Int&FLAG_BREAK)     ? 'B' : '-');
    printf("%c",(RegP.Int&FLAG_INTERRUPT) ? 'I' : '-');
    printf("%c",(RegP.Int&FLAG_DECIMAL)   ? 'D' : '-');
    printf("%c",(RegP.Int&FLAG_ZERO)      ? 'Z' : '-');
    printf("%c",(RegP.Int&FLAG_CARRY)     ? 'C' : '-');
    if( RegS.Int<0xFE ) {
	printf(" [$%04X",RWStack(1+RegS.Int));
	if( RegS.Int<0xFC )
	    printf(",$%04X",RWStack(3+RegS.Int));
	printf("] ");
    }
    printf(" Cycle(%d)",Cycle);
    printf("\n");
#else
    UpdateFlags();			/* Recalculate flags */
    printf("PC=$%04X ",RegPC);
    printf("A=$%02X ",RegA.Int);
    printf("X=$%02X ",RegX.Int);
    printf("Y=$%02X ",RegY.Int);
    printf("S=$%02X ",RegS.Int);
    printf("P=$%02X ",RegP.Int);
    printf("%c",(RegP.Int&FLAG_NEGATIVE)  ? 'N' : '-');
    printf("%c",(RegP.Int&FLAG_OVERFLOW)  ? 'O' : '-');
    printf("%c",(RegP.Int&FLAG_UNDEFINED) ? '*' : '-');
    printf("%c",(RegP.Int&FLAG_BREAK)     ? 'B' : '-');
    printf("%c",(RegP.Int&FLAG_INTERRUPT) ? 'I' : '-');
    printf("%c",(RegP.Int&FLAG_DECIMAL)   ? 'D' : '-');
    printf("%c",(RegP.Int&FLAG_ZERO)      ? 'Z' : '-');
    printf("%c",(RegP.Int&FLAG_CARRY)     ? 'C' : '-');
    if( RegS.Int<0xFE ) {
	printf(" [$%04X",RWStack(1+RegS.Int));
	if( RegS.Int<0xFC )
	    printf(",$%04X",RWStack(3+RegS.Int));
	printf("] ");
    }
    printf(" Cycle(%d)",Cycle);
    printf("\n");
    printf("$%04X: ",RegPC);
    s1=CIA1[CIA_ICR];			/* Don't destroy ICR */
    s2=CIA2[CIA_ICR];
    s3=VIC[0x1E];			/* Don't destroy collision */
    s4=VIC[0x1F];
    Dis(RegPC);
    printf("\n");
    CIA1[CIA_ICR]=s1;
    CIA2[CIA_ICR]=s2;
    VIC[0x1E]=s3;
    VIC[0x1F]=s4;
#endif
}

/*
**	Memory dump
*/
static void MonMemDump(int s,int e)
{
    int i;

    while( s<=e ) {
	printf("$%04X: ",s);
	for( i=0; i<16; ++i ) {
	    if( i==8 ) {
		putchar(' ');
	    }
	    printf("%02X ",RBMem((s+i)&0xFFFF));
	}
	putchar(' ');
	for( i=0; i<16; ++i ) {
	    putchar(Cbm2Ascii(RBMem((s+i)&0xFFFF)));
	}
	printf("\n");
	s+=16;
    }
    MonLastAddr=s;
}

#ifdef REAL1541
/*
**	Memory dump
*/
static void Mon1541MemDump(int s,int e)
{
    int i;

    while( s<=e ) {
	printf("$%04X: ",s);
	for( i=0; i<16; ++i ) {
	    if( i==8 ) {
		putchar(' ');
	    }
	    printf("%02X ",R1541RBMem((s+i)&0xFFFF));
	}
	putchar(' ');
	for( i=0; i<16; ++i ) {
	    putchar(Cbm2Ascii(R1541RBMem((s+i)&0xFFFF)));
	}
	printf("\n");
	s+=16;
    }
    MonLastAddr=s;
}
#endif

/*
**	Ascii dump
*/
static void MonAsciiDump(int s,int e)
{
    int i;
    int c;

    while( s<=e ) {
	printf("$%04X: ",s);
	for( i=0; i<16; ++i ) {
	    c=RBMem((s+i)&0xFFFF);
	    if( c>=0xA0 ) {
		printf(" ~");
		c-=0x80;
	    } else if( c==0x9F ) {
		printf("~^");
		c='?';
	    } else if( c>=0x80 ) {
		printf("~^");
		c-=0x80;
		c+='@';
	    } else if( c==0x7F ) {
		printf(" ^");
		c='?';
	    } else if( c<' ' ) {
		printf(" ^");
		c+='@';
	    } else {
		printf("  ");
	    }
	    printf("%c ",c);
	}
	printf("\n");
	s+=16;
    }
    MonLastAddr=s;
}

/*
**	Disassemble
*/
static void MonDisassemble(int s,int e)
{
    int i;

    if( e ) {
	while( s<=e ) {
	    printf("$%04X: ",s);
	    s+=Dis(s&0xFFFF);
	    printf("\n");
	}
    } else {
	for( i=0; i<16; ++i ) {
	    printf("$%04X: ",s);
	    s+=Dis(s&0xFFFF);
	    printf("\n");
	}
    }
    MonLastAddr=s;
}

#ifdef REAL1541
/*
**	Disassemble
*/
static void Mon1541Disassemble(int s,int e)
{
    int i;

    if( e ) {
	while( s<=e ) {
	    printf("$%04X: ",s);
	    s+=R1541Dis(s&0xFFFF);
	    printf("\n");
	}
    } else {
	for( i=0; i<16; ++i ) {
	    printf("$%04X: ",s);
	    s+=R1541Dis(s&0xFFFF);
	    printf("\n");
	}
    }
    MonLastAddr=s;
}
#endif

/*
**	Suspend C64 mode.
*/
static void SuspendC64(void)
{
    SuspendSound();
    SuspendKeyboard();
    SuspendVideo();
}

/*
**	Resume C64 mode.
*/
static void ResumeC64(void)
{
    ResumeVideo();
    ResumeKeyboard();
    ResumeSound();
}

/*-----------------------------------------------------------------------------
 *	Monitor parse functions
 *---------------------------------------------------------------------------*/

/*
**	Get number.
*/
static unsigned MonGetNum(CONST char* in,char** out)
{
    int n;

    n=strtol(in,out,MonDefaultBase);

    return n;
}

/*
**	Get from to.
*/
static void MonGetFromTo(CONST char* in,char** out,int n,int* s,int *e)
{
    *s=MonGetNum(in,out);
    if( in!=*out ) {
	in=*out;
	*e=MonGetNum(in,out);
	if( !*e && n ) {
	    *e=*s+n;
	}
    } else {
	*s=MonLastAddr;
	if( n ) {
	    *e=*s+n;
	} else {
	    *e=0;
	}
    }
}

/*-----------------------------------------------------------------------------
 *	Monitor main loop
 *---------------------------------------------------------------------------*/

/*
**	Monitor.
*/
static void Monitor(CONST char* txt)
{
    char buf[256];
    char cmd;
    char* cp;
    int a,s,e;

    SuspendC64();
    if( txt ) {				/* enter message */
	printf("%s\n",txt);
    }
#if defined(JOHNS) && defined(REAL1541)
    if( !FastLoaderOff ) {
	IEC_ReadPortTrap();		/* advance 1541 */
	R1541Reg();
    }
#endif
    Reg();

    if( 0 ) {
	MonAction=Cycle+1;
	MonFunction=MonSingleStep;
	UpdateAction();
	return;
    }

    for( ;; ) {
	printf(".");
	gets(buf);
	for( cp=buf; *cp && *cp<=' '; ++cp )
	    ;
	switch( cmd=*cp++ ) {

	    case '\0':
		break;

	    case 'a':
		MonGetFromTo(cp,&cp,15,&s,&e);
		MonAsciiDump(s,e);
		continue;

	    case 'b':
		a=MonGetNum(cp,&cp);
		if( a ) {
		    BreakPoint=a&0xFFFF;
		} else {
		    printf("Breakpoint $%04X\n",BreakPoint);
		}
		continue;

	    case 'B':
		BreakPoint=0;
		continue;

	    case 'c':
	    case 'C':
		a=MonGetNum(cp,&cp);
		if( a ) {
		    BreakPoint=a&0xFFFF;
		}
		if( TempBreakPoint ) {	/* old temp break point */
		    MonAction=Cycle+1;
		    MonFunction=MonTempBreakPoint;
		    UpdateAction();
		} else if( BreakPoint ) {/* Any break point */
		    MonAction=Cycle+1;
		    MonFunction=MonBreakPoint;
		    UpdateAction();
		} else {		/* none break point */
		    MonAction=-1;
		    MonFunction=0;
		    UpdateAction();
		}
		if( cmd=='c' ) {
		    ResumeC64();
		}
		return;

	    case 'd':
		MonGetFromTo(cp,&cp,0,&s,&e);
		MonDisassemble(s,e);
		continue;

#ifdef REAL1541
	    case 'D':
		MonGetFromTo(cp,&cp,0,&s,&e);
		Mon1541Disassemble(s,e);
		continue;
#endif

	    case 'e':
		s=MonGetNum(cp,&cp);
		do {
		    a=MonGetNum(cp,&cp);
		    WBMem(s&0xFFFF,a&0xFF);
		    ++s;
		} while( *cp );
		MonLastAddr=s;
		continue;

	    case 'f':
		MonGetFromTo(cp,&cp,1,&s,&e);
		a=0;
		if( *cp ) {
		    a=MonGetNum(cp,&cp);
		}
		s&=0xFFFF;
		e&=0xFFFF;
		a&=0xFF;
		while( s<e )
		    WBMem(s++,a);
		MonLastAddr=s;
		continue;

	    case 'g':
	    case 'G':
		a=MonGetNum(cp,&cp);
		if( a ) {
		    if( a>1 && a<65535 )
			RegPC=a&0xFFFF;
		    else
			printf("Bad address!\n");
		}
		if( TempBreakPoint ) {	/* old temp break point */
		    MonAction=Cycle+1;
		    MonFunction=MonTempBreakPoint;
		    UpdateAction();
		} else if( BreakPoint ) {/* Any break point */
		    MonAction=Cycle+1;
		    MonFunction=MonBreakPoint;
		    UpdateAction();
		} else {
		    MonAction=-1;
		    MonFunction=0;
		    UpdateAction();
		}
		if( cmd=='g' ) {
		    ResumeC64();
		}
		return;

	    case 'i':
		DisplayHead();
		DisplayCia1();
		DisplayCia2();
		DisplaySid();
		DisplayVic();

		MonAction=-1;
		UpdateAction();
		printf("Config %s %s %s\n",
		    (Ram[1]|~Ram[0])&4 ? "-" : "charen",
		    (Ram[1]|~Ram[0])&2 ? "-" : "hiram",
		    (Ram[1]|~Ram[0])&1 ? "-" : "lowram");
		printf("Cycle=%d Action=%d Next in=%d\n",
		    Cycle,Action,Action-Cycle);

		printf("Cia1: Mask $%02X Alarm: %c%c %02X %02X %02X %02X\n",
		    Cia1IrqMask,
		    Cia1TodBuffered ? 'B' : '-',
		    Cia1TodStopped ? 'S' : '-',
		    Cia1Alarm[0],Cia1Alarm[1],Cia1Alarm[2],Cia1Alarm[3]);
		printf("  TimerA=%8d Latch=%8d Action=%8d\n",
		    Cia1TimerA,CIA1[CIA_TALO]|(CIA1[CIA_TAHI]<<8),
		    Cia1TimerA_Action);
		printf("  TimerB=%8d Latch=%8d Action=%8d\n",
		    Cia1TimerB,CIA1[CIA_TBLO]|(CIA1[CIA_TBHI]<<8),
		    Cia1TimerB_Action);

		printf("Cia2: Mask $%02X Alarm: %c%c %02X %02X %02X %02X\n",
		    Cia2IrqMask,
		    Cia2TodBuffered ? 'B' : '-',
		    Cia2TodStopped ? 'S' : '-',
		    Cia2Alarm[0],Cia2Alarm[1],Cia2Alarm[2],Cia2Alarm[3]);
		printf("  TimerA=%8d Latch=%8d Action=%8d\n",
		    Cia2TimerA,CIA2[CIA_TALO]|(CIA2[CIA_TAHI]<<8),
		    Cia2TimerA_Action);
		printf("  TimerB=%8d Latch=%8d Action=%8d\n",
		    Cia2TimerB,CIA2[CIA_TBLO]|(CIA2[CIA_TBHI]<<8),
		    Cia2TimerB_Action);

		VicInfo();
		printf("  Updates: %d\tRefreshes: %d\tSyncSpeed: %d\n",
		    VicEmulateRate,VicRefreshRate,VideoSyncSpeed);
		continue;

	    case 'I':
		SidInfo();
		continue;

	    case 'l':
	    { int adr;
		while( *cp==' ' || *cp=='\t' ) {
		    ++cp;
		}
		/*
		**	Send filename.
		*/
		IEC_Listen(8);
		IEC_SEC_Listen(0xF0);
		while( *cp ) {
		    IEC_Write(*cp++);
		}
		IEC_Unlisten();
		if( IEC_Status ) {
		    printf("Can't open file!\n");
		    continue;
		}
		
		/*
		**	Load the file.
		*/
		IEC_Talk(8);
		IEC_SEC_Talk(0);
		adr=(IEC_Read()&0xFF);
		if( IEC_Status ) {
		    printf("Can't open file!\n");
		    continue;
		}
		adr|=(IEC_Read()&0xFF)<<8;
		printf("From $%04X ",adr);
		while( !(IEC_Status&0x40) ) {
		    WBMem(adr&0xFFFF,IEC_Read()&0xFF);
		    ++adr;
		}
		printf(" to $%04X\n",adr);
		IEC_Untalk();
		continue;
	    }
#if 0
#endif

	    case 'n':
	    case 'N':
		if( RBMem(RegPC)==JSR ) {
		    TempBreakPoint=RegPC+3;
		    MonAction=Cycle+1;
		    MonFunction=MonTempBreakPoint;
		    UpdateAction();
		    if( cmd=='n' ) {
			ResumeC64();
		    }
		    return;
		}
		break;

	    case 'm':
		MonGetFromTo(cp,&cp,15,&s,&e);
		MonMemDump(s,e);
		continue;

#ifdef REAL1541
	    case 'M':
		MonGetFromTo(cp,&cp,15,&s,&e);
		Mon1541MemDump(s,e);
		continue;
#endif

	    case 'p':
		if( *cp=='c' ) {
		    s=strtol(cp+1,&cp,16);
		    if( s>1 && s<65535 )
			RegPC=s&0xffff;
		    else
			printf("Bad address!\n");
		}
		continue;

	    case 'r':
		Reg();
		continue;

#ifdef REAL1541
	    case 'R':
		R1541Reg();
		continue;
#endif

	    case 's':
		break;

	    case 'S':
		SaveScreen();
		continue;

	    case 'T':
		MonAction=Cycle+TimeCycles;
		MonFunction=MonTiming;
		UpdateAction();
		ResumeC64();
		TIMER(StartTime);
		return;

	    case 'u':
	    case 'U':
		a=RBMem(RegPC);
		if( a==BEQ || a==BNE || a==BPL || a==BMI
			    || a==BCC || a==BCS || a==BVC || a==BVS ) {
		    TempBreakPoint=RegPC+2;
		    MonAction=Cycle+1;
		    MonFunction=MonTempBreakPoint;
		    UpdateAction();
		    if( cmd=='u' ) {
			ResumeC64();
		    }
		    return;
		}
		break;

	    case 'v':
		printf("Irq/Brk vector $%04X ($314: $%04X,$316: $%04X)\n",
		    RWVec(IRQ_VECTOR),
		    RBMem(0x314)|(RBMem(0x315)<<8),
		    RBMem(0x316)|(RBMem(0x317)<<8));
		printf("Nmi     vector $%04X ($318: $%04X)\n",
		    RWVec(NMI_VECTOR),RBMem(0x318)|(RBMem(0x319)<<8));
		continue;

	    case 'w':
		while( *cp==' ' || *cp=='\t' ) {
		    ++cp;
		}
#if 0
		if( *cp ) {
		    if( chdir(cp)==-1 ) {
			printf("Can't change directory to `%s'\n",cp);
		    } else {
			printf("New working directory `%s'\n",cp);
		    }
		} else {
		    printf("Current working directory `%s'\n",
			getcwd(buf,sizeof(buf)));
		}
#endif
		if( *cp ) {
		    --cp;
		    *cp=':';
		    if( cmd_go(cp) ) {
			if( FloppyImage ) {
			    printf("New working floppy image `%s'\n",cp+1);
			} else {
			    printf("New working directory `%s'\n",cp+1);
			}
		    } else {
			printf("Can't change directory/image to `%s'\n",cp+1);
		    }
		} else {
		    if( FloppyImage ) {
			printf("Current working floppy image `%s'\n",
			    FloppyImage);
		    } else {
			printf("Current working directory `%s'\n",
			    getcwd(buf,sizeof(buf)));
		    }
		}
		continue;

	    case 'X':
		Exit(0);

	    case '@':
		{
		    extern char errorbuf[];

		    printf("1541: %s\n",errorbuf);
		}
		continue;
	
	    case '#':
		s=strtol(cp+1,&cp,0);
		printf("$%04X, %05d\n",s,s);
		continue;

	    case '%':
		if( VerticalRetraceHook==TimePerFrame ) {
		    VerticalRetraceHook=ShowPC;
		    printf("Showing program counter\n");
		} else if( VerticalRetraceHook==ShowPC ) {
		    VerticalRetraceHook=TimePer20s;
		    Min20s=99999999;
		    Avg20s=0;
		    printf("Showing time per 20Sec\n");
		} else if( VerticalRetraceHook==TimePer20s ) {
		    VerticalRetraceHook=VerticalRetraceNoop;
		    printf("Showing nothing\n");
		} else {
		    VerticalRetraceHook=TimePerFrame;
		    printf("Showing time per frame\n");
		}
		continue;

	    case '?':
		printf("\nMonitor help\n");
		printf("@\t\tPrint 1541 error\n");
		printf("# n\tPrint number conversion\n");
		printf("%%\t\tCycle vertical retrace hook\n");
		printf("a [adr] [adr]\tAsciidump from to\n");
		printf("b [adr]\t\tset/display break-point (only one for now)\n");
		printf("B\t\tdisable break-point\n");
		printf("c [brk]\tcontinue until break-point\n");
		printf("C [brk]\tcontinue until break-point no display\n");
		printf("g [adr]\t\tGoto adr\n");
		printf("G [adr]\t\tGoto adr no display\n");
		printf("n\t\tExecute sub routine without trace\n");
		printf("N\t\tExecute sub routine without trace no display\n");
		printf("r\t\tDisplay 6510 registers\n");
		printf("s\t\tSingle step\n");
		printf("RET\t\tSingle step\n");
		printf("u\t\tStep until after branch\n");
		printf("U\t\tStep until after branch no display\n");
		printf("pc [adr]\tSet program counter\n");
		printf("d [adr] [adr]\tDisassemble from to\n");
		printf("e adr val ...\tEdit memory\n");
		printf("f adr adr val\tFill memory from to with\n");
		printf("m [adr] [adr]\tMemory dump from to\n");
		printf("i\t\tPrint emulator state infos\n");
		printf("I\t\tPrint sid info\n");
		printf("v\t\tPrint 6502 vectors\n");
		printf("w [dir]\tChange working directory to dir\n");
		printf("X\t\tExit emulator\n\n");
		continue;

	    default:
		printf("Unknown command `%s', use `?' for help\n",cp-1);
		continue;
	}
	/*
	**	Continue single step
	*/
	MonAction=Cycle+1;
	MonFunction=MonSingleStep;
	UpdateAction();
	break;
    }
}

/* FIXME: Move to own file !!! */

#include "pbm.c"

unsigned char rmap[] = {
    0 , 63, 39, 0 , 47, 0 , 0,  63, 63, 30, 63, 15, 31, 0 , 0 , 47,
};
unsigned char gmap[] = {
    0 , 63, 0 , 47, 0 , 39, 0 , 63, 22, 14, 0 , 15, 31, 63, 0 , 47,
};
unsigned char bmap[] = {
    0 , 63, 0 , 47, 47, 0 , 39, 0 , 0 , 0 , 0 , 15, 31, 0 , 63, 47,
};

void SaveScreen(void)
{
#if defined(X11) && defined(COLORMAP0)
    /* FIXME: only written for this configuration */
    FILE* fp;
    int i;
    extern unsigned char Pixel2Pen[];

    for( i=0; i<256; ++i ) {
	RMAP[i]=rmap[Pixel2Pen[i]]*4;
	GMAP[i]=gmap[Pixel2Pen[i]]*4;
	BMAP[i]=bmap[Pixel2Pen[i]]*4;
    }

    fp=fopen("screenshot.ppm","w");
    if( !fp ) {
	printf("Can't open screen shot file\n");
	return;
    }
    printf("Screenshot ");
    fflush(stdout);
    WritePBM(fp,VMEM,PIC8,HSIZE,VSIZE,RMAP,GMAP,BMAP,8,0,1,"C64 Screen Shot");
    fclose(fp);
    printf("done.\n");
#endif
}
























#if 0
			printf("Joystick 1 - DC01 = $%02X\n",JoyStick1);
			printf("Joystick 2 - DC00 = $%02X\n",JoyStick2);

			{ int i;
			for( i=0; i<8; ++i ) {
			    if( ConfigTable[i]==Config ) {
				break;
			    }
			}
			printf("Config %d\n",i);
			}

			continue;
		    case 'V':
			printf("Vic X-Offset %d ",VicXOffset);
			printf("Vic Y-Offset %d\n",VicYOffset);
			printf("Vic Scanline %d\n",VicScanLine);
			printf("Cols %02d Rows %02d\n", (VIC[0x16]&0x08) ? 40 : 38, (VIC[0x11]&0x08) ? 25 : 24);
			continue;
		    case 'M':
			{
			int i;
			int j;

			for( j=0; j<16; ++j ) {
			    printf("$%02X: ",j*16);
			    for( i=0; i<16; ++i ) {
				printf("%02X ",Matrix[j*16+i]);
			    }
			    printf("\n");
			}
			}
			continue;
			continue;
#endif

#endif  /* } MONITOR */
