/*
 *	The C64 emulator
 *
 *	Copyright 1992-96 by ALE.
 *	written by Lutz Sammer.
 *
 *	Keyboard emulation for X11.
 *	Written by Edgar Trnig
 *	R6 Version
 *------------------------------------------------------------------------------
 * $Id: key_x11.c,v 1.2 1996/07/01 22:12:20 johns Exp root $
 * $Log: key_x11.c,v $
 * Revision 1.2  1996/07/01 22:12:20  johns
 * Added keycodes for older XFree and SERVER_NUM_LOCK. Moved some common
 * functions to key.c
 *
 * Revision 1.1  1996/05/10 20:04:20  froese
 * Initial revision
 *
 *------------------------------------------------------------------------------
 */

#include "c64.h"
#include "vic.h"
#include "sid.h"

#include <stdio.h>
#include <memory.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#define XK_MISCELLANY
#define XK_LATIN1
#include <X11/keysymdef.h>

extern Display *display;
extern Atom WM_Delete_Window;
static unsigned char x11keymap[32];

static void GetLeds(void);
extern void VideoArrange(int,int);

#define KEY_PRESSED(k)  (x11keymap[(k)>>3]&(1<<((k)&7)))
#define KEY_RELEASED(k) (!KEY_PRESSED(k))


static void
C64Key(int pressed, int bit, int mask)
{
    int i, j;

    if (bit == 0 || mask == 0)
	return;

    if (pressed)
	for (j = 0; j < 256; j += bit)
	    for (i = 0; i < bit; ++i)
		KeyMatrix[j++] &= ~mask;
    else
	for (j = 0; j < 256; j += bit)
	    for (i = 0; i < bit; ++i)
		KeyMatrix[j++] |= mask;
}


/*
 *      Set/Reset C64 Leftshift key.
 */
static
void C64LeftShift(int pressed)
{
    C64Key(pressed | KEY_PRESSED(50), 0x02, 0x80);
}


static void
MapX11Key(int pressed, int keycode)
{
    char buf[256];

    if (pressed)
	x11keymap[keycode / 8] |= 1 << (keycode % 8);
    else
	x11keymap[keycode / 8] &= ~(1 << (keycode % 8));

    switch (keycode)
    {
	case  49: /* ^ */		C64Key(pressed, 0x80, 0x02);	break;
	case  10: /* 1 */		C64Key(pressed, 0x80, 0x01);	break;
	case  11: /* 2 */		C64Key(pressed, 0x80, 0x08);	break;
	case  12: /* 3 */		C64Key(pressed, 0x02, 0x01);	break;
	case  13: /* 4 */		C64Key(pressed, 0x02, 0x08);	break;
	case  14: /* 5 */		C64Key(pressed, 0x04, 0x01);	break;
	case  15: /* 6 */		C64Key(pressed, 0x04, 0x08);	break;
	case  16: /* 7 */		C64Key(pressed, 0x08, 0x01);	break;
	case  17: /* 8 */		C64Key(pressed, 0x08, 0x08);	break;
	case  18: /* 9 */		C64Key(pressed, 0x10, 0x01);	break;
	case  19: /* 0 */		C64Key(pressed, 0x10, 0x08);	break;
	case  20: /*  */		C64Key(pressed, 0x20, 0x01);	break;
	case  21: /* ' */		C64Key(pressed, 0x20, 0x08);	break;
	case  97: /* home */		C64Key(pressed, 0x40, 0x08);	break;
	case  22: /* backspace */	C64Key(pressed, 0x01, 0x01);	break;

	case  23: /* tab */		C64Key(pressed, 0x80, 0x04);	break;
	case  24: /* q */		C64Key(pressed, 0x80, 0x40);	break;
	case  25: /* w */		C64Key(pressed, 0x02, 0x02);	break;
	case  26: /* e */		C64Key(pressed, 0x02, 0x40);	break;
	case  27: /* r */		C64Key(pressed, 0x04, 0x02);	break;
	case  28: /* t */		C64Key(pressed, 0x04, 0x40);	break;
	case  29: /* z */		C64Key(pressed, 0x08, 0x02);	break;
	case  30: /* u */		C64Key(pressed, 0x08, 0x40);	break;
	case  31: /* i */		C64Key(pressed, 0x10, 0x02);	break;
	case  32: /* o */		C64Key(pressed, 0x10, 0x40);	break;
	case  33: /* p */		C64Key(pressed, 0x20, 0x02);	break;
	case  34: /*  */		C64Key(pressed, 0x20, 0x40);	break;
	case  35: /* + */		C64Key(pressed, 0x40, 0x02);	break;

	case  94: /* < (NOT on 101 keyboards!) */
	case 105: /* page-down */	C64Key(pressed, 0x40, 0x40);	break;
	case  99: /* page-up */		C64Key(pressed, 0x40, 0x01);	break;

	case  66: /* caps-lock */	C64Key(pressed, 0x80, 0x80);	break;
	case  38: /* a */		C64Key(pressed, 0x02, 0x04);	break;
	case  39: /* s */		C64Key(pressed, 0x02, 0x20);	break;
	case  40: /* d */		C64Key(pressed, 0x04, 0x04);	break;
	case  41: /* f */		C64Key(pressed, 0x04, 0x20);	break;
	case  42: /* g */		C64Key(pressed, 0x08, 0x04);	break;
	case  43: /* h */		C64Key(pressed, 0x08, 0x20);	break;
	case  44: /* j */		C64Key(pressed, 0x10, 0x04);	break;
	case  45: /* k */		C64Key(pressed, 0x10, 0x20);	break;
	case  46: /* l */		C64Key(pressed, 0x20, 0x04);	break;
	case  47: /*  */		C64Key(pressed, 0x20, 0x20);	break;
	case  48: /*  */		C64Key(pressed, 0x40, 0x04);	break;
	case  51: /* # */		C64Key(pressed, 0x40, 0x20);	break;
	case 108: /* kp-enter */
	case  36: /* return */		C64Key(pressed, 0x01, 0x02);	break;

	case  50: /* l-shift */		C64Key(pressed, 0x02, 0x80);	break;
	case  52: /* y */		C64Key(pressed, 0x02, 0x10);	break;
	case  53: /* x */		C64Key(pressed, 0x04, 0x80);	break;
	case  54: /* c */		C64Key(pressed, 0x04, 0x10);	break;
	case  55: /* v */		C64Key(pressed, 0x08, 0x80);	break;
	case  56: /* b */		C64Key(pressed, 0x08, 0x10);	break;
	case  57: /* n */		C64Key(pressed, 0x10, 0x80);	break;
	case  58: /* m */		C64Key(pressed, 0x10, 0x10);	break;
	case  59: /* , */		C64Key(pressed, 0x20, 0x80);	break;
	case  60: /* . */		C64Key(pressed, 0x20, 0x10);	break;
	case 112: /* kp-/ */
	case  61: /* - */		C64Key(pressed, 0x40, 0x80);	break;
	case  62: /* r-shift */		C64Key(pressed, 0x40, 0x10);	break;

	case  65: /* space */		C64Key(pressed, 0x80, 0x10);	break;

	case  37: /* l-control */	C64Key(pressed, 0x80, 0x04);	break;

	case  64: /* l-alt */
	case 113: /* r-alt */		C64Key(pressed, 0x80, 0x20);	break;

	case  92: /* alt-sys-req */	if (!pressed) Exit(0);		break;
	case 103: /* end */		if (!pressed) Nmi();		break;
	case 114: /* ctrl-break */	if (!pressed) Reset();		break;

	case 106: /* insert */
		C64LeftShift(pressed);
	case 107: /* delete */
		C64Key(pressed, 0x01, 0x01);	break;

	case  98: /* up */
		C64LeftShift(pressed);
	case 104:
		C64Key(pressed, 0x01, 0x80);	break;

	case 100: /* left */
		C64LeftShift(pressed);
	case 102:
		C64Key(pressed, 0x01, 0x04);	break;

	case  68: /* f2 */
		C64LeftShift(pressed);
	case  67: /* f1 */
		C64Key(pressed, 0x01, 0x10);	break;

	case  70: /* f4 */
		C64LeftShift(pressed);
	case  69: /* f3 */
		C64Key(pressed, 0x01, 0x20);	break;

	case  72: /* f6 */
		C64LeftShift(pressed);
	case  71: /* f5 */
		C64Key(pressed, 0x01, 0x40);	break;

	case  74: /* f8 */
		C64LeftShift(pressed);
	case  73: /* f7 */
		C64Key(pressed, 0x01, 0x08);	break;

	case  75: /* f9 */
		if (!pressed) {
		    if (KEY_PRESSED(50))
			++VicFetch;
		    else
			--VicFetch;
		    sprintf(buf,"%d fetch",VicFetch);
		    VicMessage(buf,FRAMES_PER_SECOND);
		}
		break;

	case  76: /* f10 */
		if (!pressed) {
		    if (KEY_PRESSED(50))
			++VicFetchAdd;
		    else
			--VicFetchAdd;
		    sprintf(buf,"%d add",VicFetchAdd);
		    VicMessage(buf,FRAMES_PER_SECOND);
		}
		break;

	case  95: /* f11 */
		if( !pressed ) {
		    ToggleSound();
		    if( SidSoundOff ) {
			VicMessage("sound off",FRAMES_PER_SECOND);
		    } else {
			VicMessage("sound on",FRAMES_PER_SECOND);
		    }
		}
		break;

	case  96: /* f12 */
		if( !pressed )
		    MonitorOn();
		break;

	case 147: /* other X11 */
	case  79: /* kp-7 */		/* Joystick Left+Up */
	    if (!pressed) {
		if( KEY_RELEASED(80) )
		    *JoyStick|=0x1;
		if( KEY_RELEASED(83) )
		    *JoyStick|=0x4;
	    } else {
		*JoyStick&=~0x5;
	    }
	    break;
	case 148: /* other X11 */
	case  80: /* kp-8 */		/* Joystick Up */
	    if( !pressed ) {
		if( KEY_RELEASED(79) && KEY_RELEASED(81) )
		    *JoyStick|=0x1;
	    } else {
		*JoyStick&=~0x1;
	    }
	    break;
	case 149: /* other X11 */
	case  81: /* kp-9 */		/* Joystick Right+Up */
	    if( !pressed ) {
		if( KEY_RELEASED(80) )
		    *JoyStick|=0x1;
		if( KEY_RELEASED(85) )
		    *JoyStick|=0x8;
	    } else {
		*JoyStick&=~0x9;
	    }
	    break;
	case 150: /* other X11 */
	case  83: /* kp-4 */		/* Joystick Left */
	    if( !pressed ) {
		if( KEY_RELEASED(79) && KEY_RELEASED(87) )
		    *JoyStick|=0x4;
	    } else {
		*JoyStick&=~0x4;
	    }
	    break;
	case 152: /* other X11 */
	case  85: /* kp-6 */		/* Joystick Right */
	    if( !pressed ) {
		if( KEY_RELEASED(81) && KEY_RELEASED(89) )
		    *JoyStick|=0x8;
	    } else {
		*JoyStick&=~0x8;
	    }
	    break;
	case 153: /* other X11 */
	case  87: /* kp-1 */		/* Joystick Left+Down */
	    if( !pressed ) {
		if( KEY_RELEASED(83) )
		    *JoyStick|=0x4;
		if( KEY_RELEASED(88) )
		    *JoyStick|=0x2;
	    } else {
		*JoyStick&=~0x6;
	    }
	    break;
	case 154: /* other X11 */
	case  88: /* kp-2 */		/* Joystick Down */
	    if( !pressed ) {
		if( KEY_RELEASED(87) && KEY_RELEASED(89) )
		    *JoyStick|=0x2;
	    } else {
		*JoyStick&=~0x2;
	    }
	    break;
	case 155: /* other X11 */
	case  89: /* kp-3 */		/* Joystick Right+Down */
	    if( !pressed ) {
		if( KEY_RELEASED(85) )
		    *JoyStick|=0x8;
		if( KEY_RELEASED(88) )
		    *JoyStick|=0x2;
	    } else {
		*JoyStick&=~0xA;
	    }
	    break;
	case 109: /* r-control */
	case   9: /* Escape */
	case 151: /* other X11 */
	case  84: /* kp-5 */		/* Joystick Fire */
	    if( !pressed ) {
		*JoyStick|=0x10;
	    } else {
		*JoyStick&=~0x10;
	    }
	    break;
	case  77: /* num-lock */	/* Joystick emulation toggle */
#ifdef XSERVER_NUM_LOCK
	    if( !pressed ) {		/* generate normal up/down */
		if( JoyStick==&JoyStick2 ) {
		    JoyStick=&JoyStick1, OtherJoyStick=&JoyStick2;
		    VicMessage("Joystick 1",FRAMES_PER_SECOND);
		} else {
		    JoyStick=&JoyStick2, OtherJoyStick=&JoyStick1;
		    VicMessage("Joystick 2",FRAMES_PER_SECOND);
		}
	    }
#else
	    if (pressed) {		/* XF86 gen. down/up depending on led */
		JoyStick = &JoyStick2, OtherJoyStick = &JoyStick1;
		VicMessage("Joystick 2",FRAMES_PER_SECOND);
	    } else {
		JoyStick = &JoyStick1, OtherJoyStick = &JoyStick2;
		VicMessage("Joystick 1",FRAMES_PER_SECOND);
	    }
#endif
	    JoyStick2 = JoyStick1 = 0xFF;
	    ShowLeds(0100);
	    break;

	case 111: /* druck */		/* Autofire on/off toggle */
	    if( !pressed ) {
		AutoFire^=1;
		if( !AutoFire ) {
		    if( KEY_PRESSED(9) )
			*JoyStick|=0x10;
		    VicMessage("Autofire off",FRAMES_PER_SECOND);
		} else {
		    VicMessage("Autofire on",FRAMES_PER_SECOND);
		}
		ShowLeds(0010);
	    }
	    break;

	case  78: /* scroll-lock */
	    if( !pressed ) {
		if( KEY_PRESSED(50) ) {
		    VicEmulateRate=(VicEmulateRate&3)+1;
		    sprintf(buf,"1/%d emulate",VicEmulateRate);
		    VicMessage(buf,FRAMES_PER_SECOND);
		    ShowLeds(0001);
		} else {
		    VicRefreshRate=(VicRefreshRate&3)+1;
		    sprintf(buf,"1/%d updates",VicRefreshRate);
		    VicMessage(buf,FRAMES_PER_SECOND);
		    ShowLeds(0001);
		}
	    }
	    break;

	case  63: /* kp-* */	HandleKey(pressed,KEY_FASTLOADER_TOGGLE);break;
	case  86: /* kp-+ */	HandleKey(pressed,KEY_NEXT_DISC);	break;
	case  82: /* kp-- */	HandleKey(pressed,KEY_PREV_DISC);	break;

	case  90: /* kp-0 */		C64Key(pressed, 0x00, 0x00);	break;
	case  91: /* kp-, */		C64Key(pressed, 0x00, 0x00);	break;
	case 110: /* pause */		C64Key(pressed, 0x00, 0x00);	break;

	case  93: /* ??? */		C64Key(pressed, 0x00, 0x00);	break;
	case 101: /* ??? */		C64Key(pressed, 0x00, 0x00);	break;
	default: break;
    }
}

/*
 * Resync the matrix with the current X11 keymap
 */
static void
RebuildMatrix()
{
    int i;

    memset(KeyMatrix, 0xFF, sizeof(KeyMatrix));

    for (i = 0; i < 256; ++i)
	if (x11keymap[i / 8] & (1 << (i % 8)))
	    MapX11Key(1, i);
}



void
EnterKey(void)
{
    GetLeds();
}

void
LeaveKey(void)
{
    if( display ) {
#ifdef XREPEATOFF
	XAutoRepeatOn(display);		/* restore autorepeat */
#endif
	ShowLeds(0000);			/* all leds off */
    }
}

/*
 *      Look for keyboard events.
 *
 *      Translate system keyboard events -> c64 keyboard matrix
 *
 *      Called with 50hz/60hz in video blank.
 */
void EmulKeyboard(void)
{
    XEvent ev;
    int n;

#if 0
    // n = XEventsQueued(display, QueuedAfterReading);
    n = XEventsQueued(display, QueuedAfterFlush);
    if( n )
	printf("Events %d\n",n);
    for (; n; --n)
#else
    for (n = XEventsQueued(display, QueuedAfterReading); n; --n)
#endif
    {
	XNextEvent(display, &ev);

	switch (ev.xany.type)
	{
	    case KeyPress:
	    case KeyRelease:
		MapX11Key(ev.xkey.type == KeyPress, ev.xkey.keycode);
		break;
	    case FocusIn:
#ifdef XREPEATOFF
		XAutoRepeatOff(display);
#endif
		memset(x11keymap, 0, sizeof(x11keymap));
		RebuildMatrix();
		break;
	    case FocusOut:
#ifdef XREPEATOFF
		XAutoRepeatOn(display);
#endif
		memset(x11keymap, 0, sizeof(x11keymap));
		RebuildMatrix();
		break;
	    case KeymapNotify:
		memcpy(x11keymap, ev.xkeymap.key_vector, sizeof(x11keymap));
		RebuildMatrix();
		break;
	    case Expose:
		VicDrawOverscan=1;
		break;
	    case ConfigureNotify:
		VideoArrange(ev.xconfigure.width,ev.xconfigure.height);
		break;
	    case ClientMessage:
		if (ev.xclient.format != 32 || ev.xclient.data.l[0] != WM_Delete_Window)
		    break;
		Exit(0);
		break;
	    default:
		break;
	}
    }

    /*
    **  Autofire on and ESCAPE pressed toggles fire button
    */
    if( AutoFire && (KEY_PRESSED(9) || KEY_PRESSED(109)) ) {
	*JoyStick^=0x10;
    }
}

/*
**	Suspend keyboard.
*/
void SuspendKeyboard(void)
{
#ifdef XREPEATOFF
    XAutoRepeatOn(display);
    XFlush(display);
#endif
}

/*
**	Resume keyboard.
*/
void ResumeKeyboard(void)
{
}

void
ShowLeds(int flags)
{
    XKeyboardControl kbctrl;

    if (flags == 0000) {
	kbctrl.led_mode = LedModeOff;
	XChangeKeyboardControl(display, KBLedMode, &kbctrl);
	return;
    }

    if (flags & 0100) {
	kbctrl.led = 2;	/* num-lock led */
	kbctrl.led_mode = (JoyStick == &JoyStick2) ? LedModeOn : LedModeOff;
	XChangeKeyboardControl(display, KBLed | KBLedMode, &kbctrl);
    }

    if (flags & 0010) {
	kbctrl.led = 1;	/* caps-lock led */
	kbctrl.led_mode = AutoFire ? LedModeOn : LedModeOff;
	XChangeKeyboardControl(display, KBLed | KBLedMode, &kbctrl);
    }

    if (flags & 0001) {
	kbctrl.led = 3;	/* scroll-lock led */
	kbctrl.led_mode = (VicRefreshRate==1) ? LedModeOn : LedModeOff;
	XChangeKeyboardControl(display, KBLed | KBLedMode, &kbctrl);
    }
}



static void
GetLeds(void)
{
    XKeyboardState kbstate;

#ifndef XSERVER_NUM_LOCK
    /* get num-lock state */

    if( !JoystickGiven ) {	/* commandline overwrites setting */
	XGetKeyboardControl(display, &kbstate);

	if (kbstate.led_mask & 2)
	    JoyStick = &JoyStick2, OtherJoyStick = &JoyStick1;
	else
	    JoyStick = &JoyStick1, OtherJoyStick = &JoyStick2;

    }
#endif
    JoyStick2=JoyStick1=0xFF;

    ShowLeds(0111);
}
