/*
 *	The C64 emulator
 *
 *	Copyright 1992-96 by ALE.
 *	written by Lutz Sammer.
 *
 *	Keyboard emulation for GO32 (MSDOS version of GNU-C)
 *-----------------------------------------------------------------------------
 * $Id: key_go32.c,v 1.15 1996/07/04 23:31:21 ari Exp root $
 * $Log: key_go32.c,v $
 * Revision 1.15  1996/07/04 23:31:21  ari
 * Rewritten DPMI keyboard interrupt handler, to remove conflict with
 * sound interrupts.
 *
 * Revision 1.14  1996/06/16 00:43:33  ari
 * Splitted into extra file.
 *
 * Revision 1.13  1995/06/01 12:08:24  ari
 * added full DPMI-compliance to GO32 version - now usable under OS/2 + Windows
 *
 * Revision 1.12  1995/02/13  11:27:22  ari
 * second linux port ( by ari ) integration
 *
 * Revision 1.11  1995/02/06  23:56:41  johns
 * go32 DPMI keyboard support
 *
 * Revision 1.10  1994/07/16  13:59:15  root
 * support go32 1.11.m5 changed keyboard interrupt handle
 *
 * Revision 1.9  1993/12/30  11:46:21  root
 * go32-changes integrated in unix version
 *
 * Revision 1.8  1993/09/07  21:40:26  johns
 * key mapping modification F12 <-> PageUp, add PageDown as Key102 alias
 *
 * Revision 1.7  1993/08/31  20:30:03  johns
 * go32 initial support
 *
 * Revision 1.6  1993/06/13  12:25:00  johns
 * final vt-switching mods
 *
 * Revision 1.5  1993/01/05  12:41:24  johns
 * Checkin before applying wuebbel patches
 *
 * Revision 1.4  1992/07/28  19:47:42  johns
 * F9/F10 changes VicFetch VicFetchAdd.
 *
 * Revision 1.3  1992/07/20  04:15:46  johns
 * Better macros for (re)set key in c64 matrix.
 * Keyboard joystick emulation improved for release.
 *
 * Revision 1.2  1992/07/13  04:36:26  johns
 * Missing led ioctl fixed.
 *
 * Revision 1.1  1992/07/11  18:34:08  johns
 * Initial revision
 *
 *-----------------------------------------------------------------------------
 */

#ifdef __GO32__	/* { */

#include <stdio.h>

#include <sys/types.h>
#include <go32.h>
#include <dos.h>
#include <dpmi.h>
#include <stdlib.h>

#include "c64.h"
#include "vic.h"
#include "sid.h"

/*---------------------------------------------------------------------------*/

extern void SaveSnapShot(CONST char*);

/*---------------------------------------------------------------------------*/

/*
 *	Reset key in C64 keyboard matrix
 */
#define MatrixReset(bit,mask) \
    do {                                        \
	int i,j;                                \
						\
	for( j=0; j<256; j+=(bit) ) {           \
	    for( i=0; i<(bit); ++i ) {          \
		KeyMatrix[j++]|=(mask);            \
	    }                                   \
	}                                       \
    } while( 0 )

/*
 *      Set key in C64 keyboard matrix
 */
#define MatrixSet(bit,mask) \
    do {                                        \
	int i,j;                                \
						\
	for( j=0; j<256; j+=(bit) ) {           \
	    for( i=0; i<(bit); ++i ) {          \
		KeyMatrix[j++]&=(mask)^0xFF;       \
	    }                                   \
	}                                       \
    } while( 0 )

/*---------------------------------------------------------------------------*/

/*
 *      Modifier set:
 */
#define KEY_S_NONE      0x00
#define KEY_S_SHIFT     0x01
#define KEY_S_CTRL      0x02
#define KEY_S_LALT      0x04
#define KEY_S_RALT      0x08
#define KEY_S_ALT       0x0C
#define KEY_S_CAPS      0x10
#define KEY_S_NUM       0x20
#define KEY_S_SCROLL    0x40
#define KEY_S_REPEAT    0x80

/*
 *      Scancodes enumerated.
 */
/*
 * definition of the AT84/MF101/MF102 Keyboard:
 * ============================================================
 *       Defined             Key Cap Glyphs       Pressed value
 *      Key Name            Main       Also       (hex)    (dec)
 *      ----------------   ---------- -------    ------    ------
 */
#define KEY_Escape      /* Escape                0x01  */    1  
#define KEY_1           /* 1           !         0x02  */    2 
#define KEY_2           /* 2           @         0x03  */    3 
#define KEY_3           /* 3           #         0x04  */    4 
#define KEY_4           /* 4           $         0x05  */    5 
#define KEY_5           /* 5           %         0x06  */    6 
#define KEY_6           /* 6           ^         0x07  */    7 
#define KEY_7           /* 7           &         0x08  */    8 
#define KEY_8           /* 8           *         0x09  */    9 
#define KEY_9           /* 9           (         0x0a  */   10 
#define KEY_0           /* 0           )         0x0b  */   11 
#define KEY_Minus       /* - (Minus)   _ (Under) 0x0c  */   12
#define KEY_Equal       /* = (Equal)   +         0x0d  */   13 
#define KEY_BackSpace   /* Back Space            0x0e  */   14 
#define KEY_Tab         /* Tab                   0x0f  */   15
#define KEY_Q           /* Q                     0x10  */   16
#define KEY_W           /* W                     0x11  */   17
#define KEY_E           /* E                     0x12  */   18
#define KEY_R           /* R                     0x13  */   19
#define KEY_T           /* T                     0x14  */   20
#define KEY_Y           /* Y                     0x15  */   21
#define KEY_U           /* U                     0x16  */   22
#define KEY_I           /* I                     0x17  */   23
#define KEY_O           /* O                     0x18  */   24
#define KEY_P           /* P                     0x19  */   25
#define KEY_LeftBrace   /* [           {         0x1a  */   26
#define KEY_RightBrace  /* ]           }         0x1b  */   27 
#define KEY_Return      /* Return                0x1c  */   28
#define KEY_LeftCtrl    /* Ctrl(left)            0x1d  */   29
#define KEY_A           /* A                     0x1e  */   30
#define KEY_S           /* S                     0x1f  */   31
#define KEY_D           /* D                     0x20  */   32 
#define KEY_F           /* F                     0x21  */   33
#define KEY_G           /* G                     0x22  */   34
#define KEY_H           /* H                     0x23  */   35
#define KEY_J           /* J                     0x24  */   36
#define KEY_K           /* K                     0x25  */   37
#define KEY_L           /* L                     0x26  */   38
#define KEY_SemiColon   /* ;(SemiColon) :(Colon) 0x27  */   39
#define KEY_Quote       /* ' (Apostr)  " (Quote) 0x28  */   40
#define KEY_Accent      /* ` (Accent)  ~ (Tilde) 0x29  */   41
#define KEY_LeftShift   /* Shift(left)           0x2a  */   42
#define KEY_BackSlash   /* \(BckSlash) |(VertBar)0x2b  */   43
#define KEY_Z           /* Z                     0x2c  */   44
#define KEY_X           /* X                     0x2d  */   45
#define KEY_C           /* C                     0x2e  */   46
#define KEY_V           /* V                     0x2f  */   47
#define KEY_B           /* B                     0x30  */   48
#define KEY_N           /* N                     0x31  */   49
#define KEY_M           /* M                     0x32  */   50
#define KEY_Comma       /* , (Comma)   < (Less)  0x33  */   51
#define KEY_Dot         /* . (Period)  >(Greater)0x34  */   52
#define KEY_Slash       /* / (Slash)   ?         0x35  */   53
#define KEY_RightShift  /* Shift(right)          0x36  */   54
#define KEY_KP_Multiply /* *                     0x37  */   55
#define KEY_LeftAlt     /* Alt(left)             0x38  */   56
#define KEY_Space       /*   (SpaceBar)          0x39  */   57
#define KEY_CapsLock    /* CapsLock              0x3a  */   58
#define KEY_F1          /* F1                    0x3b  */   59
#define KEY_F2          /* F2                    0x3c  */   60
#define KEY_F3          /* F3                    0x3d  */   61
#define KEY_F4          /* F4                    0x3e  */   62
#define KEY_F5          /* F5                    0x3f  */   63
#define KEY_F6          /* F6                    0x40  */   64
#define KEY_F7          /* F7                    0x41  */   65
#define KEY_F8          /* F8                    0x42  */   66
#define KEY_F9          /* F9                    0x43  */   67
#define KEY_F10         /* F10                   0x44  */   68
#define KEY_NumLock     /* NumLock               0x45  */   69
#define KEY_ScrollLock  /* ScrollLock            0x46  */   70
#define KEY_KP_7        /* 7           Home      0x47  */   71 
#define KEY_KP_8        /* 8           Up        0x48  */   72 
#define KEY_KP_9        /* 9           PgUp      0x49  */   73 
#define KEY_KP_Minus    /* - (Minus)             0x4a  */   74
#define KEY_KP_4        /* 4           Left      0x4b  */   75
#define KEY_KP_5        /* 5                     0x4c  */   76
#define KEY_KP_6        /* 6           Right     0x4d  */   77
#define KEY_KP_Plus     /* + (Plus)              0x4e  */   78
#define KEY_KP_1        /* 1           End       0x4f  */   79
#define KEY_KP_2        /* 2           Down      0x50  */   80
#define KEY_KP_3        /* 3           PgDown    0x51  */   81
#define KEY_KP_0        /* 0           Insert    0x52  */   82
#define KEY_KP_Dot      /* . (Decimal) Delete    0x53  */   83 
#define KEY_SysReqest   /* SysReqest             0x54  */   84
			/* NOTUSED               0x55  */
#define KEY_102         /* < (Less)   >(Greater) 0x56  */   86
#define KEY_F11         /* F11                   0x57  */   87
#define KEY_F12         /* F12                   0x58  */   88

#define KEY_Prefix0     /* special ( & 0x7f )    0x60  */   96
#define KEY_Prefix1     /* special ( & 0x7f )    0x61  */   97

/*
 * The 'scancodes' below are generated by the server, because the MF101/102
 * keyboard sends them as sequence of other scancodes
 */
#define KEY_Home        /* Home                  0x59  */   89
#define KEY_Up          /* Up                    0x5a  */   90
#define KEY_PageUp      /* PgUp                  0x5b  */   91
#define KEY_Left        /* Left                  0x5c  */   92
#define KEY_Begin       /* Begin                 0x5d  */   93
#define KEY_Right       /* Right                 0x5e  */   94
#define KEY_End         /* End                   0x5f  */   95
#define KEY_Down        /* Down                  0x60  */   96
#define KEY_PageDown    /* PgDown                0x61  */   97
#define KEY_Insert      /* Insert                0x62  */   98
#define KEY_Delete      /* Delete                0x63  */   99
#define KEY_KP_Enter    /* Enter                 0x64  */  100
#define KEY_RightCtrl   /* Ctrl(right)           0x65  */  101
#define KEY_Pause       /* Pause                 0x66  */  102
#define KEY_Print       /* Print                 0x67  */  103
#define KEY_KP_Divide   /* Didive                0x68  */  104
#define KEY_RightAlt    /* AltLang(right)        0x69  */  105
#define KEY_Break       /* Break                 0x6a  */  106

#define KEY_Released    128

#define CAPS_LED        1
#define NUM_LED         2
#define SCROLL_LED      4

#define IS_KEY_DOWN(k)	(KeyDown[(k)>>3]&(1<<((k)&7)))
#define IS_VTSW_QUAL	(0)
#define VT_ACTCON(xx) 

static int KeyState;			/* Current state */
static int KeyPrefix;			/* Current prefix */
static int KeyLeds;			/* Current leds */
static char KeyDown[128/8];		/* Keys currently pressed */ 

static _go32_dpmi_seginfo RmKHMem;
static unsigned long vecsave;
static unsigned long kbcntp;
static unsigned long kbreadp;
static unsigned long kbbufp;
static unsigned long go32kbvp;
static unsigned char keyrupt[62] = {
    0x1E, 0x50, 0x53, 0x51, 0x52, 0x0E, 0x1F, 0xE4,
    0x60, 0xBB, 0x10, 0x00, 0x8B, 0x0F, 0x3B, 0xCB,
    0x7D, 0x18, 0x8B, 0x5F, 0x02, 0x88, 0x07, 0x43,
    0x83, 0xFB, 0x10, 0x75, 0x02, 0x2B, 0xDB, 0x8B,
    0xD3, 0xBB, 0x10, 0x00, 0x89, 0x57, 0x02, 0x41,
    0x89, 0x0F, 0xBA, 0x61, 0x00, 0xEC, 0x0C, 0x80,
    0xEE, 0x24, 0x7F, 0xEE, 0xB0, 0x20, 0xE6, 0x20,
    0x5A, 0x59, 0x5B, 0x58, 0x1F, 0xCF
};

void ShowLeds(int dummy)
{
    KeyLeds=0;
    if( JoyStick==&JoyStick2 )
	KeyLeds|=NUM_LED;
    if( AutoFire )
	KeyLeds|=SCROLL_LED;   /* REALLY: CAPS_LED */
    if( VicEmulateRate==1 )
	KeyLeds|=CAPS_LED;     /* REALLY: SCROLL_LED */

    outb(0x60, 0xed);
    delay(10);
    outb(0x60, KeyLeds);
    delay(10);
}

void SuspendKeyboard(void)
{
    LeaveKey();
}

void ResumeKeyboard(void)
{
    EnterKey();
}

void EnterKey()
{
    unsigned short p1[2];
    unsigned long p2;
    unsigned long rmaddr;
    unsigned long intaddr;
    unsigned char tmpmem[32];
    
    if (RmKHMem.size == 0) {
	RmKHMem.size = 128/16;
	if (_go32_dpmi_allocate_dos_memory(&RmKHMem)) {
	    printf("Unable to allocate dos memory for KeyHandler!\n");
	    Cleanup();
	    exit(1);
	}
    }
    dosmemget(0x24, 4, &p1);
    p2 = p1[0] + p1[1] * 16ul;
    rmaddr = RmKHMem.rm_segment * 16;
    kbreadp = 0;
    kbbufp = rmaddr;
    kbcntp = rmaddr + 16;
    intaddr = (RmKHMem.rm_segment << 16) + 32;
    
    /* Heavy magic : pick go32 __ev_oldkbint savevector from go32. */
    dosmemget(p2 + 15, 2, &p1);
    go32kbvp = p1[0] + p1[1] * 16ul;
    dosmemget(go32kbvp, 4, &vecsave);

    memset(tmpmem, 0, 32);
    dosmemput(tmpmem, 32, rmaddr);
    dosmemput(keyrupt, sizeof(keyrupt), rmaddr + 32);
    disable();
    dosmemput(&intaddr, 4, go32kbvp);
    p1[0] = 1;
    dosmemput(p1, 2, go32kbvp + 4);
    enable();

    ShowLeds(0);
}

void LeaveKey()
{                        
    unsigned char oldleds;
    unsigned char bits = 0;
    unsigned short t;

    dosmemget(0x417, 1, &oldleds);
    if (oldleds & 0x10)
	bits |= 1;
    if (oldleds & 0x20)
	bits |= 2;
    if (oldleds & 0x40)
	bits |= 4;
    outb(0x60,0xed);
    delay(10);
    outb(0x60,bits);
    delay(10);

    if (RmKHMem.size) {
	disable();
	t = 0;
	dosmemput(&t, 2, go32kbvp + 4);
	dosmemput(&vecsave, 4, go32kbvp);
	enable();
	_go32_dpmi_free_dos_memory(&RmKHMem);
	RmKHMem.size = 0;
    }
}

short 
kbgetkey()
{
    short c, x;

    dosmemget(kbcntp, 2, &c);
    if (c == 0)
	return -1;
    c = 0;
    dosmemget(kbbufp + kbreadp++, 1, &c);
    if (kbreadp == 16)
	kbreadp = 0;
    disable();
    dosmemget(kbcntp, 2, &x);
    x--;
    dosmemput(&x, 2, kbcntp);
    enable();
    return c;
}

short 
kbsnoopkey()
{
    short c;

    dosmemget(kbcntp, 2, &c);
    if (c == 0)
	return -1;
    c = 0;
    dosmemget(kbbufp + kbreadp, 1, &c);
    return c;
}

/****** KeyboarLevel0 (1.00) *****
 *
 *      Level 0 Keyboard scancode processing:
 *      Map scancodes to one scancode.
 */
static int KeyboardLevel0(key)
int key;
{
    int scancode;
    int released;

    scancode=key&0x7F;
    released=key&0x80;
    switch( KeyPrefix ) {                                                       /* Handle according current prefix */
	case KEY_Prefix0:
	    KeyPrefix=0;
	    switch( scancode ) {                                                /* NUM-PAD */
		case KEY_KP_7:          scancode=KEY_Home;      break;
		case KEY_KP_8:          scancode=KEY_Up;        break;
		case KEY_KP_9:          scancode=KEY_PageUp;    break;
		case KEY_KP_4:          scancode=KEY_Left;      break;
		/* case KEY_KP_5:          scancode=KEY_Begin;     break; */
		case KEY_KP_6:          scancode=KEY_Right;     break;
		case KEY_KP_1:          scancode=KEY_End;       break;
		case KEY_KP_2:          scancode=KEY_Down;      break;
		case KEY_KP_3:          scancode=KEY_PageDown;  break;
		case KEY_KP_0:          scancode=KEY_Insert;    break;
		case KEY_KP_Dot: 	scancode=KEY_Delete;    break;
		case KEY_Return:        scancode=KEY_KP_Enter;  break;
		case KEY_LeftCtrl:      scancode=KEY_RightCtrl; break;
		case KEY_KP_Multiply:   scancode=KEY_Print;     break;
		case KEY_Slash:         scancode=KEY_KP_Divide; break;
		case KEY_LeftAlt:       scancode=KEY_RightAlt;  break;
		case KEY_ScrollLock:    scancode=KEY_Break;     break;
		default:                                                        /* Ignore other shifts */
		    return 0;
	    }
	    break;

	case KEY_Prefix1:                                                       /* Pause: Prefix1|LeftCtrl|NumLock */
	    KeyPrefix= (scancode==KEY_LeftCtrl) ? scancode : 0;
	    return 0;
	
	case KEY_LeftCtrl:                                                      /* Pause: Prefix1|LeftCtrl|NumLock */
	    KeyPrefix=0;
	    if( scancode!=KEY_NumLock )
		return 0;
	    scancode=KEY_Pause;
	    break;
	
	default:                                                                /* No prefix yet */
	    switch( scancode ) {
		case KEY_Prefix0:
		case KEY_Prefix1:
		    KeyPrefix=scancode;
		    return 0;
		default:
		    break;
	    }
	    break;
    }
    if( !released && (KeyDown[scancode>>3]&(1<<(scancode&7))) )                 /* Check for autorepeat */
	KeyState|=KEY_S_REPEAT;
    else
	KeyState&=~KEY_S_REPEAT;

    if( released )                                                              /* Set key in pressed/released set */
	KeyDown[scancode>>3]&=~(1<<(scancode&7));
    else
	KeyDown[scancode>>3]|=1<<(scancode&7);

    return released|scancode;
}

/*
 *      Set/Reset C64 Leftshift key.
 */
static void LeftShift(key)
int key;
{
    if( key&KEY_Released ) {
	if( !IS_KEY_DOWN(KEY_LeftShift) ) {
	    MatrixReset(0x02,0x80);
	}
    } else {
	MatrixSet(0x02,0x80);
    }
}

/*
 *      Convert scancode -> C64 key matrix
 */
static void EncodeKey(key)
int key;
{
    int b,m;
    char buf[256];

    switch( key&0x7F ) {
	case KEY_Accent:        b=0x80; m=0x02; break;                          /* <- */
	case KEY_1:             b=0x80; m=0x01; break;
	case KEY_2:             b=0x80; m=0x08; break;
	case KEY_3:             b=0x02; m=0x01; break;
	case KEY_4:             b=0x02; m=0x08; break;
	case KEY_5:             b=0x04; m=0x01; break;
	case KEY_6:             b=0x04; m=0x08; break;
	case KEY_7:             b=0x08; m=0x01; break;
	case KEY_8:             b=0x08; m=0x08; break;
	case KEY_9:             b=0x10; m=0x01; break;
	case KEY_0:             b=0x10; m=0x08; break;
	case KEY_Minus:         b=0x20; m=0x01; break;                          /* + */
	case KEY_Equal:         b=0x20; m=0x08; break;                          /* - */
	case KEY_Home:          b=0x40; m=0x08; break;                          /* CLR/HOME */
	case KEY_BackSpace:     b=0x01; m=0x01; break;                          /* INS/DEL */

	case KEY_Tab:           b=0x80; m=0x04; break;                          /* Control */
	case KEY_Q:             b=0x80; m=0x40; break;
	case KEY_W:             b=0x02; m=0x02; break;
	case KEY_E:             b=0x02; m=0x40; break;
	case KEY_R:             b=0x04; m=0x02; break;
	case KEY_T:             b=0x04; m=0x40; break;
	case KEY_Y:             b=0x08; m=0x02; break;
	case KEY_U:             b=0x08; m=0x40; break;
	case KEY_I:             b=0x10; m=0x02; break;
	case KEY_O:             b=0x10; m=0x40; break;
	case KEY_P:             b=0x20; m=0x02; break;
	case KEY_LeftBrace:     b=0x20; m=0x40; break;                          /* @ */
	case KEY_RightBrace:    b=0x40; m=0x02; break;                          /* * */

	case KEY_PageDown:							/* be nice to 101-keyboard-users */
	case KEY_102:           b=0x40; m=0x40; break;                          /* Arrow up */
	case KEY_PageUp: 	b=0x40; m=0x01; break;				/* pound */

	case KEY_CapsLock:      b=0x80; m=0x80; break;                          /* RUN/STOP */
	case KEY_A:             b=0x02; m=0x04; break;
	case KEY_S:             b=0x02; m=0x20; break;
	case KEY_D:             b=0x04; m=0x04; break;
	case KEY_F:             b=0x04; m=0x20; break;
	case KEY_G:             b=0x08; m=0x04; break;
	case KEY_H:             b=0x08; m=0x20; break;
	case KEY_J:             b=0x10; m=0x04; break;
	case KEY_K:             b=0x10; m=0x20; break;
	case KEY_L:             b=0x20; m=0x04; break;
	case KEY_SemiColon:     b=0x20; m=0x20; break;                          /* : */
	case KEY_Quote:         b=0x40; m=0x04; break;                          /* ; */
	case KEY_BackSlash:     b=0x40; m=0x20; break;                          /* = */
	case KEY_KP_Enter:
	case KEY_Return:        b=0x01; m=0x02; break;                          /* RETURN */

	case KEY_LeftShift:     b=0x02; m=0x80; break;                          /* Leftshift */
	case KEY_Z:             b=0x02; m=0x10; break;
	case KEY_X:             b=0x04; m=0x80; break;
	case KEY_C:             b=0x04; m=0x10; break;
	case KEY_V:             b=0x08; m=0x80; break;
	case KEY_B:             b=0x08; m=0x10; break;
	case KEY_N:             b=0x10; m=0x80; break;
	case KEY_M:             b=0x10; m=0x10; break;
	case KEY_Comma:         b=0x20; m=0x80; break;                          /* , */
	case KEY_Dot:           b=0x20; m=0x10; break;                          /* . */
	case KEY_KP_Divide:
	case KEY_Slash:         b=0x40; m=0x80; break;                          /* / */
	case KEY_RightShift:    b=0x40; m=0x10; break;                          /* Rightshift */

	case KEY_Space:         b=0x80; m=0x10; break;                          /* Space */

	case KEY_F1:            if IS_VTSW_QUAL {
				    VT_ACTCON(0);
				    return;
				} else {
				    b=0x01; m=0x10;
				} break;                                        /* F1 */
	case KEY_F3:            if IS_VTSW_QUAL {
				    VT_ACTCON(2);
				    return;
				} else {
				    b=0x01; m=0x20;
				} break;                                        /* F3 */
	case KEY_F5:            if IS_VTSW_QUAL {
				    VT_ACTCON(4);
				    return;
				} else {
				    b=0x01; m=0x40;
				} break;                                        /* F5 */
	case KEY_F7:            if IS_VTSW_QUAL {
				    VT_ACTCON(6);
				    return;
				} else {
				    b=0x01; m=0x08;
				} break;                                        /* F7 */
	case KEY_Right:         b=0x01; m=0x04; break;                          /* Right/Left */
	case KEY_Down:          b=0x01; m=0x80; break;                          /* Down/Up */

	case KEY_LeftCtrl:      b=0x80; m=0x04; break;                          /* Control */

	case KEY_LeftAlt:       b=0x80; m=0x20; break;                          /* C= */
	case KEY_RightAlt:      b=0x80; m=0x20; break;                          /* C= */

	case KEY_End:
	    if( key&KEY_Released ) Nmi();       return;                         /* Restore */

/*---------------------------------------------------------------------------*/

	case KEY_Insert:
	    LeftShift(key);     b=0x01; m=0x01; break;                          /* Shift + INS/DEL */

	case KEY_Up:
	    LeftShift(key);     b=0x01; m=0x80; break;                          /* Shift + Down/Up */

	case KEY_Left:
	    LeftShift(key);     b=0x01; m=0x04; break;                          /* Shift + Right/Left */

	case KEY_F2:            if IS_VTSW_QUAL {
				    VT_ACTCON(1);
				    return;
				} else {
	    LeftShift(key);     b=0x01; m=0x10; break;                          /* Shift + F1 */
				}
	case KEY_F4:            if IS_VTSW_QUAL {
				    VT_ACTCON(3);
				    return;
				} else {
	    LeftShift(key);     b=0x01; m=0x20; break;                          /* Shift + F3 */
				}
	case KEY_F6:            if IS_VTSW_QUAL {
				    VT_ACTCON(5);
				    return;
				} else {
	    LeftShift(key);     b=0x01; m=0x40; break;                          /* Shift + F5 */
				}
	case KEY_F8:            if IS_VTSW_QUAL {
				    VT_ACTCON(7);
				    return;
				} else {
	    LeftShift(key);     b=0x01; m=0x08; break;                          /* Shift + F7 */
				}

/*---------------------------------------------------------------------------*/

	case KEY_KP_7:                                                          /* Joystick Left+Up */
	    if( key&KEY_Released ) {
		if( !IS_KEY_DOWN(KEY_KP_8) )
		    *JoyStick|=0x1;
		if( !IS_KEY_DOWN(KEY_KP_4) )
		    *JoyStick|=0x4;
	    } else {
		*JoyStick&=~0x5;
	    }
	    return;
	case KEY_KP_8:                                                          /* Joystick Up */
	    if( key&KEY_Released ) {
		if( !IS_KEY_DOWN(KEY_KP_7) && !IS_KEY_DOWN(KEY_KP_9) )
		    *JoyStick|=0x1;
	    } else {
		*JoyStick&=~0x1;
	    }
	    return;
	case KEY_KP_9:                                                          /* Joystick Right+Up */
	    if( key&KEY_Released ) {
		if( !IS_KEY_DOWN(KEY_KP_8) )
		    *JoyStick|=0x1;
		if( !IS_KEY_DOWN(KEY_KP_6) )
		    *JoyStick|=0x8;
	    } else {
		*JoyStick&=~0x9;
	    }
	    return;
	case KEY_KP_4:                                                          /* Joystick Left */
	    if( key&KEY_Released ) {
		if( !IS_KEY_DOWN(KEY_KP_7) && !IS_KEY_DOWN(KEY_KP_1) )
		    *JoyStick|=0x4;
	    } else {
		*JoyStick&=~0x4;
	    }
	    return;
	case KEY_KP_6:                                                          /* Joystick Right */
	    if( key&KEY_Released ) {
		if( !IS_KEY_DOWN(KEY_KP_9) && !IS_KEY_DOWN(KEY_KP_3) )
		    *JoyStick|=0x8;
	    } else {
		*JoyStick&=~0x8;
	    }
	    return;
	case KEY_KP_1:                                                          /* Joystick Left+Down */
	    if( key&KEY_Released ) {
		if( !IS_KEY_DOWN(KEY_KP_4) )
		    *JoyStick|=0x4;
		if( !IS_KEY_DOWN(KEY_KP_2) )
		    *JoyStick|=0x2;
	    } else {
		*JoyStick&=~0x6;
	    }
	    return;
	case KEY_KP_2:                                                          /* Joystick Down */
	    if( key&KEY_Released ) {
		if( !IS_KEY_DOWN(KEY_KP_1) && !IS_KEY_DOWN(KEY_KP_3) )
		    *JoyStick|=0x2;
	    } else {
		*JoyStick&=~0x2;
	    }
	    return;
	case KEY_KP_3:                                                          /* Joystick Right+Down */
	    if( key&KEY_Released ) {
		if( !IS_KEY_DOWN(KEY_KP_6) )
		    *JoyStick|=0x8;
		if( !IS_KEY_DOWN(KEY_KP_2) )
		    *JoyStick|=0x2;
	    } else {
		*JoyStick&=~0xA;
	    }
	    return;
	case KEY_KP_5:
	case KEY_RightCtrl:							/* Control */
	case KEY_Escape:							/* Joystick Fire */
	    if( key&KEY_Released ) {
		*JoyStick|=0x10;
	    } else {
		*JoyStick&=~0x10;
	    }
	    return;
	case KEY_NumLock:                                                       /* Joystick emulation toggle */
	    if( key&KEY_Released ) {
		if( JoyStick==&JoyStick2 ) {
		    JoyStick = &JoyStick1, OtherJoyStick = &JoyStick2;
		    VicMessage("Joystick 1",FRAMES_PER_SECOND);
		} else {
		    JoyStick = &JoyStick2, OtherJoyStick = &JoyStick1;
		    VicMessage("Joystick 2",FRAMES_PER_SECOND);
		}
	    }
	    JoyStick2 = JoyStick1 = 0xFF;
	    ShowLeds(0);
	    return;

/*---------------------------------------------------------------------------*/

	case KEY_F9:                                                            /* Video timing test +- */
	    if IS_VTSW_QUAL 
	        VT_ACTCON(8);
	    else if( key&KEY_Released ) {
		if( IS_KEY_DOWN(KEY_LeftShift) )
		    ++VicFetch;
		else
		    --VicFetch;
		sprintf(buf,"%d fetch",VicFetch);
		VicMessage(buf,FRAMES_PER_SECOND);
	    }
	    return;

	case KEY_F10:                                                           /* Video timing test +- */
	    if IS_VTSW_QUAL 
	        VT_ACTCON(9);
	    else if( key&KEY_Released ) {
		if( IS_KEY_DOWN(KEY_LeftShift) )
		    ++VicFetchAdd;
		else
		    --VicFetchAdd;
		sprintf(buf,"%d add",VicFetchAdd);
		VicMessage(buf,FRAMES_PER_SECOND);
	    }
	    return;

	case KEY_F11:                                                           /* Sound on/off toggle */
	    if IS_VTSW_QUAL 
	        VT_ACTCON(10);
	    else if( key&KEY_Released ) {
		ToggleSound();
		if( SidSoundOff ) {
		    VicMessage("sound off",FRAMES_PER_SECOND);
		} else {
		    VicMessage("sound on",FRAMES_PER_SECOND);
		}
	    }
	    return;

	case KEY_F12:
	    if IS_VTSW_QUAL {
	        VT_ACTCON(11);
		return;
	    }
	    if( key&KEY_Released ) {
		if( IS_KEY_DOWN(KEY_RightCtrl) || IS_KEY_DOWN(KEY_LeftCtrl) ) { /* Freeze/make snapshot */
		    SaveSnapShot(C64SNAPSHOT);
		} else {                                                        /* Enter monitor */
		    MonitorOn();
		}
	    }
	    return;

	case KEY_Print:                                                         /* Autofire on/off toggle */
	    if( key&KEY_Released ) {
		if( IS_KEY_DOWN(KEY_LeftAlt) || IS_KEY_DOWN(KEY_RightAlt) )
		    Exit(0);
		AutoFire^=1;
		if( !AutoFire ) {
		    if( IS_KEY_DOWN(KEY_Escape) ) {
			*JoyStick|=0x10;
		    }
		    VicMessage("Autofire off",FRAMES_PER_SECOND);
		} else {
		    VicMessage("Autofire on",FRAMES_PER_SECOND);
		}
		ShowLeds(0);
	    }
	    return;

	case KEY_SysReqest:							/* Leave emulator */
	    if( key&KEY_Released ) {
		Exit(0);
	    }
	    return;

	case KEY_ScrollLock:                                                    /* Update speed round about */
	    if( key&KEY_Released ) {
		if( IS_KEY_DOWN(KEY_LeftShift) ) {
		    VicEmulateRate=(VicEmulateRate&3)+1;
		    sprintf(buf,"1/%d emulate",VicEmulateRate);
		    VicMessage(buf,FRAMES_PER_SECOND);
		    ShowLeds(0);
		} else {
		    VicRefreshRate=(VicRefreshRate&3)+1;
		    sprintf(buf,"1/%d updates",VicRefreshRate);
		    VicMessage(buf,FRAMES_PER_SECOND);
		    ShowLeds(0);
		}
	    }
	    return;

	case KEY_KP_Dot:                                                        /* Exit/Interrupt key 2 */
	    if( key&KEY_Released )
#ifndef __OLD__
		if (IS_KEY_DOWN(KEY_LeftAlt) && IS_KEY_DOWN(KEY_LeftCtrl))
#endif
		    Exit(0);
	    return;

	case KEY_Delete:                                                        /* DEL - Exit/Interrupt key */
	    if( key&KEY_Released ) {
#ifndef __OLD__
		if (IS_KEY_DOWN(KEY_LeftAlt) && IS_KEY_DOWN(KEY_LeftCtrl))
#endif
		    Exit(0);
	    }
	    b=0x01; m=0x01; break;                          			/* INS/DEL */

	case KEY_Pause:                                                         /* C64 emulator pause */
	    if( key&KEY_Released ) {
		SuspendSound();
		SuspendKeyboard();
		SuspendVideo();
		printf("Come back...\n");
		while( getchar()!='\n' )
		    ;
		ResumeVideo();
		ResumeKeyboard();
		ResumeSound();
	    }
	    return;

	case KEY_Break:                                                         /* C64 emulator reset */
	    if( key&KEY_Released )
		Reset();
	    return;
	
	case KEY_KP_Plus:
	    HandleKey(!(key&KEY_Released),KEY_NEXT_DISC);
	    return;
	case KEY_KP_Minus:
	    HandleKey(!(key&KEY_Released),KEY_PREV_DISC);
	    return;
	case KEY_KP_Multiply:
	    HandleKey(!(key&KEY_Released),KEY_FASTLOADER_TOGGLE);
	    return;

	default:
	    /* For keyboard debugging:
	    Cleanup();
	    printf("unkown key-code %d\n",key);
	    abort();
	    */
	    return;
    }
    if( key&KEY_Released ) {
	MatrixReset(b,m);
    } else {
	MatrixSet(b,m);
    }
}

/*
 *      Look for keyboard events.
 *
 *      Translate system keyboard events -> c64 keyboard matrix
 *
 *      Called with 50hz/60hz in video blank.
 */
void EmulKeyboard(void)
{
    int key;

    while ((key = kbgetkey()) != -1) 
    { 
	if((key=KeyboardLevel0(key)) && !(KeyState&KEY_S_REPEAT)) 
	{
	    EncodeKey(key);
	    if(!(key & KEY_Released)) 
	    { 
		while ((key = kbsnoopkey()) != -1)
		{
		    if (key & KEY_Released)
			break;
		    (void)kbgetkey();
		    if( (key=KeyboardLevel0(key)) && !(KeyState&KEY_S_REPEAT) )
			EncodeKey(key);
		}
	    }
	}
    }

    /*
    **	Autofire on and ESCAPE pressed toggle fire button
    */
    if( AutoFire && (IS_KEY_DOWN(KEY_Escape) && IS_KEY_DOWN(KEY_RightCtrl)) ) {
	*JoyStick^=0x10;
    }
}

#endif	/* } __GO32__ */

/*
 *      PORTING:          
 *
 *              EnterKey        Turn keyboard on
 *              LeaveKey        Turn keyboard off
 *              EmulKeyboard	Translate keys
 *              ShowLeds        Show some states
 */
