/*
 *	The C64 emulator
 *
 *	Copyright 1992-96 by ALE.
 *	written by Lutz Sammer.
 *
 *	Keyboard emulation.
 *-----------------------------------------------------------------------------
 * $Id: key.c,v 1.15 1996/07/01 22:02:29 johns Exp root $
 * $Log: key.c,v $
 * Revision 1.15  1996/07/01 22:02:29  johns
 * Collected first common functions.
 *
 * Revision 1.14  1996/04/28 15:01:52  johns
 * Machine dependend split into different parts.
 *
 * Revision 1.1  1992/07/11  18:34:08  johns
 * Initial revision
 *
 *-----------------------------------------------------------------------------
 */

#include "config.h"

#if 0

C64 Keyboard matrix:

Port A = DC00
Port B = DC01
 
	Port B
Port A     0     1       2       3       4       5       6       7
0       | DEL | RET | | <-> | | F7  | | F1  | | F3  | | F5  | | ^v  |
1       |  3  |  W  | |  A  | |  4  | |  Z  | |  S  | |  E  | | SHL |
2       |  5  |  R  | |  D  | |  6  | |  C  | |  F  | |  T  | |  X  |
3       |  7  |  Y  | |  G  | |  8  | |  B  | |  H  | |  U  | |  V  |
4       |  9  |  I  | |  J  | |  0  | |  M  | |  K  | |  O  | |  N  |
5       |  +  |  P  | |  L  | |  -  | |  .  | |  :  | |  @  | |  ,  |
6       | PFN |  *  | |  ;  | | HOM | | SHR | |  =  | |  ^  | |  /  |
7       |  1  | <-- | | CTR | |  2  | | SPC | | C=  | |  Q  | | RUN |

#endif

enum {
    KEY_PREV_DISC=0x100,
    KEY_NEXT_DISC,
    KEY_FASTLOADER_TOGGLE,

    KEY_MATRIX=0x200,
};

extern	void	HandleKey	(int,int);

#ifdef X11
#include "key_x11.c"
#endif

#ifdef SVGALIB
#include "key_svga.c"
#define RebuildMatrix()
#endif

#ifdef __GO32__
#include "key_go32.c"
#define RebuildMatrix()
#endif

unsigned char KeyMatrix[256];		/* C64- Key pressed matrix */
unsigned char *AutoFeedStr;		/* string of simulated keypresses */

/*---------------------------------------------------------------------------*/

/*
**      Set a key in the C64 keyboard matrix.
**		bit:	bit of port A.
**		mask:	bit of port B.
*/
void SetKeyMatrix(int bit,int mask)
{
    int i,j;

    for( j=0; j<256; j+=(bit) ) {
	for( i=0; i<(bit); ++i ) {
	    KeyMatrix[j++]&=~(mask);
	}
    }
}

/*
**      Clear a key in the C64 keyboard matrix.
**		bit:	bit of port A.
**		mask:	bit of port B.
*/
void ClrKeyMatrix(int bit,int mask)
{
    int i,j;

    for( j=0; j<256; j+=(bit) ) {
	for( i=0; i<(bit); ++i ) {
	    KeyMatrix[j++]|=(mask);
	}
    }
}

/*
**	Reset complete keyboard matrix.
*/
void ResetKeyMatrix(void)
{
    memset(KeyMatrix,0xFF,sizeof(KeyMatrix));
}

/*---------------------------------------------------------------------------*/

/*
**	Generate simulated keypresses.
*/
static void AutoFeed(void)
{
    static int AutoFeedDelay = -8;
    static unsigned char last_key = 0;
    static unsigned char CONST ascii2matrix[] = {
	0074, 0170, 0173, 0110, 0113, 0120, 0123, 0130,
	0133, 0140, 0061, 0050, 0057, 0053, 0054, 0067,
	0043, 0070, 0073, 0010, 0013, 0020, 0023, 0030,
	0033, 0040, 0055, 0062, 0157, 0065, 0154, 0167,
	0156, 0112, 0134, 0124, 0122, 0116, 0125, 0132,
	0135, 0141, 0142, 0145, 0152, 0144, 0147, 0146,
	0151, 0176, 0121, 0115, 0126, 0136, 0137, 0111,
	0127, 0131, 0114, 0155,    0, 0162, 0066,    0,
	   0, 0012, 0034, 0024, 0022, 0016, 0025, 0032,
	0035, 0041, 0042, 0045, 0052, 0044, 0047, 0046,
	0051, 0076, 0021, 0015, 0026, 0036, 0037, 0011,
	0027, 0031, 0014,    0,    0,    0, 0001, 0003,
    };
    unsigned char key;

    if (AutoFeedStr && ++AutoFeedDelay > 0)
    {
	/* release all keys */
	ResetKeyMatrix();

	/* press a new key */
	while ( (key = *AutoFeedStr++) )
	    if (key >= 0x20 && key <= 0x7f)
		if ( (key = ascii2matrix[key - 0x20]) )
		{
		    if (key == last_key)
		    {
		    	/* two same chars require an up event */
			AutoFeedStr--;
			last_key = 0;
		    }
		    else
		    {
			if (key & 0100)		/* SHIFT */
			    SetKeyMatrix(0x02, 0x80);
			SetKeyMatrix(1 << ((key>>3) & 7), 1 << (key & 7));
			last_key = key;
		    }
		    AutoFeedDelay=-1;
		    return;
		}
	AutoFeedStr = 0;
	AutoFeedDelay = 0;
	RebuildMatrix();
    }
}

/*---------------------------------------------------------------------------*/

/*
**	Advance disk image.
*/
static void ChangeDisk(int add)
{
    char buf[41];

    if( FloppyImage ) {
	char* cp;
	char* nam;

	
	nam=alloca(strlen(FloppyImage)+4);
	nam[0]=':';
	strcpy(nam+1,FloppyImage);
	cp=strrchr(nam,'.');
	if( cp && '0'-add<=cp[-1] && cp[-1]<='9'-add ) {
	    cp[-1]+=add;
	    cp=nam+1;
	    while( strlen(cp)>30 && strchr(cp,'/') ) {
		cp=strchr(cp,'/')+1;
	    }
	    if( cmd_go(nam) ) {
		sprintf(buf,"Image: %-.30s",cp);
		VicMessage(buf,FRAMES_PER_SECOND*2);
	    } else {
		sprintf(buf,"%-.30s NOT HERE",cp);
		VicMessage(buf,FRAMES_PER_SECOND*3);
	    }
	}
    }
}

/*
**	Common key functions.
**	Should become a new API for keyboard.
*/
void HandleKey(int pressed,int key)
{
    int b; int m;
    extern int DoMystic;

    b=m=1;
    switch( key ) {
	case ' ':	b=0x80; m=0x10;		break;		/* SPACE */
	case '':	b=0x80; m=0x04;		break;		/* Control */
	
	/*
	**	Emulation special keys.
	*/
	default:
	    if( KEY_MATRIX<=key && key<=(KEY_MATRIX+0x3F) ) {
		b=1 << ((key>>3) & 7);
		m=1 << (key & 7);
		break;
	    }

	    if( !pressed )  {
		return;
	    }

	    switch( key ) {
		/*
		**	Prev disk.
		*/
		case KEY_PREV_DISC:
		    ChangeDisk(-1);
		    return;
		/*
		**	Next disk.
		*/
		case KEY_NEXT_DISC:
		    ChangeDisk(1);
		    return;

#ifdef REAL1541
		/*
		**	Toggle fastloader support on/off.
		*/
		case KEY_FASTLOADER_TOGGLE:
		    if( !FastLoaderOff ) {
			if( DoMystic==1015 ) {
			    DoMystic=978;
			} else {
			    FastLoaderOff^=1;
			}
		    } else {
			DoMystic=1015;
			FastLoaderOff^=1;
		    }
		    { extern unsigned R1541Cycle;
			R1541Cycle=Cycle;
		    }
		    PatchKernel();
		    if( FastLoaderOff ) {
			VicMessage("Fastloader off",FRAMES_PER_SECOND);
		    } else {
			if( DoMystic==1015 ) {
			    VicMessage("PAL Fastloader on",FRAMES_PER_SECOND);
			} else {
			    VicMessage("NTSC Fastloader on",FRAMES_PER_SECOND);
			}
		    }
		    return;
#endif

		default:
		    return;
	    }
    }

    if( pressed ) {
	SetKeyMatrix(b,m);
    } else {
	ClrKeyMatrix(b,m);
    }
}


/*---------------------------------------------------------------------------*/

/*
**	Look for keyboard events.
**
**	Translate system keyboard events -> c64 keyboard matrix
**
**	Called with 50hz/60hz in video blank.
*/
void EmulKey(void)
{
    EmulKeyboard();			/* do machine dependend */

    if( AutoFeedStr ) {
	AutoFeed();
    }

#if 0
    /*
    **	Autofire on and ESCAPE pressed toggles fire button
    */
    if( AutoFire && KEY_PRESSED(9) ) {
	*JoyStick^=0x10;
    }
#endif
}
