/*
 *	The C64 emulator
 *
 *	Copyright 1992,96 by ALE.
 *	written by Lutz Sammer.
 *
 *	Joystick emulation
 *-----------------------------------------------------------------------------
 * $Id: joy.c,v 1.8 1996/08/11 02:09:07 johns Exp root $
 * $Log: joy.c,v $
 * Revision 1.8  1996/08/11 02:09:07  johns
 * Test for MSDOS joystick support.
 *
 * Revision 1.7  1996/07/04 02:51:02  maerz (Bernhard Maerz)
 * First version for analog PC joysticks.
 *
 * Revision 1.6  1996/07/04 02:15:40  johns
 * Cleanup.
 *
 * Revision 1.5  1994/06/02  14:54:24  johns
 * Use demosum.h, always revert meaning of '-j'.
 *
 * Revision 1.4  1993/10/01  18:54:02  johns
 * GO32 version reverts meaning of '-j'
 *
 * Revision 1.3  1993/06/13  12:24:36  johns
 * command line disable option
 *
 * Revision 1.2  1992/07/20  04:15:46  johns
 * Hardware joystick emulation for game port.
 *
 * Revision 1.1  1992/07/11  21:52:24  johns
 * Initial revision
 *
 *-----------------------------------------------------------------------------
 */

#include "c64.h"

extern int DoEmulJoy;

#define GAMEP_BASE 0x201

/*****************************************************************************/

#ifdef GAMEPORT_JOY	/* { */

/*
**  C64 Joystick using the PC game port.
*/
/* Hardware documentation :

    Adapter SUB-D 9 (male) to SUB-D 15 (male)

    The four SUB-D 9 direction bits map to the
    four fire buttons of the PC game-port (digital).
    The SUB-D 9 fire button is connected to the PC game-port
    joystick 1 up/down potentiometer and pulled by a 1k resistor.

    Pinout:

    DB-9	DB-15 (all others NC)

    1 ----------  2
    2 ----------  7
    3 ---------- 10
    4 ---------- 14
    5 NC
    6 ----------  3 ----|
    7 ----------  1	|
    8 ----------  4	1k
    9 NC		|
		  8 ----|

*/

/*
**	Initialise joystick.
*/
void EnterJoy(void)
{
    /* nothing todo */
}

/*
**	Cleanup joystick.
*/
void LeaveJoy(void)
{
    /* nothing todo */
}

/*
**	Do the joystick emulation, called each frame in the blank.
*/
void EmulJoy(void)
{
    unsigned char joyval;

    if (!DoEmulJoy)
	return;
    joyval = (unsigned char)(inb(GAMEP_BASE) &0xff);
    outb(GAMEP_BASE,0xff);	/* retrigger */
    *OtherJoyStick = 0xe0 | (joyval >> 4) | ((joyval & 1) ? 0 : 0x10);
}

#endif	/* } GAMEPORT_JOY */

/*****************************************************************************/

#ifdef LP_JOY	/* { */

/*
**	Joystick emulation on printer port.
*/

/* Hardware documentation :
	FIXME: must be written!!!
*/

/*
**	Initialise joystick.
*/
void EnterJoy(void)
{
    /* nothing todo */
}

/*
**	Cleanup joystick.
*/
void LeaveJoy(void)
{
    /* nothing todo */
}

/*
**	Do the joystick emulation, called each frame in the blank.
*/
void EmulJoy(void)
{
    unsigned joy;

    if (!DoEmulJoy)
	return;
    joy=inb(LP_STAT);
    joy>>=3;
    joy^=0xF0;
    *OtherJoy=joy;
}

#endif	/* } LP_JOY */

/*****************************************************************************/

#ifdef PC_ANALOG_JOY	/* { */

#if defined(__linux__) || defined(__GO32__)	/* { */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#ifdef __linux__

#include <linux/joystick.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#define JoyRead	read

#endif

#ifdef __GO32__

#include <dos.h>

typedef struct JS_DATA_TYPE {
    int	buttons;
    int	x;
    int	y;
} js;
#define JS_RETURN sizeof(struct JS_DATA_TYPE)

int JoyRead(int JoyDevice, struct JS_DATA_TYPE *js_data, int js_return)
{
    union REGS	in, out;

    /* Position lesen */
    in.h.ah = 0x84;			/* Funktion:     Joystick lesen */
    in.x.dx = 0x0001;			/* Sub-Funktion: Koordinaten lesen */
    int86(0x15, &in, &out);		/* Interrupt ausloesen */  

    /* wenn der Joystick erfolgreich ausgelesen werden konnte, ist cflag 0 */
    if(out.x.cflag)
	return js_return+1;		/* Fehler ! */

    if(!JoyDevice)		/* falls Joystick A gelesen werden sollte */
    {
	//js_data->x = out.h.al&0xFF;
	//js_data->y = out.h.bl&0xFF;
	js_data->x = (int) out.h.al + ((int) out.h.ah << 8);
	js_data->y = (int) out.h.bl + ((int) out.h.bh << 8);
    }
    else			/* ansonsten B */
    {
	//js_data->x = out.h.cl&0xFF;
	//js_data->y = out.h.dl&0xFF;
	js_data->x = (int) out.h.cl + ((int) out.h.ch << 8);
	js_data->y = (int) out.h.dl + ((int) out.h.dh << 8);
    }

    /* Buttons lesen */
    in.h.ah = 0x84;			/* Funktion:     Joystick lesen */
    in.x.dx = 0x0000;			/* Sub-Funktion: Buttons lesen */
    int86(0x15, &in, &out);		/* Interrupt ausloesen */  

    if(!JoyDevice)
	js_data->buttons =  0xF - (out.h.al>>4);
    else
	js_data->buttons =  0xF - (out.h.al>>6);

    return js_return;			/* alles OK */
}

#endif

static int JoyDevice0=-1;		/* joystick game port 0 */
static int JoyDevice1=-1;		/* joystick game port 1 */
static int Joy0Button2=0;		/* joystick button 2 was pressed */
static int Joy0Key=0x200|074;		/* key on button 2 (SPACE) */
static int Joy1Button2=0;		/* joystick button 2 was pressed */
static int Joy1Key=0x200|074;		/* key on button 2 (SPACE) */


/* calibration */
struct {int Up, Down, Left, Right, Button1, Button2;} JoyData0;
struct {int Up, Down, Left, Right, Button1, Button2;} JoyData1;

/*
**	Initialise joystick.
*/
void EnterJoy(void)
{
    char* config;

    if (!DoEmulJoy) {			/* no emulation wanted */
	return;
    }

#ifdef __linux__	/* { */
{
    long tmpl;
    int status;
    struct JS_DATA_TYPE js_data;
    char fname0[] = "/dev/js0";
    char fname1[] = "/dev/js1";

    JoyDevice0 = open (fname0, O_RDONLY);
    if (JoyDevice0 < 0) {
	printf("Can't get joystick port 1\n");
    } else {
	tmpl = 2;				/* 20 ms */
	status = ioctl (JoyDevice0, JS_SET_TIMELIMIT, &tmpl);
	if (status == -1) {
	    perror ("jscal");
	    exit (1);
	}
	js_data.x = 0;
	js_data.y = 0;
	status = ioctl (JoyDevice0, JS_SET_CAL, &js_data);
	if (status == -1) {
	    perror ("jscal");
	    exit (1);
	}
    }

    JoyDevice1=-1;			/* didn't detect no second! */
    if( GetDefault("C64JOY2",0,0) ) {
	JoyDevice1 = open (fname1, O_RDONLY);
    }
    if (JoyDevice1 < 0) {
	printf("Can't get joystick port 2\n");
    } else {
	tmpl = 2;				/* 20 ms */
	status = ioctl (JoyDevice1, JS_SET_TIMELIMIT, &tmpl);
	if (status == -1) {
	    perror ("jscal");
	    exit (1);
	}
	js_data.x = 0;
	js_data.y = 0;
	status = ioctl (JoyDevice1, JS_SET_CAL, &js_data);
	if (status == -1) {
	    perror ("jscal");
	    exit (1);
	}

    }
}
#endif	/* } __linux__ */

#ifdef __GO32__	/* { */
{
    union REGS	in, out;

    in.h.ah = 0x84;
    in.x.dx = 0x0000;
    int86(0x15, &in, &out);

    if(out.x.cflag)
      {
       printf("failed\n");
       switch(out.h.ah)
	 {
	  case 0x80 :
	    printf("invalid command\n");
	  case 0x86 :
	    printf("function not supported\n");
	  default :
	    printf("ah: %d\n", out.h.ah);
	 }
       DoEmulJoy=0;
       return;
      }
    JoyDevice0=0;
    if( GetDefault("C64JOY2",0,0) ) {
	JoyDevice1=1;
    }
}
#endif	/* } __GO32__ */

    /* Buttons koennen fuer Linkshaender - oder falls lieber - vertauscht */
    /* werden. */
    JoyData0.Button1 = 1;     /* 1 oder 2 */
    JoyData0.Button2 = 2;     /* 2 oder 1 */

    JoyData1.Button1 = 1;     /* 1 oder 2 */
    JoyData1.Button2 = 2;     /* 2 oder 1 */

    /*
    **	Get joystick configuration.
    */
    config=GetDefault("C64JOY1",0,0);
    if( config ) {
	if( sscanf(config,"%d,%d,%d,%d",
	    &JoyData0.Up,&JoyData0.Down,&JoyData0.Left,&JoyData0.Right)==4 ) {
#ifdef JOHNS
	    printf("Joy0:%d,%d,%d,%d\n",
		JoyData0.Up,JoyData0.Down,JoyData0.Left,JoyData0.Right);
#endif
	    return;
	}
    }
    config=GetDefault("C64JOY1KEY",0,0);
    if( config ) {
	Joy0Key=0x200|(atol(config)&0x3F);	/* key on button 2 (SPACE) */
    }

    config=GetDefault("C64JOY2",0,0);
    if( config ) {
	if( sscanf(config,"%d,%d,%d,%d",
	    &JoyData1.Up,&JoyData1.Down,&JoyData1.Left,&JoyData1.Right)==4 ) {
#ifdef JOHNS
	    printf("Joy1:%d,%d,%d,%d\n",
		JoyData1.Up,JoyData1.Down,JoyData1.Left,JoyData1.Right);
#endif
	    return;
	}
    }
    config=GetDefault("C64JOY2KEY",0,0);
    if( config ) {
	Joy1Key=0x200|(atol(config)&0x3F);	/* key on button 2 (SPACE) */
    }

    /* Joystick calibration muss angepasst werden:  */

#ifdef JOHNS
    /*  Mein Digitaljoystick: */
    JoyData0.Up    =  350;
    JoyData0.Down  =  980;
    JoyData0.Left  =  360;
    JoyData0.Right = 1024;
#endif

#ifdef MAERZ
    /*  Mein Digitaljoystick: Sensibility 50   */
    JoyData0.Up    =  718;
    JoyData0.Down  = 2036;
    JoyData0.Left  =  767;
    JoyData0.Right = 2168;

    /*  Mein Analogjoystick: Sensibility 10   */
    /* JoyData0.Up    =  399;
    JoyData0.Down  =  512;
    JoyData0.Left  = 1025;
    JoyData0.Right = 1160; */

    /*  Mein Analogjoystick: Sensibility 50   */
    /* JoyData0.Up    =  338;
    JoyData0.Down  =  915;
    JoyData0.Left  =  752;
    JoyData0.Right = 1413; */
#endif
#if !defined(JOHNS) && !defined(MAERZ)
    /* FIXME: DEFAULTS: Was ist gut, What is correct ? */
#ifdef __linux__
    JoyData0.Up    =JoyData1.Up    = 900;
    JoyData0.Left  =JoyData1.Left  = 900;
    JoyData0.Down  =JoyData1.Down  = 1000;
    JoyData0.Right =JoyData1.Right = 1000;
#else
    JoyData0.Up    =JoyData1.Up    = 60;
    JoyData0.Left  =JoyData1.Left  = 60;
    JoyData0.Down  =JoyData1.Down  = 100;
    JoyData0.Right =JoyData1.Right = 100;
#endif
#endif
}

/*
**	Cleanup joystick.
*/
void LeaveJoy (void)
{
#ifdef __linux__
    if( JoyDevice0>=0 ) {
	close (JoyDevice0);
    }
    if( JoyDevice1>=0 ) {
	close (JoyDevice1);
    }
#endif
}

/*
**	Do the joystick emulation, called each frame in the blank.
*/
void EmulJoy(void)
{
    long tmpl;
    struct JS_DATA_TYPE js_data;

    if( JoyDevice0>=0 ) {
	tmpl = 0xFF;
	if (JoyRead (JoyDevice0, &js_data, JS_RETURN) != JS_RETURN) {
	    perror ("jscal");
	} else {
	    if (js_data.y < JoyData0.Up)    tmpl &= ~1;
	    if (js_data.y > JoyData0.Down)  tmpl &= ~2;
	    if (js_data.x < JoyData0.Left)  tmpl &= ~4;
	    if (js_data.x > JoyData0.Right) tmpl &= ~8;
	    if (js_data.buttons & JoyData0.Button1) tmpl &= ~16;
	    if (js_data.buttons & JoyData0.Button2) {
		Joy0Button2=1;
		HandleKey(1,Joy0Key);
	    } else if( Joy0Button2 ) {	/* must release the keys */
		Joy0Button2=0;
		HandleKey(0,Joy0Key);
	    }
	}
	*OtherJoyStick = tmpl;
    }
    if( JoyDevice1>=0 ) {
	tmpl = 0xFF;
	if (JoyRead (JoyDevice1, &js_data, JS_RETURN) != JS_RETURN) {
	    perror ("jscal");
	} else {
	    if (js_data.y < JoyData1.Up)    tmpl &= ~1;
	    if (js_data.y > JoyData1.Down)  tmpl &= ~2;
	    if (js_data.x < JoyData1.Left)  tmpl &= ~4;
	    if (js_data.x > JoyData1.Right) tmpl &= ~8;
	    if (js_data.buttons & JoyData1.Button1) tmpl &= ~16;
	    if (js_data.buttons & JoyData1.Button2) {
		Joy1Button2=1;
		HandleKey(1,Joy1Key);
	    } else if( Joy0Button2 ) {	/* must release the keys */
		Joy1Button2=0;
		HandleKey(0,Joy1Key);
	    }
	}
	*JoyStick = tmpl;
    }
}

#endif	/* } __linux__ || __GO32__ */

#ifdef __GO32__	/* { */
#if 0

#include <stdio.h>
#include <dos.h>

static int Joy0Button2=0;		/* joystick 0 button 2 was pressed */
static int Joy1Button2=0;		/* joystick 1 button 2 was pressed */

struct {int Up, Down, Left, Right, Button1, Button2;} JoyData0;
struct {int Up, Down, Left, Right, Button1, Button2;} JoyData1;

/*
**	Initialise joystick.
*/
void EnterJoy (void)
{
    union REGS    in, out;
    char* config;

    if (!DoEmulJoy) {			/* no emulation wanted */
	return;
    }

    /* Ersma schaun, ob da wirklich n Schoistick is... */

    in.h.ah = 0x84;
    in.x.dx = 0x0000;
    int86(0x15, &in, &out);

    if(out.x.cflag)
      {
       printf("failed\n");
       switch(out.h.ah)
	 {
	  case 0x80 :
	    printf("invalid command\n");
	  case 0x86 :
	    printf("function not supported\n");
	  default :
	    printf("ah: %d\n", out.h.ah);
	 }
       DoEmulJoy=0;
       return;
      }

    /* OK! */

    /* FIXME: The same for the other port? Das selbe fuer den zweiten Port? */

    /* Buttons koennen fuer Linkshaender - oder falls lieber - vertauscht */
    /* werden. */
    JoyData0.Button1 = 1;     /* 1 oder 2 */
    JoyData0.Button2 = 2;     /* 2 oder 1 */

    JoyData1.Button1 = 1;     /* 1 oder 2 */
    JoyData1.Button2 = 2;     /* 2 oder 1 */

    /*
    **	Get joystick configuration.
    */
    config=GetDefault("C64JOY1",0,0);
    if( config ) {
	if( sscanf(config,"%d,%d,%d,%d",
	    &JoyData0.Up,&JoyData0.Down,&JoyData0.Left,&JoyData0.Right)==4 ) {
	    return;
	}
    }
    config=GetDefault("C64JOY2",0,0);
    if( config ) {
	if( sscanf(config,"%d,%d,%d,%d",
	    &JoyData1.Up,&JoyData1.Down,&JoyData1.Left,&JoyData1.Right)==4 ) {
	    return;
	}
    }

    /* FIXME: DEFAULTS: Was ist gut, What is correct ? */
    JoyData0.Up    =JoyData1.Up    = 60;
    JoyData0.Left  =JoyData1.Left  = 60;
    JoyData0.Down  =JoyData1.Down  = 100;
    JoyData0.Right =JoyData1.Right = 100;
}

/*
**	Cleanup joystick.
*/
void LeaveJoy (void)
{
}

/*
**	Do the joystick emulation, called each frame in the blank.
*/
void EmulJoy(void)
{
    long tmpl;
    int x,y;
    int buttons;
    union REGS    in, out;

    if (!DoEmulJoy) {			/* no emulation wanted */
	return;
    }

    in.x.ax = 0x8400;              /* Funktion:     Joystick lesen */
    in.x.dx = 0x0001;              /* Sub-Funktion: Koordinaten lesen */
    int86(0x15, &in, &out);        /* Interrupt ausloesen */

    /* wenn der Joystick erfolgreich ausgelesen werden konnte,
       ist cflag 0 */
    if(out.x.cflag)
      return;				/* Fehler ! */

#if 0 /* Spter :) */
    if(joystick == 1)   /* falls Joystick A gelesen werden sollte */
      {
       *x = out.x.ax;
       *y = out.x.bx;
      }
    else                /* ansonsten B */
      {
       *x = out.x.cx;
       *y = out.x.dx;
      }
#endif

    x = out.h.al&0xFF;
    y = out.h.bl&0xFF;

    in.x.ax = 0x8400;              /* Funktion:     Joystick lesen */
    in.x.dx = 0x0000;              /* Sub-Funktion: ?????????????? */
    int86(0x15, &in, &out);        /* Interrupt ausloesen */

    buttons = ~out.x.ax>>4;	   /* FIXME: Wo sind die Buttons? */

    tmpl = 0xFF;
    if (y < JoyData0.Up)    tmpl &= ~1;
    if (y > JoyData0.Down)  tmpl &= ~2;
    if (x < JoyData0.Left)  tmpl &= ~4;
    if (x > JoyData0.Right) tmpl &= ~8;
    if (buttons & JoyData0.Button1) tmpl &= ~16;
    if (buttons & JoyData0.Button2) {
	Joy0Button2=1;
	HandleKey(1,' ');
	// HandleKey(1,'');
    } else if( Joy0Button2 ) {	/* must release the keys */
	Joy0Button2=0;
	HandleKey(0,' ');
	// HandleKey(0,'');
    }
    *OtherJoyStick = tmpl;

    /* FIXME: The same for the other joystick port! */
}

#endif
#endif	/* } __GO32__ */

#endif	/* } PC_ANALOG_JOY */

/*****************************************************************************/

#if !defined(GAMEPORT_JOY)	\
	&& !defined(LP_JOY) && !defined(PC_ANALOG_JOY)	/* { */

/*
**	Initialise joystick.
*/
void EnterJoy(void)
{
    /* nothing todo */
}

/*
**	Cleanup joystick.
*/
void LeaveJoy(void)
{
    /* nothing todo */
}

/*
**	Do the joystick emulation, called each frame in the blank.
*/
void EmulJoy(void)
{
    /* nothing todo */
}

#endif	/* } no joystick support */
