/*
 *	The C64 emulator
 *
 *	Copyright 1992,93,96 by ALE.
 *	written by Lutz Sammer.
 *
 *	6502/6510 Disassembler
 *-----------------------------------------------------------------------------
 * $Id: dis.c,v 1.8 1996/07/06 18:53:55 johns Exp root $
 * $Log: dis.c,v $
 * Revision 1.8  1996/07/06 18:53:55  johns
 * Disassemble of illegal opcodes, emulator traps.
 *
 * Revision 1.7  1994/12/18  00:32:11  root
 * non-gcc feature macro support, icc support
 *
 * Revision 1.6  1993/09/17  11:27:39  johns
 * support for wide ( > 80 char ) disassembly
 *
 * Revision 1.5  1993/01/05  12:41:24  johns
 * Checkin before applying wuebbel patches
 *
 * Revision 1.4  1992/07/28  19:47:42  johns
 * Name change of 6502 registers.
 *
 * Revision 1.3  1992/07/20  04:15:46  johns
 * RWMem becomes a macro.
 *
 * Revision 1.1  1992/07/11  18:30:17  johns
 * Initial revision
 *
 *-----------------------------------------------------------------------------
 */

#include "c64.h"
#include <stdio.h>

/*---------------------------------------------------------------------------*/

#ifdef __GNUC__

#define RW(adr) \
    ({ unsigned _x=adr,_y; _y=RB(_x)|(RB(_x+1)<<8); _y; })

#else

#define RW(adr) \
    (__utmpvar__=(int)(adr), RB(__utmpvar__)|(RB(__utmpvar__+1)<<8))
    static unsigned __utmpvar__;

#endif

int DisStringLen;			/* disassembler length */

/*---------------------------------------------------------------------------*/

/*
**	Name of emulator traps.
*/
static CONST char* TrapName(int trap)
{
    switch( trap ) {
	case RETURN_EMUL:
	    return "leave";

	case PIEC_TALK:
	    return "k-talk";
	case PIEC_SECTALK:
	    return "k-sectalk";
	case PIEC_UNTALK:
	    return "k-untalk";
	case PIEC_READ:
	    return "k-iecin";
	case PIEC_LISTEN:
	    return "k-listen";
	case PIEC_SECLISTEN:
	    return "k-seclisten";
	case PIEC_UNLISTEN:
	    return "k-unlisten";
	case PIEC_WRITE:
	    return "k-iecout";

	case IEC_TALK:
	    return "TALK";
	case IEC_SECTALK:
	    return "SECTALK";
	case IEC_UNTALK:
	    return "UN-TALK";
	case IEC_READ:
	    return "IEC-IN";
	case IEC_LISTEN:
	    return "LISTEN";
	case IEC_SECLISTEN:
	    return "SECLISTEN";
	case IEC_UNLISTEN:
	    return "UN-LISTEN";
	case IEC_WRITE:
	    return "IECOUT";
	case IEC_STATUS:
	    return "STATUS";

	default:
	    return "unknown illegal";

    }
}

/*
**	Universal 6502 Disassambler
*/
int Disassemble(int ip,int RegX,int RegY,int RegS,int (*RB)(int))
{
    int b;
    char* opc;
    char buf[32];
    char info[32];

#define P_NO()	*buf='\0'; \
		*info='\0'
#define P_AC()	sprintf(buf,"A"); \
		*info='\0'
#define P_IM()	sprintf(buf,"#$%02X",RB(ip)); \
		*info='\0'
#define P_AB()	sprintf(buf,"$%04X",RW(ip)); \
		*info='\0'
#define P_AI()	sprintf(buf,"($%04X)",RW(ip)); \
		*info='\0'
#define P_DP()	sprintf(buf,"$%02X",RB(ip)); \
		*info='\0'
#define P_AX()	sprintf(buf,"$%04X,X",RW(ip)); \
		*info='\0'
#define P_AY()	sprintf(buf,"$%04X,Y",RW(ip)); \
		*info='\0'
#define P_DX()	sprintf(buf,"$%02X,X",RB(ip)); \
		*info='\0'
#define P_DY()	sprintf(buf,"$%02X,Y",RB(ip)); \
		*info='\0'
#define P_IX()	sprintf(buf,"($%02X,X)",RB(ip)); \
		*info='\0'
#define P_IY()	sprintf(buf,"($%02X),Y",RB(ip)); \
		*info='\0'
#define P_R()	sprintf(buf,"%+03d",(char)RB(ip))

#define I_PR()	sprintf(info,"$%04X",RWStack(1+RegS)+1)
#define I_PI()	sprintf(info,"$%04X",RWStack(2+RegS))

#define I_AB()	sprintf(info,"$%02X",RB(RW(ip)))
#define I_AX()	sprintf(info,"[$%04X]=$%02X",RW(ip)+RegX,RB(RW(ip)+RegX))
#define I_AY()	sprintf(info,"[$%04X]=$%02X",RW(ip)+RegY,RB(RW(ip)+RegY))
#define I_AI()	sprintf(info,"$%04X",RB(RW(ip))|(RB((RW(ip)&0xFF00)|((RW(ip)+1)&0x00FF))<<8))
#define I_DP()	sprintf(info,"$%02X",RB(RB(ip)))
#define I_DX()	sprintf(info,"$%02X",RB((RB(ip)+RegX)&0xFF))
#define I_DY()	sprintf(info,"$%02X",RB((RB(ip)+RegY)&0xFF))
#define I_IY()	sprintf(info,"[$%04X]=$%02X",RW(RB(ip))+RegY,RB(RW(RB(ip))+RegY))
#define I_IX()	sprintf(info,"[$%04X]=$%02X",RW((RB(ip)+RegX)&0xFF),RB(RW((RB(ip)+RegX)&0xFF)))
#define I_R()	sprintf(info,"$%04X",ip+1+(char)RB(ip))
#define I_EM()	sprintf(info,"(%s)",TrapName(RB(ip)))

    b=1;
    switch( RB(ip++) ) {
	case ADC_IM:	opc="ADC";	P_IM();		b=2;	break;
	case ADC_A:	opc="ADC";	P_AB();	I_AB();	b=3;	break;
	case ADC_D:	opc="ADC";	P_DP();	I_DP();	b=2;	break;
	case ADC_AX:	opc="ADC";	P_AX();	I_AX();	b=3;	break;
	case ADC_AY:	opc="ADC";	P_AY();	I_AY();	b=3;	break;
	case ADC_DX:	opc="ADC";	P_DX();	I_DX();	b=2;	break;
	case ADC_IX:	opc="ADC";	P_IX();	I_IX();	b=2;	break;
	case ADC_IY:	opc="ADC";	P_IY();	I_IY();	b=2;	break;

	case AND_IM:	opc="AND";	P_IM();		b=2;	break;
	case AND_A:	opc="AND";	P_AB();	I_AB();	b=3;	break;
	case AND_D:	opc="AND";	P_DP();	I_DP();	b=2;	break;
	case AND_AX:	opc="AND";	P_AX();	I_AX();	b=3;	break;
	case AND_AY:	opc="AND";	P_AY();	I_AY();	b=3;	break;
	case AND_DX:	opc="AND";	P_DX();	I_DX();	b=2;	break;
	case AND_IX:	opc="AND";	P_IX();	I_IX();	b=2;	break;
	case AND_IY:	opc="AND";	P_IY();	I_IY();	b=2;	break;

	case BRK:	opc="BRK";	P_NO();		b=1;	break;

	case ILL02:	opc="Emul";	P_IM();	I_EM();	b=2;	break;

	case ASL:	opc="ASL";	P_AC();		b=1;	break;
	case ASL_A:	opc="ASL";	P_AB();	I_AB();	b=3;	break;
	case ASL_D:	opc="ASL";	P_DP();	I_DP();	b=2;	break;
	case ASL_AX:	opc="ASL";	P_AX();	I_AX();	b=3;	break;
	case ASL_DX:	opc="ASL";	P_DX();	I_DX();	b=2;	break;

	case LSR:	opc="LSR";	P_AC();		b=1;	break;
	case LSR_A:	opc="LSR";	P_AB();	I_AB();	b=3;	break;
	case LSR_D:	opc="LSR";	P_DP();	I_DP();	b=2;	break;
	case LSR_AX:	opc="LSR";	P_AX();	I_AX();	b=3;	break;
	case LSR_DX:	opc="LSR";	P_DX();	I_DX();	b=2;	break;

	case ROL:	opc="ROL";	P_AC();		b=1;	break;
	case ROL_A:	opc="ROL";	P_AB();	I_AB();	b=3;	break;
	case ROL_D:	opc="ROL";	P_DP();	I_DP();	b=2;	break;
	case ROL_AX:	opc="ROL";	P_AX();	I_AX();	b=3;	break;
	case ROL_DX:	opc="ROL";	P_DX();	I_DX();	b=2;	break;

	case ROR:	opc="ROR";	P_AC();		b=1;	break;
	case ROR_A:	opc="ROR";	P_AB();	I_AB();	b=3;	break;
	case ROR_D:	opc="ROR";	P_DP();	I_DP();	b=2;	break;
	case ROR_AX:	opc="ROR";	P_AX();	I_AX();	b=3;	break;
	case ROR_DX:	opc="ROR";	P_DX();	I_DX();	b=2;	break;

	case CMP_IM:	opc="CMP";	P_IM();		b=2;	break;
	case CMP_A:	opc="CMP";	P_AB();	I_AB();	b=3;	break;
	case CMP_D:	opc="CMP";	P_DP();	I_DP();	b=2;	break;
	case CMP_AX:	opc="CMP";	P_AX();	I_AX();	b=3;	break;
	case CMP_AY:	opc="CMP";	P_AY();	I_AY();	b=3;	break;
	case CMP_DX:	opc="CMP";	P_DX();	I_DX();	b=2;	break;
	case CMP_IX:	opc="CMP";	P_IX();	I_IX();	b=2;	break;
	case CMP_IY:	opc="CMP";	P_IY();	I_IY();	b=2;	break;

	case EOR_IM:	opc="EOR";	P_IM();		b=2;	break;
	case EOR_A:	opc="EOR";	P_AB();	I_AB();	b=3;	break;
	case EOR_D:	opc="EOR";	P_DP();	I_DP();	b=2;	break;
	case EOR_AX:	opc="EOR";	P_AX();	I_AX();	b=3;	break;
	case EOR_AY:	opc="EOR";	P_AY();	I_AY();	b=3;	break;
	case EOR_DX:	opc="EOR";	P_DX();	I_DX();	b=2;	break;
	case EOR_IX:	opc="EOR";	P_IX();	I_IX();	b=2;	break;
	case EOR_IY:	opc="EOR";	P_IY();	I_IY();	b=2;	break;

	case ORA_IM:	opc="ORA";	P_IM();		b=2;	break;
	case ORA_A:	opc="ORA";	P_AB();	I_AB();	b=3;	break;
	case ORA_D:	opc="ORA";	P_DP();	I_DP();	b=2;	break;
	case ORA_AX:	opc="ORA";	P_AX();	I_AX();	b=3;	break;
	case ORA_AY:	opc="ORA";	P_AY();	I_AY();	b=3;	break;
	case ORA_DX:	opc="ORA";	P_DX();	I_DX();	b=2;	break;
	case ORA_IX:	opc="ORA";	P_IX();	I_IX();	b=2;	break;
	case ORA_IY:	opc="ORA";	P_IY();	I_IY();	b=2;	break;

	case SBC_IM:	opc="SBC";	P_IM();		b=2;	break;
	case SBC_A:	opc="SBC";	P_AB();	I_AB();	b=3;	break;
	case SBC_D:	opc="SBC";	P_DP();	I_DP();	b=2;	break;
	case SBC_AX:	opc="SBC";	P_AX();	I_AX();	b=3;	break;
	case SBC_AY:	opc="SBC";	P_AY();	I_AY();	b=3;	break;
	case SBC_DX:	opc="SBC";	P_DX();	I_DX();	b=2;	break;
	case SBC_IX:	opc="SBC";	P_IX();	I_IX();	b=2;	break;
	case SBC_IY:	opc="SBC";	P_IY();	I_IY();	b=2;	break;

	case LDA_IM:	opc="LDA";	P_IM();		b=2;	break;
	case LDA_A:	opc="LDA";	P_AB();	I_AB();	b=3;	break;
	case LDA_D:	opc="LDA";	P_DP();	I_DP();	b=2;	break;
	case LDA_AX:	opc="LDA";	P_AX();	I_AX();	b=3;	break;
	case LDA_AY:	opc="LDA";	P_AY();	I_AY();	b=3;	break;
	case LDA_DX:	opc="LDA";	P_DX();	I_DX();	b=2;	break;
	case LDA_IX:	opc="LDA";	P_IX();	I_IX();	b=2;	break;
	case LDA_IY:	opc="LDA";	P_IY();	I_IY();	b=2;	break;

	case LDX_IM:	opc="LDX";	P_IM();		b=2;	break;
	case LDX_A:	opc="LDX";	P_AB();	I_AB();	b=3;	break;
	case LDX_D:	opc="LDX";	P_DP();	I_DP();	b=2;	break;
	case LDX_AY:	opc="LDX";	P_AY();	I_AY();	b=3;	break;
	case LDX_DY:	opc="LDX";	P_DY();	I_DY();	b=2;	break;

	case LDY_IM:	opc="LDY";	P_IM();		b=2;	break;
	case LDY_A:	opc="LDY";	P_AB();	I_AB();	b=3;	break;
	case LDY_D:	opc="LDY";	P_DP();	I_DP();	b=2;	break;
	case LDY_AX:	opc="LDY";	P_AX();	I_AX();	b=3;	break;
	case LDY_DX:	opc="LDY";	P_DX();	I_DX();	b=2;	break;

	case STA_A:	opc="STA";	P_AB();	I_AB();	b=3;	break;
	case STA_D:	opc="STA";	P_DP();	I_DP();	b=2;	break;
	case STA_AX:	opc="STA";	P_AX();	I_AX();	b=3;	break;
	case STA_AY:	opc="STA";	P_AY();	I_AY();	b=3;	break;
	case STA_DX:	opc="STA";	P_DX();	I_DX();	b=2;	break;
	case STA_IX:	opc="STA";	P_IX();	I_IX();	b=2;	break;
	case STA_IY:	opc="STA";	P_IY();	I_IY();	b=2;	break;

	case STX_A:	opc="STX";	P_AB();	I_AB();	b=3;	break;
	case STX_D:	opc="STX";	P_DP();	I_DP();	b=2;	break;
	case STX_DY:	opc="STX";	P_DY();	I_DY();	b=2;	break;

	case STY_A:	opc="STY";	P_AB();	I_AB();	b=3;	break;
	case STY_D:	opc="STY";	P_DP();	I_DP();	b=2;	break;
	case STY_DX:	opc="STY";	P_DX();	I_DX();	b=2;	break;

	case DEC_A:	opc="DEC";	P_AB();	I_AB();	b=3;	break;
	case DEC_D:	opc="DEC";	P_DP();	I_DP();	b=2;	break;
	case DEC_AX:	opc="DEC";	P_AX();	I_AX();	b=3;	break;
	case DEC_DX:	opc="DEC";	P_DX();	I_DX();	b=2;	break;

	case INC_A:	opc="INC";	P_AB();	I_AB();	b=3;	break;
	case INC_D:	opc="INC";	P_DP();	I_DP();	b=2;	break;
	case INC_AX:	opc="INC";	P_AX();	I_AX();	b=3;	break;
	case INC_DX:	opc="INC";	P_DX();	I_DX();	b=2;	break;

	case DEX:	opc="DEX";	P_NO();		b=1;	break;
	case DEY:	opc="DEY";	P_NO();		b=1;	break;

	case INX:	opc="INX";	P_NO();		b=1;	break;
	case INY:	opc="INY";	P_NO();		b=1;	break;

	case BCC:	opc="BCC";	P_R();	I_R();	b=2;	break;
	case BCS:	opc="BCS";	P_R();	I_R();	b=2;	break;
	case BEQ:	opc="BEQ";	P_R();	I_R();	b=2;	break;
	case BNE:	opc="BNE";	P_R();	I_R();	b=2;	break;
	case BMI:	opc="BMI";	P_R();	I_R();	b=2;	break;
	case BPL:	opc="BPL";	P_R();	I_R();	b=2;	break;
	case BVC:	opc="BVC";	P_R();	I_R();	b=2;	break;
	case BVS:	opc="BVS";	P_R();	I_R();	b=2;	break;

	case BIT_A:	opc="BIT";	P_AB();	I_AB();	b=3;	break;
	case BIT_D:	opc="BIT";	P_DP();	I_DP();	b=2;	break;

	case NOP:	opc="NOP";	P_NO();		b=1;	break;
	case RTI:	opc="RTI";	P_NO();	I_PI();	b=1;	break;
	case RTS:	opc="RTS";	P_NO();	I_PR();	b=1;	break;
	case TAX:	opc="TAX";	P_NO();		b=1;	break;
	case TAY:	opc="TAY";	P_NO();		b=1;	break;
	case TSX:	opc="TSX";	P_NO();		b=1;	break;
	case TXA:	opc="TXA";	P_NO();		b=1;	break;
	case TXS:	opc="TXS";	P_NO();		b=1;	break;
	case TYA:	opc="TYA";	P_NO();		b=1;	break;

	case CLC:	opc="CLC";	P_NO();		b=1;	break;
	case CLD:	opc="CLD";	P_NO();		b=1;	break;
	case CLI:	opc="CLI";	P_NO();		b=1;	break;
	case CLV:	opc="CLV";	P_NO();		b=1;	break;

	case SEC:	opc="SEC";	P_NO();		b=1;	break;
	case SED:	opc="SED";	P_NO();		b=1;	break;
	case SEI:	opc="SEI";	P_NO();		b=1;	break;

	case JMP_A:	opc="JMP";	P_AB();		b=3;	break;
	case JMP_AI:	opc="JMP";	P_AI();	I_AI();	b=3;	break;
	case JSR:	opc="JSR";	P_AB();		b=3;	break;

	case PHA:	opc="PHA";	P_NO();		b=1;	break;
	case PHP:	opc="PHP";	P_NO();		b=1;	break;
	case PLA:	opc="PLA";	P_NO();		b=1;	break;
	case PLP:	opc="PLP";	P_NO();		b=1;	break;

	case CPX_IM:	opc="CPX";	P_IM();		b=2;	break;
	case CPX_A:	opc="CPX";	P_AB();	I_AB();	b=3;	break;
	case CPX_D:	opc="CPX";	P_DP();	I_DP();	b=2;	break;

	case CPY_IM:	opc="CPY";	P_IM();		b=2;	break;
	case CPY_A:	opc="CPY";	P_AB();	I_AB();	b=3;	break;
	case CPY_D:	opc="CPY";	P_DP();	I_DP();	b=2;	break;

	case ILL03:	opc="ASLORA";	P_IX();	I_IX();	b=2;	break;
	case ILL07:	opc="ASLORA";	P_DP();	I_DP();	b=2;	break;
	case ILL0F:	opc="ASLORA";	P_AB();	I_AB();	b=3;	break;
	case ILL13:	opc="ASLORA";	P_IY();	I_IY();	b=2;	break;
	case ILL17:	opc="ASLORA";	P_DX();	I_DX();	b=2;	break;
	case ILL1B:	opc="ASLORA";	P_AY();	I_AY();	b=3;	break;
	case ILL1F:	opc="ASLORA";	P_AX();	I_AX();	b=3;	break;

	case ILL23:	opc="ROLAND";	P_IX();	I_IX();	b=2;	break;
	case ILL27:	opc="ROLAND";	P_DP(); I_DP();	b=2;	break;
	case ILL2F:	opc="ROLAND";	P_AB();	I_AB();	b=3;	break;
	case ILL33:	opc="ROLAND";	P_IY();	I_IY();	b=2;	break;
	case ILL37:	opc="ROLAND";	P_DX();	I_DX();	b=2;	break;
	case ILL3B:	opc="ROLAND";	P_AY();	I_AY();	b=3;	break;
	case ILL3F:	opc="ROLAND";	P_AX();	I_AX();	b=3;	break;

	case ILL43:	opc="LSREOR";	P_IX();	I_IX();	b=2;	break;
	case ILL47:	opc="LSREOR";	P_DP();	I_DP();	b=2;	break;
	case ILL4F:	opc="LSREOR";	P_AB();	I_AB();	b=3;	break;
	case ILL53:	opc="LSREOR";	P_IY();	I_IY();	b=2;	break;
	case ILL57:	opc="LSREOR";	P_DX();	I_DX();	b=2;	break;
	case ILL5B:	opc="LSREOR";	P_AY();	I_AY();	b=3;	break;
	case ILL5F:	opc="LSREOR";	P_AX();	I_AX();	b=3;	break;

	case ILL63:	opc="RORADC";	P_IX(); I_IX();	b=2;	break;
	case ILL67:	opc="RORADC";	P_DP(); I_DP();	b=2;	break;
	case ILL6F:	opc="RORADC";	P_AB(); I_AB();	b=3;	break;
	case ILL73:	opc="RORADC";	P_IY(); I_IY();	b=2;	break;
	case ILL77:	opc="RORADC";	P_DX(); I_DX();	b=2;	break;
	case ILL7B:	opc="RORADC";	P_AY(); I_AY();	b=3;	break;
	case ILL7F:	opc="RORADC";	P_AX(); I_AX();	b=3;	break;

	case ILLA3:	opc="LDAX";	P_IX();	I_IX();	b=2;	break;
	case ILLA7:	opc="LDAX";	P_DP(); I_DP();	b=2;	break;
	case ILLAF:	opc="LDAX";	P_AB(); I_AB();	b=3;	break;
	case ILLB3:	opc="LDAX";	P_IY(); I_IY();	b=2;	break;
	case ILLB7:	opc="LDAX";	P_DY(); I_DY();	b=2;	break;
	case ILLBF:	opc="LDAX";	P_AY(); I_AY();	b=3;	break;

	case ILLC3:	opc="DECCMP";	P_IX();	I_IX();	b=2;	break;
	case ILLC7:	opc="DECCMP";	P_DP();	I_DP();	b=2;	break;
	case ILLCF:	opc="DECCMP";	P_AB();	I_AB();	b=3;	break;
	case ILLD3:	opc="DECCMP";	P_IY();	I_IY();	b=2;	break;
	case ILLD7:	opc="DECCMP";	P_DX();	I_DX();	b=2;	break;
	case ILLDB:	opc="DECCMP";	P_AY();	I_AY();	b=3;	break;
	case ILLDF:	opc="DECCMP";	P_AX(); I_AX();	b=3;	break;

	case ILLE3:	opc="INCSBC";	P_IX();		b=2;	break;
	case ILLE7:	opc="INCSBC";	P_DP();		b=2;	break;
	case ILLEF:	opc="INCSBC";	P_AB();		b=3;	break;
	case ILLF3:	opc="INCSBC";	P_IY();		b=2;	break;
	case ILLF7:	opc="INCSBC";	P_DX();		b=2;	break;
	case ILLFB:	opc="INCSBC";	P_AY();		b=3;	break;
	case ILLFF:	opc="INCSBC";	P_AX();		b=3;	break;

	case ILL0B:	opc="ANDN->C";	P_IM();		b=2;	break;
	case ILL4B:	opc="ANDLSR";	P_IM();		b=2;	break;
	case ILL87:	opc="STANDX";	P_DP();		b=2;	break;
	case ILL8F:	opc="STANDX";	P_AB();		b=3;	break;
	case ILL97:	opc="STANDX";	P_DY();	I_DY();	b=2;	break;
	case ILL9B:	opc="SHS:A&X->S";P_AY(); I_AY();b=3;	break;
	case ILL9C:	opc="ANDY+HI+1";	P_AX();	b=3;	break;
	case ILL9E:	opc="ANDX+HI+1";	P_AY();	b=3;	break;
	case ILL9F:	opc="ANDA+X+HI+1";	P_AY();	b=3;	break;
	case ILLAB:	opc="ANDTAX";	P_IM();		b=2;	break;
	case ILLEB:	opc="SUB";	P_IM();		b=2;	break;

	case ILL1A: case ILL3A: case ILL5A:
	case ILL7A: case ILLDA: case ILLFA:
			opc="NOOP";	P_NO();		b=1;	break;
	case ILL80: case ILL82: case ILL89:
	case ILLC2: case ILLE2:
	case ILL04: case ILL44: case ILL64:
	case ILL14: case ILL34: case ILL54:
	case ILL74: case ILLD4: case ILLF4:
			opc="NOOP";	P_IM();		b=2;	break;
	case ILL0C:
	case ILL1C: case ILL3C: case ILL5C:
	case ILL7C: case ILLDC: case ILLFC:
			opc="NOOP";	P_AB();		b=3;	break;
	/*
	default:	Cleanup(); abort();		break;
	*/
	default:	opc="ILL";	P_NO();		b=1;	break;
    }

    printf("%s\t%s",opc,buf);
    DisStringLen = strlen(opc);
    DisStringLen = ((DisStringLen + 7)/8)*8;		/* calc tab */
    DisStringLen += strlen(buf);
	
    if( *info ) {
	DisStringLen = ((DisStringLen + 7)/8)*8 + 1;	/* calc tab + '=' */
	printf("\t=%s",info);
	DisStringLen += strlen(info);
    }
    return b;
}

/*---------------------------------------------------------------------------*/

/*
**	FIXME: Should become a nondestructive C64 memory read.
*/
static int C64RBMem(int adr)
{
    return RBMem(adr);
}

/*
**	C64 disassembler
*/
int Dis(int ip)
{
    return Disassemble(
	ip,
	Integer(GlobalRegX),
	Integer(GlobalRegY),
	Integer(GlobalRegS),
	C64RBMem);
}
