/*
 *	The C64 emulator
 *
 *	Copyright 1992-96 by ALE.
 *	written by Lutz Sammer.
 *
 *	Main header file
 *-----------------------------------------------------------------------------
 * $Id: c64.h,v 1.19 1996/07/01 21:38:52 johns Exp root $
 * $Log: c64.h,v $
 * Revision 1.19  1996/07/01 21:38:52  johns
 * Moved emulator traps to 6502.h and added 1541 prototypes.
 *
 * Revision 1.18  1996/06/12 23:55:43  ari
 * Moved PCSOLARIS autodection to config.h.
 *
 * Revision 1.17  1996/06/12 23:27:21  johns
 * Moved sid prototypes and defines to own header file.
 *
 * Revision 1.16  1996/05/12 20:48:55  johns
 * Cleanup, Started new-memory interface, many changes.
 *
 * Revision 1.15  1995/06/01  12:08:10  ari
 * new version
 *
 * Revision 1.14  1995/02/06  23:56:41  johns
 * new minor 1.0.8
 *
 * Revision 1.13  1994/12/18  00:32:11  root
 * non-gcc feature macro support, icc support
 *
 * Revision 1.12  1994/06/02  14:56:27  johns
 * version 1.07 ci
 *
 * Revision 1.11  1993/10/02  13:14:49  root
 * new version (1.05) for GO32 dirfile enforcement
 *
 * Revision 1.10  1993/09/04  14:25:19  johns
 * new version (1.04) for dirfile support
 *
 * Revision 1.9  1993/08/31  20:30:45  johns
 * go32 initial support
 *
 * Revision 1.8  1993/06/13  12:44:53  johns
 * new version 1.03
 *
 * Revision 1.7  1993/06/13  12:25:22  johns
 * new version
 *
 * Revision 1.6  1993/01/05  12:42:43  johns
 * Checkin before wuebbel patches
 *
 * Revision 1.5  1992/07/28  19:47:42  johns
 * Global argument parsing to config parsing.
 *
 * Revision 1.4  1992/07/20  04:30:20  johns
 * R3Mem, RLMem rewritten for GNU compiler bug.
 * All other from changes in other files.
 *
 * Revision 1.3  1992/07/13  04:36:26  johns
 * R3Mem opcode fetch for big endian.
 *
 * Revision 1.1  1992/07/11  18:24:04  johns
 * Initial revision
 *
 *-----------------------------------------------------------------------------
 */

#ifndef __C64_H__	/* { */
#define __C64_H__

#define C64Version "The C64 Emulator Version 1.13"

/*
**	Config:
*/

#include "config.h"			/* configuration */

#include "md.h"

/*
**	Non public config:
*/
/* DIDN'T improve anything */
#define noNEW_ACTION			/* testing new action-handling */

/*
**	Emulator registers + memory.
*/
extern	unsigned char	Ram[64*1024];
extern	unsigned char	BasicRom[8*1024];
extern	unsigned char	CharacterRom[4*1024];
extern	unsigned char	KernelRom[8*1024];

extern	unsigned char	VIC[47];
extern	unsigned char	SID[29];
extern	unsigned char	ColorRam[1024+80];
extern	unsigned char	CIA1[16];
extern	unsigned char	CIA2[16];

extern	unsigned char	KeyMatrix[256];	/* C64- Key pressed matrix */
extern	unsigned char	*AutoFeedStr;	/* string of simulated keypresses */

extern	int		AutoFire;
extern	unsigned	JoyStick1;
extern	unsigned	JoyStick2;
extern	int		JoystickGiven;
extern	unsigned*	JoyStick;
extern	unsigned*	OtherJoyStick;

extern	int		Cia1TimerA;
extern	unsigned	Cia1TimerA_Action;
extern	int		Cia1TimerB;
extern	unsigned	Cia1TimerB_Action;
extern	int		Cia1IrqMask;
extern	int		Cia1TodBuffered;
extern	int		Cia1TodStopped;
extern	unsigned char	Cia1Time[4];	/* Time buffer */
extern	unsigned char	Cia1Alarm[4];	/* Alarm time */

extern	int		Cia2TimerA;
extern	unsigned	Cia2TimerA_Action;
extern	int		Cia2TimerB;
extern	unsigned	Cia2TimerB_Action;
extern	int		Cia2IrqMask;
extern	int		Cia2TodBuffered;
extern	int		Cia2TodStopped;
extern	unsigned char	Cia2Time[4];	/* Time buffer */
extern	unsigned char	Cia2Alarm[4];	/* Alarm time */

extern	int		FastLoaderOff;	/* fast-loader support turned off */

/*---------------------------------------------------------------------------*/

/*
**	Emulator traps.
**		Special instructions, for emulation.
*/

enum emul_traps {
    RETURN_EMUL,			/* leave emulator */

    PIEC_TALK,				/* kernal patches IEC-FUNCTIONS */
    PIEC_SECTALK,
    PIEC_UNTALK,
    PIEC_READ,
    PIEC_LISTEN,
    PIEC_SECLISTEN,
    PIEC_UNLISTEN,
    PIEC_WRITE,

    IEC_TALK,				/* file patches IEC-FUNCTION */
    IEC_SECTALK,
    IEC_UNTALK,
    IEC_READ,
    IEC_LISTEN,
    IEC_SECLISTEN,
    IEC_UNLISTEN,
    IEC_WRITE,
    IEC_STATUS,
};

/*---------------------------------------------------------------------------*/

/*
**	Hardware	c64.c
*/
extern	void	DisplayHead	(void);
extern	void	DisplayCia1	(void);
extern	void	DisplayCia2	(void);
extern	void	DisplaySid	(void);
extern	void	DisplayVic	(void);
extern	void	EmulCia1Tod	(void);
extern	void	EmulCia2Tod	(void);
extern	void	EmulatorTrap	(int);
extern	void	Cleanup		(void);
extern	void	Exit		(int);
extern	void	UpdateAction	(void);
extern	void	FullHardwareReset(void);
extern	void	HardwareReset	(void);
extern	void	PatchKernel	(void);
extern	char*	GetDefault	(CONST char*,char*,char*);

/*
**	Disassembler	dis.c
*/
extern	int	Disassemble	(int,int,int,int,int (*)(int));
extern	int	Dis		(int);

#ifdef MONITOR	/* { */

/*
**	Monitor:	mon.c
*/
extern	unsigned MonAction;
extern	void	(*MonFunction)	(void);
extern	void	MonitorOn	(void);
extern	void	SaveScreen	(void);

#endif	/* } MONITOR */

/*
**	6502/6510:	6502.c
*/
#include "6502.h"

extern	void	UpdateFlags	(void);
extern	void	Reset		(void);
extern	void	Nmi		(void);
extern	void	Irq		(void);
extern	void	Emul6502Loop	(void);

/*
**	IEC Emulation:	iec.c
*/
extern	int	IEC_ReadPortTrap	(void);
extern	void	IEC_WritePortTrap	(int);

/*
**	1541:		1541.c
*/
extern int IEC_Status;
extern char* FloppyImage;

extern void Init_IECDos(void);
extern void IEC_Talk(int);
extern void IEC_SEC_Talk(int);
extern void IEC_Untalk(void);
extern int IEC_Read(void);
extern void IEC_Listen(int);
extern void IEC_SEC_Listen(int);
extern void IEC_Unlisten(void);
extern void IEC_Write(int);

extern int cmd_go(CONST char*);

/*
**	1541:		real1541.c
*/
extern int SerialTimeout;		/* timeout for restart of 1541 */

extern int R1541Dis(int);		/* 6502 disassembler */
extern void R1541Reg(void);		/* 6502 register show */

extern void ResetReal1541(void);
extern void EnterReal1541(void);
extern void LeaveReal1541(void);
extern void EmulReal1541(void);

/*
**	Keyboard:	key.c
*/
extern	void	ShowLeds	(int);
extern	void	EnterKey	(void);
extern	void	LeaveKey	(void);
extern	void	SuspendKeyboard	(void);
extern	void	ResumeKeyboard	(void);
extern	void	EmulKey		(void);

extern	void	HandleKey	(int,int);
extern	void	EmulKeyboard	(void);

/*
**	Joystick	joy.c
*/
extern	void	EnterJoy	(void);
extern	void	LeaveJoy	(void);
extern	void	EmulJoy		(void);

/*---------------------------------------------------------------------------*/

enum CiaReg {
    CIA_PRA,
    CIA_PRB,
    CIA_DDRA,
    CIA_DDRB,
    CIA_TALO,
    CIA_TAHI,
    CIA_TBLO,
    CIA_TBHI,
    CIA_TOD10THS,
    CIA_TODSEC,
    CIA_TODMIN,
    CIA_TODHR,
    CIA_SDR,
    CIA_ICR,
    CIA_CRA,
    CIA_CRB
};

/*---------------------------------------------------------------------------*/

#define ChangeAction(a,f)	UpdateAction()
#define DeleteAction(a,f)	UpdateAction()
#define InsertAction(a,f)	UpdateAction()

/*---------------------------------------------------------------------------*/

/*
**	C64 Memory addresses.
*/

#define ADR_EXTERNAL	0x8000		/* External ROM */
#define ADR_BASIC	0xA000		/* Basic ROM */
#define ADR_VIC		0xD000		/* Vic-chip */
#ifdef ALE_RAM
#   define ADR_ALE_RAM1	0xD200		/* Ale additional ram */
#endif
#define ADR_SID		0xD400		/* Sid-chip */
#ifdef ALE_RAM
#   define ADR_ALE_RAM2	0xD600		/* Ale additional ram */
#endif
#define ADR_CRAM	0xD800		/* Color Ram */
#define ADR_CIA1	0xDC00		/* Cia1-chip */
#define ADR_CIA2	0xDD00		/* Cia2-chip */
#define ADR_KERNEL	0xE000		/* Kernel ROM */

/*
**	Configuration:	Memory access tokens.
*/
enum config {
    C_PDIR,				/* Processor port direction */
    C_PDATA,				/* Processor port data */
    C_ZERO,				/* Read RAM for 64K round about */
    C_RW_RAM,				/* Read RAM		- Write RAM */
    C_BROM,				/* Read BASIC ROM	- Write RAM */
    C_KROM,				/* Read KERNEL ROM	- Write RAM */
    C_CROM,				/* Read Character ROM	- Write Ram */

    C_VIC00,				/* VIC Register 00 */
    C_VIC01,				/* VIC Register 01 */
    C_VIC02,				/* VIC Register 02 */
    C_VIC03,				/* VIC Register 03 */
    C_VIC04,				/* VIC Register 04 */
    C_VIC05,				/* VIC Register 05 */
    C_VIC06,				/* VIC Register 06 */
    C_VIC07,				/* VIC Register 07 */
    C_VIC08,				/* VIC Register 08 */
    C_VIC09,				/* VIC Register 09 */
    C_VIC0A,				/* VIC Register 0A */
    C_VIC0B,				/* VIC Register 0B */
    C_VIC0C,				/* VIC Register 0C */
    C_VIC0D,				/* VIC Register 0D */
    C_VIC0E,				/* VIC Register 0E */
    C_VIC0F,				/* VIC Register 0F */
    C_VIC10,				/* VIC Register 10 */
    C_VIC11,				/* VIC Register 11 */
    C_VIC12,				/* VIC Register 12 */
    C_VIC13,				/* VIC Register 13 */
    C_VIC14,				/* VIC Register 14 */
    C_VIC15,				/* VIC Register 15 */
    C_VIC16,				/* VIC Register 16 */
    C_VIC17,				/* VIC Register 17 */
    C_VIC18,				/* VIC Register 18 */
    C_VIC19,				/* VIC Register 19 */
    C_VIC1A,				/* VIC Register 1A */
    C_VIC1B,				/* VIC Register 1B */
    C_VIC1C,				/* VIC Register 1C */
    C_VIC1D,				/* VIC Register 1D */
    C_VIC1E,				/* VIC Register 1E */
    C_VIC1F,				/* VIC Register 1F */
    C_VIC20,				/* VIC Register 20 */
    C_VIC21,				/* VIC Register 21 */
    C_VIC22,				/* VIC Register 22 */
    C_VIC23,				/* VIC Register 23 */
    C_VIC24,				/* VIC Register 24 */
    C_VIC25,				/* VIC Register 25 */
    C_VIC26,				/* VIC Register 26 */
    C_VIC27,				/* VIC Register 27 */
    C_VIC28,				/* VIC Register 28 */
    C_VIC29,				/* VIC Register 29 */
    C_VIC2A,				/* VIC Register 2A */
    C_VIC2B,				/* VIC Register 2B */
    C_VIC2C,				/* VIC Register 2C */
    C_VIC2D,				/* VIC Register 2D */
    C_VIC2E,				/* VIC Register 2E */

    C_SID00,				/* SID Register 00 */
    C_SID01,				/* SID Register 01 */
    C_SID02,				/* SID Register 02 */
    C_SID03,				/* SID Register 03 */
    C_SID04,				/* SID Register 04 */
    C_SID05,				/* SID Register 05 */
    C_SID06,				/* SID Register 06 */
    C_SID07,				/* SID Register 07 */
    C_SID08,				/* SID Register 08 */
    C_SID09,				/* SID Register 09 */
    C_SID0A,				/* SID Register 0A */
    C_SID0B,				/* SID Register 0B */
    C_SID0C,				/* SID Register 0C */
    C_SID0D,				/* SID Register 0D */
    C_SID0E,				/* SID Register 0E */
    C_SID0F,				/* SID Register 0F */
    C_SID10,				/* SID Register 10 */
    C_SID11,				/* SID Register 11 */
    C_SID12,				/* SID Register 12 */
    C_SID13,				/* SID Register 13 */
    C_SID14,				/* SID Register 14 */
    C_SID15,				/* SID Register 15 */
    C_SID16,				/* SID Register 16 */
    C_SID17,				/* SID Register 17 */
    C_SID18,				/* SID Register 18 */
    C_SID19,				/* SID Register 19 */
    C_SID1A,				/* SID Register 1A */
    C_SID1B,				/* SID Register 1B */
    C_SID1C,				/* SID Register 1C */

    C_CRAM,				/* Color ram */

    C_CIA1_0,				/* CIA1 Register 00 */
    C_CIA1_1,				/* CIA1 Register 01 */
    C_CIA1_2,				/* CIA1 Register 02 */
    C_CIA1_3,				/* CIA1 Register 03 */
    C_CIA1_4,				/* CIA1 Register 04 */
    C_CIA1_5,				/* CIA1 Register 05 */
    C_CIA1_6,				/* CIA1 Register 06 */
    C_CIA1_7,				/* CIA1 Register 07 */
    C_CIA1_8,				/* CIA1 Register 08 */
    C_CIA1_9,				/* CIA1 Register 09 */
    C_CIA1_A,				/* CIA1 Register 0A */
    C_CIA1_B,				/* CIA1 Register 0B */
    C_CIA1_C,				/* CIA1 Register 0C */
    C_CIA1_D,				/* CIA1 Register 0D */
    C_CIA1_E,				/* CIA1 Register 0E */
    C_CIA1_F,				/* CIA1 Register 0F */

    C_CIA2_0,				/* CIA2 Register 00 */
    C_CIA2_1,				/* CIA2 Register 01 */
    C_CIA2_2,				/* CIA2 Register 02 */
    C_CIA2_3,				/* CIA2 Register 03 */
    C_CIA2_4,				/* CIA2 Register 04 */
    C_CIA2_5,				/* CIA2 Register 05 */
    C_CIA2_6,				/* CIA2 Register 06 */
    C_CIA2_7,				/* CIA2 Register 07 */
    C_CIA2_8,				/* CIA2 Register 08 */
    C_CIA2_9,				/* CIA2 Register 09 */
    C_CIA2_A,				/* CIA2 Register 0A */
    C_CIA2_B,				/* CIA2 Register 0B */
    C_CIA2_C,				/* CIA2 Register 0C */
    C_CIA2_D,				/* CIA2 Register 0D */
    C_CIA2_E,				/* CIA2 Register 0E */
    C_CIA2_F,				/* CIA2 Register 0F */

    C_FREE,				/* free i/o address */

#ifdef ALE_RAM
    C_ALE_RAM1,				/* Ale ram at D200 */
    C_ALE_RAM2,				/* Ale ram at D600 */
#endif

    C_MAX				/* Number of tokens */
};

#if defined(USE_REGPARM) && defined(__GNUC__)
/* FIXME: move to md.h */
/*
**	Support for register parameter.
*/
#define REGPARM1 __attribute__((regparm (1)))
#define REGPARM2 __attribute__((regparm (2)))

#else

#define REGPARM1
#define REGPARM2

#endif

typedef int REGPARM1 regparm1(int);
typedef void REGPARM2 regparm2(int,int);

#ifdef GLOBAL_ARGS

extern	int (*R_Config[C_MAX])(void);	/* Read  memory */
extern	void (*W_Config[C_MAX])(void);	/* Write memory */

#else

extern	regparm1 *R_Config[C_MAX];	/* Read  memory */
extern	regparm2 *W_Config[C_MAX];	/* Write memory */

#endif

extern	unsigned char* Config;		/* Current memory configuration */
extern	unsigned char* CONST ConfigTable[8];	/* Config table */

/*
**	Change config, if written to processor port.
*/
#define ChangeConfig(x) \
    (Config=ConfigTable[x])

/* FIXME: use of GNU features */

/*
**	Memory access
*/
#ifdef GLOBAL_ARGS

extern int GlobalAdr;
extern int GlobalVal;

#define RBMem(adr) \
    R_Config[Config[GlobalAdr=(adr)]]()

#define WBMem(adr,val) \
    do{ GlobalVal=(val); W_Config[Config[GlobalAdr=(adr)]](); }while(0)

#else

#ifdef __GNUC__
#define RBMem(adr) \
    ({	int rx=adr; R_Config[Config[rx]](rx); })
#else
#define RBMem(adr) \
    ( __tmpvar__=(int)(adr), R_Config[Config[__tmpvar__]](__tmpvar__))
#endif

#define WBMem(adr,val) \
    do{ int wx=adr; W_Config[Config[wx]](wx,val); }while(0)

#endif

/*
**	Instruction fetch.
*/
#ifdef BIG_ENDIAN	/* { */

#define R3Mem(adr,to)	\
    {						\
	if( Config[adr]==C_RW_RAM ) {		\
	    to[0]=Ram[adr+0];			\
	    to[1]=Ram[adr+1];			\
	    to[2]=Ram[adr+2];			\
	} else if( Config[adr]==C_KROM ) {	\
	    to[0]=KernelRom[adr+0-ADR_KERNEL];	\
	    to[1]=KernelRom[adr+1-ADR_KERNEL];	\
	    to[2]=KernelRom[adr+2-ADR_KERNEL];	\
	} else if( Config[adr]==C_BROM ) {	\
	    to[0]=BasicRom[adr+0-ADR_BASIC];	\
	    to[1]=BasicRom[adr+1-ADR_BASIC];	\
	    to[2]=BasicRom[adr+2-ADR_BASIC];	\
	} else {				\
	    to[0]=RBMem(adr+0);			\
	    to[1]=RBMem(adr+1);			\
	    to[2]=RBMem(adr+2);			\
	}					\
    }

#endif	/* } BIG_ENDIAN */

#ifdef LITTLE_ENDIAN	/* { */

#define RLMem(adr,val) \
    {									\
	if( Config[adr]==C_RW_RAM )					\
	    val=*(unsigned long*)(Ram+adr);				\
	else if( Config[adr]==C_KROM )					\
	    val=*(unsigned long*)(KernelRom-ADR_KERNEL+adr);		\
	else if( Config[adr]==C_BROM )					\
	    val=*(unsigned long*)(BasicRom-ADR_BASIC+adr);		\
	else								\
	    val=RBMem(adr+0)|(RBMem(adr+1)<<8)|(RBMem(adr+2)<<16);	\
    }

#endif	/* } LITTLE_ENDIAN */

/*
**	Processor vector fetch.
*/
#ifdef BIG_ENDIAN	/* { */

#define RWVec(adr) \
    ({	int vx=adr;						\
	int val;						\
								\
	if( Config[vx]==C_RW_RAM )				\
	    val=Ram[vx]|(Ram[vx+1]<<8);				\
	else							\
	    val=KernelRom[vx-ADR_KERNEL]			\
		    |(KernelRom[vx-ADR_KERNEL+1]<<8);		\
	val;							\
    })

#endif	/* } BIG_ENDIAN */

#ifdef LITTLE_ENDIAN	/* { */

#ifdef __GNUC__
#define RWVec(adr) \
    ({	int vx=adr;						\
	int val;						\
								\
	if( Config[vx]==C_RW_RAM )				\
	    val=*(unsigned short*)(Ram+vx);			\
	else							\
	    val=*(unsigned short*)(KernelRom+vx-ADR_KERNEL);	\
	val;							\
    })
#else
#define RWVec(adr) \
    (__tmpvar__=(int)(adr), Config[__tmpvar__] == C_RW_RAM ?	\
	*(unsigned short*)(Ram+__tmpvar__) :			\
	*(unsigned short*)(KernelRom+__tmpvar__-ADR_KERNEL))
#endif

#endif	/* } LITTLE_ENDIAN */

/*-----------------------------------------------------------------------------
 *	Memory interface
 *---------------------------------------------------------------------------*/

#ifdef NEW_MEMORY

#define MEM_BANK_SIZE	4096		/* size of memory segments */
#define MEM_BANKS	32		/* number of segments */

extern	regparm1 *RdConfig[8][MEM_BANKS];	/* Read  memory */
extern	regparm2 *WrConfig[8][MEM_BANKS];	/* Write memory */

extern	regparm1 (*(*RdIt)[MEM_BANKS]);		/* Read  memory */
extern	regparm2 (*(*WrIt)[MEM_BANKS]);		/* Write memory */

/*
**	Read a byte from memory with current configuration.
*/
#if 0
#define RB_MEM(adr) \
    ((*RdIt)[(adr)>>12]((adr)))
#else
    /* A good c-compiler should produce this */
#define RB_MEM(adr) \
    ( (*(regparm1**) ((unsigned)(*RdIt) + (((unsigned)(adr)>>10)&~3)) ) ((adr)))
#endif

/*
**	Write a byte to memory with current configuration.
*/
#if 0
#define WB_MEM(adr,val) \
    ((*WrIt)[(adr)>>12]((adr),(val)))
#else
    /* A good c-compiler should produce this */
#define WB_MEM(adr,val) \
    (( *(regparm2**) ((unsigned)(*WrIt)	\
	    + (((unsigned)(adr)>>10)&~3)) ) ((adr),(val)))
#endif

#undef RBMem
#ifdef __GNUC__
#define RBMem(adr) \
    ({	int __rx=(adr); RB_MEM(__rx); })
#else
#define RBMem(adr) \
    ( __tmpvar__=(int)(adr), RB_MEM(__tmpvar__) )
#endif

#undef WBMem
#define WBMem(adr,val) \
    do{ int __wx=(adr); WB_MEM(__wx,(val)); }while(0)

#endif

/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/

/*-----------------------------
**	Zero/Direct page access
*/

/*
**	Read byte from zero page.
*/
#define RBDP(adr) \
    Ram[adr]

/*
**	Write byte to zero page. Special Processor port data+direction
*/
#define WBDP(adr,val) \
    do {						\
	int __x=(adr);					\
	Ram[__x]=(val);					\
	if( __x<=1 ) {					\
	    if( __x==1 )				\
		Ram[1]|=0x10;				\
	    ChangeConfig((Ram[1]|(~Ram[0]))&7);		\
	}						\
    } while( 0 )

/*
**	Read word from zero page for indirect-x and indirect-y.
*/
#ifdef __GNUC__
#define RWDP(adr) \
    ({ int __x=(adr); Ram[__x]|(Ram[(__x+1)&0xFF]<<8); })
#else
static int __tmpvar__;
#define RWDP(adr) \
    (__tmpvar__=(int)(adr), Ram[__tmpvar__]|(Ram[(__tmpvar__+1)&0xFF]<<8))
#endif

/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/

/*
**	Video card include file.
*/
#ifdef X11
#   define VIDEO_H	"video_x11.h"
#endif
#ifdef SVGALIB
#   define VIDEO_H	"video_svga.h"
#endif
#ifdef GO32
#   define VIDEO_H	"video_go32.h"
#endif

/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/

/*
**	Different timing constants for PAL or NTSC version.
*/

#if !defined(PAL) && !defined(NTSC)
#error "PAL or NTSC must be defined"
#endif

#ifdef PAL
#define CPU_CLK			985248 	/* PAL=  17734472/18 =  985248 Hz */
#define FRAMES_PER_SECOND	50	/* Frames per second or TOD clock */
#define LINES_PER_FRAME		312	/* scanlines per screen frame */
#define CYCLES_PER_LINE		63	/* cpu cycles per scanline */
#endif

#ifdef NTSC
#define CPU_CLK			1022727	/* NTSC= 14318181/14 = 1022727 Hz */
#define FRAMES_PER_SECOND	60	/* Frames per second or TOD clock */
#define LINES_PER_FRAME		262	/* scanlines per screen frame */
#define CYCLES_PER_LINE		65	/* cpu cycles per scanline */
#endif

#endif	/* } !__C64_H__ */
